/*
 * PPMEffecTV - Realtime Video Effector
 * Copyright (C) 2001-2002 FUKUCHI Kentaro
 * Adapted 2003 by Charles Yates
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdint.h>
#include <ctype.h>

#include "EffecTV.h"
#include "effects.h"
#include "utils.h"
#include "palette.h"

int debug = 0; /* 0 = off, 1 = less debug messages, 2 = more debug messages. */
int autoplay = 0;
int autoplay_counter;

static effectRegistFunc *effects_register_list[] =
{
	dumbRegister,
	quarkRegister,
	fireRegister,
	burnRegister,
	blurzoomRegister,
	streakRegister,
	baltanRegister,
	onedRegister,
	dotRegister,
	mosaicRegister,
	puzzleRegister,
	predatorRegister,
	spiralRegister,
	simuraRegister,
	edgeRegister,
	shagadelicRegister,
	noiseRegister,
	agingRegister,
	TransFormRegister,
	lifeRegister,
	sparkRegister,
	warpRegister,
	holoRegister,
	cycleRegister,
	rippleRegister,
	diceRegister,
	dizzyRegister,
	DeinterlaceRegister,
	nervousRegister,
	rndmRegister,
	revRegister,
	rdsRegister,
	lensRegister,
	diffRegister,
	scrollRegister,
	warholRegister
};

static effect **effectsList;
static int effectMax;
static int currentEffectNum;
static effect *currentEffect;
static int frame_count = 0;

static void usage()
{
	printf("ppmeffectv - Batch Video Effector\n");
	printf("Version: %s\n", VERSION_STRING);
	printf("Usage: ppmeffectv [options...]\n");
	printf("Options:\n");
	printf("  --list     list effects available\n" );
	printf("  -a NUMBER  changes effects automatically every NUMBER frames\n");
	printf("  -e number  start with effect number\n" );
	exit( 0 );
}

static int registEffects( int output )
{
	int i, n;
	effect *entry;

	n = sizeof(effects_register_list)/sizeof(effectRegistFunc *);
	effectsList = (effect **)malloc(n*sizeof(effect *));
	effectMax = 0;
	for(i=0;i<n;i++) {
		entry = (*effects_register_list[i])();
		if(entry) {
			if ( output )
				printf( "%2d: %.40s\n",i,entry->name);
			effectsList[effectMax] = entry;
			effectMax++;
		}
	}
	if ( output )
		printf( "%d effects are available.\n",effectMax);
	return effectMax;
}

static int changeEffect(int num)
{
/* return value:
 *  0: fatal error
 *  1: success
 *  2: not available
 */
	if(currentEffect)
		currentEffect->stop();
	currentEffectNum = num;
	if(currentEffectNum < 0)
		currentEffectNum += effectMax;
	if(currentEffectNum >= effectMax)
		currentEffectNum -= effectMax;
	currentEffect = effectsList[currentEffectNum];
	if(currentEffect->start() < 0)
		return 2;

	return 1;
}

int ReadData( uint8_t *data, int length )
{
	return fread( data, 1, length, stdin );
}

int ReadNumber( )
{
	int eof = 0;
	int value = 0;
	uint8_t c = '\0';

	do 
	{
		eof = ReadData( &c, 1 ) == 0;

		while( !eof && !isdigit( c ) && c != '#' )
			eof = ReadData( &c, 1 ) == 0;

		if ( c == '#' )
			while( !eof && c != '\n' )
				eof = ReadData( &c, 1 ) == 0;
	}
	while ( !isdigit( c ) && !eof );

	while( isdigit( c ) && !eof )
	{
		value = value * 10 + ( c - '0' );
		eof = ReadData( &c, 1 ) == 0;
	}

	return value;
}

int ReadHeader( int *width, int *height )
{
	uint8_t type[ 2 ];
	int maxval;
	
	if ( ReadData( type, 2 ) == 2 && !strncmp( ( char * )type, "P6", 2 ) )
	{
		*width = ReadNumber( );
		*height = ReadNumber( );
		maxval = ReadNumber( );
	}

	return !strncmp( ( char * )type, "P6", 2 ) && *width != 0 && *height != 0;
}

int ReadImage( uint8_t **image, int *width, int *height )
{
	int ret = 1;
	int new_width = 0;
	int new_height = 0;
	if ( ReadHeader( &new_width, &new_height ) )
	{
		int image_size = new_width * new_height * 3;

		if ( *image == NULL || new_width != *width || new_height != *height )
		{
			*width = new_width;
			*height = new_height;
			*image = realloc( *image, new_width * new_height * 3 );
		}
		if ( *image != NULL )
		{
			ret = ReadData( *image, image_size ) != image_size;
		}
	}
	return ret;
}

int stretch = 0;
int video_area = 0;
int screen_width = 0;
int screen_height = 0;
int video_width = 0;
int video_height = 0;
int doublebuf = 0;
int screen_scale = 1;

static uint8_t *ppm_frame = NULL;
static uint8_t *video_frame = NULL;
static uint8_t *screen_frame = NULL;

int screen_mustlock( )
{
	return 0;
}

int screen_lock( )
{
	return 0;
}

int screen_unlock( )
{
	return 0;
}

void screen_setcaption(const char *str)
{
}

void screen_clear( int color )
{
	int i;
	int *p = (int *)screen_frame;
	for ( i = 0; i < video_area; i ++ )
		*p ++ = color;
}

static void convert_RGB24toRGB32(unsigned char *src, RGB32 *dest, int width, int height)
{ 
    int i, length;
    
    length = width * height;
    for(i=0; i<length; i++) {
        *dest++ = *(unsigned int *)src & 0xffffff;
        src += 3;
    }
}

static int in = 0;
static int out = 0;

int screen_update( )
{
	RGB32 *p = (RGB32 *)screen_frame;
	int stride = screen_width * 3;
	uint8_t *row = malloc( stride );
	int i, j;
	printf( "P6\n%d %d\n255\n", screen_width, screen_height );
	for ( i = 0; i < screen_height; i ++ )
	{
		for ( j = 0; j < stride; )
		{
			row[ j ++ ] = ( (uint8_t *)p )[ 0 ];
			row[ j ++ ] = ( (uint8_t *)p )[ 1 ];
			row[ j ++ ] = ( (uint8_t *)p )[ 2 ];
			p ++;
		}
		fwrite( row, stride, 1, stdout );
	}
	out ++;
	free( row );
	return fflush( stdout );
}

unsigned char *screen_getaddress( )
{
	return screen_frame;
}

int video_grabstart( )
{
	return 0;
}

int video_grabframe( )
{
	if ( frame_count == 0 && video_frame != NULL )
	{
		frame_count ++;
		return 0;
	}
	else if ( ReadImage( &ppm_frame, &video_width, &video_height ) == 0 )
	{
		if ( screen_width != video_width || screen_height != video_height )
		{
			screen_width = video_width;
			screen_height = video_height;
			video_area = screen_width * screen_height;
			video_frame = realloc( video_frame, screen_width * screen_height * sizeof( RGB32 ) );
			screen_frame = realloc( screen_frame, screen_width * screen_height * sizeof( RGB32 ) );
		}
		convert_RGB24toRGB32( ppm_frame, (RGB32 *)video_frame, screen_width, screen_height );
		in ++;
		return 0;
	}
	else
	{
		return -1;
	}
}

int video_grabstop( )
{
	return 0;
}

int video_syncframe( )
{
	return 0;
}

int video_grab_check( int palette )
{
	return 0;
}

unsigned char *video_getaddress( )
{
	return video_frame;
}

static int startTV()
{
	int ret;
	int flag;
	int frames=0;

	currentEffect = NULL;
	flag = changeEffect(currentEffectNum);
	if(autoplay) {
		autoplay_counter = autoplay;
	}

	while(flag) {
		if(flag == 1) {
			ret = currentEffect->draw();
			if(ret < 0) {
				flag = 2;
			} else if(ret == 0) {
				screen_update();
			}
		}
		if (flag == 2) {
			if(currentEffect)
				currentEffect->stop();
			break;
		}
		if(autoplay) {
			autoplay_counter--;
			if(autoplay_counter == 0) {
				autoplay_counter = autoplay;
				flag = changeEffect(currentEffectNum+1);
			}
		}
		frames ++;
	}
	currentEffect->stop();
	return 0;
}

int main(int argc, char **argv)
{
	int i = 0;

	srand(time(NULL));
	fastsrand(time(NULL));

	for ( i = 1; i < argc; i ++)
	{
		if ( !strcmp( argv[ i ], "--help" ) )
			usage( );
		else if ( !strcmp( argv[ i ], "--list" ) )
		{
			screen_width = video_width = 720;
			screen_height = video_height = 576;
			video_area = 720 * 576;
			sharedbuffer_init();
			utils_init();
			registEffects( 1 );
			exit( 0 );
		}
		else if ( !strcmp( argv[ i ], "-e" ) )
			currentEffectNum = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-a" ) )
			autoplay = atoi( argv[ ++ i ] );
	}

	if ( video_grabframe( ) )
	{
		fprintf( stderr, "Unable to obtain a frame\n" );
		return 1;
	}

	if(sharedbuffer_init()){
		fprintf(stderr, "Memory allocation failed.\n");
		exit(1);
	}
	if(utils_init()) {
		fprintf(stderr, "Utility function initialization failed.\n");
		exit(1);
	}
	if(registEffects( 0 ) == 0) {
		fprintf(stderr, "No available effect.\n");
		exit(1);
	}

	startTV();

	return 0;
}
