/* sigit.c  -- The thing to get it roling here. 

   Copyright (C) Kenneth 'Redhead' Nielsen <kn@redhead.dk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


/****
 * $Id: sigit.c,v 1.3 2003/07/22 22:03:42 redhead Exp $ 
 * $Log: sigit.c,v $
 * Revision 1.3  2003/07/22 22:03:42  redhead
 * Cleaned up the code, which included rewriting every structure in the program.
 * Fixed some error checking, to make sure theres no undocumentet parts that might
 * produce an error.
 *
 * Revision 1.2  2002/10/18 18:25:23  redhead
 * Added the use of boxes to create a surrounding box around the signature
 * This introduced the --with-boxes flag to configure.
 * If that flag isn't used it will default be assumed boxes isn't installed
 * Else the configure script will search for boxes, if none should be found
 * in the search path, you can use --with-boxes=/where to specify it.
 *
 ****/

#include "sigit.h"

  
#ifdef sigit


/* 
 * Well what do you know, its time to fireup the baby... 
 */

int main(int argc, char* argv[])
{
  char buffer[LINE_SIZE +1], file_name[LINE_SIZE +1], opt;
  int fifo_count = 0, counter = 0, verbose = 0, deamon = 0;
  FIFO* cur;
  FIFO* next;
  FILE* file;
  pthread_t* threads;
  /*  pthread_attr_t* attrs; */
  

  /* this is essential */
  file_name[0] = '\0';

  /* there allways gotta be room for one fifo */

  cur = (FIFO*) USE__MALLOC(sizeof(FIFO));
  cur->next = NULL;
  cur->prev = NULL;

  if(argc > 1)
    { /* if we were run with an argument, lets findout what that was.. */
      while( (opt = getopt(argc, argv, "f:cdehH:vV")) > 0)
        {
          switch ((char)opt)
            {
            case 'V':
	      print_copyright();
	      return SUCCESS;
	    case 'h':
              print_help("");
              return SUCCESS;
	    case 'H':
              print_help(optarg);
              return SUCCESS;
            case 'c':
              if(deamon == 1)
                {
		  printf("[main]\t\t\t: Error: Can't be run as a deamon and a cron job at the same time.\n");
		  return FAILURE;
		}
	      deamon = 2;
              break;
            case 'd':
              if(deamon == 2)
                {
		  printf("[main]\t\t\t: Error: Can't be run as a deamon and a cronjob at the same time.\n");
		  return FAILURE;
		}
	      deamon = 1;
              break;
            case 'f':
#ifdef I__HAVE__STRNCPY
	      strncpy(file_name, optarg, LINE_SIZE);
#else
	      strcpy(file_name, optarg);
#endif
              break;
            case 'v':
	      verbose++;
              break;
            default:
              print_help("");
              return FAILURE;
            }
        }
    }
  if(verbose > 1)
    {
      printf("[main]\t\t\t: Starting extra verbose run of\n");
      printf("[main]\t\t\t: " PACKAGE "-" VERSION " (" CALLSIGN ") <" HOMEPAGE ">\n");
      printf("[main]\t\t\t: Build [" __DATE__ " " __TIME__ "]\n");
#ifdef BOXING_SUPPORT
      printf("[main]\t\t\t: With " BOX_VERSION " Support buildin.\n");
      printf("[main]\t\t\t:      (c) Thomas Jensen <tsjensen@stud.informatik.uni-erlangen.de>\n");
      printf("[main]\t\t\t:      Web page: http://boxes.thomasjensen.com/\n");
#endif
    }
  file = open_config_file(file_name, verbose);
  if(!file)
    return FAILURE;
  for(; !feof(file); )
    {
      fgets(buffer, LINE_SIZE, file); 
      if(!strncmp(buffer, "fifo{", 5))
	{
	  cur->verbose = verbose;
	  if(!read_config_fifo(cur, file))
	    {
	      /* it all went well, lets do some mengeling.. */
	      fifo_count++;
	      /* delete what ever previus debug file there was. */
	      unlink(cur->debug_file);
	      /* better make room for another one. */
	      next = (FIFO*) USE__MALLOC(sizeof(FIFO));
	      cur->next = next;
	      next->prev = cur;
	      next->next = NULL;
	      cur = cur->next;
	    } 
	  else
	    {
	      /* something went wrong */
	      if(verbose > 0)
		printf("[main]\t\t\t: Error: An unexpected exit from read_config_fifo().\n");
	      return FAILURE;
	    }
	} /* if("fifo{") */
	  /* it's just a non interresting line, could be a comment */
    } /* for() */
  
  install_sig_handler(verbose);
  
  if(deamon == 1)
    deamon_init(argv[0], 0);
  if(create_pid_file() != SUCCESS && deamon != 1)
    {
      printf("[main]\t\t\t: Error: Failed to kill the allready running Sigit, either the program\n");
      printf("[main]\t\t\t: Error: died unexpected, or the execution of rsh(1) failed.\n");
    }
  /* should never end up here, since we exit with a FAILURE in this case,
   * but for "magic switch" effect we keep the code.
   */
  if(deamon == 2)
    { /* we must be in a cronjob case */
      for(; cur->prev != NULL;)
	{
	  /* no init, its a cronjob just remove old fifos */
	  unlink(cur->prev->fifo_name);
	  if(SUCCESS == get_sig(cur->prev))
	    pretty_print(cur->prev);
	  else
	    {
#ifdef I__HAVE__STRNCPY
	      strncpy(cur->prev->sig.sig, DEFAULT_SIG, NUM_LINES*LINE_SIZE);
	      strncpy(cur->prev->sig.ident, "DEFAULT", HEXCOUNT);
#else
	      strcpy(cur->prev->sig.sig, DEFAULT_SIG);
	      strcpy(cur->prev->sig.ident, "DEFAULT");
#endif
  	      cur->prev->sig.count = htonl(0);
	      pretty_print(cur->prev);
	    }
	  sleep(1); /* better make sure the random can come up with a new sig for next time */
	  cur = cur->prev;
	  /* no unlinking, since its cronjob, the file must remain */
	}
      remove_pid_file();
    }
  else
    {
      /* this is the part which in the end will remain as the favorit, no if required */
      
      threads = (pthread_t*) USE__MALLOC(fifo_count * sizeof(pthread_t));
      /*  attrs = (pthread_attr_t*) USE__MALLOC(fifo_count * sizeof(pthread_attr_t));*/
      
      for(; cur->prev->prev != NULL; counter++)
	{
	  cur = cur->prev; 
	  if(verbose > 0 && deamon != 1)
	    printf("[main]\t\t\t: Creating thread {%d} in a total of {%d} threads.\n", counter+1, fifo_count+1);
	  
	  pthread_mutex_init(&mutex, NULL);
	  /*
	   * pthread_attr_init(&attrs[counter]);
	   * pthread_attr_setdetachstate(&attrs[counter], PTHREAD_CREATE_DETACHED);
	   * pthread_attr_setscope(&attrs[counter], PTHREAD_SCOPE_SYSTEM);
	   */
	  if(cur->nfs_delay > 0)
	    pthread_create(&threads[counter], NULL/*&attrs[counter]*/, monitor_nfs_fifo, (void *) cur);
	  else
	    pthread_create(&threads[counter], NULL/*&attrs[counter]*/, monitor_fifo, (void *) cur);
	  /* there should be some form of sleep() here, else the signature 
	   * for all threads will be the same number in each. 
	   * The random generator can't change within the split second 
	   * it takes to spawn a thread 
	   */
	}
      if(verbose > 0 && deamon != 1)
	{
	  if(fifo_count < 2)
	    printf("[main]\t\t\t: Creating thread {%d} in a total of {%d} threads.\n", counter+1, fifo_count);
	  else
	    printf("[main]\t\t\t: Creating thread {%d} in a total of {%d} threads.\n", counter+1, fifo_count+1);
	  if(fifo_count > 1)
	    printf("[main]\t\t\t: Creating thread {%d} in a total of {%d} threads.\n", counter+2, fifo_count+1);
	}
	  if(cur->prev->nfs_delay > 0)
	    monitor_nfs_fifo(cur->prev);
	  else
	    monitor_fifo(cur->prev);
    }
  return SUCCESS;
}

/*
 * A help function to print any info, on how to use the program.
 */

int print_help(char* option)
{
  if('\0' == option[0])
    {
      printf("\n");
      printf("The normal use of this program would be:\n");
      printf(PACKAGE " [-f file] [-cdhvV] [-H option]\n");
      printf("\n"),
      printf("\t-c\n");
      printf("\t   This will execute " PACKAGE " as beeing run from a,\n");
      printf("\t   cronjob entry, it will change the signature once,\n");
      printf("\t   and then exit.\n");
      printf("\t-d\n");
      printf("\t   This will launch " PACKAGE " into the background,\n");
      printf("\t   and thereby create a deamon, which will run even\n");
      printf("\t   after you are loged out.\n");
      printf("\t-f  <file>\n");
      printf("\t   This will invoke the program, with the <file>\n");
      printf("\t   beeing treated as the orriginal configuration file.\n");
      printf("\t-h\n");
      printf("\t   Print this help screen and exit.\n");
      printf("\t-H  <option>\n");
      printf("\t   Print explanatory help, if no option given,\n");
      printf("\t   it will print this help instruction and exit.\n");
      printf("\t   If an option is given, it will print further help\n");
      printf("\t   regarding that option.\n");
      printf("\t-v\n");
      printf("\t   This is used, to produce verbose output.\n");
      printf("\t-V\n");
      printf("\t   This will print info about the program version.\n\n");
   }
  else
    {
      if(!strcmp(option, "c")||!strcmp(option, "-c"))
	{
	  printf("\n\tsigit -c\n\n");
	  printf("\t   If you have a setup, where it's realy rare the signature has to be changed,\n");
	  printf("\tThen it might be a good idear to run Sigit as a cron job, this meaning adding\n");
	  printf("\tSigit to you cron list. if you dont know how to use cron, then consult the\n");
	  printf("\t`man crontab`, or `info crontab` pages, they will have sufficient info regarding\n");
	  printf("\twhat you'll need.\n");
	  printf("\t   By using the '-c' flag, Sigit will execute once, filling the fifo given files\n");
	  printf("\tas if they were simple text files, and exit. This is also a usefull flag, if you\n");
	  printf("\twant to see, wether the program actualy reads the correct assigned data file.\n");
	  printf("\tSince it will exit without having initiated the file as a fifo, it is possible to\n");
	  printf("\ttake a closer look at what actualy gets written. Thus you can examin the output\n");
	  printf("\tmore thoroughly, in order to see exactly what line couldn't be handled correct.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "d")||!strcmp(option, "-d"))
	{
	  printf("\n\tsigit -d\n\n");
	  printf("\t   If you like to remotely login onto an account where you check your mail.\n");
	  printf("\tIt might be convenient to have sigit running in the background without you\n");
	  printf("\thaving to be concerned with executing it once you've loged in.\n");
	  printf("\t   With the '-d' flag, sigit will close any file discriptor (stdin/stdout/etc)\n");
	  printf("\tpointing to it from the system, and fork itself into the background, thereby\n");
	  printf("\tbecomming a shell less program, which means even you logging out, won't kill it.\n");
	  printf("\t   So next time you login, sigit is allready running and just waiting for the fifo\n");
	  printf("\tto be touched.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "f")||!strcmp(option, "-f"))
	{
	  printf("\n\tsigit -f config-file\n\n");
	  printf("\t   If you have a quick test configuration file lying around, and you don't want\n");
	  printf("\tto rename/overwrite the allready good and existing one, then invoke Sigit with\n");
	  printf("\tthe '-f' flag, to parse the file as the configuration file for the current run.\n");
	  printf("\t   Sigit will _not_ try and see, if there should be a %s file or ~/.sigitrc\n", DEFAULT_CONF_FILE);
	  printf("\tit will assume the _only_ configuration file in existance on the system is your\n");
	  printf("\tmanualy told one. So if the file dosn't exist there will _not_ be made any attempt\n");
	  printf("\tto read any of the usual standard configuration files.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "h")||!strcmp(option, "-h"))
	{
	  printf("\n\tsigit -h\n\n");
	  printf("\t   Get the usual help informations, if you should be in need for a quick guide\n");
	  printf("\ton all the flags this program handles. Then this is the flag to be using.\n");
	  printf("\t  (Since you're seeing this, you might allready have found the use of '-h')\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "H")||!strcmp(option, "-H"))
	{
	  printf("\n\tsigit -H flag\n\n");
	  printf("\t   You're allready there... Talk about recursive calls..\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "v")||!strcmp(option, "-v"))
	{
	  printf("\n\tsigit -v\n\n");
	  printf("\t   It's for verbose output, as it usualy is, do I realy need to say more ?\n");
	  printf("\tWell I guess I could give a description of the usage anyway..\n");
	  printf("\t   When using the '-v' flag, Sigit will print the debugging information requested\n");
	  printf("\tin the giving verbose level, the level is controlled with the number of '-v's\n");
	  printf("\tThus if you were to use 'sigit -v' it would produce less debug info, than 'sigit -vv'\n");
	  printf("\t    And should you be a total morron, like I tend to be, when debugging stuff,\n");
	  printf("\t'sigit -vvvvv' migth give you just the info you were looking for.\n");
	  printf("\tIts a nice way, to see, if the buffers and current state the program is in, is\n");
	  printf("\tsuffering from an overflow, or miss configuration, or in some cases a small \"feature\".\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "V")||!strcmp(option, "-V"))
	{
	  printf("\n\tsigit -V\n\n");
	  printf("\t   Look ma' its the version info... Weee, I didn't know I was running this version.\n\n");
	  return SUCCESS;
	}
      printf("There is no option what so ever in this program, that ressembles\n");
      printf("the given {%s} option.\n", option);
      return FAILURE;
    }
  return SUCCESS;
}


#endif


#ifdef sigitdb

/* 
 * Should it be the sigitdb program, this main function is beeing used.
 */

int main(int argc, char* argv[])
{
  int index = 0, verbose = 0, add = 0, del = 0, edit = 0, rebuild = 0, opt;
  static struct option opts[] = {
    {"add", 1, 0, 0},
    {"delete", 1, 0, 0},
    {"edit", 1, 0, 0},
    {"rebuild", 1, 0, 0},
    {"append", 1, 0, 0},
    {"list", 1, 0, 0},
    {0, 0, 0, 0}
  };  

  char reason[12];
  FIFO* fifo;
  fifo = (FIFO*) USE__MALLOC(sizeof(FIFO));

  fifo->fifo_name[0] = '\0';
  fifo->data_name[0] = '\0';
  fifo->delete[0] = '\0';

  if(argc > 1)
    { /* if we were run with an argument, lets find out what that was.. */
      while((index = 6) && (opt = getopt_long(argc, argv, "vVhH:", opts, &index)) >= 0)
        { 
	  switch (opt)
	    {
	    case 0:
	      if(!strcmp("add", opts[index].name))
		{
		  /* Just see if the file is there, should be a text file */
		  if(del || edit || rebuild)
		    {
		      printf("[main]\t\t\t: Error: The {%s} option can only be accompanied by {-v}\n", opts[index].name);
		      return FAILURE;
		    }
		  if(!access(optarg, F_OK))
		    {
#ifdef I__HAVE__STRNCPY
		      strncpy(fifo->fifo_name, optarg, LINE_SIZE);
#else
		      strcpy(fifo->fifo_name, optarg);
#endif
		      if(argv[optind] != '\0' && !error_db_file(argv[optind]))
			{
#ifdef I__HAVE__STRNCPY  
			  strncpy(fifo->data_name, argv[optind], LINE_SIZE);
#else
			  strcpy(fifo->data_name, argv[optind]);
#endif
			  add = 1;
			  optind++;
			}
		      else
			{
			  printf("[main]\t\t\t: Error: Theres something wrong with the given db-file, use {-v} for further info.\n");
			  return FAILURE;
			}
		    }
		  else
		    {
		      /* should we endup here, theres no need to check any more, it gotta be an error */
		      printf("[main]\t\t\t: Error: The given file [%s] to fetch signaures from dosn't exist.\n", argv[optind]);
		      return FAILURE;
		    }
		  break;
		}
	      if(!strcmp("append", opts[index].name))
		{
		  /* Just see if the file is there, shoudl be a db-file file */
		  if(del || edit || rebuild)
		    {
		      printf("[main]\t\t\t: Error: The {%s} option can only be accompanied by {-v}\n",  opts[index].name);
		      return FAILURE;
		    }
		  if(!error_db_file(optarg))
		    {
#ifdef I__HAVE__STRNCPY
		      strncpy(fifo->data_name, optarg, LINE_SIZE);
#else
		      strcpy(fifo->data_name, optarg);
#endif
		      add = 1;
		      fifo->delete[0] = 'S';
		      fifo->delete[1] = '\0';
		    }
		  else
		    {
		      /* should we endup here, theres no need to check any more, it gotta be an error */
		      
		      printf("[main]\t\t\t: Error: The given db-file {%s} isn't a valid db-file.\n", optarg);
		      return FAILURE;
		    }
		  break;
		}
	      if(!strcmp("delete",  opts[index].name))
		{
		  if(add || edit || rebuild)
		    {
		      printf("[main]\t\t\t: Error: The {%s} option can only be accompanied by {-v}\n", opts[index].name);
		      return FAILURE;
		    }
		  if(optarg != '\0')
		    strncpy(fifo->delete, optarg, HEXCOUNT);
		  if(argv[optind] != '\0' && !error_db_file(argv[optind]))
		    {
#ifdef I__HAVE__STRNCPY
		      strncpy(fifo->data_name, argv[optind], LINE_SIZE);
#else
		      strcpy(fifo->data_name, argv[optind]);
#endif
#ifdef I__HAVE__SNPRINTF
		      snprintf(fifo->fifo_name, LINE_SIZE, "/tmp/.sigitdb-d.%d", getpid());
#else
		      sprintf(fifo->fifo_name, "/tmp/.sigitdb-d.%d", getpid());
#endif
		      del = 1;
		      optind++;
		    }
		  else
		    {
		      /* if we end here, it must be a mistake, no need to check for it.. */
		      printf("[main]\t\t\t: Error: The required signature to delete, can not be decided from the given number.\n");
		      return FAILURE;
		    }
		  break;
		}
	      
	      if(!strcmp("edit",  opts[index].name))
		{
		  if(!add && !del && !rebuild && optarg != '\0' && !error_db_file(optarg))
		    {
#ifdef I__HAVE__STRNCPY
		      strncpy(fifo->data_name, optarg, LINE_SIZE);
#else
		      strcpy(fifo->data_name, optarg);
#endif
		      
#ifdef I__HAVE__SNPRINTF
		      snprintf(fifo->fifo_name, LINE_SIZE, "/tmp/.sigitdb-e.%d", getpid());
#else
		      sprintf(fifo->fifo_name, "/tmp/.sigitdb-e.%d", getpid());
#endif
		      edit = 1;
		      break;
		    }
		  else
		    return FAILURE;
		  break;
		}
	      if(!strcmp("list",  opts[index].name))
		{
		  /* it's just a --list */
		  if(!add && !del && !edit && !rebuild && optarg != '\0' && !error_db_file(optarg))
		    {
#ifdef I__HAVE__STRNCPY
		      strncpy(fifo->data_name, optarg, LINE_SIZE);
#else
		      strcpy(fifo->data_name, optarg);
#endif
		      fifo->delete[0] = 'L';
		      fifo->delete[1] = '\0';
		      edit = 2;
		    }
		  else
		    return FAILURE;
		  break;
		}
	      if(!strcmp("rebuild",  opts[index].name))
		{
		  if(add || del || edit)
		    {
		      printf("[main]\t\t\t: Error: The {%s} option can only be accompanied by {-v}\n", opts[index].name);
		      return FAILURE;
		    }
		  /* Just see if the file is there, should be a text file */
		  if(optarg != '\0')
		    {
		      if(0 != error_db_file(optarg))
			{
			  if(verbose > 0)
			    printf("[main]\t\t\t: Assuming argument {%s} is a plain text file.\n", optarg);
#ifdef I__HAVE__STRNCPY
			  strncpy(fifo->fifo_name, optarg, LINE_SIZE);
#else
			  strcpy(fifo->fifo_name, optarg);
#endif
			  if(argv[optind] != '\0')
			    {
#ifdef I__HAVE__STRNCPY
			      strncpy(fifo->data_name, argv[optind], LINE_SIZE);
#else
			      strcpy(fifo->data_name, argv[optind]);
#endif
			      rebuild = 1;
			      optind++;
			    }
			  else
			    {
			      printf("[main]\t\t\t: Error: There wasn't given any db-file.\n");
			      return FAILURE;
			    }
			}
		      else
			{
			  /* could be a db-file as first argument */
			  if(!error_db_file(optarg))
			    {
			      /* Just a db file, lets try a few things to recover */
			      strcpy(fifo->delete, "DEFAULT");
#ifdef I__HAVE__SNPRINTF
			      snprintf(fifo->fifo_name, LINE_SIZE, "/tmp/.sigitdb-r.%d", getpid());
#else
			      sprintf(fifo->fifo_name, "/tmp/.sigitdb-r.%d", getpid());
#endif	  
#ifdef I__HAVE__STRNCPY
			      strncpy(fifo->data_name, optarg, LINE_SIZE); 
#else
			      strcpy(fifo->data_name, optarg); 
#endif
			      strcpy(fifo->delete, "DEFAULT");
			      rebuild = 2;
			      break;
			    }
			  else
			    {
			      printf("[main]\t\t\t: Error: Couldn't decide, if the first given file is a db-file or a text file.\n");
			      printf("[main]\t\t\t: Error: My guess is it dosn't even exist.\n");
			      return FAILURE;
			    } 
			}
		    }
		  else
		    {
		      printf("[main]\t\t\t: Error: Couldn't decide, if the first given file is a db-file or a text file.\n");
		      printf("[main]\t\t\t: Error: My guess is it dosn't even exist.\n");
		      return FAILURE;
		    }
		  break;
		}
	      printf ("%s: invalid option -- %s\n", argv[0], optarg);
	      print_help("");		  
	      return FAILURE;
	      
	    case 'V':
	      print_copyright();
	      return SUCCESS;
	    case 'h':
	      print_help("");
	      return SUCCESS;
	    case 'H':
	      print_help(optarg);
	      return SUCCESS;
	    case 'v':
	      verbose++;
	      break;
	    default:
	      print_help(""); 
	      return FAILURE;
	    }
	}

      if(verbose > 1)
	{
	  printf("[main]\t\t\t: Starting extra verbose run of\n");
	  printf("[main]\t\t\t: " PACKAGE "-" VERSION " (" CALLSIGN ") <" HOMEPAGE ">\n");
	  printf("[main]\t\t\t: Build [" __DATE__ " " __TIME__ "]\n");
	}
      fifo->verbose = verbose;
      if(add)
	{
	  if(add_sig(fifo))
	    {
	    printf("[main]\t\t\t: Error: Terminating due to errors while adding signature\n");
	    return FAILURE;
	    }
	}
      if(edit || del || rebuild == 2)
	{
	  if(edit == 1)
	    strcpy(reason, "editing");
	  else 
	    if (del)
	      strcpy(reason, "deleting");
	    else
	      if(edit == 2)
		{
		  fifo->verbose = 0;
		  strcpy(reason, "listing");
		}
	      else
		strcpy(reason, "rebuilding");
	  if(edit_db_file(fifo))
	    {
	      printf("[main]\t\t\t: Error: Terminating due to errors while %s signature\n", reason);
	      printf("[main]\t\t\t: Orriginal data file {%s} kept for future investigation\n", fifo->data_name);
	      return FAILURE;
	    }
	  if(edit == 2)
	    {
	      return SUCCESS;
	    }
	  unlink(fifo->data_name);
	  if(add_sig(fifo))
	    {
	      printf("[main]\t\t\t: Error: Terminating due to errors while %s signature\n", reason);
	      printf("[main]\t\t\t: A plain text version of the db-file might be found in {%s}\n", fifo->fifo_name);
	      return FAILURE;
	    }
	  unlink(fifo->fifo_name);
	}
      if(rebuild == 1)
	{
	  unlink(fifo->data_name);
	  if(add_sig(fifo))
	    {
	      printf("[main]\t\t\t: Error: Terminating due to errors while rebuilding signature\n");
	      printf("[main]\t\t\t: A plain text version of the db-file might be found in {%s}\n", fifo->fifo_name);
	      return FAILURE;
	    }
	  if(NULL != strstr(fifo->fifo_name, "/tmp/.sigitdb-r"))
	    unlink(fifo->fifo_name);
	}
      if(!add && !edit && !del && !rebuild)
	printf("[main]\t\t\t: Error: The given arguments, couldn't be used to decide what to do.\n");
      return SUCCESS;
    }
  else
    print_help("");
  return SUCCESS;
}


/*
 * A function to check, if the given file is valid as a db-file, it will 
 * see if the file exists, if it can read a counter structure out of it
 * and see if the integrity in the counter seems to fit the needed for 
 * it to be a valid db-file
 */

int error_db_file(char* file_name)
{
  COUNTER count;
  int mini = -1, minor = -1, maxi = -1;
  gzFile file;
  if(access(file_name, F_OK))
    {
      printf("[error_db_file]\t\t: Error: The db-file {%s} dosn't exist.\n", file_name);
      return CONF_MI;
    }
  if(!(file = gzopen(file_name, "rb")))
    {
      printf("[error_db_file]\t\t: Error: The db-file {%s} can't be opened for reading.\n", file_name);
      return NO_READ;
    }     
    if(gzseek(file, 0, SEEK_SET) != 0)
    {
      printf("[error_db_file]\t\t: Error: Cant seek in file.\n");
      gzclose(file);
      return SEEK_ERR;
    }
    /* hopefully even an old_counter stored in this file, will have the ver and counter number 
       placed the same way as the new counter does. */

    if(0 >= gzread(file, (void*) &count, sizeof(COUNTER)))
      {
	printf("[error_db_file]\t\t: Error: Can't read from db-file {%s}\n", file_name);
	gzclose(file);
	return NO_READ;
      }
    count.count = (int) ntohl(count.count);
    if(count.count < 0)
      {
	printf("[error_db_file]\t\t: Error: The db-file {%s} has a wrong counter.\n", file_name);
	gzclose(file);
	return COUNT_MISS;
      }
    if(EOF == sscanf(count.ver, "%d.%d.%d", &mini, &minor, &maxi))
      {
	printf("[error_db_file]\t\t: Error: The db-file {%s} has illformed version numbering.\n", file_name);
	gzclose(file);
	return COUNT_MISS;	
      }
    if(mini < 0)
      {
	printf("[error_db_file]\t\t: Error: The db-file {%s} has illformed version numbering.\n", file_name);
	gzclose(file);
	return COUNT_MISS;	
      }
    if(minor < 0)
      {
	printf("[error_db_file]\t\t: Error: The db-file {%s} has illformed version numbering.\n", file_name);
	gzclose(file);
	return COUNT_MISS;	
      }
    if(maxi < 0)
      {
	printf("[error_db_file]\t\t: Error: The db-file {%s} has illformed version numbering.\n", file_name);
	gzclose(file);
	return COUNT_MISS;	
      }
    gzclose(file);
    return SUCCESS;
}


/*
 * A help function to print any info, on how to use the program.
 */

int print_help(char* option)
{
  if('\0' == option[0])
    {
      printf("\n");
      printf("The normal use of this program would be:\n");
      printf(PACKAGE " [-hvV] [--add file db-file] [--append db-file] [--edit db-file] [--list db-file]\n");
      printf("\t[-H option] [--rebuild [file] db-file] [--delete sig-number db-file]\n");
      printf("\n"),
      printf("\t--add  <file> <db-file>\n");
      printf("\t   This is used, when you want to add a\n");
      printf("\t   signature to the desired database <db-fie>.\n");
      printf("\t   The desired signature is stored in the\n");
      printf("\t   <file> as plain text.\n");
      printf("\t--append <db-file>\n");
      printf("\t   This will invoke sigitdb as if you were adding a signature,\n");
      printf("\t   but instead of reading from a file, it will read the signature\n");
      printf("\t   from stdin. And the usual ^D will end the reading.\n");
      printf("\t--delete <sig-number>  <db-file>\n");
      printf("\t   This will delete the signature with the given\n");
      printf("\t   sig-number from the db-file.\n");
      printf("\t--edit  <db-file>\n");
      printf("\t   This will retrieve all signatures from the\n");
      printf("\t   db-file, and load them into your $EDITOR\n");
      printf("\t   where you can edit them all.\n");
      printf("\t   Upon exit the program will rebuild the data file\n");
      printf("\t   with the new updated signatures.\n");
      printf("\t--list  <db-file>\n");
      printf("\t   In order to get a quick look through the contence of the db-file,\n");
      printf("\t   use this flag, and a listing of the signatures will be printed to stdout.\n");
      printf("\t--rebuild  [file] <db-file>\n");
      printf("\t   Erase the given db-file, and rebuild it with the\n");
      printf("\t   contence of the given file, where this will be a\n");
      printf("\t   strict text file, where each signature is seperated\n");
      printf("\t   by a '@@' This is good, if the db-file is currupted.\n");
      printf("\t   If you only provide the 'db-file' argument, and it matches\n");
      printf("\t   a 'db-file' structure, then the program will try and\n");
      printf("\t   recreate the file as if it was a conversion of an earlier\n");
      printf("\t   Sigit db-file version.\n");
      printf("\t-h\n");
      printf("\t   Print this help screen and exit.\n");
      printf("\t-H  <option>\n");
      printf("\t   Print explanatory help, if no option given,\n");
      printf("\t   it will print this help instruction and exit.\n");
      printf("\t   If an option is given, it will print further help\n");
      printf("\t   regarding that option.\n");
      printf("\t-v\n");
      printf("\t   This is used, to produce verbose output.\n");
      printf("\t-V\n");
      printf("\t   This will print info about the program version.\n\n");
 
   }
  else
    {
      if(!strcmp(option, "add")||!strcmp(option, "--add"))
	{
	  printf("\n\tsigitdb --add <file> <db-file>\n\n");
	  printf("\t   With this flag, sigitdb will read the given signatur(es) in file assuming\n");
	  printf("\tit's a regular text file, and add them to the given db-file.\n");
	  printf("\t   This text file can either have been made by a 'sigitdb --list old-db-file'\n");
	  printf("\tor custom made, if the first line is containing '@@' it is assumed that there will\n");
	  printf("\tbe more than one signature in the file to add. Thus Sigit will see if the file was\n");
	  printf("\tmade from a '--list' or just a usermade one, where each signature is seperated by\n");
	  printf("\t'@@', should the file assume to have been made from a '--list', but is infact usermade.\n");
	  printf("\tThen it will try and correct it, but if theres more than three signatures, that dosn't\n");
	  printf("\tfollow the '--list' structure, it will give up on correcting the errors.\n");
	  printf("\t   A usermade file containing more than one signature, should follow the structure:\n");
	  printf("\t@@\n");
	  printf("\tThis is the first signature.\n");
	  printf("\tWhere there can be just as many\n");
	  printf("\tlines as you want.\n");
	  printf("\t@@\n");
	  printf("\tAnd the second signature\n");
	  printf("\tJust as intrigent\n");
	  printf("\t@@\n");
	  printf("\t   Should the first line _not_ contain '@@', it will be assumed the entire file is\n");
	  printf("\tjust one single signature intended to add to the db-file.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "append")||!strcmp(option, "--append"))
	{
	  printf("\n\tsigitdb --append <db-file>\n\n");
	  printf("\t   As a request from Sean, this flag was introduced, you use it for a quick adding of a\n");
	  printf("\tsignature. Sigitdb will read from stdin and add the text parsed to it as a single signature\n");
	  printf("\tto the db-file. This is a very usefull flag, if you find just the one you need, and don't\n");
	  printf("\twant to spend time saving it to a file, and then executing sigitdb with the --add flag.\n");
	  printf("\t   Just use --append, and paste the signature into the terminal, uppon ^D the program will\n");
	  printf("\tstop reading and add what has been read up untill then.\n");
	  printf("\t   Should the signature consist of more than %d chars, the program will only add the first\n", LINE_SIZE*NUM_LINES);
	  printf("\t%d chars. But since verbose is turned off during this, it can be difficult to see, when that\n",  LINE_SIZE*NUM_LINES);
	  printf("\tlimit is reached.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "delete")||!strcmp(option, "--delete"))
	{
	  printf("\n\tsigitdb --delete <sig-number> <db-file>\n\n");
	  printf("\t   You just mailed a friend, and found, that the signature appended to the letter\n");
	  printf("\tJust dosn't cut it, for beeing desirable in your signature collection.\n");
	  printf("\t   The time is pushing it pretty close, and you dont have time to invoke\n");
	  printf("\t'sigitdb --eit dbfile' and then manualy find the signature, adn remove it.\n");
	  printf("\tThen just do a 'sigitdb --list db-file' and locate where the signature is at,\n");
	  printf("\tseeing what number is displayed for it as \"Identifier\" and invoke a '--delete'\n");
	  printf("\twith that given number. It's as easy as that, that specific signature will no\n");
	  printf("\tlonger be pressent in your db-file.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "edit")||!strcmp(option, "--edit"))
	{
	  printf("\n\tsigitdb --edit <db-file>\n\n");
	  printf("\t   This is a realy simple flag, if you have your $EDITOR variable set to anything\n");
	  printf("\tusefull, it will invoke that with the signatures currently stored in db-file.\n");
	  printf("\tSo you're ready to intuatively edit them, and make sure, they are preserved just\n");
	  printf("\tlike you want them.\n");
	  printf("\t   Basicly, it is just the same as running a 'sigitdb --list db-file>file' and then\n");
	  printf("\tedit that file in your default editor, then run 'sigitdb --rebuild file db-file'.\n");
	  printf("\t   Should it be found, that you havn't set your $EDITOR, it will invoke vim(1)\n");
	  printf("\tBut if vi/vim isn't found on your system, it will terminate with an error, and\n");
	  printf("\tthe amount of verbosity will reveal what info you will be pressented with.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "list")||!strcmp(option, "--list"))
	{
	  printf("\n\tsigitdb --list <db-file>\n\n");
	  printf("\t   Use '--list' if you want a fast look through the db-file. It will display every\n");
	  printf("\tsignature found in it, in an easy to read converted streaming to stdout.\n");
	  printf("\t   Don't worry, the db-file itself will _not_ be changed or harmed in any way during\n");
	  printf("\tthis readout.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "rebuild")||!strcmp(option, "--rebuild"))
	{
	  printf("\n\tsigitdb --rebuild [file] <db-file>\n\n");
	  printf("\t   If you find an error in your db-file structure, which makes sigit unable to\n");
	  printf("\tuse it, then you need to rebuild the db-file, with the '--rebuild' flag, sigitdb\n");
	  printf("\twill see, if the first argument given is a db-file structure, if it is, there will\n");
	  printf("\tbe taken no concern what so ever to the next argument, and sigitdb will try to\n");
	  printf("\trebuild the db-file from what is found in it.\n");
	  printf("\t   Should you supply the first argument as beeing a text file, it is assumed, that\n");
	  printf("\tthe signatures needed to rebuild the db-file with, is gathered in this file, and it\n");
	  printf("\twill be the same as usign 'sigitdb --add file db-file', where the db-file never existed.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "h")||!strcmp(option, "-h"))
	{
	  printf("\n\tsigitdb -h\n\n");
	  printf("\t   Get the usual help informations, if you should be in need for a quick guide\n");
	  printf("\ton all the flags this program handles. Then this is the flag to be using.\n");
	  printf("\tto read any of the usual standard configuration files.\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "H")||!strcmp(option, "-H"))
	{
	  printf("\n\tsigitdb -H flag\n\n");
	  printf("\t   You're allready there... Talk about recursive calls..\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "v")||!strcmp(option, "-v"))
	{
	  printf("\n\tsigitdb -v\n\n");
	  printf("\t   It's for verbose output, as it usualy is, do I realy need to say more ?\n");
	  printf("\tWell I guess I could give a description of the usage anyway..\n");
	  printf("\t   When using the '-v' flag, Sigit will print the debugging information requested\n");
	  printf("\tin the giving verbose level, the level is controlled with the number of '-v's\n");
	  printf("\tThus if you were to use 'sigitdb -v' it would produce less debug info, than 'sigitdb -vv'\n");
	  printf("\t    And should you be a total morron, like I tend to be, when debugging stuff,\n");
	  printf("\t'sigitdb -vvvvv' might give you just the info you were looking for.\n");
	  printf("\tIts a nice way, to see, if the buffers and current state the program is in, is\n");
	  printf("\tsuffering from an overflow, or miss configuration, or in some cases a small \"feature\".\n\n");
	  return SUCCESS;
	}
      if(!strcmp(option, "V")||!strcmp(option, "-V"))
	{
	  printf("\n\tsigitdb -V\n\n");
	  printf("\t   Look ma' its the version info... Weee, I didn't know I was running this version.\n\n");
	  return SUCCESS;
	}
      printf("There is no option what so ever in this program, that ressembles\n");
      printf("the given {%s} option.\n", option);
      return FAILURE;
    }
  return SUCCESS;
}

#endif








