/**
 * SIEGE socket library
 *
 * Copyright (C) 2000, 2001, 2002 by
 * Jeffrey Fulmer - <jdfulmer@armstrong.com>
 * This file is distributed as part of Siege 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef  HAVE_CONFIG_H
# include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <setup.h> 
 
#include <sock.h>
#include <joedog/joedog.h>
#include <joedog/boolean.h>
#include <joedog/defs.h>
#include <pthread.h>
#include <fcntl.h>

#ifdef  HAVE_UNISTD_H
# include <unistd.h>
#endif/*HAVE_UNISTD_H*/

#ifdef  HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif/*HAVE_ARPA_INET_H*/
 
#ifdef  HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif/*HAVE_SYS_SOCKET_H*/ 

#ifdef  HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif/*HAVE_NETINET_IN_H*/
 
#ifdef  HAVE_NETDB_H
# include <netdb.h>
#endif/*HAVE_NETDB_H*/

#ifdef  HAVE_SSL
# include <openssl/rand.h>
#endif/*HAVE_SSL*/

/** 
 * local prototypes 
 */
private int     __socket_block(int socket, int nonblock);
private ssize_t __socket_write(int sock, const void *vbuf, size_t len);  
private ssize_t __ssl_socket_write(CONN *C, const void *vbuf, size_t len);

/**
 * new_socket
 * returns int, socket handle
 */
int
new_socket(CONN *C, const char *hostparam, int portparam)
{
  int conn;
  struct timeval timeout;
  int res;
  int opt;
  int herrno;
  struct sockaddr_in cli; 
  struct hostent     *hp;
  char   hn[256];
  int    port;
#if defined(_AIX)
  char *aixbuf;
  int  rc;
#endif/*_AIX*/ 

  C->sock    = -1;
  C->encrypt = FALSE;
 
  /* if we are using a proxy, then we make a socket
     connection to that server rather then a httpd */ 
  if(my.proxy.required){
    snprintf(hn, sizeof(hn), "%s", my.proxy.hostname);
    port = my.proxy.port;
  } else {
    snprintf(hn, sizeof(hn), "%s", hostparam);
    port = portparam;
  }
 
  /* create a socket, return -1 on failure */
  if(( C->sock = socket( AF_INET, SOCK_STREAM, 0 )) < 0 ){
    switch( errno ){
      case EPROTONOSUPPORT: { joe_error("unsupported protocol");      break; }
      case EMFILE:          { joe_error("descriptor table is full");  break; }
      case ENFILE:          { joe_error("system file table is full"); break; }
      case EACCES:          { joe_error("permission denied");         break; }
      case ENOBUFS:         { joe_error("insufficient buffer space"); break; }
      default:              { joe_error("unknown socket error");      break; }
    } socket_close( C ); return -1;
  }

#if defined(__GLIBC__)
  {
    struct hostent hent; 
    char hbf[9000]; 
    /* for systems using GNU libc */
    if((gethostbyname_r( hn, &hent, hbf, sizeof(hbf), &hp, &herrno ) < 0)){
      hp = NULL;
    }
  }
#elif defined(sun)
# ifdef HAVE_GETIPNODEBYNAME
  hp = getipnodebyname( hn, AF_INET, 0, &herrno );
# else /* default use gethostbyname_r*/
  {
    struct hostent hent; 
    char hbf[9000]; 
    hp = gethostbyname_r( hn, &hent, hbf, sizeof(hbf), &herrno ); 
  }
# endif/*HAVE_GETIPNODEBYNAME*/
#elif defined(_AIX)
  aixbuf = (char*)xmalloc( 9000 );
  rc  = gethostbyname_r(hn, (struct hostent *)aixbuf,
                       (struct hostent_data *)(aixbuf + sizeof(struct hostent)));
  hp = (struct hostent*)aixbuf;
#elif ( defined(hpux) || defined(__hpux) || defined(__osf__) )
  hp = gethostbyname( hn );
  herrno = h_errno;
#else
  /* simply hoping that gethostbyname is thread-safe */
  hp = gethostbyname( hn );
  herrno = h_errno;
#endif/*OS SPECIFICS*/ 

  if( hp == NULL ){ return -1; } 
  memset((void*) &cli, 0, sizeof( cli ));
  memcpy( &cli.sin_addr, hp->h_addr, hp->h_length );
#if defined(sun)
# ifdef  HAVE_FREEHOSTENT
  freehostent(hp);
# endif/*HAVE_FREEHOSTENT*/ 
#endif
  cli.sin_family = AF_INET;
  cli.sin_port = htons( port );

  if( C->connection.keepalive ){
    opt = 1; 
    if(setsockopt(C->sock,SOL_SOCKET,SO_KEEPALIVE,(char *)&opt,sizeof(opt))<0){
      switch( errno ){
        case EBADF:         { joe_error("invalid descriptor");       break; }
        case ENOTSOCK:      { joe_error("not a socket");             break; }
        case ENOPROTOOPT:   { joe_error("not a protocol option");    break; }
        case EFAULT:        { joe_error("setsockopt unknown error"); break; }
        default:            { joe_error("unknown sockopt error");    break; }
      } socket_close( C ); return -1;
    }
  }

  if(( __socket_block(C->sock, TRUE)) < 0 ){
    joe_error( "socket: unable to set socket to non-blocking." );
    return -1; 
  }

  /**
   * connect to the host 
   * evaluate the server response and check for
   * readability/writeability of the socket....
   */ 
  conn = connect( C->sock, (struct sockaddr *)&cli, sizeof(struct sockaddr_in));
  if( conn < 0 && errno != EINPROGRESS ){
    switch( errno ){
      case EACCES:        {joe_error("socket: EACCES");                  break;}
      case EADDRNOTAVAIL: {joe_error("socket: address is unavailable."); break;}
      case ETIMEDOUT:     {joe_error("socket: connection timed out.");   break;}
      case ECONNREFUSED:  {joe_error("socket: connection refused.");     break;}
      case ENETUNREACH:   {joe_error("socket: network is unreachable."); break;}
      default:            {joe_error("socket: unknown network error." ); break;}
    } socket_close( C ); return -1;
  }
  else{
    /** 
     * we have a connection, set the
     * socket to non-blocking and test
     * its integrity.
     */
    FD_ZERO(&C->rset);
    FD_ZERO(&C->wset);
    FD_SET( C->sock, &C->rset );    
    FD_SET( C->sock, &C->wset );    
    memset((void *)&timeout, '\0', sizeof( struct timeval ));
    /**
     * the default timeout is set in init.c, it's
     * value can be changed by the user in .siegerc,
     * but we'll still use a ternary condition since
     * you can't be too safe....
     */ 
    timeout.tv_sec  = (my.timeout > 0)?my.timeout:30;
    timeout.tv_usec = 0;

    if(( res = select( C->sock+1, &C->rset, &C->wset, NULL, &timeout )) < 1 ){
      joe_error( "socket: connection timed out." );
      close( C->sock );
      return -1; 
    }
    else{ 
      C->status = S_READING; 
    }
  } /* end of connect conditional */

  /**
   * make the socket blocking again.
   */
  if(( __socket_block( C->sock, FALSE )) < 0 ){
    joe_error( "socket: unable to set socket to non-blocking." );
    return -1;
  } 
 
  C->connection.status = 1; 
  return( C->sock );
}

/**
 * makes the socket non-blocking,
 * calls select with timeout and
 * returns the socket to blocking.
 */
BOOLEAN
socket_check(CONN *C, SDSET test)
{
  int res;
  struct timeval timeout;

  if( C->connection.tested >= RDWR ){
    return TRUE; /* already tested */
  }

  C->connection.tested |= test;  

  memset((void *)&timeout, '\0', sizeof( struct timeval ));

  if(( __socket_block( C->sock, TRUE )) < 0 ){
    return FALSE;
  }
 
  timeout.tv_sec  = (my.timeout > 0)?my.timeout:15;
  timeout.tv_usec = 0;
 
  switch( test ){
  case READ:
    FD_ZERO(&C->rset);
    FD_SET( C->sock, &C->rset );
    if(( res = select( C->sock+1, &C->rset, NULL, NULL, &timeout )) < 1 ){
      close( C->sock );
      return FALSE;
    }
    break;
  case WRITE:
    FD_ZERO(&C->wset);
    FD_SET( C->sock, &C->wset );
    if(( res = select( C->sock+1, NULL, &C->wset, NULL, &timeout )) < 1 ){
      close( C->sock );
      return FALSE;
    }
    break;
  case RDWR:
    FD_ZERO(&C->rset);
    FD_ZERO(&C->wset);
    FD_SET( C->sock, &C->rset );
    FD_SET( C->sock, &C->wset );
    if(( res = select( C->sock+1, &C->rset, &C->wset, NULL, &timeout )) < 1 ){
      close( C->sock );
      return FALSE;
    }
    break;
  }
 
  if(( __socket_block( C->sock, FALSE )) < 0 ){
    joe_error( "socket: unable to set socket to non-blocking." );
    return FALSE;
  }
  FD_CLR( C->sock, &C->rset );
  FD_CLR( C->sock, &C->wset ); 
  return TRUE;
}

/**
 * local function
 * set socket to non-blocking
 */
private int
__socket_block(int sock, int nonblock)
{
#if HAVE_FCNTL_H 
  int flags;
  int retval;

  if(( flags = fcntl( sock, F_GETFL, 0 )) < 0 ){
    switch( errno ){
      case EACCES:          { joe_error( "fcntl: EACCES");                  break; }
      case EBADF:           { joe_error( "fcntl: bad file descriptor");     break; }
      case EAGAIN:          { joe_error( "fcntl: address is unavailable."); break; }
      default:              { joe_error( "fcntl: unknown network error." ); break; }
    } return -1;
  }
  if( nonblock ){ 
    flags |=  O_NDELAY;
  }
  else{
    flags &= ~O_NDELAY;
  }
  if(( retval = fcntl( sock, F_SETFL, flags)) < 0 ){
    joe_error("fcntl: unable to set fcntl flags\n");
    return -1;
  } 
  return retval;

#elif defined( FIONBIO )
  ioctl_t status;
 
  status = nb ? 1 : 0;
  return ioctl( sd, FIONBIO, &status );
#endif
}  

/**
 * returns ssize_t
 * writes vbuf to sock
 */
private ssize_t
__socket_write( int sock, const void *vbuf, size_t len )
{
  size_t      n;
  ssize_t     w;
  const char *buf;
 
  buf = vbuf;
  n   = len;
  while( n > 0 ){
    if(( w = write( sock, buf, n )) <= 0 ){
      if( errno == EINTR )
        w = 0;
      else
        return( -1 );
    }
    n   -= w;
    buf += w;
  }
  return( len );
}

/**
 * local function
 * returns ssize_t
 * writes vbuf to sock
 */
private ssize_t
__ssl_socket_write( CONN *C, const void *vbuf, size_t len )
{
  size_t      n;
  ssize_t     w;
  const char *buf;

  buf = vbuf;
  n   = len;
#ifdef HAVE_SSL
  while( n > 0 ){
    if(( w = SSL_write( C->ssl, buf, n )) <= 0 ){
      if( errno == EINTR )
        w = 0;
      else
        return( -1 );
    }
    n   -= w;
    buf += w;
  }
  return( len );
#else
  w = C->sock; /* silenct the compiler */
  joe_error( "protocol not supported" );
  return -1;
#endif/*HAVE_SSL*/
}

ssize_t
socket_read( CONN *C, void *vbuf, size_t len )
{
  int type;
  size_t      n;
  ssize_t     r;
  char *buf;
  
  pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, &type ); 
  
  buf = vbuf;
  n   = len;
  if(C->encrypt == TRUE){
  #ifdef HAVE_SSL
    while( n > 0 ){ 
      if(( r = SSL_read( C->ssl, buf, n )) < 0 ){
        if( errno == EINTR )
          r = 0;
        else
          return( -1 );
      }
      else if( r == 0 ) break; 
      n   -= r;
      buf += r;
    }   /* end of while    */
  #endif/*HAVE_SSL*/
  }  
  else{
    while( n > 0 ){
      if( C->inbuffer <  n ){
        int lidos;
        memmove(C->buffer,&C->buffer[C->pos_ini],C->inbuffer);
        C->pos_ini = 0;
        lidos = read( C->sock, &C->buffer[C->inbuffer], sizeof(C->buffer)-C->inbuffer );
        if( lidos < 0 ){
          if(errno==EINTR || errno==EAGAIN)
            lidos = 0;
          if( errno==EPIPE ){
            return 0;
          }
          else{
            joe_error( "socket: read error %s:", strerror( errno ));
            return 0; /* was return -1 */
          }
        }
        C->inbuffer += lidos;
      }
      if( C->inbuffer >= n ){
        r = n;
      }
      else{
        r = C->inbuffer;
      }
      if( r == 0 ) break; 
      memmove(buf,&C->buffer[C->pos_ini],r);
      C->pos_ini  += r;
      C->inbuffer -= r;
      n   -= r;
      buf += r;
    } /* end of while */ 
  }   /* end of else  */

  pthread_setcanceltype(type,NULL);
  pthread_testcancel(); 
  return( len - n );  
}  

/**
 * this function is used for chunked
 * encoding transfers to acquire the 
 * size of the message check.
 */
ssize_t
socket_readline(CONN *C, char *ptr, size_t maxlen)
{
  int type;
  int n, len, res;
  char c;

  len = maxlen;
  pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, &type ); 

  for( n = 1; n < len; n ++ ){
    if(( res = socket_read(C, &c, 1)) == 1 ){
      *ptr++ = c;
      if(( c == '\n')||( c == '\r')) {
        break;
      }
    }
    else if( res == 0 ){
      if( n == 1 ) return ( 0 ); 
      else 
        break; 
    }
    else{
      return -1; /* something bad happened */
    }
  } /* end of for loop */

  *ptr=0;
  
  pthread_setcanceltype(type,NULL);
  pthread_testcancel(); 

  return( n );
}

/**
 * returns void
 * socket_write wrapper function.
 */
int
socket_write( CONN *C, const void *buf, size_t len )
{
  int     type;
  size_t bytes;

  pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, &type ); 

  if(C->encrypt == TRUE){
    /* handle HTTPS protocol */
    #ifdef HAVE_SSL
    if((bytes = __ssl_socket_write( C, buf, len )) != len){
      joe_error( "ssl_socket_write: ERROR" );
      return -1;
    }
    #else
      joe_error( "ssl_socket_write: protocol NOT supported" );
      return -1;
    #endif/*HAVE_SSL*/
  }
  else{
    /* assume HTTP */
    if((bytes = __socket_write( C->sock, buf, len )) != len){
      joe_error( "__socket_write" );
      return -1;
    }
  }

  pthread_setcanceltype(type,NULL); 
  pthread_testcancel(); 

  return bytes;
} 

/**
 * returns void
 * frees ssl resources if using ssl and
 * closes the connection and the socket.
 */
void
socket_close(CONN *C)
{
  int type;
#ifdef  HAVE_SSL
  int ret   = 0;
  int tries = 0;
#endif/*HAVE_SSL*/
  
  if(C==NULL) return;

  /* HELP PLEASE: Is this necessary? */ 
  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &type); 

  if(C->encrypt == TRUE){
#ifdef  HAVE_SSL
    if(!C->connection.reuse || C->connection.max == 1){
      if(C->ssl != NULL){
        do{
          if((ret = SSL_shutdown(C->ssl))==1) 
            break;
          tries++;
        } while(tries < 5);
      }
      SSL_free(C->ssl);
      C->ssl = NULL;
      SSL_CTX_free(C->ctx);
      C->ctx = NULL;
      close(C->sock);
      C->connection.status = 0;
      C->connection.max    = 0;
      C->connection.tested = 0;
    }
#endif/*HAVE_SSL*/
  }
  else {
    if(C->connection.reuse == 0 || C->connection.max == 1){
      close(C->sock);
      C->connection.status    = 0;
      C->connection.max       = 0;
      C->connection.tested    = 0;
    }
  }
  C = NULL;
  pthread_setcanceltype(type,NULL);
  pthread_testcancel(); 

  return;
} 


