/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_setup_ui.cpp,v 1.42 2004/08/21 14:32:14 suzhe Exp $
 *
 */

#include <cstring>
#include <cstdio>

#define Uses_SCIM_COMPOSE_KEY
#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_MODULE
#define Uses_SCIM_IMENGINE_MODULE
#define Uses_STL_MAP
#include "scim_private.h"
#include "scim.h"
#include "scim_setup_module.h"
#include "scim_setup_ui.h"

#define LIST_ICON_SIZE 20 

const gchar * scim_setup_module_categories [] =
{
    N_("SCIM"),
    N_("FrontEnd"),
    N_("IMEngine"),
    N_("Panel"),
    N_("Extra"),
    NULL
};

enum
{
    MODULE_LIST_LABEL = 0,
    MODULE_LIST_CATEGORY,
    MODULE_LIST_MODULE,
    MODULE_LIST_WIDGET,
    MODULE_LIST_NUM_COLUMNS
};

enum
{
    FACTORY_LIST_ENABLE = 0,
    FACTORY_LIST_ICON,
    FACTORY_LIST_NAME,
    FACTORY_LIST_UUID,
    FACTORY_LIST_INCONSISTENT,
    FACTORY_LIST_NUM_COLUMNS
};

static GdkPixbuf *
scale_pixbuf (GdkPixbuf **pixbuf,
               int         width,
               int         height)
{
    if (pixbuf && *pixbuf) {
        if (gdk_pixbuf_get_width (*pixbuf) != width ||
            gdk_pixbuf_get_height (*pixbuf) != height) {
            GdkPixbuf *dest = gdk_pixbuf_scale_simple (*pixbuf, width, height, GDK_INTERP_BILINEAR);
            gdk_pixbuf_unref (*pixbuf);
            *pixbuf = dest;
        }
        return *pixbuf;
    }
    return 0;
}

SetupUI::SetupUI (int argc, char *argv [])
    : m_main_window (0),
      m_work_area (0),
      m_apply_button (0),
      m_restore_button (0),
      m_status_bar (0),
      m_module_list_view (0),
      m_module_list_selection (0),
      m_module_list_model (0),
      m_current_widget (0),
      m_current_module (0),
      m_config (NULL),
      m_config_name (""),
      m_query_changed_timeout (0),
      m_changes_applied (false)
{
    String def_config = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));

    int i = 0;

    while (i < argc) {
        if (++i >= argc) break;
        if (String ("-c") == argv [i] ||
            String ("--config") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "No argument for option " << argv [i-1] << "\n";
                exit (-1);
            }
            def_config = argv [i];
        }
        if (String ("-h") == argv [i] ||
            String ("--help") == argv [i]) {
            std::cout << "Usage: " << argv [0] << " [option]...\n\n"
                      << "The options are:\n"
                      << "  -c, --config name    Load specified config module by default.\n"
                      << "  -h, --help           Show this help message.\n";
            exit (0);
        }

        if (String ("--") == argv [i])
            break;
    }

    gtk_init (&argc, &argv);

    load_config_modules (def_config);

    create_main_ui ();
    create_module_list_model ();

    m_query_changed_timeout = gtk_timeout_add (500, query_changed_timeout_cb, this);
}

SetupUI::~SetupUI ()
{
    g_source_remove (m_query_changed_timeout);
    gtk_widget_destroy (m_main_window);
}

bool
SetupUI::add_module (SetupModule *module)
{
    if (!module || !module->valid ()) return false;

    GtkTreeIter iter, parent;

    GtkWidget *module_widget   = module->create_ui ();
    String     module_label    = module->get_name ();
    String     module_category = module->get_category ();

    if (!module_widget || !module_label.length () || !module_category.length ())
        return false;

    if (!m_config.null ())
        module->load_config (m_config);

    gtk_box_pack_start (GTK_BOX (m_work_area), module_widget, TRUE, TRUE, 0);
    gtk_widget_hide (module_widget);

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (m_module_list_model), &parent)) {
        gchar *category;

        do {
            gtk_tree_model_get (GTK_TREE_MODEL (m_module_list_model), &parent,
                                MODULE_LIST_CATEGORY, &category, -1);

            // Find the right category and append the module.
            if (category && !strcmp (category, module_category.c_str ())) {
                gtk_tree_store_append (m_module_list_model, &iter, &parent);
                gtk_tree_store_set (
                    m_module_list_model, &iter,
                    MODULE_LIST_LABEL,    module_label.c_str (),
                    MODULE_LIST_CATEGORY, NULL,
                    MODULE_LIST_MODULE,   module,
                    MODULE_LIST_WIDGET,   module_widget,
                    -1);

                g_free (category);

                gtk_tree_view_expand_all (GTK_TREE_VIEW (m_module_list_view));

                return true;
            }

            g_free (category);

        } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (m_module_list_model), &parent));
    }

    GtkWidget *cover;

    if (module_category == String ("IMEngine"))
        cover = create_factory_list_view ();
    else
        cover = create_setup_cover (module_category.c_str ());

    gtk_box_pack_start (GTK_BOX (m_work_area), cover, TRUE, TRUE, 0);

    // No suitable category available, add one.
    gtk_tree_store_append (m_module_list_model, &parent, NULL);
    gtk_tree_store_set (
        m_module_list_model, &parent,
        MODULE_LIST_LABEL,    _(module_category.c_str ()),
        MODULE_LIST_CATEGORY, module_category.c_str (),
        MODULE_LIST_MODULE,   NULL,
        MODULE_LIST_WIDGET,   cover,
        -1);

    gtk_tree_store_append (m_module_list_model, &iter, &parent);
    gtk_tree_store_set (
        m_module_list_model, &iter,
        MODULE_LIST_LABEL,    module_label.c_str (),
        MODULE_LIST_CATEGORY, NULL,
        MODULE_LIST_MODULE,   module,
        MODULE_LIST_WIDGET,   module_widget,
        -1);

    gtk_tree_view_expand_all (GTK_TREE_VIEW (m_module_list_view));

    return true;
}

bool
SetupUI::run ()
{
    if (m_main_window) {
        gtk_widget_show (m_main_window);

        if (!m_all_configs.size ()) {
            GtkWidget *dialog =
                gtk_message_dialog_new (GTK_WINDOW (m_main_window),
                                         GTK_DIALOG_MODAL,
                                         GTK_MESSAGE_WARNING,
                                         GTK_BUTTONS_OK,
                                         _("Failed to load any Config Modules!"));

            gtk_dialog_run (GTK_DIALOG (dialog));
            gtk_widget_destroy (dialog);
        }

        gtk_main ();
    }
    return m_changes_applied;
}

void
SetupUI::create_main_ui ()
{
    GtkWidget *hpaned1;
    GtkWidget *scrolledwindow1;
    GtkWidget *vbox1;
    GtkWidget *vbox2;
    GtkWidget *frame1;
    GtkWidget *hbox1;
    GtkWidget *ok_button;
    GtkWidget *exit_button;
    GtkWidget *vseparator1;

    GtkCellRenderer *module_list_cell;
    GtkTreeViewColumn *module_list_column;

    // Create main window.
    m_main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (m_main_window), _("SCIM Input Method Setup"));
    gtk_window_set_position (GTK_WINDOW (m_main_window), GTK_WIN_POS_CENTER);
    gtk_window_set_modal (GTK_WINDOW (m_main_window), TRUE);
    gtk_window_set_destroy_with_parent (GTK_WINDOW (m_main_window), TRUE);
    gtk_window_set_resizable (GTK_WINDOW (m_main_window), TRUE);

    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox1);
    gtk_container_add (GTK_CONTAINER (m_main_window), vbox1);

    // Create paned window.
    hpaned1 = gtk_hpaned_new ();
    gtk_widget_show (hpaned1);
    gtk_box_pack_start (GTK_BOX (vbox1), hpaned1, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hpaned1), 4);

    // Create statusbar.
    m_status_bar = gtk_statusbar_new ();
    gtk_statusbar_set_has_resize_grip (GTK_STATUSBAR (m_status_bar), TRUE);
    gtk_widget_show (m_status_bar);
    gtk_box_pack_start (GTK_BOX (vbox1), m_status_bar, FALSE, FALSE, 0);

    // Create scrollwindow for module list.
    scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow1);
    gtk_paned_pack1 (GTK_PANED (hpaned1), scrolledwindow1, FALSE, FALSE);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow1),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow1), 
                                         GTK_SHADOW_ETCHED_IN);

    // Create module list view.
    m_module_list_view = gtk_tree_view_new ();
    gtk_widget_show (m_module_list_view);
    gtk_container_add (GTK_CONTAINER (scrolledwindow1), m_module_list_view);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (m_module_list_view), FALSE);
    gtk_tree_view_set_enable_search (GTK_TREE_VIEW (m_module_list_view), FALSE);

    // Get module list selection.
    m_module_list_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (m_module_list_view));
    gtk_tree_selection_set_mode (m_module_list_selection, GTK_SELECTION_BROWSE);

    // Create module list column.
    module_list_cell = gtk_cell_renderer_text_new ();
    module_list_column = gtk_tree_view_column_new_with_attributes (
                            NULL, module_list_cell, "text", MODULE_LIST_LABEL, NULL);

    gtk_tree_view_append_column (GTK_TREE_VIEW (m_module_list_view), module_list_column);

    // Create vbox for work area and button area.
    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox2);
    gtk_paned_pack2 (GTK_PANED (hpaned1), vbox2, TRUE, FALSE);

    // Create frame for work area.
    frame1 = gtk_frame_new (NULL);
    gtk_widget_show (frame1);
    gtk_box_pack_start (GTK_BOX (vbox2), frame1, TRUE, TRUE, 0);

    m_work_area = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (m_work_area);
    gtk_container_add (GTK_CONTAINER (frame1), m_work_area);

    // Create hbox for button area.
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox1);
    gtk_box_pack_end (GTK_BOX (vbox2), hbox1, FALSE, FALSE, 8);

    ok_button = gtk_button_new_from_stock ("gtk-ok");
    gtk_widget_show (ok_button);
    gtk_box_pack_end (GTK_BOX (hbox1), ok_button, FALSE, FALSE, 4);

    exit_button = gtk_button_new_from_stock ("gtk-quit");
    gtk_widget_show (exit_button);
    gtk_box_pack_end (GTK_BOX (hbox1), exit_button, FALSE, FALSE, 4);

    vseparator1 = gtk_vseparator_new ();
    gtk_widget_show (vseparator1);
    gtk_box_pack_end (GTK_BOX (hbox1), vseparator1, FALSE, FALSE, 4);

    m_apply_button = gtk_button_new_from_stock ("gtk-apply");
    gtk_widget_show (m_apply_button);
    gtk_box_pack_end (GTK_BOX (hbox1), m_apply_button, FALSE, FALSE, 4);
    GTK_WIDGET_SET_FLAGS (m_apply_button, GTK_CAN_DEFAULT);
    gtk_widget_set_sensitive (m_apply_button, FALSE);

    m_restore_button = gtk_button_new_from_stock ("gtk-revert-to-saved");
    gtk_widget_show (m_restore_button);
    gtk_box_pack_end (GTK_BOX (hbox1), m_restore_button, FALSE, FALSE, 4);
    gtk_widget_set_sensitive (m_restore_button, FALSE);

    g_signal_connect ((gpointer) ok_button, "clicked",
                      G_CALLBACK (SetupUI::ok_button_clicked_callback),
                      this);
    g_signal_connect ((gpointer) exit_button, "clicked",
                      G_CALLBACK (SetupUI::exit_button_clicked_callback),
                      this);
    g_signal_connect ((gpointer) m_apply_button, "clicked",
                      G_CALLBACK (SetupUI::apply_button_clicked_callback),
                      this);
    g_signal_connect ((gpointer) m_restore_button, "clicked",
                      G_CALLBACK (SetupUI::restore_button_clicked_callback),
                      this);
    g_signal_connect (G_OBJECT (m_main_window), "delete_event",
                      G_CALLBACK (main_window_delete_callback),
                      this);

    g_signal_connect (G_OBJECT (m_module_list_selection), "changed",
                      G_CALLBACK (module_list_selection_changed_callback),
                      this);

    gtk_widget_grab_default (m_apply_button);
}

GtkWidget *
SetupUI::create_splash_view ()
{
    GtkRequisition size;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *view;
    GtkWidget *combo;
    GtkWidget *label;
    GtkWidget *separator;
    GList     *config_list = NULL;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    view = gtk_label_new (NULL);
    gtk_widget_show (view);
    gtk_label_set_justify (GTK_LABEL (view), GTK_JUSTIFY_CENTER);
    gtk_label_set_markup (GTK_LABEL (view), _(
                " <span size=\"20000\">Smart Common Input Method platform</span> \n\n"
                "<span size=\"16000\" style=\"italic\">GUI Setup Utility</span>\n\n\n\n"
                "<span size=\"12000\">Copyright 2002-2004, James Su &lt;suzhe@tsinghua.org.cn&gt;</span>"));
    gtk_box_pack_start (GTK_BOX (vbox), view, TRUE, TRUE, 4);

    separator = gtk_hseparator_new ();
    gtk_widget_show (separator);
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 4);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);

    label = gtk_label_new (NULL);
    gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Current Config Module:"));
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 8);

    combo = gtk_combo_new ();
    gtk_widget_show (combo);
    gtk_combo_set_value_in_list (GTK_COMBO (combo), TRUE, FALSE);
    gtk_combo_set_case_sensitive (GTK_COMBO (combo), TRUE);
    gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (combo)->entry), FALSE);
    gtk_box_pack_start (GTK_BOX (hbox), combo, FALSE, FALSE, 8);
    gtk_label_set_mnemonic_widget (GTK_LABEL (label), GTK_COMBO (combo)->entry);

    for (size_t i = 0; i < m_all_configs.size (); ++ i)
        config_list = g_list_append (config_list, g_strdup (m_all_configs [i].first.c_str ()));

    if (!config_list)
        config_list = g_list_append (config_list, g_strdup (_("<Not found>")));

    gtk_combo_set_popdown_strings (GTK_COMBO (combo), config_list);

    for (GList *ptr = config_list; ptr != NULL; ptr = g_list_next (ptr))
        g_free (ptr->data);

    g_list_free (config_list);

    if (m_config_name.length ()) 
        gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry), m_config_name.c_str ());

    g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry), "changed",
                      G_CALLBACK (config_changed_callback),
                      this);

    gtk_widget_size_request (vbox, &size);

    if (size.width  < 320) size.width = 320;
    if (size.height < 240) size.height = 240;

    gtk_widget_set_size_request (vbox, size.width, size.height);

    gtk_widget_hide (vbox);

    return vbox;
}

GtkWidget *
SetupUI::create_setup_cover (const char *category)
{
    GtkRequisition size;
    GtkWidget *cover;
    char buf [128];

    snprintf (buf, 127,
              _("<span size=\"x-large\">The Setup for %s modules.</span>"),
              _(category));

    cover = gtk_label_new (NULL);

    gtk_label_set_markup (GTK_LABEL (cover), buf);
    gtk_label_set_justify (GTK_LABEL (cover), GTK_JUSTIFY_CENTER);

    gtk_widget_show (cover);

    gtk_widget_size_request (cover, &size);

    if (size.width  < 320) size.width = 320;
    if (size.height < 240) size.height = 240;

    gtk_widget_set_size_request (cover, size.width, size.height);

    gtk_widget_hide (cover);

    return cover;
}

void
SetupUI::create_module_list_model ()
{
    GtkTreeIter iter;
    GtkWidget  *widget;

    widget = create_splash_view ();

    gtk_box_pack_start (GTK_BOX (m_work_area), widget, TRUE, TRUE, 0);

    m_module_list_model = gtk_tree_store_new (
                            MODULE_LIST_NUM_COLUMNS,
                            G_TYPE_STRING,
                            G_TYPE_STRING,
                            G_TYPE_POINTER,
                            GTK_TYPE_WIDGET);

    gtk_tree_store_append (m_module_list_model, &iter, NULL);

    gtk_tree_store_set (m_module_list_model, &iter,
                        MODULE_LIST_LABEL,    _(scim_setup_module_categories [0]),
                        MODULE_LIST_CATEGORY, scim_setup_module_categories [0],
                        MODULE_LIST_MODULE,   NULL,
                        MODULE_LIST_WIDGET,   widget,
                        -1);

    gtk_tree_view_set_model (GTK_TREE_VIEW (m_module_list_view),
                             GTK_TREE_MODEL (m_module_list_model));
}

GtkWidget *
SetupUI::create_factory_list_view ()
{
    GtkRequisition size;

    GtkWidget *vbox;
    GtkWidget *scrolledwindow;
    GtkWidget *label;
    GtkWidget *sep;

    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    char buf [128];

    snprintf (buf, 127,
              _("<span size=\"x-large\">The Setup for %s modules.</span>"),
              _("IMEngine"));

    vbox = gtk_vbox_new (FALSE, 4);
    gtk_widget_show (vbox);

    label = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (label), buf);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 4);

    sep = gtk_hseparator_new ();
    gtk_widget_show (sep);
    gtk_box_pack_start (GTK_BOX (vbox), sep, FALSE, TRUE, 2);

    label = gtk_label_new (_("The installed input method services:"));
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 4);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), 
                                         GTK_SHADOW_NONE);
 
    m_factory_list_view = gtk_tree_view_new ();
    gtk_widget_show (m_factory_list_view);
    gtk_container_add (GTK_CONTAINER (scrolledwindow), m_factory_list_view);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (m_factory_list_view), TRUE);
    gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (m_factory_list_view), TRUE);

    // Name column
    column = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
    gtk_tree_view_column_set_resizable (column, TRUE);

    gtk_tree_view_column_set_title (column, _("Name"));

    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_set_attributes (column, renderer,
                                         "pixbuf", FACTORY_LIST_ICON, NULL);

    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (column, renderer, TRUE);
    gtk_tree_view_column_set_attributes (column, renderer,
                                         "text", FACTORY_LIST_NAME, NULL);

    gtk_tree_view_append_column (GTK_TREE_VIEW (m_factory_list_view), column);

    // Enable column
    column = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_resizable (column, FALSE);
    gtk_tree_view_column_set_title (column, _("Enable"));

    renderer = gtk_cell_renderer_toggle_new ();
    gtk_cell_renderer_toggle_set_radio (GTK_CELL_RENDERER_TOGGLE (renderer), FALSE);
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_set_attributes (column, renderer,
                                         "active", FACTORY_LIST_ENABLE,
                                         "inconsistent", FACTORY_LIST_INCONSISTENT,
                                         NULL);

    g_signal_connect (G_OBJECT (renderer), "toggled",
                      G_CALLBACK (SetupUI::factory_list_changed_callback),
                      this);

    gtk_tree_view_append_column (GTK_TREE_VIEW (m_factory_list_view), column);

    create_factory_list_model ();

    gtk_tree_view_collapse_all (GTK_TREE_VIEW (m_factory_list_view));

    gtk_widget_size_request (vbox, &size);

    if (size.width  < 360) size.width = 360;
    if (size.height < 300) size.height = 300;

    gtk_widget_set_size_request (vbox, size.width, size.height);

    gtk_widget_hide (vbox);

    return vbox;
}

void
SetupUI::create_factory_list_model ()
{
    std::vector <String> uuids;
    std::vector <String> names;
    std::vector <String> langs;
    std::vector <String> icons;

    std::map <String, std::vector <size_t> > groups;

    GtkTreeIter iter;
    GtkTreeIter parent;

    String      lang_name;
    GdkPixbuf   *pixbuf;
    size_t i;

    m_factory_list_model = gtk_tree_store_new (FACTORY_LIST_NUM_COLUMNS,
                                              G_TYPE_BOOLEAN,
                                              GDK_TYPE_PIXBUF,
                                              G_TYPE_STRING,
                                              G_TYPE_STRING,
                                              G_TYPE_BOOLEAN);

    get_factory_list (uuids, names, langs, icons);

    for (i = 0; i < uuids.size (); ++i) {
        groups [langs [i]].push_back (i);
    }

    // Add language group
    for (std::map <String, std::vector <size_t> >::iterator it = groups.begin ();
         it != groups.end (); ++ it) {
        lang_name = scim_get_language_name (it->first);
        gtk_tree_store_append (m_factory_list_model, &parent, NULL);
        gtk_tree_store_set (m_factory_list_model, &parent,
                            FACTORY_LIST_ENABLE, true,
                            FACTORY_LIST_ICON, NULL,
                            FACTORY_LIST_NAME, lang_name.c_str (),
                            FACTORY_LIST_UUID, NULL,
                            FACTORY_LIST_INCONSISTENT, FALSE,
                            -1);

        // Add factories for this group
        for (i = 0; i < it->second.size (); ++i) {
            pixbuf = gdk_pixbuf_new_from_file (icons [it->second [i]].c_str (), NULL);
            scale_pixbuf (&pixbuf, LIST_ICON_SIZE, LIST_ICON_SIZE);
            gtk_tree_store_append (m_factory_list_model, &iter, &parent);
            gtk_tree_store_set (m_factory_list_model, &iter,
                                FACTORY_LIST_ENABLE, true,
                                FACTORY_LIST_ICON, pixbuf,
                                FACTORY_LIST_NAME, names [it->second [i]].c_str (),
                                FACTORY_LIST_UUID, uuids [it->second [i]].c_str (),
                                FACTORY_LIST_INCONSISTENT, FALSE,
                                -1);
            if (pixbuf)
                g_object_unref (pixbuf);
        }
    }

    load_factory_list_config ();

    gtk_tree_view_set_model (GTK_TREE_VIEW (m_factory_list_view),
                             GTK_TREE_MODEL (m_factory_list_model));
}

void
SetupUI::get_factory_list (std::vector <String> &uuids,
                          std::vector <String> &names,
                          std::vector <String> &langs,
                          std::vector <String> &icons)
{
    std::vector<String>    module_list;
    IMEngineFactoryPointer factory;
    IMEngineModule         module;
    ConfigPointer          config;

    scim_get_imengine_module_list (module_list);

    uuids.clear ();
    names.clear ();
    langs.clear ();
    icons.clear ();

    if (!m_config.null ()) config = m_config;
    else config = new DummyConfig ("scim");

    // Add "English/European" factory first.
    factory = new ComposeKeyFactory ();
    uuids.push_back (factory->get_uuid ());
    names.push_back (utf8_wcstombs (factory->get_name ()));
    langs.push_back (scim_get_normalized_language (factory->get_language ()));
    icons.push_back (factory->get_icon_file ());

    for (size_t i = 0; i < module_list.size (); ++ i) {

        module.load (module_list [i], config);

        if (module.valid ()) {
            for (size_t j = 0; j < module.number_of_factories (); ++j) {
                try {
                    factory = module.create_factory (j);
                } catch (...) {
                    factory.reset ();
                }

                if (!factory.null ()) {
                    if (std::find (uuids.begin (), uuids.end (), factory->get_uuid ()) == uuids.end ()) {
                        uuids.push_back (factory->get_uuid ());
                        names.push_back (utf8_wcstombs (factory->get_name ()));
                        langs.push_back (scim_get_normalized_language (factory->get_language ()));
                        icons.push_back (factory->get_icon_file ());
                    }
                    factory.reset ();
                }
            }
            module.unload ();
        }
    }
}

gboolean
SetupUI::factory_list_load_config_func (GtkTreeModel *model,
                                       GtkTreePath  *path,
                                       GtkTreeIter  *iter,
                                       gpointer      data)
{
    gchar *uuid;
    std::vector <String> *disabled = static_cast <std::vector<String> *> (data);

    gtk_tree_model_get (model, iter,
                        FACTORY_LIST_UUID, &uuid,
                        -1);
    if (uuid &&
        std::find (disabled->begin (), disabled->end (), String (uuid)) != disabled->end ()) {
        gtk_tree_store_set (GTK_TREE_STORE (model), iter,
                            FACTORY_LIST_ENABLE, FALSE,
                            -1);
    } else {
        gtk_tree_store_set (GTK_TREE_STORE (model), iter,
                            FACTORY_LIST_ENABLE, TRUE,
                            -1);
    }

    if (uuid) g_free (uuid);

    return FALSE;
}

void
SetupUI::load_factory_list_config ()
{
    std::vector <String> disabled;

    if (m_factory_list_model) {
        disabled = scim_global_config_read (String (SCIM_GLOBAL_CONFIG_DISABLED_IMENGINE_FACTORIES), disabled);

        gtk_tree_model_foreach (GTK_TREE_MODEL (m_factory_list_model),
                                factory_list_load_config_func,
                                static_cast <gpointer> (&disabled));

        factory_list_update_inconsistent ();
    }

    m_factory_list_changed = false;
}

gboolean
SetupUI::factory_list_save_config_func (GtkTreeModel *model,
                                       GtkTreePath  *path,
                                       GtkTreeIter  *iter,
                                       gpointer      data)
{
    gchar *uuid;
    gboolean enable;
    std::vector <String> *disabled = static_cast <std::vector<String> *> (data);

    gtk_tree_model_get (model, iter,
                        FACTORY_LIST_UUID, &uuid,
                        FACTORY_LIST_ENABLE, &enable,
                        -1);

    if (!enable && uuid)
    disabled->push_back (String (uuid));

    if (uuid) g_free (uuid);

    return FALSE;
}

void
SetupUI::save_factory_list_config ()
{
    std::vector <String> disabled;

    if (m_factory_list_model && m_factory_list_changed) {
        gtk_tree_model_foreach (GTK_TREE_MODEL (m_factory_list_model),
                                factory_list_save_config_func,
                                static_cast <gpointer> (&disabled));

        scim_global_config_write (String (SCIM_GLOBAL_CONFIG_DISABLED_IMENGINE_FACTORIES), disabled);
    }

    m_factory_list_changed = false;
}

bool
SetupUI::ask_quit () const
{
    GtkWidget *dialog;
    gint result;

    dialog = gtk_message_dialog_new (GTK_WINDOW (m_main_window),
                                     GTK_DIALOG_MODAL,
                                     GTK_MESSAGE_WARNING,
                                     GTK_BUTTONS_OK_CANCEL,
                                     _("Are you sure to quit SCIM Setup?"));

    result = gtk_dialog_run (GTK_DIALOG (dialog));

    gtk_widget_destroy (dialog);

    return result == GTK_RESPONSE_OK;
}

gboolean
SetupUI::module_list_hide_widget_iter_func  (GtkTreeModel *model,
                                             GtkTreePath *path,
                                             GtkTreeIter *iter,
                                             gpointer data)
{
    GtkWidget    *widget;
    gtk_tree_model_get (model, iter, MODULE_LIST_WIDGET, &widget, -1);

    if (widget)
        gtk_widget_hide (widget);

    g_object_unref (widget);

    return FALSE;
}

void
SetupUI::module_list_selection_changed_callback (GtkTreeSelection *selection, gpointer user_data)
{
    GtkTreeModel *model;
    GtkTreeIter   iter;
    GtkWidget    *widget;
    SetupModule  *module;
    gchar        *label;
    gchar        *category;

    SetupUI * ui = (SetupUI *) user_data;

    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        gtk_tree_model_get (model, &iter,
                            MODULE_LIST_LABEL,    &label,
                            MODULE_LIST_CATEGORY, &category,
                            MODULE_LIST_MODULE,   &module,
                            MODULE_LIST_WIDGET,   &widget,
                            -1);

        if (widget != ui->m_current_widget) {
            //Hide all other widgets
            gtk_tree_model_foreach (model, module_list_hide_widget_iter_func, NULL); 
            gtk_widget_show (widget);
            ui->m_current_widget = widget;
        }

        if (module != ui->m_current_module || !module) {
            gtk_statusbar_pop (GTK_STATUSBAR (ui->m_status_bar), 1);
            gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
            gtk_widget_set_sensitive (ui->m_restore_button, FALSE);

            if (module) {
                String desc = module->get_description ();
                if (desc.length ())
                    gtk_statusbar_push (GTK_STATUSBAR (ui->m_status_bar), 1, desc.c_str ());

                if (module->query_changed () && !ui->m_config.null ()) {
                    gtk_widget_set_sensitive (ui->m_apply_button, TRUE);
                    gtk_widget_set_sensitive (ui->m_restore_button, TRUE);
                }
            } else if (category && String (category) == String ("IMEngine")){
                gtk_statusbar_push (GTK_STATUSBAR (ui->m_status_bar), 1, _("You can enable/disable installed input methods here."));
                if (ui->m_factory_list_changed) {
                    gtk_widget_set_sensitive (ui->m_apply_button, TRUE);
                    gtk_widget_set_sensitive (ui->m_restore_button, TRUE);
                }
            }

            ui->m_current_module = module;
        }

        g_free (label);
        if (category) g_free (category);
        g_object_unref (widget);
    }
}

void
SetupUI::restore_button_clicked_callback (GtkButton *button, gpointer user_data)
{
    SetupUI *ui = (SetupUI*) user_data;

    if (ui->m_config.null ()) return;

    if (ui->m_current_module) {
        ui->m_current_module->load_config (ui->m_config);

        gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
        gtk_widget_set_sensitive (ui->m_restore_button, FALSE);
    } else {
        GtkTreeSelection *selection;
        GtkTreeModel     *model;
        GtkTreeIter       iter;
        gchar            *category;

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ui->m_module_list_view));

        if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
            gtk_tree_model_get (model, &iter,
                                MODULE_LIST_CATEGORY, &category,
                                -1);
            if (category && String (category) == String ("IMEngine")) {
                ui->load_factory_list_config ();
                gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
                gtk_widget_set_sensitive (ui->m_restore_button, FALSE);
            }

            if (category) g_free (category);
        }
    }
}

void
SetupUI::apply_button_clicked_callback (GtkButton *button, gpointer user_data)
{
    SetupUI *ui = (SetupUI*) user_data;

    if (ui->m_config.null ()) return;

    if (ui->m_current_module) {
        ui->m_current_module->save_config (ui->m_config);

        ui->m_config->flush ();

        ui->m_changes_applied = true;

        gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
        gtk_widget_set_sensitive (ui->m_restore_button, FALSE);
    } else {
        GtkTreeSelection *selection;
        GtkTreeModel     *model;
        GtkTreeIter       iter;
        gchar            *category;

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ui->m_module_list_view));

        if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
            gtk_tree_model_get (model, &iter,
                                MODULE_LIST_CATEGORY, &category,
                                -1);
            if (category && String (category) == String ("IMEngine")) {
                ui->save_factory_list_config ();
                gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
                gtk_widget_set_sensitive (ui->m_restore_button, FALSE);
            }

            if (category) g_free (category);
        }
    }
}

void 
SetupUI::factory_list_update_inconsistent(void)
{
    GtkTreeIter iter;
    GtkTreeIter childiter;
    gboolean    enable;
    gboolean    inconsistent;

    if (!m_factory_list_model || !gtk_tree_model_get_iter_first (GTK_TREE_MODEL (m_factory_list_model), &iter))
        return;

    do {
        gtk_tree_model_get (GTK_TREE_MODEL (m_factory_list_model), &iter,
                            FACTORY_LIST_ENABLE, &enable,
                            FACTORY_LIST_INCONSISTENT, &inconsistent,
                            -1);
        if (gtk_tree_model_iter_children (GTK_TREE_MODEL (m_factory_list_model), &childiter, &iter)) {
            gint enable_count = 0;
            gint total_count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL (m_factory_list_model), &iter);
            do {
                gboolean child_enable;
                gtk_tree_model_get (GTK_TREE_MODEL (m_factory_list_model), &childiter,
                                    FACTORY_LIST_ENABLE, &child_enable,
                                    -1);
                if (child_enable) ++ enable_count;
            } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (m_factory_list_model), &childiter));
            enable = (enable_count && enable_count >= ((total_count+1) >> 1));
            inconsistent = (enable_count && enable_count < total_count);
        }
        gtk_tree_store_set (GTK_TREE_STORE (m_factory_list_model), &iter,
                            FACTORY_LIST_ENABLE, enable,
                            FACTORY_LIST_INCONSISTENT, inconsistent,
                            -1);
    } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (m_factory_list_model), &iter));
}

void
SetupUI::factory_list_changed_callback (GtkCellRendererToggle *cell, gchar *arg1, gpointer data)
{
    SetupUI *ui = (SetupUI*) data;
    GtkTreePath *path;
    GtkTreeIter iter, childiter;
    gboolean enable;

    path = gtk_tree_path_new_from_string (arg1);
    if (gtk_tree_model_get_iter (GTK_TREE_MODEL (ui->m_factory_list_model), &iter, path)) {
        if (gtk_tree_model_iter_children(GTK_TREE_MODEL (ui->m_factory_list_model), &childiter, &iter)) {
            gtk_tree_model_get (GTK_TREE_MODEL (ui->m_factory_list_model), &iter,
                                FACTORY_LIST_ENABLE, &enable,
                                -1);
            enable = !enable;
            gtk_tree_store_set (ui->m_factory_list_model, &iter,
                                FACTORY_LIST_ENABLE, enable,
                                FACTORY_LIST_INCONSISTENT, FALSE,
                                -1);
            do {
                gtk_tree_store_set (ui->m_factory_list_model, &childiter,
                                    FACTORY_LIST_ENABLE, enable,
                                    -1);
            } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (ui->m_factory_list_model), &childiter));
        } else {
            gtk_tree_model_get (GTK_TREE_MODEL (ui->m_factory_list_model), &iter,
                                FACTORY_LIST_ENABLE, &enable,
                                -1);
            gtk_tree_store_set (ui->m_factory_list_model, &iter,
                                FACTORY_LIST_ENABLE, !enable,
                                -1);
            ui->factory_list_update_inconsistent ();
        }
    }
    gtk_tree_path_free (path);
 
    ui->m_factory_list_changed = true;
}

gboolean
SetupUI::module_list_save_config_iter_func  (GtkTreeModel *model,
                                             GtkTreePath *path,
                                             GtkTreeIter *iter,
                                             gpointer data)
{
    SetupModule *module;

    SetupUI *ui = (SetupUI *) data;

    gtk_tree_model_get (model, iter, MODULE_LIST_MODULE, &module, -1);

    if (module && module->query_changed () && ui && !ui->m_config.null ()) {
        module->save_config (ui->m_config);
        ui->m_changes_applied = true;
    }

    return FALSE;
}

gboolean
SetupUI::module_list_load_config_iter_func  (GtkTreeModel *model,
                                             GtkTreePath *path,
                                             GtkTreeIter *iter,
                                             gpointer data)
{
    SetupModule *module;

    SetupUI *ui = (SetupUI *) data;

    gtk_tree_model_get (model, iter, MODULE_LIST_MODULE, &module, -1);

    if (module && ui && !ui->m_config.null ())
        module->load_config (ui->m_config);

    return FALSE;
}


void
SetupUI::ok_button_clicked_callback (GtkButton *button, gpointer user_data)
{
    SetupUI *ui = (SetupUI *) user_data;

    if (!ui->m_config.null ()) {
        gtk_tree_model_foreach (GTK_TREE_MODEL (ui->m_module_list_model),
                                module_list_save_config_iter_func,
                                user_data); 

        ui->save_factory_list_config ();

        ui->m_config->flush ();
        if (ui->m_changes_applied)
            ui->show_restart_hint ();
    }

    gtk_main_quit ();
}

void
SetupUI::exit_button_clicked_callback (GtkButton *button, gpointer user_data)
{
    SetupUI *ui = (SetupUI*) user_data;

    if (ui->ask_quit ()) {
        if (ui->m_changes_applied)
            ui->show_restart_hint ();
        gtk_main_quit ();
    }
}

void
SetupUI::config_changed_callback (GtkEditable *editable, gpointer user_data)
{
    SetupUI *ui = (SetupUI*) user_data;

    String name = gtk_entry_get_text (GTK_ENTRY (editable));

    for (size_t i = 0; i < ui->m_all_configs.size (); ++ i) {
        if (name == ui->m_all_configs [i].first) {
            ui->m_config_name = name;
            ui->m_config      = ui->m_all_configs [i].second;

            gtk_tree_model_foreach (GTK_TREE_MODEL (ui->m_module_list_model),
                            module_list_load_config_iter_func,
                            user_data); 

            ui->load_factory_list_config ();

            GtkWidget *dialog;

            dialog = gtk_message_dialog_new (GTK_WINDOW (ui->m_main_window),
                            GTK_DIALOG_MODAL,
                            GTK_MESSAGE_INFO,
                            GTK_BUTTONS_OK,
                            _("Current Config Module has been changed to %s."),
                            name.c_str ());

            gtk_dialog_run (GTK_DIALOG (dialog));

            gtk_widget_destroy (dialog);
        }
    }
}

gboolean
SetupUI::main_window_delete_callback (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    SetupUI *ui = (SetupUI*) user_data;

    if (ui->ask_quit ()) {

        if (ui->m_changes_applied)
            ui->show_restart_hint ();

        gtk_main_quit ();
    }

    return TRUE;
}

gboolean
SetupUI::query_changed_timeout_cb (gpointer data)
{
    SetupUI *ui = (SetupUI *) data;

    gtk_widget_set_sensitive (ui->m_apply_button, FALSE);
    gtk_widget_set_sensitive (ui->m_restore_button, FALSE);

    if (ui->m_config.null ())
        return TRUE;

    if (ui->m_current_module) {
        if (ui->m_current_module->query_changed ()) {
            gtk_widget_set_sensitive (ui->m_apply_button, TRUE);
            gtk_widget_set_sensitive (ui->m_restore_button, TRUE);
        }
    } else {
        GtkTreeSelection *selection;
        GtkTreeModel     *model;
        GtkTreeIter       iter;
        gchar            *category;

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ui->m_module_list_view));

        if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
            gtk_tree_model_get (model, &iter,
                                MODULE_LIST_CATEGORY, &category,
                                -1);
            if (category && String (category) == String ("IMEngine") && ui->m_factory_list_changed) {
                gtk_widget_set_sensitive (ui->m_apply_button, TRUE);
                gtk_widget_set_sensitive (ui->m_restore_button, TRUE);
            }

            if (category) g_free (category);
        }
    }

    return TRUE;
}

void
SetupUI::show_restart_hint () const
{
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new (GTK_WINDOW (m_main_window),
                            GTK_DIALOG_MODAL,
                            GTK_MESSAGE_INFO,
                            GTK_BUTTONS_OK,
                            _("Not all configuration can be reloaded on the fly. "
                              "Don't forget to restart SCIM in order to let all of "
                              "the new configuration take effect."));

    gtk_dialog_run (GTK_DIALOG (dialog));

    gtk_widget_destroy (dialog);
}

void
SetupUI::load_config_modules (const String &def_config)
{
    std::vector<String>  config_list;
    ConfigModule        *module;
    ConfigPointer        config;

    m_all_configs.clear ();
    m_config_name = "";
    m_config.reset ();

    if (scim_get_config_module_list (config_list)) {
        size_t i;

        for (i = 0; i < config_list.size (); ++ i) {
            std::cout << "Loading Config Module " << config_list [i] << "\n";
            module = new ConfigModule (config_list [i]);
            if (module && module->valid ()) {
                config = module->create_config ("scim");
                if (!config.null ()) {
                    m_all_configs.push_back (
                        std::make_pair (config_list [i], config));
                    continue;
                }
            }
            std::cerr << "  Failed to load Config Module " << config_list [i] << "\n";
        }

        if (!m_all_configs.size ()) {
            std::cerr << "Can not load any config module!\n";
            exit (-1);
        }

        for (i = 0; i < m_all_configs.size (); ++ i) {
            if (m_all_configs [i].first == def_config) {
                m_config_name = def_config;
                m_config = m_all_configs [i].second;
                return;
            }
        }
        m_config_name = m_all_configs [0].first;
        m_config = m_all_configs [0].second;
    }
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
