/** @file gtkimcontextscim.cpp
 *  @brief immodule for GTK2.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: gtkimcontextscim.cpp,v 1.116.2.3 2004/11/15 14:41:50 suzhe Exp $
 */

#define Uses_SCIM_DEBUG
#define Uses_SCIM_BACKEND
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_IMENGINE_MODULE
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_SCIM_PANEL
#include <iostream>

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "scim_private.h"
#include "scim.h"

using namespace scim;

#include "gtkimcontextscim.h"

#define GTK_TYPE_IM_CONTEXT_SCIM             _gtk_type_im_context_scim
#define GTK_IM_CONTEXT_SCIM(obj)             (GTK_CHECK_CAST ((obj), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIM))
#define GTK_IM_CONTEXT_SCIM_CLASS(klass)     (GTK_CHECK_CLASS_CAST ((klass), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIMClass))
#define GTK_IS_IM_CONTEXT_SCIM(obj)          (GTK_CHECK_TYPE ((obj), GTK_TYPE_IM_CONTEXT_SCIM))
#define GTK_IS_IM_CONTEXT_SCIM_CLASS(klass)  (GTK_CHECK_CLASS_TYPE ((klass), GTK_TYPE_IM_CONTEXT_SCIM))
#define GTK_IM_CONTEXT_SCIM_GET_CLASS(obj)   (GTK_CHECK_GET_CLASS ((obj), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIMClass))

#ifndef SCIM_KEYBOARD_ICON_FILE
  #define SCIM_KEYBOARD_ICON_FILE            (SCIM_ICONDIR "/keyboard.png")
#endif

typedef std::vector<KeyEvent> KeyEventVector;

/* Typedef */
struct _GtkIMContextSCIMImpl
{
    int                      id;
    SocketClient             panel;
    SocketTransaction        send;
    uint32                   panel_magic_key;
    GdkWindow               *client_window;
    WideString               preedit_string;
    AttributeList            preedit_attrlist;
    gint                     preedit_caret;
    gint                     cursor_x;
    gint                     cursor_y;
    guint                    panel_source_id;
    gboolean                 use_preedit;
    bool                     is_on;
};

struct _GSCIMPanelSource
{
    GSource                source;
    GtkIMContextSCIM      *context;
    GPollFD                fd;
};

typedef struct _GSCIMPanelSource GSCIMPanelSource;

/* Methods declaration */
static void     gtk_im_context_scim_class_init         (GtkIMContextSCIMClass *klass,
                                                        gpointer              *klass_data);
static void     gtk_im_context_scim_init               (GtkIMContextSCIM      *context_scim,
                                                        GtkIMContextSCIMClass *klass);
static void     gtk_im_context_scim_finalize           (GObject               *obj);
static void     gtk_im_context_scim_finalize_partial   (GtkIMContextSCIM      *context_scim);
static void     gtk_im_context_scim_set_client_window  (GtkIMContext          *context,
                                                        GdkWindow             *client_window);
static gboolean gtk_im_context_scim_filter_keypress    (GtkIMContext          *context,
                                                        GdkEventKey           *key);
static void     gtk_im_context_scim_reset              (GtkIMContext          *context);
static void     gtk_im_context_scim_focus_in           (GtkIMContext          *context);
static void     gtk_im_context_scim_focus_out          (GtkIMContext          *context);
static void     gtk_im_context_scim_set_cursor_location(GtkIMContext          *context,
                                                        GdkRectangle          *area);
static void     gtk_im_context_scim_set_use_preedit    (GtkIMContext          *context,
                                                        gboolean               use_preedit);
static void     gtk_im_context_scim_get_preedit_string (GtkIMContext          *context,
                                                        gchar                **str,
                                                        PangoAttrList        **attrs,
                                                        gint                  *cursor_pos);

static gboolean gtk_scim_key_snooper                   (GtkWidget             *grab_widget,
                                                        GdkEventKey           *event,
                                                        gpointer               data);

static void     gtk_im_slave_commit_cb                 (GtkIMContext          *context,
                                                        const char            *str,
                                                        GtkIMContextSCIM      *context_scim);

/* private functions */
static void     launch_panel ();

static bool     panel_open_connection                 (GtkIMContextSCIM      *ic);
static bool     panel_connect_server                  (GtkIMContextSCIM      *ic);
static bool     panel_prepare_transaction             (GtkIMContextSCIM      *ic);
static bool     panel_send_request                    (GtkIMContextSCIM      *ic);
static bool     panel_receive_reply                   (GtkIMContextSCIM      *ic);

static void     panel_req_turn_on_panel               (GtkIMContextSCIM      *ic);
static void     panel_req_turn_off_panel              (GtkIMContextSCIM      *ic);
static void     panel_req_update_display              (GtkIMContextSCIM      *ic);
static void     panel_req_update_screen               (GtkIMContextSCIM      *ic);
static void     panel_req_show_help                   (GtkIMContextSCIM      *ic);
static void     panel_req_show_factory_menu           (GtkIMContextSCIM      *ic);
static void     panel_req_focus_in                    (GtkIMContextSCIM      *ic);
static void     panel_req_focus_out                   (GtkIMContextSCIM      *ic);
static void     panel_req_update_factory_info         (GtkIMContextSCIM      *ic);
static void     panel_req_update_spot_location        (GtkIMContextSCIM      *ic);
static void     panel_req_show_preedit_string         (GtkIMContextSCIM      *ic);
static void     panel_req_hide_preedit_string         (GtkIMContextSCIM      *ic);
static void     panel_req_update_preedit_string       (GtkIMContextSCIM      *ic,
                                                       const WideString      &str,
                                                       const AttributeList   &attrs);
static void     panel_req_update_preedit_caret        (GtkIMContextSCIM      *ic,
                                                       int                    caret);
static void     panel_req_show_aux_string             (GtkIMContextSCIM      *ic);
static void     panel_req_hide_aux_string             (GtkIMContextSCIM      *ic);
static void     panel_req_update_aux_string           (GtkIMContextSCIM      *ic,
                                                       const WideString      &str,
                                                       const AttributeList   &attrs);
static void     panel_req_show_lookup_table           (GtkIMContextSCIM      *ic);
static void     panel_req_hide_lookup_table           (GtkIMContextSCIM      *ic);
static void     panel_req_update_lookup_table         (GtkIMContextSCIM      *ic,
                                                       const LookupTable     &table);
static void     panel_req_register_properties         (GtkIMContextSCIM      *ic,
                                                       const PropertyList    &properties);
static void     panel_req_update_property             (GtkIMContextSCIM      *ic,
                                                       const Property        &property);


/* Panel Source relates functions */
static gboolean panel_source_prepare                  (GSource               *source,
                                                       gint                  *timeout);
static gboolean panel_source_check                    (GSource               *source);
static gboolean panel_source_dispatch                 (GSource               *source,
                                                       GSourceFunc            callback,
                                                       gpointer               user_data);
static bool     panel_check_connection                (GtkIMContextSCIM      *ic);

/* utility functions */
static void     set_focus_ic                           (GtkIMContextSCIM      *ic);

static bool     match_key_event                        (const KeyEventVector  &keys,
                                                        const KeyEvent        &key);

static KeyEvent keyevent_gdk_to_scim                   (const GdkEventKey     &gdkevent);

static void     keyevent_scim_to_gdk                   (GdkEventKey           &gdkevent,
                                                        const KeyEvent        &scimkey,
                                                        GtkIMContextSCIM      *ic);

static void     initialize                             (void);

static void     finalize                               (void);

static bool     open_default_factory                   (GtkIMContextSCIM      *ic,
                                                        int                    icid);
static void     open_next_factory                      (GtkIMContextSCIM      *ic);
static void     open_previous_factory                  (GtkIMContextSCIM      *ic);
static void     open_specific_factory                  (GtkIMContextSCIM      *ic,
                                                        const String           &uuid);

static String   get_help_info                          (GtkIMContextSCIM      *ic);

/* slot functions */
static void     slot_show_preedit_string               (int                    id);
static void     slot_show_aux_string                   (int                    id);
static void     slot_show_lookup_table                 (int                    id);

static void     slot_hide_preedit_string               (int                    id);
static void     slot_hide_aux_string                   (int                    id);
static void     slot_hide_lookup_table                 (int                    id);

static void     slot_update_preedit_caret              (int                    id,
                                                        int                    caret);
static void     slot_update_preedit_string             (int                    id,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     slot_update_aux_string                 (int                    id,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     slot_commit_string                     (int                    id,
                                                        const WideString      &str);
static void     slot_forward_key_event                 (int                    id,
                                                        const KeyEvent        &key);
static void     slot_update_lookup_table               (int                    id,
                                                        const LookupTable     &table);

static void     slot_register_properties               (int                    id,
                                                        const PropertyList    &properties);
static void     slot_update_property                   (int                    id,
                                                        const Property        &property);

static void     reload_config_callback                 (const ConfigPointer   &config);

static void     fallback_commit_string_cb              (IMEngineInstanceBase  *si,
                                                        const WideString      &str);

/* Local variables declaration */
static GType                 _gtk_type_im_context_scim   = 0;
static GObjectClass         *_parent_klass               = 0;

static KeyEventVector        _trigger_keys;
static KeyEventVector        _next_factory_keys;
static KeyEventVector        _previous_factory_keys;
static KeyEventVector        _show_factory_menu_keys;
static int                   _valid_key_mask;

static ConfigModule         *_config_module              = 0;
static ConfigPointer         _config;
static BackEndPointer        _backend;

static String                _default_factory;

static GtkIMContextSCIM     *_focused_ic                 = 0;

static bool                  _scim_initialized           = false;

static String                _panel_address;
static int                   _panel_timeout;
static bool                  _panel_launched             = false;

static String                _config_module_name         = "socket";

static GdkColor              _normal_bg;
static GdkColor              _normal_text;
static GdkColor              _active_bg;
static GdkColor              _active_text;

static gint                  _snooper_id                 = 0;
static bool                  _snooper_installed          = false;

static int                   _input_context_count        = 0;

static IMEngineFactoryPointer  _fallback_factory;
static IMEngineInstancePointer _fallback_instance;

static GSourceFuncs          _panel_source_funcs         =
{
    panel_source_prepare,
    panel_source_check,
    panel_source_dispatch,
    NULL
};

/* Function Implementations */

/* Public functions */
GtkIMContext *
gtk_im_context_scim_new (void)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_new...\n";

    GtkIMContextSCIM *result;
    
    result = GTK_IM_CONTEXT_SCIM (g_object_new (GTK_TYPE_IM_CONTEXT_SCIM, NULL));

    return GTK_IM_CONTEXT (result);
}

void
gtk_im_context_scim_register_type (GTypeModule *type_module)
{
    static const GTypeInfo im_context_scim_info =
    {
        sizeof               (GtkIMContextSCIMClass),
        (GBaseInitFunc)       NULL,
        (GBaseFinalizeFunc)   NULL,
        (GClassInitFunc)      gtk_im_context_scim_class_init,
        NULL,                 /* class_finalize */
        NULL,                 /* class_data */
        sizeof               (GtkIMContextSCIM),
        0,
        (GtkObjectInitFunc)  gtk_im_context_scim_init,
    };

    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_register_type...\n";

    if (!_gtk_type_im_context_scim) {
        _gtk_type_im_context_scim = 
            g_type_module_register_type (type_module,
                                     GTK_TYPE_IM_CONTEXT,
                                     "GtkIMContextSCIM",
                                     &im_context_scim_info,
                                     (GTypeFlags) 0);
    }
}

void
gtk_im_context_scim_shutdown (void)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_shutdown...\n";

    finalize ();
}

/* Private functions */
static void
gtk_im_context_scim_class_init (GtkIMContextSCIMClass *klass,
                                gpointer              *klass_data)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_class_init...\n";

    GtkIMContextClass *im_context_klass = GTK_IM_CONTEXT_CLASS (klass);
    GObjectClass      *gobject_klass    = G_OBJECT_CLASS (klass);

    _parent_klass = (GObjectClass *) g_type_class_peek_parent (klass);

    im_context_klass->set_client_window   = gtk_im_context_scim_set_client_window;
    im_context_klass->filter_keypress     = gtk_im_context_scim_filter_keypress;
    im_context_klass->reset               = gtk_im_context_scim_reset;
    im_context_klass->get_preedit_string  = gtk_im_context_scim_get_preedit_string;
    im_context_klass->focus_in            = gtk_im_context_scim_focus_in;
    im_context_klass->focus_out           = gtk_im_context_scim_focus_out;
    im_context_klass->set_cursor_location = gtk_im_context_scim_set_cursor_location;
    im_context_klass->set_use_preedit     = gtk_im_context_scim_set_use_preedit;
    gobject_klass->finalize               = gtk_im_context_scim_finalize;

    if (!_scim_initialized) {
        initialize ();
        _scim_initialized = true;
    }
}

static void
gtk_im_context_scim_init (GtkIMContextSCIM      *context_scim,
                          GtkIMContextSCIMClass *klass)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_init...\n";

    int   icid = -1;

    context_scim->impl = NULL;

    /* slave exists for using gtk+'s table based input method */
    context_scim->slave = gtk_im_context_simple_new ();
    g_signal_connect(G_OBJECT(context_scim->slave),
                     "commit",
                     G_CALLBACK(gtk_im_slave_commit_cb),
                     context_scim);

    if (_backend.null () || !_default_factory.length ())
        return;

    icid = _backend->new_instance (_default_factory, "UTF-8");

    if (icid < 0) return;

    context_scim->impl = new GtkIMContextSCIMImpl;

    context_scim->impl->id = icid;
    context_scim->impl->client_window = NULL;
    context_scim->impl->preedit_caret = 0;
    context_scim->impl->cursor_x = 0;
    context_scim->impl->cursor_y = 0;
    context_scim->impl->is_on = FALSE;
    context_scim->impl->use_preedit = TRUE;
    context_scim->impl->panel_source_id = 0;

    ++ _input_context_count;
}

static void
gtk_im_context_scim_finalize_partial (GtkIMContextSCIM *context_scim)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_finalize_partial...\n";

    if (context_scim->impl) {
        _backend->delete_instance (context_scim->impl->id);

        if (context_scim->impl->panel.is_connected ())
            context_scim->impl->panel.close ();
        if (context_scim->impl->panel_source_id)
            g_source_remove (context_scim->impl->panel_source_id);

        delete context_scim->impl;

        context_scim->impl = NULL;

        -- _input_context_count;
    }

    if (context_scim == _focused_ic)
        _focused_ic = 0;
}

static void
gtk_im_context_scim_finalize (GObject *obj)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_finalize...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (obj);

    g_signal_handlers_disconnect_by_func(context_scim->slave,
                                         (void *)gtk_im_slave_commit_cb,
                                         (void *)context_scim);
    g_object_unref(context_scim->slave);

    gtk_im_context_scim_finalize_partial (context_scim);

    _parent_klass->finalize (obj);
}

static void
gtk_im_context_scim_set_client_window (GtkIMContext *context,
                                       GdkWindow    *client_window)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_set_client_window...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        if (client_window)
            g_object_ref (client_window);

        if (context_scim->impl->client_window)
            g_object_unref (context_scim->impl->client_window);

        context_scim->impl->client_window = client_window;
    }
}

static gboolean
gtk_im_context_scim_filter_keypress (GtkIMContext *context,
                                     GdkEventKey  *event)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_filter_keypress...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->slave)
        return gtk_im_context_filter_keypress (context_scim->slave, event);

    return FALSE;
}

static void
gtk_im_context_scim_reset (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_reset...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl && context_scim == _focused_ic) {
        panel_prepare_transaction (context_scim);
        _backend->reset (context_scim->impl->id);
        panel_send_request (context_scim);
    }
}

static void
gtk_im_context_scim_focus_in (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_focus_in...\n";

    if (!_snooper_installed) {
        _snooper_id = gtk_key_snooper_install ((GtkKeySnoopFunc)gtk_scim_key_snooper, NULL);
        _snooper_installed = true;
    }

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        panel_prepare_transaction (context_scim);
        set_focus_ic (context_scim);
        panel_send_request (context_scim);
    }
}

static void
gtk_im_context_scim_focus_out (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_focus_out...\n";

    if (_snooper_installed) {
        gtk_key_snooper_remove (_snooper_id);
        _snooper_installed = false;
    }

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl && context_scim == _focused_ic) {
        panel_prepare_transaction (context_scim);
        _backend->focus_out (context_scim->impl->id);
        panel_req_turn_off_panel (context_scim);
        panel_req_focus_out (context_scim);
        panel_send_request (context_scim);
        _focused_ic = 0;
    }
}

static void
gtk_im_context_scim_set_cursor_location (GtkIMContext *context,
                                         GdkRectangle *area)
{
    SCIM_DEBUG_FRONTEND(3) << "gtk_im_context_scim_set_cursor_location...\n";

    gint x, y;
    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);
 
    if (context_scim && context_scim->impl && context_scim->impl->client_window && context_scim == _focused_ic) {
        gdk_window_get_origin(context_scim->impl->client_window, &x, &y);
        if (context_scim->impl->cursor_x != x + area->x + area->width ||
            context_scim->impl->cursor_y != y + area->y + area->height + 8) {
            context_scim->impl->cursor_x = x + area->x + area->width;
            context_scim->impl->cursor_y = y + area->y + area->height + 8;
            panel_prepare_transaction (context_scim);
            panel_req_update_spot_location (context_scim);
            panel_send_request (context_scim);
        } 
    }
}

static void
gtk_im_context_scim_set_use_preedit (GtkIMContext *context,
                                     gboolean      use_preedit)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_set_use_preedit...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        context_scim->impl->use_preedit = use_preedit;
        if (context_scim == _focused_ic && context_scim->impl->preedit_string.length ()) {
            panel_prepare_transaction (context_scim);
            slot_show_preedit_string (context_scim->impl->id);
            panel_send_request (context_scim);
        }
    }
}

static void
gtk_im_context_scim_get_preedit_string (GtkIMContext   *context,
                                        gchar         **str,
                                        PangoAttrList **attrs,
                                        gint           *cursor_pos)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_get_preedit_string...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        String mbs = utf8_wcstombs (context_scim->impl->preedit_string);

        if (str) {
            if (mbs.length () && context_scim->impl->is_on)
                *str = g_strdup (mbs.c_str ());
            else
                *str = g_strdup ("");
        }

        if (cursor_pos) {
            if (context_scim->impl->is_on)
                *cursor_pos = context_scim->impl->preedit_caret;
            else
                *cursor_pos = 0;
        }

        if (attrs) {
            *attrs = pango_attr_list_new ();

            if (mbs.length () && context_scim->impl->is_on) {
                guint start_index, end_index;
                guint wlen = context_scim->impl->preedit_string.length ();

                PangoAttribute *attr;
                AttributeList::const_iterator i;
                bool underline = false;

                for (i = context_scim->impl->preedit_attrlist.begin ();
                     i != context_scim->impl->preedit_attrlist.end (); ++i) {
                    start_index = i->get_start ();
                    end_index = i->get_end ();

                    if (end_index <= wlen && start_index < end_index) {
                        start_index = g_utf8_offset_to_pointer (mbs.c_str (), i->get_start ()) - mbs.c_str ();
                        end_index = g_utf8_offset_to_pointer (mbs.c_str (), i->get_end ()) - mbs.c_str ();

                        if (i->get_type () == SCIM_ATTR_DECORATE) {
                            if (i->get_value () == scim::SCIM_ATTR_DECORATE_UNDERLINE) {
                                attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                                underline = true;
                            } else if (i->get_value () == SCIM_ATTR_DECORATE_REVERSE) {
                                attr = pango_attr_foreground_new (_normal_bg.red, _normal_bg.green, _normal_bg.blue);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
   
                                attr = pango_attr_background_new (_normal_text.red, _normal_text.green, _normal_text.blue);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                            } else if (i->get_value () == SCIM_ATTR_DECORATE_HIGHLIGHT) {
                                attr = pango_attr_foreground_new (_active_text.red, _active_text.green, _active_text.blue);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
   
                                attr = pango_attr_background_new (_active_bg.red, _active_bg.green, _active_bg.blue);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                            }
                        } else if (i->get_type () == SCIM_ATTR_FOREGROUND) {
                            unsigned int color = i->get_value ();
   
                            attr = pango_attr_foreground_new (SCIM_RGB_COLOR_RED(color) * 256, SCIM_RGB_COLOR_GREEN(color) * 256, SCIM_RGB_COLOR_BLUE(color) * 256);
                            attr->start_index = start_index;
                            attr->end_index = end_index;
                            pango_attr_list_insert (*attrs, attr);
                        } else if (i->get_type () == SCIM_ATTR_BACKGROUND) {
                            unsigned int color = i->get_value ();
   
                            attr = pango_attr_background_new (SCIM_RGB_COLOR_RED(color) * 256, SCIM_RGB_COLOR_GREEN(color) * 256, SCIM_RGB_COLOR_BLUE(color) * 256);
                            attr->start_index = start_index;
                            attr->end_index = end_index;
                            pango_attr_list_insert (*attrs, attr);
                        }
                    }
                }

                // If there is no underline at all, then draw underline under the whole preedit string.
                if (!underline) {
                    attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
                    attr->start_index = 0;
                    attr->end_index = mbs.length ();
                    pango_attr_list_insert (*attrs, attr);
                }
            }
        }
    } else {
        if (str)
            *str = g_strdup ("");

        if (attrs)
            *attrs = pango_attr_list_new ();
    }
}

static gboolean
gtk_scim_key_snooper (GtkWidget    *grab_widget,
                      GdkEventKey  *event,
                      gpointer      data)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_scim_key_snooper...\n";

    gboolean ret = FALSE;

    if (_focused_ic && _focused_ic->impl && (event->type == GDK_KEY_PRESS || event->type == GDK_KEY_RELEASE)) {
        KeyEvent key = keyevent_gdk_to_scim (*event);

        key.mask &= _valid_key_mask;

        panel_prepare_transaction (_focused_ic);

        if (match_key_event (_trigger_keys, key)) {
            if (!_focused_ic->impl->is_on) {
                _focused_ic->impl->is_on = TRUE;
                set_focus_ic (_focused_ic);
            } else { 
                _focused_ic->impl->is_on = FALSE;
                set_focus_ic (_focused_ic);
                if (_focused_ic->impl->use_preedit)
                    g_signal_emit_by_name(_focused_ic, "preedit_changed");
            }
            ret = TRUE;
        } else if (match_key_event (_show_factory_menu_keys, key)) {
            panel_req_show_factory_menu (_focused_ic);
            ret = TRUE;
        } else if (_focused_ic->impl->is_on) {
            if (match_key_event (_next_factory_keys, key)) {
                open_next_factory (_focused_ic);
                set_focus_ic (_focused_ic);
                ret = TRUE;
            } else if (match_key_event (_previous_factory_keys, key)) {
                open_previous_factory (_focused_ic);
                set_focus_ic (_focused_ic);
                ret = TRUE;
            } else if (!_backend->process_key_event (_focused_ic->impl->id, key)) {
                ret = _fallback_instance->process_key_event (key);
            } else {
                ret = TRUE;
            }
        } else {
            ret = _fallback_instance->process_key_event (key);
        }

        panel_send_request (_focused_ic);
    }

    return ret;
}

static void
gtk_im_slave_commit_cb (GtkIMContext     *context,
                        const char       *str,
                        GtkIMContextSCIM *context_scim)
{
    g_return_if_fail(str);
    g_signal_emit_by_name(context_scim, "commit", str);
}

static void
launch_panel ()
{
    char * my_argv [2] = {"--no-stay", 0};

    scim_launch_panel (true, _config_module_name, 0, "none", my_argv);
}

static bool
panel_open_connection (GtkIMContextSCIM *ic)
{
    if (!ic || !ic->impl || !ic->impl->panel.is_connected ()) return false;

    if (!scim_socket_trans_open_connection (ic->impl->panel_magic_key,
                                            String ("FrontEnd"),
                                            String ("Panel"),
                                            ic->impl->panel,
                                            _panel_timeout)) {
        ic->impl->panel.close ();
        return false;
    }

    return true;
}

static bool
panel_connect_server (GtkIMContextSCIM *ic)
{
    if (!ic || !ic->impl) return false;

    SocketAddress address (_panel_address);
    bool ret = true;

    if (!ic->impl->panel.connect (address)) {
        ret = false;
        launch_panel ();
        for (int i=0; i<200; ++i) {
            if (ic->impl->panel.connect (address)) {
                ret = true;
                break;
            }
            scim_usleep (100000);
        }
    }

    if (ret)
        ret = panel_open_connection (ic);

    if (ret) {
        SCIM_DEBUG_FRONTEND(2) << " Attach panel source...\n";
        GSCIMPanelSource *source =
            (GSCIMPanelSource*) g_source_new (&_panel_source_funcs,
                                               sizeof (GSCIMPanelSource));

        source->context = ic;
        source->fd.fd = ic->impl->panel.get_id ();
        source->fd.events = G_IO_IN;
        g_source_add_poll ((GSource *)source, &(source->fd));
        g_source_set_can_recurse ((GSource *)source, FALSE);

        if (ic->impl->panel_source_id)
            g_source_remove (ic->impl->panel_source_id);

        ic->impl->panel_source_id = g_source_attach ((GSource *)source, NULL);

        g_source_unref ((GSource *)source);
    }

    return ret;
}

static bool
panel_prepare_transaction (GtkIMContextSCIM *ic)
{
    int cmd;
    uint32 data;

    ic->impl->send.clear ();
    ic->impl->send.put_command (SCIM_TRANS_CMD_REQUEST);
    ic->impl->send.put_data (ic->impl->panel_magic_key);
    ic->impl->send.put_data ((uint32) 0); // context id

    ic->impl->send.get_command (cmd);
    ic->impl->send.get_data (data);
    ic->impl->send.get_data (data);
    return true;
}

static bool
panel_send_request (GtkIMContextSCIM *ic)
{
    if (!ic->impl->panel.is_connected () && !panel_connect_server (ic))
        return false;

    if (ic->impl->panel.is_connected () &&
        ic->impl->send.get_data_type () != SCIM_TRANS_DATA_UNKNOWN)
        return ic->impl->send.write_to_socket (ic->impl->panel, 0x4d494353);

    return false;
}

static bool
panel_receive_reply (GtkIMContextSCIM *ic)
{
    SocketTransaction receive;

    if (!ic->impl->panel.is_connected () || !receive.read_from_socket (ic->impl->panel, _panel_timeout))
        return false;

    if (_backend.null ())
        return false;

    int cmd;
    uint32 context;

    if (!receive.get_command (cmd) || !cmd == SCIM_TRANS_CMD_REPLY)
        return false;

    // No context id available, so there will be some global command.
    if (receive.get_data_type () == SCIM_TRANS_DATA_COMMAND) {
        while (receive.get_command (cmd)) {
            switch (cmd) {
                case SCIM_TRANS_CMD_RELOAD_CONFIG:
                    {
                        if (!_config.null ())
                            _config->reload ();
                    }
                    break;
                case SCIM_TRANS_CMD_EXIT:
                    {
                        gtk_im_context_scim_finalize_partial (ic);

                        // All IC are finalized, then close all things.
                        if (_input_context_count == 0) {
                            _default_factory = String ();
                            _backend.reset ();
                            _config.reset ();
                        }
                    }
                    break;
                default:
                    break;
            }
        }
        return true;
    }

    if (!receive.get_data (context) || context != 0)
        return false;

    panel_prepare_transaction (ic);

    while (receive.get_command (cmd)) {
        switch (cmd) {
            case SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE:
                {
                    uint32 size;
                    if (receive.get_data (size)) {
                        _backend->update_lookup_table_page_size (ic->impl->id, size);
                    }
                }
                break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_UP:
                {
                    _backend->lookup_table_page_up (ic->impl->id);
                }
                break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_DOWN:
                {
                    _backend->lookup_table_page_down (ic->impl->id);
                }
                break;
            case SCIM_TRANS_CMD_TRIGGER_PROPERTY:
                {
                    String prop;
                    if (receive.get_data (prop) && ic->impl->is_on)
                        _backend->trigger_property (ic->impl->id, prop);
                }
                break;
            case SCIM_TRANS_CMD_MOVE_PREEDIT_CARET:
                {
                    uint32 caret;
                    if (receive.get_data (caret)) {
                        _backend->move_preedit_caret (ic->impl->id, caret);
                    }
                }
                break;
            case SCIM_TRANS_CMD_SELECT_CANDIDATE:
                {
                    uint32 item;
                    if (receive.get_data (item)) {
                        _backend->select_candidate (ic->impl->id, item);
                    }
                }
                break;
            case SCIM_TRANS_CMD_PROCESS_KEY_EVENT:
                {
                    KeyEvent key;
                    if (receive.get_data (key)) {
                        if (_focused_ic != ic)
                            set_focus_ic (ic);

                        if (!_backend->process_key_event (ic->impl->id, key) &&
                            !_fallback_instance->process_key_event (key)) {
                            GdkEventKey gdkevent;
                            keyevent_scim_to_gdk (gdkevent, key, ic);
                            if (!gtk_im_context_filter_keypress (ic->slave, &gdkevent)) {
                                ucs4_t ucs = key.get_unicode_code ();
                                if (ucs && key.is_key_press ()) {
                                    unsigned char utf8[7];
                                    utf8 [utf8_wctomb (utf8, ucs, 7)] = 0;
                                    g_signal_emit_by_name (ic, "commit", utf8);
                                }
                            }
                        }
                    }
                }
                break;
            case SCIM_TRANS_CMD_COMMIT_STRING:
                {
                    String str;
                    if (receive.get_data (str)) {
                        g_signal_emit_by_name (ic, "commit", str.c_str ());
                    }
                }
                break;
            case SCIM_TRANS_CMD_FORWARD_KEY_EVENT:
                {
                    KeyEvent key;
                    if (receive.get_data (key)) {
                        GdkEventKey gdkevent;
                        keyevent_scim_to_gdk (gdkevent, key, ic);
                        if (!_fallback_instance->process_key_event (key) &&
                            !gtk_im_context_filter_keypress (ic->slave, &gdkevent)) {
                            ucs4_t ucs = key.get_unicode_code ();
                            if (ucs && key.is_key_press ()) {
                                unsigned char utf8[7];
                                utf8 [utf8_wctomb (utf8, ucs, 7)] = 0;
                                g_signal_emit_by_name (ic, "commit", utf8);
                            }
                        }
                    }
                }
                break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_HELP:
                {
                    panel_req_show_help (ic);
                }
                break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_FACTORY_MENU:
                {
                    panel_req_show_factory_menu (ic);
                }
                break;
            case SCIM_TRANS_CMD_PANEL_CHANGE_FACTORY:
                {
                    String uuid;
                    if (receive.get_data (uuid)) {
                        if (!uuid.length () && ic->impl->is_on) {
                            ic->impl->is_on = FALSE;
                            set_focus_ic (ic);
                            if (ic->impl->use_preedit)
                                g_signal_emit_by_name(ic, "preedit_changed");
                        } else if (uuid.length ()) {
                            ic->impl->is_on = TRUE;
                            open_specific_factory (ic, uuid);
                            set_focus_ic (ic);
                        }
                    }
                }
            default:
                break;
        }
    }
    panel_send_request (ic);
    return true;
}

static void
panel_req_turn_on_panel (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_TURN_ON);
}

static void
panel_req_turn_off_panel (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_TURN_OFF);
}

static void
panel_req_update_display (GtkIMContextSCIM *ic)
{
#if GDK_MULTIHEAD_SAFE
    if (ic->impl->client_window) {
        GdkDisplay *display = gdk_drawable_get_display (GDK_DRAWABLE (ic->impl->client_window));
        if (display) {
            String name (gdk_display_get_name (display));
            if (name.length ()) {
                ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_DISPLAY);
                ic->impl->send.put_data (name);
            }
        }
    }
#endif
}

static void
panel_req_update_screen (GtkIMContextSCIM *ic)
{
#if GDK_MULTIHEAD_SAFE
    if (ic->impl->client_window) {
        GdkScreen *screen = gdk_drawable_get_screen (GDK_DRAWABLE (ic->impl->client_window));
        if (screen) {
            uint32 number = gdk_screen_get_number (screen);
            ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SCREEN);
            ic->impl->send.put_data (number);
        }
    }
#endif
}

static void
panel_req_show_help (GtkIMContextSCIM *ic)
{
    String help = get_help_info (ic);
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_SHOW_HELP);
    ic->impl->send.put_data (help);
}

static void
panel_req_show_factory_menu (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_SHOW_FACTORY_MENU);

    std::vector<String> uuids;
    _backend->get_factory_list (uuids);

    for (size_t i = 0; i < uuids.size (); ++ i) {
        ic->impl->send.put_data (uuids [i]);
        ic->impl->send.put_data (utf8_wcstombs (_backend->get_factory_name (uuids [i])));
        ic->impl->send.put_data (_backend->get_factory_language (uuids [i]));
        ic->impl->send.put_data (_backend->get_factory_icon_file (uuids [i]));
    }
}

static void
panel_req_update_factory_info (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_FACTORY_INFO);
    if (ic->impl->is_on) {
        ic->impl->send.put_data (utf8_wcstombs (_backend->get_instance_name (ic->impl->id)));
        ic->impl->send.put_data (_backend->get_instance_icon_file (ic->impl->id));
    } else {
        ic->impl->send.put_data (String (_("English/Keyboard")));
        ic->impl->send.put_data (String (SCIM_KEYBOARD_ICON_FILE));
    }
}

static void
panel_req_focus_in (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_FOCUS_IN);
}

static void
panel_req_focus_out (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_FOCUS_OUT);
}

static void
panel_req_update_spot_location (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SPOT_LOCATION);
    ic->impl->send.put_data ((uint32) ic->impl->cursor_x);
    ic->impl->send.put_data ((uint32) ic->impl->cursor_y);
}


static void
panel_req_show_preedit_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_PREEDIT_STRING);
}

static void
panel_req_hide_preedit_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_PREEDIT_STRING);
}

static void
panel_req_update_preedit_string (GtkIMContextSCIM    *ic,
                                  const WideString    &str,
                                  const AttributeList &attrs)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING);
    ic->impl->send.put_data (utf8_wcstombs (str));
    ic->impl->send.put_data (attrs);
}

static void
panel_req_update_preedit_caret (GtkIMContextSCIM *ic,
                                int               caret)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET);
    ic->impl->send.put_data ((uint32) caret);
}

static void
panel_req_show_aux_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_AUX_STRING);
}

static void
panel_req_hide_aux_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_AUX_STRING);
}

static void
panel_req_update_aux_string (GtkIMContextSCIM    *ic,
                              const WideString    &str,
                              const AttributeList &attrs)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_AUX_STRING);
    ic->impl->send.put_data (utf8_wcstombs (str));
    ic->impl->send.put_data (attrs);
}

static void
panel_req_show_lookup_table (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE);
}

static void
panel_req_hide_lookup_table (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE);
}

static void
panel_req_update_lookup_table (GtkIMContextSCIM  *ic,
                               const LookupTable &table)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE);
    ic->impl->send.put_data (table);
}

static void
panel_req_register_properties (GtkIMContextSCIM   *ic,
                               const PropertyList &properties)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_REGISTER_PROPERTIES);
    ic->impl->send.put_data (properties);
}

static void
panel_req_update_property (GtkIMContextSCIM *ic,
                           const Property   &property)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_PROPERTY);
    ic->impl->send.put_data (property);
}

static gboolean
panel_source_prepare (GSource *source,
                       gint    *timeout)
{
    *timeout = -1;

    return FALSE;
}

static gboolean
panel_source_check (GSource *source)
{
    GSCIMPanelSource *panel_source = (GSCIMPanelSource *) source;

    if (panel_source && (panel_source->fd.revents & G_IO_IN))
        return TRUE;
    return FALSE;
}

static gboolean
panel_source_dispatch (GSource     *source,
                        GSourceFunc  callback,
                        gpointer     user_data)
{
    SCIM_DEBUG_FRONTEND(1) << "panel_source_dispatch..\n";

    GSCIMPanelSource *panel_source = (GSCIMPanelSource *) source;

    if (panel_source &&
        panel_source->context &&
        panel_source->context->impl &&
        panel_source->context->impl->panel.is_connected ()) {

        GtkIMContextSCIM *context = panel_source->context;

        if (!panel_check_connection (context))
            if (!panel_connect_server (context))
                return FALSE;

        panel_receive_reply (context);
        return TRUE;
    }
    return FALSE;
}

static bool
panel_check_connection (GtkIMContextSCIM *ic)
{
    unsigned char buf [sizeof(uint32)];

    int nbytes = 0;
 
    if (ic && ic->impl && ic->impl->panel.is_connected ())
        nbytes = ic->impl->panel.read_with_timeout (buf, sizeof(uint32), _panel_timeout);

    if (nbytes == sizeof (uint32))
        return true;

    return false;
}

static void
set_focus_ic (GtkIMContextSCIM *ic)
{
    if (!ic || !ic->impl || ic->impl->id < 0)
        return;

    if (_focused_ic && _focused_ic->impl && _focused_ic != ic)
        _backend->focus_out (_focused_ic->impl->id);

    _focused_ic = ic;

    panel_req_focus_in (ic);
    panel_req_update_display (ic);
    panel_req_update_screen (ic);
    panel_req_update_spot_location (ic);
    panel_req_update_factory_info (ic);

    if (ic->impl->is_on) {
        panel_req_turn_on_panel (ic);
        _backend->focus_in (ic->impl->id);
    } else {
        _backend->focus_out (ic->impl->id);
        panel_req_turn_off_panel (ic);
    }
}

static bool
match_key_event (const KeyEventVector &keys,
                 const KeyEvent       &key)
{
    KeyEventVector::const_iterator kit; 

    for (kit = keys.begin (); kit != keys.end (); ++kit) {
        if (key.code == kit->code &&
            (key.mask & kit->mask) == kit->mask &&
			(key.mask & SCIM_KEY_ReleaseMask) == (kit->mask & SCIM_KEY_ReleaseMask))
            return true;
    }
    return false;
}

static KeyEvent
keyevent_gdk_to_scim (const GdkEventKey &gdkevent)
{
    return KeyEvent (gdkevent.keyval,
                     (gdkevent.type == GDK_KEY_PRESS)?
                     (gdkevent.state & ~ SCIM_KEY_ReleaseMask):
                     (gdkevent.state | SCIM_KEY_ReleaseMask));
}

static guint32
get_time (void)
{
    guint32 tint;
    struct timeval tv;
    struct timezone tz;           /* is not used since ages */
    gettimeofday (&tv, &tz); 
    tint = (guint32) tv.tv_sec * 1000;
    tint = tint / 1000 * 1000; 
    tint = tint + tv.tv_usec / 1000;
    return tint;
}

static GdkKeymap *
get_gdk_keymap (GdkWindow *window)
{
    GdkKeymap *keymap;

#if GDK_MULTIHEAD_SAFE
    if (window)
        keymap = gdk_keymap_get_for_display (gdk_drawable_get_display (window));
    else
#endif
        keymap = gdk_keymap_get_default ();

    return keymap;
}

static void
keyevent_scim_to_gdk (GdkEventKey      &gdkevent,
                      const KeyEvent   &scimkey,
                      GtkIMContextSCIM *ic)
{
    gdkevent.type = ((scimkey.mask & SCIM_KEY_ReleaseMask) ? GDK_KEY_RELEASE : GDK_KEY_PRESS);
    gdkevent.window = ((ic && ic->impl) ? ic->impl->client_window : 0);
    gdkevent.send_event = TRUE;
    gdkevent.time = get_time ();
    gdkevent.state = scimkey.mask;
    gdkevent.keyval = scimkey.code;
    gdkevent.length = 0;
    gdkevent.string = 0;

    GdkKeymap *keymap = get_gdk_keymap (gdkevent.window);

    GdkKeymapKey *keys;
    gint          n_keys;

    if (gdk_keymap_get_entries_for_keyval (keymap, gdkevent.keyval, &keys, &n_keys)) {
        gdkevent.hardware_keycode = keys [0].keycode;
        gdkevent.group = keys [0].group;
    } else {
        gdkevent.hardware_keycode = 0;
        gdkevent.group = 0;
    }

    if (keys) g_free (keys);
}

static bool
check_socket_frontend ()
{
	SocketAddress address;
	SocketClient client;

    uint32 magic;

    address.set_address (scim_get_default_socket_frontend_address ());

    if (!client.connect (address))
        return false;

    if (!scim_socket_trans_open_connection (magic,
                                            String ("ConnectionTester"),
                                            String ("SocketFrontEnd"),
                                            client,
                                            1000)) {
        return false;
    }

    return true;
}

static void
initialize (void)
{
    std::vector<String>     config_list;
    std::vector<String>     engine_list;
    std::vector<String>     load_engine_list;

    std::vector<String>     debug_mask_list;
    int                     debug_verbose = 0;

    size_t                  i;

    bool                    manual = false;

    bool                    socket = true;

    // Get debug info
    {
        char *str;

        str = getenv ("GTK_IM_SCIM_DEBUG_VERBOSE");

        if (str != NULL)
            debug_verbose = atoi (str);

        DebugOutput::set_verbose_level (debug_verbose);

        str = getenv ("GTK_IM_SCIM_DEBUG_MASK");

        if (str != NULL) {
            scim_split_string_list (debug_mask_list, String (str), ',');
    		if (debug_mask_list.size ()) {
                DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
		    	for (i=0; i<debug_mask_list.size (); i++)
			    	DebugOutput::enable_debug_by_name (debug_mask_list [i]);
    		}
        }
    }

    //get modules list
    scim_get_imengine_module_list (engine_list);
    scim_get_config_module_list (config_list);

    if (std::find (engine_list.begin (), engine_list.end (), "socket") == engine_list.end () ||
        std::find (config_list.begin (), config_list.end (), "socket") == config_list.end ())
        socket = false;

    //Use simple config module as default if available.
    if (config_list.size ()) {
        _config_module_name = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));
        if (std::find (config_list.begin (), config_list.end (), _config_module_name) == config_list.end ())
            _config_module_name = config_list [0];
    } else {
        _config_module_name = "dummy";
    }

    const char *engine_list_str = getenv ("GTK_IM_SCIM_IMENGINE_MODULES");
    if (engine_list_str != NULL) {
        std::vector <String> spec_engine_list;
        scim_split_string_list (spec_engine_list, engine_list_str, ',');

        load_engine_list.clear ();

        for (size_t i = 0; i < spec_engine_list.size (); ++i)
            if (std::find (engine_list.begin (), engine_list.end (), spec_engine_list [i]) != engine_list.end ())
                load_engine_list.push_back (spec_engine_list [i]);

        manual = true;
    }

    // If you try to use the socket feature manually,
    // then let you do it by yourself.
    if (_config_module_name == "socket" ||
        std::find (load_engine_list.begin (), load_engine_list.end (), "socket") != load_engine_list.end ())
        socket = false;

    // Try to start a SCIM SocketFrontEnd process if necessary.
    if (scim_get_default_socket_frontend_address () != scim_get_default_socket_imengine_address () &&
        scim_get_default_socket_frontend_address () != scim_get_default_socket_config_address ())
        socket = false;

    if (socket) {
        // If no Socket FrontEnd is running, then launch one.
        // And set manual to false.
        if (!check_socket_frontend ()) {
            std::cout << "Launching a SCIM daemon with Socket FrontEnd...\n";
            char *new_argv [] = { "--no-stay", 0 };
            scim_launch (true,
                         _config_module_name,
                         (load_engine_list.size () ? scim_combine_string_list (load_engine_list, ',') : "all"),
                         "socket",
                         0,
                         "none",
                         "none",
                         new_argv);
            manual = false;
        }

        // If there is one Socket FrontEnd running and it's not manual mode,
        // then just use this Socket Frontend.
        if (!manual) {
            for (int i = 0; i < 100; ++i) {
                if (check_socket_frontend ()) {
                    _config_module_name = "socket";
                    load_engine_list.clear ();
                    load_engine_list.push_back ("socket");
                    break;
                }
                scim_usleep (100000);
            }
        }
    }

    //load config module
    SCIM_DEBUG_FRONTEND(1) << "Loading Config module: " << _config_module_name << "...\n";
    _config_module = new ConfigModule (_config_module_name);

    //create config instance
    if (_config_module != NULL && _config_module->valid ())
        _config = _config_module->create_config ("scim");
    else {
        SCIM_DEBUG_FRONTEND(1) << "Config module cannot be loaded, using dummy Config.\n";

        if (_config_module) delete _config_module;
        _config_module = NULL;

        _config = new DummyConfig ();
        _config_module_name = "dummy";
    }

    //Read settings.
    _panel_address = scim_get_default_panel_socket_address ();
    _panel_timeout = scim_get_default_socket_timeout ();

    // Init colors.
    gdk_color_parse ("gray92",     &_normal_bg);
    gdk_color_parse ("black",      &_normal_text);
    gdk_color_parse ("light blue", &_active_bg);
    gdk_color_parse ("black",      &_active_text);

    if (!_config.null ()) {
        scim_string_to_key_list (_trigger_keys,
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_TRIGGER),
                                       String ("Control+space")));

        scim_string_to_key_list (_next_factory_keys, 
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_NEXT_FACTORY),
                                       String ("Control+Alt+Down,Control+Shift_R,Control+Shift_L")));

        scim_string_to_key_list (_previous_factory_keys, 
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_FACTORY),
                                       String ("Control+Alt+Up,Shift+Control_R,Shift+Control_L")));

        scim_string_to_key_list (_show_factory_menu_keys, 
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_SHOW_FACTORY_MENU),
                                       String ("Control+Alt+l,Control+Alt+m,Control+Alt+s,Control+Alt+Right")));

        KeyEvent key;
        scim_string_to_key (key,
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_VALID_KEY_MASK),
                                       String ("Shift+Control+Alt+Lock")));

        _valid_key_mask = (key.mask > 0)?(key.mask):0xFFFF;
        _valid_key_mask |= SCIM_KEY_ReleaseMask;
    }

    _config->signal_connect_reload (slot (reload_config_callback));

    // create backend
    _backend = new CommonBackEnd (_config, load_engine_list.size () ? load_engine_list : engine_list);

    if (_backend.null () || _backend->number_of_factories () == 0) {
        fprintf (stderr, "GTK IM Module SCIM: Cannot create BackEnd Object!\n");
        _backend.reset ();
    } else {
        std::vector<String> uuids;
        _backend->get_factory_list (uuids);

        String lang = scim_get_locale_language (scim_get_current_locale ());
        String def_uuid = uuids [0];

        for (size_t i = 0; i < uuids.size (); ++i) {
            if (_backend->get_factory_language (uuids [i]) == lang) {
                def_uuid = uuids [i];
                break;
            }
        }

        _default_factory = scim_global_config_read (String (SCIM_GLOBAL_CONFIG_DEFAULT_IMENGINE_FACTORY) +
                                                    String ("/") + lang,
                                                    def_uuid);

        if (std::find (uuids.begin (), uuids.end (), _default_factory) == uuids.end ())
            _default_factory = def_uuid;

        _backend->signal_connect_show_preedit_string   (slot (slot_show_preedit_string));
        _backend->signal_connect_show_aux_string       (slot (slot_show_aux_string));
        _backend->signal_connect_show_lookup_table     (slot (slot_show_lookup_table));
        _backend->signal_connect_hide_preedit_string   (slot (slot_hide_preedit_string));
        _backend->signal_connect_hide_aux_string       (slot (slot_hide_aux_string));
        _backend->signal_connect_hide_lookup_table     (slot (slot_hide_lookup_table));
        _backend->signal_connect_update_preedit_caret  (slot (slot_update_preedit_caret));
        _backend->signal_connect_update_preedit_string (slot (slot_update_preedit_string));
        _backend->signal_connect_update_aux_string     (slot (slot_update_aux_string));
        _backend->signal_connect_update_lookup_table   (slot (slot_update_lookup_table));
        _backend->signal_connect_commit_string         (slot (slot_commit_string));
        _backend->signal_connect_forward_key_event     (slot (slot_forward_key_event));
        _backend->signal_connect_register_properties   (slot (slot_register_properties));
        _backend->signal_connect_update_property       (slot (slot_update_property));
    }

    _fallback_factory = new ComposeKeyFactory ();
    _fallback_instance = _fallback_factory->create_instance (String ("UTF-8"), 0);
    _fallback_instance->signal_connect_commit_string (slot (fallback_commit_string_cb));

    /* XXX:This is not recommended way!! */
    _snooper_id = gtk_key_snooper_install ((GtkKeySnoopFunc)gtk_scim_key_snooper, NULL);
    _snooper_installed = true;
}

static void
finalize (void)
{
    int i;

    SCIM_DEBUG_FRONTEND(1) << "Finalizing GTK2 SCIM IMModule...\n";

    gtk_key_snooper_remove(_snooper_id);

    if (_default_factory.length ()) {
        String locale = scim_get_current_locale ();
        scim_global_config_write (String (SCIM_GLOBAL_CONFIG_DEFAULT_IMENGINE_FACTORY) +
                                  String ("/") + scim_get_locale_language (locale),
                                  _default_factory);
    }

    _fallback_instance.reset ();

    _fallback_factory.reset ();

    SCIM_DEBUG_FRONTEND(2) << " Releasing BackEnd...\n";
    _backend.reset ();

    SCIM_DEBUG_FRONTEND(2) << " Releasing Config...\n";
    _config.reset ();

    if (_config_module) {
        SCIM_DEBUG_FRONTEND(2) << " Deleting _config_module...\n";
        delete _config_module;
    }
}

static void
open_next_factory (GtkIMContextSCIM *ic)
{
    String sf_uuid = _backend->get_instance_uuid (ic->impl->id);

    std::vector<String> uuids;

    _backend->get_factory_list (uuids);

    String new_uuid = uuids [0];

    for (size_t i = 0; i < uuids.size ()-1; ++i) {
        if (sf_uuid == uuids [i]) {
            new_uuid = uuids [i+1];
            break;
        }
    }

    if (_backend->replace_instance (ic->impl->id, new_uuid))
        _default_factory = new_uuid;
}

static void
open_previous_factory (GtkIMContextSCIM *ic)
{
    String sf_uuid = _backend->get_instance_uuid (ic->impl->id);

    std::vector<String> uuids;

    _backend->get_factory_list (uuids);

    String new_uuid = uuids [uuids.size ()-1];

    for (size_t i = uuids.size () - 1; i > 0; --i) {
        if (sf_uuid == uuids [i]) {
            new_uuid = uuids [i-1];
            break;
        }
    }

    if (_backend->replace_instance (ic->impl->id, new_uuid))
        _default_factory = new_uuid;
}

static void
open_specific_factory (GtkIMContextSCIM *ic,
                       const String     &uuid)
{
    if (_backend->replace_instance (ic->impl->id, uuid))
        _default_factory = uuid;
}

static String
get_help_info (GtkIMContextSCIM *ic)
{
    String help;
    String tmp;

    help =  String (_("Smart Common Input Method platform ")) +
            String (SCIM_VERSION) +
            String (_("\n(C) 2002-2004 James Su <suzhe@tsinghua.org.cn>\n\n"
                          "Hot keys:\n\n  "));

    scim_key_list_to_string (tmp, _trigger_keys);
    help += tmp + String (_(":\n    open/close the input method.\n\n  "));
    
    scim_key_list_to_string (tmp, _next_factory_keys);
    help += tmp + String (_(":\n    switch to the next input method.\n\n  "));

    scim_key_list_to_string (tmp, _previous_factory_keys);
    help += tmp + String (_(":\n    switch to the previous input method.\n\n\n"));

    if (ic && ic->impl) {
        help += utf8_wcstombs (_backend->get_instance_name (ic->impl->id));
        help += String (_(":\n\n"));

        help += utf8_wcstombs (_backend->get_instance_authors (ic->impl->id));
        help += String (_("\n\n"));

        help += utf8_wcstombs (_backend->get_instance_help (ic->impl->id));
        help += String (_("\n\n"));

        help += utf8_wcstombs (_backend->get_instance_credits (ic->impl->id));
    }
    return help;
}

// Implementation of slot functions
static void
slot_show_preedit_string (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_preedit_string...\n";

    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id) {
        if (_focused_ic->impl->use_preedit)
            g_signal_emit_by_name(_focused_ic, "preedit_changed");
        else
            panel_req_show_preedit_string (_focused_ic);
    }
}

static void 
slot_show_aux_string (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_aux_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_show_aux_string (_focused_ic);
}

static void 
slot_show_lookup_table (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_lookup_table...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_show_lookup_table (_focused_ic);
}

static void 
slot_hide_preedit_string (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_preedit_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id) {
        bool emit=false;
        if (_focused_ic->impl->preedit_string.length ()) {
            _focused_ic->impl->preedit_string = WideString ();
            _focused_ic->impl->preedit_caret = 0;
            _focused_ic->impl->preedit_attrlist.clear ();
            emit = true;
        }
        if (_focused_ic->impl->use_preedit) {
            if (emit) g_signal_emit_by_name(_focused_ic, "preedit_changed");
        } else {
            panel_req_hide_preedit_string (_focused_ic);
        }
    }
}

static void 
slot_hide_aux_string (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_aux_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_hide_aux_string (_focused_ic);
}

static void 
slot_hide_lookup_table (int id)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_lookup_table...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_hide_lookup_table (_focused_ic);
}

static void 
slot_update_preedit_caret (int id, int caret)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_preedit_caret...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id) {
        if (_focused_ic->impl->preedit_caret != caret) {
            _focused_ic->impl->preedit_caret = caret;
            if (_focused_ic->impl->use_preedit)
                g_signal_emit_by_name(_focused_ic, "preedit_changed");
            else
                panel_req_update_preedit_caret (_focused_ic, caret);
        }
    }
}

static void 
slot_update_preedit_string (int id,
                            const WideString & str,
                            const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_preedit_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id) {
        _focused_ic->impl->preedit_string   = str;
        _focused_ic->impl->preedit_attrlist = attrs;
        _focused_ic->impl->preedit_caret    = str.length ();
        if (_focused_ic->impl->use_preedit)
            g_signal_emit_by_name(_focused_ic, "preedit_changed");
        else
            panel_req_update_preedit_string (_focused_ic, str, attrs);
    }
}

static void 
slot_update_aux_string (int id,
                        const WideString & str,
                        const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_aux_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_update_aux_string (_focused_ic, str, attrs);
}

static void 
slot_commit_string (int id,
                    const WideString & str)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_commit_string...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        g_signal_emit_by_name (_focused_ic, "commit", utf8_wcstombs (str).c_str ());
}

static void 
slot_forward_key_event (int id,
                        const KeyEvent & key)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_forward_key_event...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id) {
        GdkEventKey gdkevent;
        keyevent_scim_to_gdk (gdkevent, key, _focused_ic);
        if (!_fallback_instance->process_key_event (key) &&
            !gtk_im_context_filter_keypress (GTK_IM_CONTEXT (_focused_ic), &gdkevent)) {
            ucs4_t ucs = key.get_unicode_code ();
            if (ucs && key.is_key_press ()) {
                unsigned char utf8[7];
                utf8 [utf8_wctomb (utf8, ucs, 7)] = 0;
                g_signal_emit_by_name (_focused_ic, "commit", utf8);
            }
        }
    }
}

static void 
slot_update_lookup_table (int id,
                          const LookupTable & table)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_lookup_table...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_update_lookup_table (_focused_ic, table);
}

static void 
slot_register_properties (int id,
                          const PropertyList & properties)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_register_properties...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_register_properties (_focused_ic, properties);
}

static void 
slot_update_property (int id,
                      const Property & property)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_property ...\n";
    if (_focused_ic && _focused_ic->impl && _focused_ic->impl->id == id)
        panel_req_update_property (_focused_ic, property);
}

static void
reload_config_callback (const ConfigPointer &config)
{
    SCIM_DEBUG_FRONTEND(1) << "reload_config_callback...\n";

    scim_string_to_key_list (_trigger_keys,
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_TRIGGER),
                      String ("Control+space")));

    scim_string_to_key_list (_next_factory_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_NEXT_FACTORY),
                      String ("Control+Alt+Down,Control+Shift_R,Control+Shift_L")));

    scim_string_to_key_list (_previous_factory_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_FACTORY),
                      String ("Control+Alt+Up,Shift+Control_R,Shift+Control_L")));

    scim_string_to_key_list (_show_factory_menu_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_SHOW_FACTORY_MENU),
                      String ("Control+Alt+l,Control+Alt+m,Control+Alt+s,Control+Alt+Right")));

    KeyEvent key;
    scim_string_to_key (key,
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_VALID_KEY_MASK),
                      String ("Shift+Control+Alt+Lock")));

    _valid_key_mask = (key.mask > 0)?(key.mask):0xFFFF;
    _valid_key_mask |= SCIM_KEY_ReleaseMask;
}

static void
fallback_commit_string_cb (IMEngineInstanceBase  *si,
                           const WideString      &str)
{
    if (_focused_ic && _focused_ic->impl)
        g_signal_emit_by_name (_focused_ic, "commit", utf8_wcstombs (str).c_str ());
}

/*
vi:ts=4:expandtab:nowrap
*/
