# Samizdat message controller
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/helpers/message_helper'

class MessageController < Controller
  include MessageHelper

  def index
    @request.redirect('message/publish')
  end

  def source
    @message = Message.cached(@id)
    @title = @message.content.title
    @content_for_layout = box(@title,
      form(nil,
        [:label, 'content', _('Content')],
        [:textarea, 'content', @message.content.body]))
  end

  def hide
    assert_moderate

    @message = Message.cached(@id)
    @message.assert_current_version

    if @request.has_key? 'confirm'
      save('hide') do
        @message.hide!(true)
      end
      @request.redirect(@id)
    else
      @title = _('Hide Message')
      @content_for_layout = box(@title,
        '<p class="moderation">' + _('The message will be hidden from public view.') + '</p>' +
        hide_message(Resource.new(@request, @id).full, true) +
        secure_form(nil, [:submit, 'confirm', _('Confirm')]))
    end
  end

  def unhide
    assert_moderate

    @message = Message.cached(@id)
    @message.assert_current_version

    if @request.has_key? 'confirm'
      save('unhide') do
        @message.hide!(false)
      end
      @request.redirect(@id)
    else
      @title = _('Unhide Message')
      @content_for_layout = box(@title,
        '<p class="moderation">' << _('The message will not be hidden from public view.') << '</p>' <<
        Resource.new(@request, @id).full <<
        secure_form(nil, [:submit, 'confirm', _('Confirm')]))
    end
  end

  def reparent
    assert_moderate

    @message = Message.cached(@id)
    @message.assert_current_version

    new_parent = @request['new_parent']
    new_parent &&= normalize_reference(new_parent)
    new_parent &&= @message.validate_reference(new_parent)

    if @request.has_key? 'confirm'
      save('reparent') do
        @message.reparent!(new_parent)
      end
      @request.redirect(@message.id)
    else
      @title = _('Reparent Message')
      @content_for_layout = box(@title,
        '<p class="moderation">'  << _('This message will be moved to new parent') << '</p>' <<
        Resource.new(@request, @id).full <<
        secure_form(nil,
          [:label, 'new_parent', _('New Parent')],
            [:text, 'new_parent', @message.parent],
          [:submit, 'confirm', _('Confirm')]))
    end
  end

  def publish
    assert_post

    @message = Message.new
    set_creator
    set_content
    set_lang
    set_desc
    set_open
    set_focus

    if @request.has_key? 'confirm'
      check_content
      save do
        @message.insert!
        @message.content.move_upload(@request)
        update_focus
      end
      @request.redirect(@message.id)
    elsif @request.has_key? 'preview'
      preview
    else
      @title = _('New Message')
      @content_for_layout = box(@title, form(nil, *edit_form(:show_focuses)))
    end
  end

  def reply
    assert_post

    @message = Message.new
    set_creator
    set_parent
    set_content
    set_lang(@parent.lang)
    set_desc
    set_open(@parent.open)

    if @request.has_key? 'confirm'
      check_content
      save do
        @message.insert!
        @message.content.move_upload(@request)
      end
      @request.redirect(location_under_parent)
    elsif @request.has_key? 'preview'
      preview
    else
      @title = _('Reply')
      @content_for_layout = box(@title, form(nil, *edit_form)) <<
        box(_('Parent Message'), Resource.new(@request, @parent.id).full)
    end
  end

  def edit
    assert_post

    @message = Message.cached(@id)
    if @session.member.nil? or
      (@message.creator.id != @session.member and not @message.open)

      raise UserError, _('You are not allowed to edit this message') 
    end
    @message.assert_current_version

    set_creator
    set_content
    set_lang(@message.lang)
    set_desc

    if @request.has_key? 'confirm'
      check_content
      save do
        version_id = @message.edit!
        @old_content.move_upload(@request, @id, version_id)
        @message.content.move_upload(@request)
      end
      @request.redirect(@message.id)
    elsif @request.has_key? 'preview'
      preview
    else
      @title = _('Edit Message')
      @content_for_layout = box(@title, form(nil, *edit_form(:disable_open)))
    end
  end

  def takeover
    assert_moderate

    @message = Message.cached(@id)
    @message.assert_current_version

    set_content
    set_lang(@message.lang)
    set_desc
    set_open

    if @request.has_key? 'confirm'
      check_content
      save('takeover') do
        version_id = @message.edit!
        @old_content.move_upload(@request, @id, version_id)
        @message.content.move_upload(@request)
      end
      @request.redirect(@message.id)
    elsif @request.has_key? 'preview'
      preview
    else
      @title = _('Take Over Message')
      @content_for_layout =
        '<p class="moderation">' <<
        _('Edit message content and open-for-all status, message will remain attributed to the current creator.') <<
        '</p>' <<
        box(@title, form(nil, *edit_form))
    end
  end

  def replace
    assert_moderate

    @message = Message.cached(@id)
    set_content
    set_lang(@message.lang)
    set_desc
    set_open(false)

    if @request.has_key? 'confirm'
      check_content
      save('replace') do
        @message.replace!
        # fixme: remove old file
        @message.content.move_upload(@request)
      end
      @request.redirect(@message.id)
    elsif @request.has_key? 'preview'
      preview
    else
      @title = _('Replace Message')
      @content_for_layout =
        '<p class="moderation">' <<
        _('MESSAGE WILL BE COMPLETELY REPLACED, NO RECOVERY WILL BE POSSIBLE. PLEASE PROVIDE DETAILED JUSTIFICATION FOR THIS ACTION.') <<
        '</p>' <<
        box(@title, form(nil, *edit_form))
    end
  end

  def request_moderation
    assert_post

    if @request.has_key? 'confirm'
      save do
        Moderation.request!(@id)
      end
      @request.redirect(@id)

    else
      Moderation.check_request(@id)

      @title = _('Request Moderation')
      @content_for_layout = box(@title,
        '<p>' <<
        _('Please confirm that you want to request moderation of this message:') <<
        '</p>' <<
        Resource.new(@request, @id).full <<
        secure_form(nil, [:submit, 'confirm', _('Confirm')]))
    end
  end

  def acknowledge
    assert_moderate

    if @request.has_key? 'confirm'
      save('acknowledge') { }
      @request.redirect_when_done

    else
      unless @request.cookie('redirect_when_done')
        @request.set_cookie('redirect_when_done', @request.referer)
      end

      @title = _('Acknowledge Moderation Request')
      @content_for_layout = box(@title,
        '<p class="moderation">' <<
        _('Moderation request will be marked as acknowledged without moderatorial action.') <<
        '</p>' <<
        Resource.new(@request, @id).full <<
        secure_form(nil, [:submit, 'confirm', _('Confirm')]))
    end
  end

  private

  # check if user is allowed to post messages
  #
  def assert_post
    @request.access('post') or raise AuthError,
      sprintf(_('Your current access level (%s) does not allow to publish messages'), _(@request.role))
  end

  FORM_ATTRIBUTES = %w[lang desc open]
  REFERENCE_PARAMS = %w[desc new_parent]

  def set_creator
    @message.creator = Member.cached(@session.member)
  end

  def set_parent
    @parent = Message.cached(@id)
    @parent.assert_current_version
    @message.parent = @id
  end

  def location_under_parent
    if not @parent.kind_of? Message
      @message.id
    elsif @parent.nreplies + 1 > limit_page
      "resource/#{@parent.id}?page=#{((@parent.nreplies + 1) / limit_page) + 1}#id#{@message.id}"
    else
      "#{@parent.id}#id#{@message.id}"
    end
  end

  def set_content
    title, format, body, @upload = @request.values_at %w[title format body upload]

    if title.nil? and @parent
      title = @parent.content.title
    end

    new_content = Content.new(nil, @session.login, title, format, body)
    @old_content = (@message.content or new_content)
    @message.content = new_content
  end

  def set_lang(default = nil)
    @message.lang = @message.validate_lang(
      (@request['lang'] or default or @request.language))
  end

  # replace request base prefix in place
  #
  def normalize_reference(ref)
    if ref.kind_of?(String) and ref =~ /[^0-9]/
      ref = ref.gsub(Regexp.new('\A' + Regexp.escape(@request.base) + '(.+)\z'), '\1')
    end

    ref
  end

  def set_desc
    # fixme: this will not allow to blank it
    @message.desc ||= @message.validate_desc(
      normalize_reference(@request['desc']))
  end

  def set_open(default = false)
    @message.open = @message.validate_open(
      (@request['open'] or default))
  end

  def set_focus
    return unless @request.access('vote')

    @focus, = @request.values_at %w[focus]
    @focus = Resource.validate_id(@focus)

    if @focus
      @message.focuses = [@focus]   # get it displayed in preview page
    end
  end

  def check_content
    @message.content.title or raise UserError,
      _('Message title is required')
    @upload or @message.content.body.kind_of? String or raise UserError,
      _('Message body is required')

    persistent_cache.fetch_or_add('antispam:' + site_name) {
      Antispam.new(config['antispam'])
    }.spam?(@request.role, @message.content.body) and
      raise SpamError, _('Your message looks like spam')
  end

  def edit_form(*options)
    fields = [
      [:label, 'title', _('Title')],
        [:text, 'title', @old_content.title],
      [:label, 'body', _('Content')],
        [:textarea, 'body', @old_content.body]
    ]

    # file upload: list of supported formats
    formats = (config['format']['image'].to_a + config['format']['other'].to_a).collect {|format|
      Content.format_extension(format)
    }.join(', ')

    fields.push(
      [:label, 'file', sprintf(_('Alternatively, upload a file (formats supported: %s)'), formats)],
        [:file, 'file']
    )

    if options.include? :show_focuses
      # select focus for new message
      fields.push(*focus_select)
    end

    # more optional options
    fields.push(
      [:label, 'lang', _('Language of the message')],
        [:select, 'lang',
          ([@request.language] + config['locale']['languages'].to_a).uniq!,
          @message.lang],
      [:label, 'format', _('Format')],
        [:select, 'format', [[nil, _('Default')]] +
          config['format']['inline'].to_a, @old_content.format ],
      [:label, 'open', _("Editing is open for all members (excluding guests who can't edit anyone's messages)")],
        [:checkbox, 'open', @message.open,
          (options.include?(:disable_open) ? :disabled : nil)]
    )

    # advanced parameters
    if @request.advanced_ui?
      fields.push(
        [:label, 'desc',
          _('Reference to description (ID or URL of another message on this site)')],
          [:text, 'desc', @message.desc]
      )
    end

    fields.push([:br], [:submit, 'preview', _('Preview')])

    fields
  end

  def preview
    @upload ||= @message.content.upload(@request)
    check_content

    body = @message.content.body
    if body and body != limit_string(body, config['limit']['short'])
      cut_warning = '<p>'+sprintf(_('Warning: content is longer than %s characters. In some situations, it will be truncated.'), config['limit']['short'])+'</p>'
    end

    @title = _('Message Preview')
    @content_for_layout = box(
      @message.content.title,
      message(@message, :full) <<
        cut_warning.to_s <<
        '<p>' << _("Press 'Back' button to change the message.") << '</p>' <<
        secure_form(
          nil,
          [:submit, 'confirm', _('Confirm')],
          [:hidden, 'title', @message.content.title],
          [:hidden, 'upload', @upload],
          [:hidden, 'body', @upload ? nil : body],
          [:hidden, 'focus', @focus],
          [:hidden, 'lang', @message.lang],
          [:hidden, 'format', @message.content.format],
          [:hidden, 'desc', @message.desc],
          [:hidden, 'open', @message.open],
          [:hidden, 'action', @action.to_s]
        )
    )
  end

  def update_focus
    if @focus
      # when publishing new message, focus rating is always '1'
      Focus.new(@request, @focus, @message.id).rating = 1
    end
  end

  # wrap save actions in a transaction, log moderatorial action if requested,
  # and ensure cache is flushed
  #
  def save(log_action = nil)
    return unless action_confirmed?
    db.transaction do |db|
      log_moderation(log_action) if log_action
      yield
      cache.flush
    end
  end
end
