
# Copyright (C) 2003 Laurent Sansonetti <lrz@gnome.org>
#
# This file is part of Ruby/GStreamer.
# 
# Ruby/GStreamer is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# Ruby/GStreamer is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with Ruby/GStreamer; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

require 'tests/misc'

class TC_clock < Test::Unit::TestCase
    def debug(c)
        puts "Clock info: time #{c.time}" if $VERBOSE
    end
    def test_system_clock
        assert_instance_of(Gst::Clock, c = Gst::SystemClock.obtain)
        c.test(self)
    end
    def test_desynch_clock
      # Test that the clock actually runs at normal speed
        src  = Gst::ElementFactory.make("fakesrc",  "source")
        id   = Gst::ElementFactory.make("identity", "filter")
        sink = Gst::ElementFactory.make("fakesink", "sink")
        
        pipeline = Gst::Pipeline.new("pipeline")
        
        src >> id >> sink
        pipeline.add(src, id, sink) 
        
        assert_instance_of(Gst::Clock, clock = pipeline.clock)
        debug(clock)        
        
        pipeline.play
        debug(clock)        
        t = clock.time
        5.times do |x|
            sleep 1 
            a = t + (x.succ - 0.1) * Gst::SECOND
            b = t + (x.succ + 0.1) * Gst::SECOND 
            debug(clock)
            assert clock.time >= a
            assert clock.time <= b
        end
        pipeline.stop
    end

    def test_timer_sync
        ret_codes = [
            Gst::Clock::OK,
            Gst::Clock::EARLY,
            Gst::Clock::UNSCHEDULED,
            Gst::Clock::BUSY,
            Gst::Clock::BADTIME,
            Gst::Clock::ERROR,
            Gst::Clock::UNSUPPORTED
        ]   
        clock = Gst::SystemClock.obtain
        assert_instance_of(Gst::Clock, clock)

        # one shot
        id = Gst::ClockEntry.new(clock, 2 * Gst::SECOND)
        assert id.single_shot?
        assert !id.periodic?
        assert_equal(2 * Gst::SECOND, id.time)
        assert_equal(Gst::Clock::Time::NONE, id.interval)  
        assert_equal(clock, id.clock)
        [ 2, 0, 0, 0 ].each do |desired_interval|
            now = Time.now.to_i
            assert ret_codes.include?(c = id.wait)
            elapsed = Time.now.to_i - now
            # FIXME make a better test
            #assert_equal(desired_interval, elapsed)
        end

        # periodic (does not work yet, because of 'interval' bug.
        # http://bugzilla.gnome.org/show_bug.cgi?id=121015

#        clock.reset!
#        clock.active = true
#        id = Gst::ClockEntry.new(clock, 1 * Gst::SECOND, 2 * Gst::SECOND)
#        assert id.periodic?
#        assert !id.single_shot?
#        assert_equal(2 * Gst::SECOND, id.time)
#        assert_equal(1 * Gst::SECOND, id.interval)
#        assert_equal(clock, id.clock)
#        [ 2, 2 ].each do |desired_interval|
#            now = Time.now.to_i
#            assert ret_codes.include?(c = id.wait)
#            elapsed = Time.now.to_i - now
#            assert_equal(desired_interval, elapsed)
#        end
    end
end

