
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org> and
 * Jim Meier <fatjim@home.com>
 *
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */


#include <roy.h>



/* ThermlState state/condition/action table (state digraph) */

static ThermlState states[] = {
  {0, NULL,
   {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {1, "opening node",
   {'<', ACT_R, 2}, {ANYTH, ACT_A, 1}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {2, "one of '/?!', or element name",
   {'/', ACT_H, 29}, {'?', ACT_D, 13}, {'!', ACT_E, 8}, {NWHSP, ACT_F, 3}, {-1, 0, 0}},

  {3, "element or node closing",
   {WHTSP, ACT_B, 4}, {'>', ACT_I, 1}, {'/', ACT_H, 15}, {ANYTH, ACT_G, 3}, {-1, 0, 0}},

  {4, "new attribute or node closing",
   {WHTSP, ACT_B, 4}, {'>', ACT_I, 1}, {'/', ACT_H, 16}, {ANYTH, ACT_J, 5}, {-1, 0, 0}},

  {5, "'=(' or whitespace",
   {'=', ACT_B, 6}, {'(', ACT_B, 18}, {WHTSP, ACT_B, 28}, {'>', ACT_C, 1}, {ANYTH, ACT_K, 5}},

  {6, "whitespace or value quotation character",
   {WHTSP, ACT_B, 6}, {ANYTH, ACT_Q, 7}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {7, "end of attribute delimiter",
   {ENDAT, ACT_B, 4}, {ANYTH, ACT_L, 7}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {8, "'-' opening comment character (1st of two)",
   {'-', ACT_B, 9}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {9, "'-' opening comment character (2nd of two)",
   {'-', ACT_B, 10}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {10, "'-' close comment character, or more comment",
   {'-', ACT_B, 11}, {ANYTH, ACT_B, 10}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {11, "'-' close comment character or more comment",
   {'-', ACT_B, 12}, {ANYTH, ACT_B, 10}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {12, "'>' close comment character or more comment",
   {'>', ACT_I, 1}, {ANYTH, ACT_B, 10}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {13, "'?>' closing processing command sequence",
   {'?', ACT_B, 14}, {ANYTH, ACT_A, 13}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {14, "'>' end of node character",
   {'>', ACT_I, 1}, {ANYTH, ACT_A, 13}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {15, "'>' end of element, or string",
   {'>', ACT_I, 1}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {16, "'>' node closure character",
   {'>', ACT_I, 1}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {17, "element name for closing node",
   {'>', ACT_I, 1}, {ANYTH, ACT_G, 17}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {18, "end quotation character, or numeric length specifier",
   {DIGIT, ACT_M, 19}, {ENDCH, ACT_O, 22}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {19, "')' or numeric length specifier",
   {')', ACT_N, 20}, {DIGIT, ACT_M, 19}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {20, "'=' character after numeric length specifier",
   {'=', ACT_B, 26}, {WHTSP, ACT_B, 20}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {21, "more attribute value characters",
   {CNTR1, ACT_L, 27}, {ANYTH, ACT_L, 21}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {22, "')' or end quote character string",
   {')', ACT_B, 23}, {ENDCH, ACT_O, 22}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {23, "'=' character after end quote sequence specifier",
   {'=', ACT_B, 25}, {WHTSP, ACT_B, 23}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {24, "more value string, or end quotation sequence",
   {ENDT1, ACT_B, 4}, {ANYTH, ACT_L, 24}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {25, "starting '\"' double quote before value string",
   {'"', ACT_B, 24}, {WHTSP, ACT_B, 25}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {26, "starting '\"' double quote before length specified value string",
   {'"', ACT_B, 21}, {WHTSP, ACT_B, 26}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {27, "ending '\"' double quote after length specified value string",
   {'"', ACT_B, 4}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}},

  {28, "whitespace after attribute name or '='",
   /* {WHTSP, ACT_B, 28}, {'=', ACT_B, 6}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}}, */
   {WHTSP, ACT_B, 28}, {'=', ACT_B, 6}, {NWHSP, ACT_P, 4}, {-1, 0, 0}, {-1, 0, 0}},

  {29, "ending node name",
   {NWHSP, ACT_G, 17}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}, {-1, 0, 0}}
    
};


static void
act_a_func (ThermlParser *parser, char c);

static void
act_b_func (ThermlParser *parser, char c);

static void
act_c_func (ThermlParser *parser, char c);

static void
act_d_func (ThermlParser *parser, char c);

static void
act_e_func (ThermlParser *parser, char c);

static void
act_f_func (ThermlParser *parser, char c);

static void
act_g_func (ThermlParser *parser, char c);

static void
act_h_func (ThermlParser *parser, char c);

static void
act_i_func (ThermlParser *parser, char c);

static void
act_j_func (ThermlParser *parser, char c);

static void
act_k_func (ThermlParser *parser, char c);

static void
act_l_func (ThermlParser *parser, char c);

static void
act_m_func (ThermlParser *parser, char c);

static void
act_n_func (ThermlParser *parser, char c);

static void
act_o_func (ThermlParser *parser, char c);

static void
act_p_func (ThermlParser *parser, char c);

static void
act_q_func (ThermlParser *parser, char c);

static void
act_r_func (ThermlParser *parser, char c);

static void
error_act (ThermlParser *parser, char c);

static ThermlActFuncEntry action_table[] = {
    {error_act,  "Error Act",},
    {act_a_func, "char -> data"},
    {act_b_func, "NOP"},
    {act_c_func, "close tag after valueless attrib"}, /* ACT_C is no longer used */
    {act_d_func, "flag pcmd"},
    {act_e_func, "flag comment"},
    {act_f_func, "flag start"},
    {act_g_func, "char -> nodename"},
    {act_h_func, "flag end"},
    {act_i_func, "signal"},
    {act_j_func, "new attr"},
    {act_k_func, "char->attr name"},
    {act_l_func, "char->attr val"},
    {act_m_func, "char->countstr"},
    {act_n_func, "count conv"},
    {act_o_func, "char->endstr"},
    {act_p_func, "chop attr val"},
    {act_q_func, "char -> endchar"},
    {act_r_func, "push new node"}
};

static void
therml_node_free (ThermlNode *node);

static void
therml_node_new (ThermlParser *parser);

static void
therml_attrib_value_pair_done (ThermlParser *parser);

static void 
push_char (ThermlParser *parser, char c);

static int 
try_match (ThermlParser *parser, char c, int match);

static int 
check_end_quote_match (ThermlParser *parser, char c);

static void 
error_act (ThermlParser *parser, char c);

static void
signal_error (ThermlParser *parser, int severity, const char *description);


void
therml_parser_free (ThermlParser *parser)
{
    ThermlNode *tnode;

    while (parser->nodes) {

        tnode = parser->nodes;
        therml_node_free (tnode);

        parser->nodes = parser->nodes->prev;
    }

    if (parser->current_attribute)
        rbuf_free (parser->current_attribute);

    if (parser->current_value)
        rbuf_free (parser->current_value);

    if (parser->count_string)
        rbuf_free (parser->count_string);

    if (parser->end_of_attribute_string)
        rbuf_free (parser->end_of_attribute_string);


    rmem_free (parser);
}


ThermlParser *
therml_parser_new (void)
{
    ThermlParser *parser = rmem_alloc0 (sizeof (ThermlParser));

    parser->state = 1;
    parser->linenum = 1;
    parser->last_state_linenum = 1;

    parser->current_attribute = rbuf_new ();
    parser->current_value = rbuf_new ();

    parser->count_string = rbuf_new ();
    parser->end_of_attribute_string = rbuf_new ();

    return (parser);
}

void
therml_set_user_data (ThermlParser * parser, void *user_data)
{
    parser->user_data = user_data;
}

void *
therml_get_user_data (ThermlParser * parser)
{
    return parser->user_data;
}

void
therml_set_error_handler (ThermlParser * parser, ThermlErrorHandler handler)
{
    parser->error_handler = handler;
}

void
therml_set_pcommand_handler (ThermlParser * parser,
    ThermlPCommandHandler handler)
{
    parser->pcommand_handler = handler;
}

void
therml_set_node_start_handler (ThermlParser * parser,
    ThermlNodeStartHandler handler)
{
    parser->node_start_handler = handler;
}

void
therml_set_node_end_handler (ThermlParser * parser, ThermlNodeEndHandler handler)
{
    parser->node_end_handler = handler;
}


int
therml_parse_chunk (ThermlParser *parser, char *chunk, int len,
                    int is_last_chunk)
{
    int rem = len;
    char *p = chunk;

    while (rem > 0) 
    {
        push_char (parser, *p);
        if (parser->fatal_error) {
            parser = NULL;					
            return (-1);
        }

        p++;
        rem--;
    }

    /* Various special things to do if this is the last chunk */
    if (is_last_chunk) {

        /* if this is the last chunk, we should be back to parser state 1 */
        if (parser->state != 1) {
            char buf[2048];
            ThermlState *tstate = &states[parser->state];
            snprintf (buf, sizeof (buf), 
                      "Parse error. "
                      "End of document reached, but still expecting %s. "
                      "Started line %d, column %d",
                      tstate->description, parser->last_state_linenum, 
                      parser->last_state_colnum-1);
            signal_error (parser, 99, buf);
        }

        /* check node stack to see if we didn't have some closed */
        if (parser->nodes != NULL) {
            signal_error (parser, 99, "Parse error. "
                          "End of document reached, but some nodes not closed.");
        }
    }

    parser = NULL;	

    return (0);
}


#if DEBUG

static void
print_node (ThermlNode * tnode)
{
    ThermlAttribute *attrib;
    RBuf *flags = rbuf_new ();

    if (!tnode) {
        printf ("NULL node.\n");
        return;
    }


    printf ("ThermlNode %s\n", rbuf_str(tnode->type));

    if (RFLAG_ISSET (tnode, NODE_TYPE_PCMD))
        rbuf_append_str (flags, "PCOMMAND ");

    if (RFLAG_ISSET (tnode, NODE_TYPE_COMMENT))
        rbuf_append_str (flags, "COMMENT ");

    if (RFLAG_ISSET (tnode, NODE_TYPE_START))
        rbuf_append_str (flags, "START ");

    if (RFLAG_ISSET (tnode, NODE_TYPE_END))
        rbuf_append_str (flags, "END ");

    if (RFLAG_ISSET (tnode, NODE_TYPE_PARTIAL))
        rbuf_append_str (flags, "PARTIAL ");

    printf ("\tflags %s\n", flags->str);
    rbuf_free (flags);

    RBHASH_FOREACH (tnode->attribs, attrib) {
        
        printf ("  '%s'='%s'\n",
                rbuf_str (rbhash_entry_getkey (attrib)),
                rbuf_str (attrib->value));
        
    } RFOREACH_CLOSE;

    printf ("DATA: '%s'\n\n", tnode->data ? rbuf_str (tnode->data) : "(NULL)");
}


#endif

static void
push_char (ThermlParser *parser, char c)
{
    ThermlState *tstate = &states[parser->state];
    int prev_state;
    ThermlTransition *ttrans;
    int i;
    char buf[2048];

#if DEBUG
    printf ("**parsing char '%c'**\n", c);
#endif

    /* keep track of what line/column we are on */
    if (c == '\n' || c == '\r') {
        parser->linenum++;
        parser->colnum = 0;
    } else {
        parser->colnum++;
    }

    prev_state = parser->state;

    /* which transition matches? */
    for (i = 5, ttrans = &(tstate->t1); i--; ttrans++)
        if (try_match (parser, c, ttrans->match)) {
#if DEBUG
            printf ("match: %2d(%s):%c:ACT_%c(%s):%2d(%s)\n", 
                    parser->state, states[parser->state].description,
                    c, 'A' - 1 + ttrans->act,
                    action_table[ttrans->act].description,
                    ttrans->newstate,
                    states[ttrans->newstate].description);
#endif
            /* dispatch action */
            action_table[ttrans->act].act(parser, c);

#if DEBUG
            print_node (parser->nodes);
#endif

            /* If the state changes, keep track of the last line num
             * where it did.  For better error reporting */
            parser->state = ttrans->newstate;
            if (prev_state != parser->state) {
                parser->last_state_linenum = parser->linenum;
                parser->last_state_colnum =  parser->colnum;
            }

            return;
        }

    /* reaching this means no match. */
    snprintf (buf, sizeof (buf), "Parse error. Found '%c', expected %s.", c,
              tstate->description);
    signal_error (parser, 99, buf);
    parser->state = 1;		/* return to initial state */
}

static void
signal_error (ThermlParser *parser, int severity, const char *description)
{
    int col = parser->colnum;

    if (severity > 50)
        parser->fatal_error = 1;

    /* ensure that col points to the charater, not
     * to the next one. */


    if (col > 0)
        col--;	
    
    if (parser->error_handler) {
        parser->error_handler (parser, severity, 
                               parser->linenum, col,
                               description);
    }
}



static int
try_match (ThermlParser *parser, char c, int match)
{
    if (c == match)
        return 1;

    switch (match)
    {
        case ANYTH:
            break;
        case ENDAT:
            if (c != parser->quote_char) {
                return 0;
            } else {
                therml_attrib_value_pair_done (parser);
            }
            break;
        case ALPHA:
            if (!isalpha (c))
                return 0;
            break;
        case WHTSP:
            if (!isspace (c))
                return 0;
            break;
        case NWHSP:
            if (isspace (c))
                return 0;
            break;			
        case DIGIT:
            if (!isdigit (c))
                return 0;
            break;
        case ENDCH:
            /* the only char not allowed in attrib end
             * markers is ')' */
            if (c == ')')
                return 0;
            break;
        case CNTR1:
            if (rbuf_len (parser->current_value) < parser->value_length) {
                return 0;
            } else {
                rbuf_append_char (parser->current_value, c);
                therml_attrib_value_pair_done (parser);
            }
            break;
        case ENDT1:
            if (!check_end_quote_match (parser, c)) {
                return 0;
            } else {

                /* Adjust length to remove the previous characters of the
                 * end of attribute string. */
                int len = rbuf_len (parser->current_value) -
                          rbuf_len (parser->end_of_attribute_string) + 1;
                
                if (len < 0)
                    len = 0;
            
                rbuf_truncate (parser->current_value, len);

                therml_attrib_value_pair_done (parser);
            }
            break;
        default:
            return 0;
    }

    return 1;
}

static int
check_end_quote_match (ThermlParser *parser, char c)
{
    char *endstr = rbuf_str (parser->end_of_attribute_string);
    char *val = rbuf_str (parser->current_value);
    int endlen = rbuf_len (parser->end_of_attribute_string);
    int vallen = rbuf_len (parser->current_value);
    int l = 1;

    if (endlen > vallen)
        return 0;

    if (c != endstr[endlen - l])
        return 0;

    l++;

    while (l <= endlen) {
        if (val[vallen + 1 - l] != endstr[endlen - l])
            return 0;
        l++;
    }

    return 1;
}

static void
therml_attrib_value_pair_done (ThermlParser *parser)
{
    ThermlAttribute *attrib;

    attrib = rchunk_alloc (sizeof (ThermlAttribute));
    
    attrib->value = parser->current_value;
    rbhash_insert (parser->nodes->attribs, attrib, parser->current_attribute);
   
    parser->current_value = rbuf_new ();
    parser->current_attribute = rbuf_new ();

    rbuf_truncate (parser->end_of_attribute_string, 0);
    rbuf_truncate (parser->count_string, 0);
}


static void
therml_node_free (ThermlNode *tnode)
{
    ThermlAttribute *attrib;

    /* free all attributes */    
    RBHASH_FOREACH (tnode->attribs, attrib) {
        rbuf_free (rbhash_entry_getkey (attrib));
        rbuf_free (attrib->value);

        rchunk_free (attrib, sizeof (RBHashEntry));
    } RFOREACH_CLOSE;

    if (tnode->type)
        rbuf_free (tnode->type);

    if (tnode->data)
        rbuf_free (tnode->data);

    rmem_free (tnode);
}


static void
therml_node_new (ThermlParser *parser)
{
    ThermlNode *node;
    
    node = rmem_alloc0 (sizeof (ThermlNode)); 
    node->type = rbuf_new ();
    node->attribs = rbhash_new ();

    node->data = rbuf_new ();

    node->prev = parser->nodes;
    parser->nodes = node;
}

static ThermlNode *
therml_node_pop (ThermlParser *parser)
{
    ThermlNode *node;

    /* keep the current node for later free'ing */
    node = parser->nodes;
    /* pop to next node on stack */
    parser->nodes = parser->nodes->prev;

    return (node);
}


/* ACTION FUNCTIONS */

/* A - Add char to 'data' attribute of cur node  */
static void
act_a_func (ThermlParser *parser, char c)
{
    if (!parser->nodes)
        return;

    if (!parser->nodes->data)
        return;

    rbuf_append_char (parser->nodes->data, c);
}


/* B - NOP */
static void
act_b_func (ThermlParser *parser, char c)
{
    return; /* do nothing */
}

static void
act_i_func (ThermlParser *parser, char c);

/* C - close node after valueless attrib */
static void
act_c_func (ThermlParser *parser, char c)
{
    therml_attrib_value_pair_done (parser);

    act_i_func (parser, c);
}


/* D - mark cur node pcmd */
static void
act_d_func (ThermlParser *parser, char c)
{
    RFLAG_SET (parser->nodes, NODE_TYPE_PCMD);
}

/* E - mark cur node comment   */
static void
act_e_func (ThermlParser *parser, char c)
{
    RFLAG_SET (parser->nodes, NODE_TYPE_COMMENT);
}

/* F - mark cur node start  */
static void
act_f_func (ThermlParser *parser, char c)
{
    RFLAG_SET (parser->nodes, NODE_TYPE_START);
    rbuf_append_char (parser->nodes->type, c);
}


/* G - Add char to cur node type */
static void
act_g_func (ThermlParser *parser, char c)
{
    rbuf_append_char (parser->nodes->type, c);
}

/* H - Mark cur node end */
static void
act_h_func (ThermlParser *parser, char c)
{
    RFLAG_SET (parser->nodes, NODE_TYPE_END);
}


/* I - signal this node (if end node or comment or pcmd, pop cur node) */
/*     also verify path (ie, pop the last path, and if we have a specified path
       here ensure that it is consistent) if this is a closing node*/
static void
act_i_func (ThermlParser *parser, char c)
{
    unsigned int was_start_node = FALSE;

    /* node popping rules */
    if (RFLAG_ISSET (parser->nodes, NODE_TYPE_PCMD) ||
        RFLAG_ISSET (parser->nodes, NODE_TYPE_COMMENT)) {

        /* pop pcommands and comments */
        ThermlNode *node;
        node = therml_node_pop (parser);
        therml_node_free (node);

        return;

    }


    if (RFLAG_ISSET (parser->nodes, NODE_TYPE_START)) {

        /* signal events */
        if (parser->node_start_handler) {

            parser->node_start_handler (parser, parser->nodes->type,
                parser->nodes->attribs);
        }

        was_start_node = TRUE;

        RFLAG_UNSET (parser->nodes, NODE_TYPE_START);
    }

    if (RFLAG_ISSET (parser->nodes, NODE_TYPE_END)) {
        ThermlNode *current_node;
        ThermlNode *popped_node;

        popped_node = therml_node_pop (parser);
        current_node = parser->nodes;

        /* If this is not a self-closing node, we check for
         * proper nesting. */
        if (!was_start_node) {
            int ret;

            /* nesting checks: node name check */
            ret = rbuf_equal_rbuf (current_node->type, popped_node->type);
            if (!ret) {
                char buf[1024];

                snprintf (buf, sizeof (buf), "closing element '%s' did not match expected close of element '%s'",
                    rbuf_str (popped_node->type), rbuf_str (current_node->type));
                signal_error (parser, 100, buf);
                return;
            }
        }

        if (parser->nodes)
            RFLAG_UNSET (parser->nodes, NODE_TYPE_END);


        if (!was_start_node) {
            therml_node_free (popped_node);
            /* The previous pop just pops the 'end node' which is the node containing the
             * type of the closing node.  This one actually pops the original node */
            popped_node = therml_node_pop (parser);

            if (parser->node_end_handler) {
                parser->node_end_handler (parser, popped_node->type, popped_node->data);
            }

            therml_node_free (popped_node);
        } else {
            if (parser->node_end_handler) {
                parser->node_end_handler (parser, popped_node->type, popped_node->data);
            }
            therml_node_free (popped_node);
        }
    }
}


/* J - add attr to cur node, name = char  */
static void
act_j_func (ThermlParser *parser, char c)
{
    /* Just keep track of it so we can use it when the attribute
     * is done */
    rbuf_append_char (parser->current_attribute, c);
}

/* K - add char to cur attribute name */
static void
act_k_func (ThermlParser *parser, char c)
{
    if (c == '>') {
        therml_attrib_value_pair_done (parser);
    }
    rbuf_append_char (parser->current_attribute, c);
}

/* L - add char to cur attr val */
static void
act_l_func (ThermlParser *parser, char c)
{
    rbuf_append_char (parser->current_value, c);
}


/* M - add char to count_str of cur attr */
static void
act_m_func (ThermlParser *parser, char c)
{
    rbuf_append_char (parser->count_string, c);
}


/* N - turn count_str to count  */
static void
act_n_func (ThermlParser *parser, char c)
{
    parser->value_length = atoi (rbuf_str (parser->count_string)) - 1;

#if DEBUG
    printf ("the attribute %s's val should be of length %d +1 (string was %s)\n",
            rbuf_str (parser->current_attribute), parser->value_length,
            rbuf_str (parser->count_string));
#endif
}

/* O - add char to end of attribute string for current attribute */
static void
act_o_func (ThermlParser *parser, char c)
{
    rbuf_append_char (parser->end_of_attribute_string, c);
}

/* P - Attribute was listed without value (didn't have an '=' after it). */
static void
act_p_func (ThermlParser *parser, char c)
{
    printf ("attribute %s with no value setting\n", rbuf_str (parser->current_attribute));
    
    therml_attrib_value_pair_done (parser);
    rbuf_append_char (parser->current_attribute, c);
}

/* Q - remember end-of-attr-char  */
static void
act_q_func (ThermlParser *parser, char c)
{
    parser->quote_char = c;
}

/* R - push new node */
static void
act_r_func (ThermlParser *parser, char c)
{
    therml_node_new (parser);
}


/* the error act */
static void
error_act (ThermlParser *parser, char c)
{
    /* Is this used at all ? */
    signal_error (parser, 99, "An unexpected internal error has a occurred.");
}



