
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef __RHASH_H__
#define __RHASH_H__

/* Documentation!
 *
 * In order to use this hash table implementation, you must create
 * your own structure to be used as an entry in the hash table.
 * This structure *MUST* start with the first entry as a
 * 'RHASH_HEADER;'.
 *
 * For Example:
 *
 * struct _MyHashEntry {
 *	    RHASH_HEADER;
 *	    RBuf *data;
 *	    ...
 * }
 *
 * You can create a toplevel RHash structure by using rhash_new,
 * or you can init an already allocated RHash struct using
 * RHASH_INIT(hash).
 *
 * In all cases, the entire hash entry is passed around where applicable.
 * This lets you do whatever you want in terms of the type of key, the
 * data stored in the struct etc.
 */


/* Function to hash the type you are using for your key */
typedef unsigned long (*RHashHashFunction) (const void *key);

/* Function to compare 2 keys.  The first argument is the key the hash system
 * thinks might be the matching one, and the second argument is the key you
 * passed into the lookup function.  This should return
 * TRUE on match, FALSE on no match. 
 *
 * The hash values are saved for each hash entry in Roy Hashes.  So, if you
 * know you have a unique key for each entry (for example, if you are storing
 * pointers), then you can pass in NULL for the compare function, and it will
 * rely solely on the hash value comparison. */
typedef unsigned int (*RHashCompareFunction) (const void *entry_key, const void *key);


typedef struct _RHashHeader	RHashHeader;
typedef struct _RHashEntry	RHashEntry;
typedef struct _RHash		RHash;

struct _RHashHeader {
    RHashEntry **prev_next_ptr; /* >: ) */
    RHashEntry *next;
    unsigned long hashval;
    void *key;
};
#define RHASH_HEADER RHashHeader rhash_header

struct _RHashEntry {
    RHASH_HEADER; 
};

struct _RHash {
    RHashEntry **entries;
    RHashHashFunction hash_func;
    RHashCompareFunction compare_func;
    unsigned int num_entries;
    unsigned int table_size;
};

#define rhash_entry_getkey(hash_entry) \
    (hash_entry)->rhash_header.key

#define rhash_entry_setkey(hash_entry,rhash_entry_key) \
    (hash_entry)->rhash_header.key = rhash_entry_key

/* Initialize a newly created list structure */
#define RHASH_INIT(hash, hash_function, compare_function) \
    (hash)->num_entries = 0; \
    (hash)->table_size = 0; \
    (hash)->hash_func = (RHashHashFunction) hash_function; \
    (hash)->compare_func = (RHashCompareFunction) compare_function; \
    (hash)->entries = NULL

/* Cleans out a hash struct.  This frees the entries table,
 * but does not remove the entries from the hash table itself.  I
 * recommend doing that with a RHASH_FOREACH and rhash_remove_entry,
 * freeing the appropriate items in the structure, and the
 * structure itself, then call this to clean up the Hash. (if you
 * didn't use rhash_new() that is. */
#define RHASH_DESTRUCT(hash) \
    rmem_free ((hash)->entries); \
    (hash)->entries = NULL; \
    (hash)->num_entries = 0; \
    (hash)->table_size = 0


/* This will foreach through all the entries in the hash table, assigning each
 * one to the 'variable' argument.  Unfortunately, we require the extra 'counter'
 * argument, which should be an integer for use in the for loop */
#define RHASH_FOREACH(hash, variable) \
    do { \
	RHashEntry *tmp_rhash_next_entry; \
        unsigned int tmp_rhash_counter; \
        for (tmp_rhash_counter = 0; tmp_rhash_counter < hash->table_size; tmp_rhash_counter++) \
            for (variable = (void *) hash->entries[tmp_rhash_counter], \
                tmp_rhash_next_entry = (void *) variable ? variable->rhash_header.next : NULL; \
                variable != NULL; \
                variable = (void *) tmp_rhash_next_entry, \
                tmp_rhash_next_entry = tmp_rhash_next_entry ? tmp_rhash_next_entry->rhash_header.next : NULL)



/* This just unlinks a hash entry from the hash table, and may be done at any time
 * from anywhere */
#define rhash_remove_entry(hash, entry) \
    do { \
        RHash *roy_hash_tmp = (hash); \
        RHashEntry *roy_hash_entry_tmp = (void *) (entry); \
        \
	if (!roy_hash_entry_tmp) \
	    break; \
	if (roy_hash_entry_tmp->rhash_header.next) { \
            roy_hash_entry_tmp->rhash_header.next->rhash_header.prev_next_ptr = \
                                roy_hash_entry_tmp->rhash_header.prev_next_ptr; \
        } \
    	\
        *roy_hash_entry_tmp->rhash_header.prev_next_ptr = roy_hash_entry_tmp->rhash_header.next; \
	roy_hash_entry_tmp->rhash_header.next = NULL; \
	roy_hash_entry_tmp->rhash_header.prev_next_ptr = NULL; \
	roy_hash_tmp->num_entries--; \
    } while (0)
/* (You get all that ?  Do I know how to have fun or what ? :) ) */


/* Create a newly allocated hash table structure */
RHash *
rhash_new (RHashHashFunction hash_func, RHashCompareFunction compare_func);

/* Free a hash table, as created with rhash_new.  Again, this does not free
 * the entries in the table. */
void
rhash_free (RHash *hash);

/* Lookup an entry given a key.  The return value is a
 * pointer to the entry you provided for insert, or NULL
 * if it was not found */
void *
rhash_lookup (RHash *hash, void *key);

/* Insert a hash entry (as described above).  The key should already
 * be set for the entry using rhash_entry_setkey(). */
void
rhash_insert (RHash *hash, void *entry, void *key);

/* Remove an entry from the hash table.  This only
 * unlinks it from the table, and does not free
 * the key, or any data in your entry struct */
void *
rhash_remove (RHash *hash, void *key);

/* Return out the size of the rhash. */
#define rhash_len(hash) (hash->num_entries)

#endif /* __RHASH_H__ */



