
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#ifndef RCHUNK_ENABLED

void
rchunk_free (void *chunk, unsigned long size)
{
    free (chunk);
}


void *
rchunk_alloc (unsigned long size)
{
    return (malloc (size));
}

void *
rchunk_alloc0 (unsigned long size)
{
    return (calloc (size, 1));
}


#else



#define ROY_CHUNK_MAX_SIZE sizeof (void *) * 8192


/* We are really cruel and brute force about our free lists, and
 * how we are recycling memory */

/* divide by the size of a pointer because we only allocate on
 * multiples of pointer size */
static char *roy_chunk_lists[ROY_CHUNK_MAX_SIZE / sizeof (void *)];


void
rchunk_free (void *chunk, unsigned long size)
{
    int listnum = 0;
    char *next;
    char *ptr = chunk;
    char **ptr2;

    if (chunk == NULL)
        return;

    /* Size must be a multiple of the size of a pointer on some
     * systems. The common case is that the size is already divisable by
     * the size of a pointer. If it's not however, we must adjust
     * it.  We let integer rounding take care of it. */
    if (size % sizeof (void *)) {
        size = ((size / sizeof (void *)) + 1) * sizeof (void *);
    }

    listnum = size / sizeof (void *);

    /* keep a pointer to the first block on the list */
    next = roy_chunk_lists[listnum];

    /* point the list at the chunk we were just given 
     * (minus the 'next' pointer of course) */
    roy_chunk_lists[listnum] = ptr - sizeof (void *);

    /* make the 'next' pointer for this block, point to the
     * previously first block */
    ptr2 = (char **) roy_chunk_lists[listnum];
    *ptr2 = next;
}


void *
rchunk_alloc0 (unsigned long size)
{
    void *mem;
    mem = rchunk_alloc (size);
    memset (mem, 0, size);
    return (mem);
}


void *
rchunk_alloc (unsigned long size)
{
    int listnum = 0;
    char *mem;
    char **ptr;
    
    /* Size must be a multiple of the size of a pointer on some
     * systems. The common case is that the size is already divisable by
     * the size of a pointer. If it's not however, we must adjust
     * it.  We let integer rounding take care of it. */
    if (size % sizeof (void *)) {
        size = ((size / sizeof (void *)) + 1) * sizeof (void *);
    }

    /* We find our list by looking straight into the lists array, after a
     * single shift (divide by 2), because all allocations should be
     * an even amount, and this buys us twice the density in our
     * chunk lists */
    listnum = size / sizeof (void *);

    if (!roy_chunk_lists[listnum]) {
        char *newmem;
        int numchunks;
        unsigned int chunksize;
        int i;

        /* Alloc enough room for a pointer on each chunk */
        chunksize = size + sizeof (void *);
        /* figure out approximately how many chunks will fit
         * in 10 pages (picked arbitrarily) */
        numchunks = (10 * 4096) / chunksize;

        /* Allocate chunks */
        newmem = malloc (numchunks * chunksize);

        /* Since we didn't previously have a list for this chunk
         * size, we set it to the first entry of the new mem
         * we just got */
        roy_chunk_lists[listnum] = newmem;
        for (i = 0; i < numchunks; i++) {

            /* Update the first pointer of this block to point
             * to the start of the next block */
            ptr = (char **) &newmem[i * chunksize];
            *ptr = &newmem[(i + 1) * chunksize];
        }

        /* insure that the last block has a NULL pointer */
        ptr = (char **) &newmem[(numchunks - 1) * chunksize];
        *ptr = NULL;
    }

    /* Get our memory from the first available block */
    mem =  roy_chunk_lists[listnum] + sizeof (void *);

    /* Update list pointer to point to next block */
    ptr = (char **) roy_chunk_lists[listnum];
    roy_chunk_lists[listnum] = *ptr;

    return ((void *)mem);
}

#endif /* RCHUNK_ENABLED */


