
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

static RArray   *int_to_rbuf_map = NULL;
static RBHash    *rbuf_to_int_map = NULL;
static RArray   *available_intern_vals = NULL;

static unsigned int intern_val_count = 0;


typedef struct
{
   RBHASH_HEADER;
   unsigned int intern_val;
   int refcount;
} RInternEntry;

void
rbuf_intern_cleanup__P (void);

void
rbuf_intern_init__P (void)
{
    int_to_rbuf_map = rarray_new (sizeof (void *), 64);
    available_intern_vals = rarray_new (sizeof (unsigned int), 64);
    rbuf_to_int_map = rbhash_new ();

    rcleanup_add_hook (rbuf_intern_cleanup__P);
}


void
rbuf_intern_cleanup__P (void)
{
    RInternEntry *entry;

    RBHASH_FOREACH (rbuf_to_int_map, entry) {
        rbuf_release (rbhash_entry_getkey (entry));
        rbuf_free (rbhash_entry_getkey (entry));
        rchunk_free (entry, sizeof (RInternEntry));
    } RFOREACH_CLOSE;

    rbhash_free (rbuf_to_int_map);
    rarray_free (int_to_rbuf_map);
    rarray_free (available_intern_vals);
}


unsigned int
rbuf_intern (RBuf *buf)
{
    RInternEntry *entry;
    RInternEntry **tmp_entry;
    unsigned int *valp;
    unsigned int val;
    RBuf *key;

    /* First thing to do is to check to see if this entry doesn't already
     * exist */
    entry = rbhash_lookup (rbuf_to_int_map, buf);
    if (entry) {
        /* We have an entry already, so we ref it, and return the value. */
        entry->refcount++;
        /* We swallow rbuf's (I think it's right.. :)), so free the one
         * handed to us - this is of course a nop if it's owned */
        rbuf_free (buf);

        return (entry->intern_val);
    }

    /* if we make it this far, there's no previous entry, so we set one up. */

    entry = rchunk_alloc (sizeof (RInternEntry));
    if (!rbuf_owned (buf)) {
        key = buf;
    } else {
        key = rbuf_new_with_rbuf (buf);
    }


    rbuf_own (key);
    rbuf_set_rdonly (key);
    
    entry->refcount = 1;

    rbhash_insert (rbuf_to_int_map, entry, key);
    
    valp = rarray_pop (available_intern_vals);
    if (valp) {
        /* We are recycling a previous entry, so insert into
         * the array using nth() */
        val = *valp;
        tmp_entry = rarray_nth (int_to_rbuf_map, val);
    } else {
        /* We are using a new value, so we use append()
         * on the array so it can grow */
        val = intern_val_count++;
        tmp_entry = rarray_append (int_to_rbuf_map);
    }
    *tmp_entry = entry;
    entry->intern_val = val;
    
    return (val);
}


RBuf *
rbuf_intern_get_buf (unsigned int val)
{
    RInternEntry **tmp_entry;
    RInternEntry *entry;
    
    tmp_entry = rarray_nth (int_to_rbuf_map, val);
    entry = *tmp_entry;

    if (entry) {
        return (rbhash_entry_getkey (entry));
    } else {
        return (NULL);
    }
}


void
rbuf_intern_unref (unsigned int val)
{
    RInternEntry **tmp_entry;
    RInternEntry *entry;
    
    tmp_entry = rarray_nth (int_to_rbuf_map, val);
    entry = *tmp_entry;

    if (entry) {
        entry->refcount--;

        if (entry->refcount <= 0) {
            unsigned int *val;
            RInternEntry **tmp_entry;
            RBuf *key = rbhash_entry_getkey (entry);
            
            rbuf_release (key);
            rbuf_free (key);
           
            /* Save this value for recycling */
            val = rarray_push (available_intern_vals);
            *val = entry->intern_val;
            
            /* Remove from the hash table */
            rbhash_remove_entry (rbuf_to_int_map, entry);
            
            /* Remove from array */
            tmp_entry = rarray_nth (int_to_rbuf_map, entry->intern_val);
            *tmp_entry = NULL;
            
            rchunk_free (entry, sizeof (RInternEntry));
        }
    }
}


