/* 
 * ------------------------------------------------------------------
 * Role PlayingDB V2.0 by Deepwoods Software
 * ------------------------------------------------------------------
 * Space.cc - Squares and Hexes -- where things happen
 * Created by Robert Heller on Fri Aug 21 11:07:24 1998
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: Space.cc,v $
 * Revision 1.5  2000/02/11 00:31:27  heller
 * MS-Windows fixes...
 *
 * Revision 1.4  1998/12/30 01:04:49  heller
 * Add comments to the code.
 *
 * Revision 1.3  1998/12/21 14:47:59  heller
 * Added image field to "Item"
 *
 * Revision 1.2  1998/12/18 21:08:25  heller
 * Changed "NextSpaceFilename" to NextSpaceIndexString"
 *
 * Revision 1.1  1998/08/23 14:46:27  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Role Playing DB -- A database package that creates and maintains
 * 		       a database of RPG characters, monsters, treasures,
 * 		       spells, and playing environments.
 * 
 *     Copyright (C) 1995,1998  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

#include <Space.h>
#include <math.h>
#include <stdio.h>

static char rcsid[] = "$Id: Space.cc,v 1.5 2000/02/11 00:31:27 heller Rel $";

#ifdef _MSC_VER
const float GeoConstants::Width = 100.0;
const float GeoConstants::HexSideLength = 57.735;
const float GeoConstants::HexPeakHeight = 28.8675;
#endif

#ifdef _MSC_VER
/* Thank you Intel! Fun with the old 8086 *segmented* address space,
   which lives on in VC++ for no good reason, unless MS really believes
   people are still wanting to write 16-bit code (I thought MS-Windows
   3.11 was dead? */
#define near near_
#endif

/*
 * Exit Vector constructor
 */

ExitVector::ExitVector()
{
	evect = new Exit[initSize];
	vSize = initSize;
	vCount = 0;
}

/*
 * Exit Vector destructor
 */
 
ExitVector::~ExitVector()
{
	delete [] evect;
}

/*
 * Exit Vector insertion function
 */
 
void ExitVector::InsertExit(const Exit& source)
{
	if (vCount == vSize)
	{
		Exit *newvect = new Exit[vSize+growSize];
		for (int i = 0; i < vCount; i++) newvect[i] = evect[i];
		delete [] evect;
		evect = newvect;
		vSize += growSize;
	}
	evect[vCount++] = source;
}

/*
 * Helper function to compute the deltaXY between to coordinates.
 * This function is used to find the "nearest" exit or item.
 */

static double DeltaXY (double x1, double y1, double x2, double y2)
{
	double dx = x1 - x2, dy = y1 - y2;
	return sqrt((dx*dx)+(dy*dy));
}

/*
 * ExitVector "Nearest" operator (function operator)
 */
 
const Exit* ExitVector::operator () (double x,double y) const
{
	int near = NearestIndex(x,y);
	if (near < 0) return NULL;
	else return &evect[near];
}

/*
 * Function to delete an exit at a specified index
 */
 
void ExitVector::DeleteAtIndex(int index)
{
	if (index < 0 || index >= vCount) return;
	for (int i = index+1; i < vCount;i++)
	{
		evect[i-1] = evect[i];
	}
	vCount--;
}

/*
 * Function to delete an exit near a point
 */
 
void ExitVector::DeleteNear(double x,double y)
{
	int near = NearestIndex(x,y);
	if (near < 0) return;
	else DeleteAtIndex(near);
}

/*
 * Function to return the index of the nearest Exit to a point
 */
 
int ExitVector::NearestIndex(double x,double y) const
{
	if (vCount == 0) return -1;
	int ni = 0;
	double dx = DeltaXY(x,y,evect[ni].XCenter(),evect[ni].YCenter());
	for (int i = ni+1;i < vCount; i++)
	{
		double newDx = DeltaXY(x,y,evect[i].XCenter(),evect[i].YCenter());
		if (newDx < dx)
		{
			dx = newDx;
			ni = i;
		}
	}
	return ni;
}

/*
 * Item Vector Constructor
 */

ItemVector::ItemVector()
{
	ivect = new Item[initSize];
	vSize = initSize;
	vCount = 0;
}

/*
 * Item Vector Destructor
 */
 
ItemVector::~ItemVector()
{
	delete [] ivect;
}

/*
 * Function to insert an item
 */
 
void ItemVector::InsertItem(const Item& source)
{
	if (vCount == vSize)
	{
		Item *newvect = new Item[vSize+growSize];
		for (int i = 0; i < vCount; i++) newvect[i] = ivect[i];
		delete [] ivect;
		ivect = newvect;
		vSize += growSize;
	}
	ivect[vCount++] = source;
}

/*
 * ItemVector "Nearest" operator (function operator)
 */
 
const Item* ItemVector::operator () (double x,double y) const
{
	int near = NearestIndex(x,y);
	if (near < 0) return NULL;
	else return &ivect[near];
}

/*
 * Function to delete an item at a specified index
 */
 
void ItemVector::DeleteAtIndex(int index)
{
	if (index < 0 || index >= vCount) return;
	for (int i = index+1; i < vCount;i++)
	{
		ivect[i-1] = ivect[i];
	}
	vCount--;
}

/*
 * Function to delete an item near a specified point
 */
 
void ItemVector::DeleteNear(double x,double y)
{
	int near = NearestIndex(x,y);
	if (near < 0) return;
	else DeleteAtIndex(near);
}

/*
 * Function to return the index of the item nearest a specified point
 */
 
int ItemVector::NearestIndex(double x,double y) const
{
	if (vCount == 0) return -1;
	int ni = 0;
	double dx = DeltaXY(x,y,ivect[ni].XCenter(),ivect[ni].YCenter());
	for (int i = ni+1;i < vCount; i++)
	{
		double newDx = DeltaXY(x,y,ivect[i].XCenter(),ivect[i].YCenter());
		if (newDx < dx)
		{
			dx = newDx;
			ni = i;
		}
	}
	return ni;
}

/*
 * Use the internal record buffer to popular the current space instance
 */
 
void Space::RecordToSpace()
{
	char* str = rawData.buffer;    // Data string
	int   bytesleft = rawData.size; // Number of bytes
	bool skipped;

	shape = Undefined;
	centerX = 0.0;
	centerY = 0.0;
	int cnt = exitList.ElementCount();
	int i;
	for (i = cnt-1; i >= 0; i--)
	{
		exitList.DeleteAtIndex(i);
	}
	cnt = itemList.ElementCount();
	for (i = cnt-1; i >= 0; i--)
	{
		itemList.DeleteAtIndex(i);
	}
	name = NULL;
	descr = NULL;
	bgcolor = NULL;
	// while bytes remain...
	while (bytesleft > 0) {
		// scan for field specifier
		while (*str != '%' && bytesleft > 0) { str++; bytesleft--; }
		// skip over specified prefix
		str++; bytesleft--;
		// while there still are bytes...
		if (bytesleft <= 0) break;
		skipped = false;
		// fan out based on field key
		switch (*str++) {
			case 's': { 
				if (strcmp(str,"Square") == 0) shape = Square; 
				else if (strcmp(str,"Hexagon") == 0) shape = Hexagon;
				else shape = Undefined;
				break;
			}
			case 'x': centerX = atof(str); break; 
			case 'y': centerY = atof(str); break;
			case 'e': ProcessRecordToExit(bytesleft,str); skipped = true; break;
			case 'i': ProcessRecordToItem(bytesleft,str); skipped = true; break;
			case 'n': name = str; break;
			case 'C': descr = str; break;
			case 'b': bgcolor = str; break;
		}
		if (!skipped)
		{
			// count specifier
			bytesleft--;
			// find strlen plus nul byte
			int slen = strlen(str) + 1;
			// update counter and pointer
			bytesleft -= slen;
			str += slen;
		}
		// EOR marker?
		if (*str == '\n') break;
	}
}

/*
 * Process an serialized exit into the exit vector.
 */

void Space::ProcessRecordToExit(int &bytesleft, char *& str)
{
	Exit newExit;
	int l;
	bytesleft--;
	newExit.xCenter = atof(str);
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newExit.yCenter = atof(str);
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	if (*str == 'T') newExit.wallAligned = true;
	else if (*str == 'F') newExit.wallAligned = false;
	str += 2;
	bytesleft -= 2;
	if (strcmp(str,"Doorway") == 0) newExit.type = Exit::Doorway;
	else if (strcmp(str,"Door") == 0) newExit.type = Exit::Door;
	else if (strcmp(str,"LockedDoor") == 0) newExit.type = Exit::LockedDoor;
	else if (strcmp(str,"SecretDoor") == 0) newExit.type = Exit::SecretDoor;
	else if (strcmp(str,"OnewayDoor") == 0) newExit.type = Exit::OnewayDoor;
	else if (strcmp(str,"TrapDoorUp") == 0) newExit.type = Exit::TrapDoorUp;
	else if (strcmp(str,"TrapDoorDown") == 0) newExit.type = Exit::TrapDoorDown;
	else if (strcmp(str,"StairsUp") == 0) newExit.type = Exit::StairsUp;
	else if (strcmp(str,"StairsDown") == 0) newExit.type = Exit::StairsDown;
	else if (strcmp(str,"WindowUnglazed") == 0) newExit.type = Exit::WindowUnglazed;
	else if (strcmp(str,"WindowGlazed") == 0) newExit.type = Exit::WindowGlazed;
	else if (strcmp(str,"Chimney") == 0) newExit.type = Exit::Chimney;
	else if (strcmp(str,"Pit") == 0) newExit.type = Exit::Pit;
	else if (strcmp(str,"Unspecified") == 0) newExit.type = Exit::Unspecified;
	else newExit.type = Exit::Unspecified;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newExit.descr = str;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newExit.image = str;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newExit.nextSpaceIndexString = str;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	while (*str != '\r' && bytesleft > 0)
	{
		str++;
		bytesleft--;
	}
	str += 2;
	bytesleft -= 2;
	exitList.InsertExit(newExit);
}

/*
 * Process a serialized item to the ItemVector.
 */
 
void Space::ProcessRecordToItem(int &bytesleft, char *& str)
{
	Item newItem;
	int l;
	bytesleft--;
	newItem.xCenter = atof(str);
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newItem.yCenter = atof(str);
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	if (strcmp(str,"Character") == 0) newItem.type = Item::Character;
	else if (strcmp(str,"Monster") == 0) newItem.type = Item::Monster;
	else if (strcmp(str,"Treasure") == 0) newItem.type = Item::Treasure;
	else if (strcmp(str,"TrickTrap") == 0) newItem.type = Item::TrickTrap;
	else if (strcmp(str,"Dressing") == 0) newItem.type = Item::Dressing;
	else if (strcmp(str,"Unspecified") == 0) newItem.type = Item::Unspecified;
	else newItem.type = Item::Unspecified;
	newItem.image = str;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	newItem.filename = str;
	l = strlen(str) + 1;
	bytesleft -= l;
	str += l;
	while (*str != '\r' && bytesleft > 0)
	{
		str++;
		bytesleft--;
	}
	str += 2;
	bytesleft -= 2;
	itemList.InsertItem(newItem);	
}

/*
 * Convert all of the class slots to the internal record buffer. Fix all
 * pointers to point to the internal memory store.
 */
 
void Space::UpdateRecord()
{
	static char centerXS[12], centerYS[12];
	char *shapeS = "Undefined";
	char **exits, **items;
	int  *nExits, *nItems;
	// formnumeric strings
	sprintf(centerXS,"%f",centerX);
	sprintf(centerYS,"%f",centerY);
	switch (shape)
	{
		case Square: shapeS = "Square"; break;
		case Hexagon: shapeS = "Hexagon"; break;
		case Undefined: shapeS = "Undefined"; break;
	}
	int rsize = 7 + 	// "*Space\0"
		strlen(centerXS) + 3 + // "%xnnn\0"
		strlen(centerYS) + 3 + // "%ynnn\0"
		strlen(shapeS) + 3 +   // "%sxxx\0"
		((name)?(strlen(name)):(0)) + 3 + // "%nxxx\0"
		((descr)?(strlen(descr)):(0)) + 3 + // "%Cxxx\0"
		((bgcolor)?(strlen(bgcolor)):(0)) + 3; // "%bxxx\0"
	int i, cnt;
	cnt = exitList.ElementCount();
	exits = new char *[cnt];
	nExits = new int[cnt];
	for (i = 0; i < cnt; i++)
	{
		nExits[i] = CreateExitRecord(*(exitList[i]),exits[i]); // "%e...\0"
		rsize += nExits[i];
	}
	cnt = itemList.ElementCount();
	items = new char *[cnt];
	nItems = new int[cnt];
	for (i = 0; i < cnt; i++)
	{
		nItems[i] = CreateItemRecord(*(itemList[i]),items[i]); // "%i...\0"
		rsize += nItems[i];
	}
	rsize += strlen("\n") + 1;     // "\n\0"
	// allocate a string
	char* str = new char[rsize]; char* p = str;
	strcpy(p,"*Space"); p += strlen(p) + 1;
	*p++ = '%'; *p++ = 'n'; if (name) strcpy(p,name); else strcpy(p,""); name = p;p += strlen(p) + 1;
	*p++ = '%'; *p++ = 'C'; if (descr) strcpy(p,descr); else strcpy(p,""); descr = p;p += strlen(p) + 1;
	*p++ = '%'; *p++ = 'b'; if (bgcolor) strcpy(p,bgcolor); else strcpy(p,""); bgcolor = p;p += strlen(p) + 1;
	*p++ = '%'; *p++ = 's'; strcpy(p,shapeS); p += strlen(p) + 1;
	*p++ = '%'; *p++ = 'x'; strcpy(p,centerXS); p += strlen(p) + 1;
	*p++ = '%'; *p++ = 'y'; strcpy(p,centerYS); p += strlen(p) + 1;
	cnt = exitList.ElementCount();
	for (i = 0; i < cnt; i++)
	{
		memcpy(p,exits[i],nExits[i]);
		UpdateExitRecord(exitList.evect[i],p);
		p += nExits[i];
		delete exits[i];
	}
	delete [] exits;
	delete [] nExits;
	cnt = itemList.ElementCount();
	for (i = 0; i < cnt; i++)
	{
		memcpy(p,items[i],nItems[i]);
		UpdateItemRecord(itemList.ivect[i],p);
		p += nItems[i];
		delete items[i];
	}
	delete [] items;
	delete [] nItems;
	*p++ = '\n'; *p++ = 0;
	// free up old buffer
	rawData.NewBuffer(0);
	// paste in new buffer
	rawData.size = rsize;
	rawData.buffer = str;
}

/*
 * Serialize an exit
 */
 
int Space::CreateExitRecord(const Exit &element,char *& buffer)
{
	char *type = "Unspecified";
	static char xCenterS[12], yCenterS[12];
	switch(element.type)
	{
		case Exit::Doorway : type = "Doorway"; break;
		case Exit::Door : type = "Door"; break;
		case Exit::LockedDoor : type = "LockedDoor"; break;
		case Exit::SecretDoor : type = "SecretDoor"; break;
		case Exit::OnewayDoor : type = "OnewayDoor"; break;
		case Exit::TrapDoorUp : type = "TrapDoorUp"; break;
		case Exit::TrapDoorDown : type = "TrapDoorDown"; break;
		case Exit::StairsUp : type = "StairsUp"; break;
		case Exit::StairsDown : type = "StairsDown"; break;
		case Exit::WindowUnglazed : type = "WindowUnglazed"; break;
		case Exit::WindowGlazed : type = "WindowGlazed"; break;
		case Exit::Chimney : type = "Chimney"; break;
		case Exit::Pit : type = "Pit"; break;
		case Exit::Unspecified : type = "Unspecified"; break;
	}
	sprintf(xCenterS,"%f",element.xCenter);
	sprintf(yCenterS,"%f",element.yCenter);
	int rsize = 2 + 	// "%e"
		strlen(xCenterS) + 1 +
		strlen(yCenterS) + 1 +
		2 +
		strlen(type) + 1 +
		((element.descr)?(strlen(element.descr)):(0)) + 1 +
		((element.image)?(strlen(element.image)):(0)) + 1 +
		((element.nextSpaceIndexString)?(strlen(element.nextSpaceIndexString)):(0)) + 1 +
		strlen("\r") + 1;
	buffer = new char[rsize];
	char *p = buffer;
	*p++ = '%'; *p++ = 'e';
	strcpy(p,xCenterS); p += strlen(p) + 1;
	strcpy(p,yCenterS); p += strlen(p) + 1;
	if (element.wallAligned) *p++ = 'T';
	else *p++ = 'F';
	*p++ = '\0';
	strcpy(p,type); p += strlen(p) + 1;
	if (element.descr) strcpy(p,element.descr); else strcpy(p,""); p += strlen(p) + 1;
	if (element.image) strcpy(p,element.image); else strcpy(p,""); p += strlen(p) + 1;
	if (element.nextSpaceIndexString) strcpy(p,element.nextSpaceIndexString); else strcpy(p,""); p += strlen(p) + 1;
	*p++ = '\r'; *p = '\0';
	return rsize;	
}

/*
 * Fix the exit's internal pointers to point to the internal record.
 */
 
void Space::UpdateExitRecord(Exit &element,const char * p)
{
	p += 2;
	p += strlen(p) + 1;
	p += strlen(p) + 1;
	p += 2;
	p += strlen(p) + 1;
	element.descr = p; p += strlen(p) + 1;
	element.image = p; p += strlen(p) + 1;
	element.nextSpaceIndexString = p;
}

/*
 * Serialize an item into the internal record structure.
 */
 
int Space::CreateItemRecord(const Item &element,char *& buffer)
{
	char *type = "Unspecified";
	static char xCenterS[12], yCenterS[12];
	switch(element.type)
	{
		case Item::Character: type = "Character"; break;
		case Item::Monster: type = "Monster"; break;
		case Item::Treasure: type = "Treasure"; break;
		case Item::TrickTrap: type = "TrickTrap"; break;
		case Item::Dressing: type = "Dressing"; break;
		case Item::Unspecified: type = "Unspecified"; break;
	}
	sprintf(xCenterS,"%f",element.xCenter);
	sprintf(yCenterS,"%f",element.yCenter);
	int rsize = 2 +         // "%i"
		strlen(xCenterS) + 1 +
		strlen(yCenterS) + 1 +
		strlen(type) + 1 +
		((element.image)?(strlen(element.image)):(0)) + 1 +
		((element.filename)?(strlen(element.filename)):(0)) + 1 +
		strlen("\r") + 1;
	buffer = new char[rsize];
	char *p = buffer;
	*p++ = '%'; *p++ = 'i';
	strcpy(p,xCenterS); p += strlen(p) + 1;
	strcpy(p,yCenterS); p += strlen(p) + 1;
	strcpy(p,type); p += strlen(p) + 1;
	if (element.image) strcpy(p,element.image); else strcpy(p,""); p += strlen(p) + 1;
	if (element.filename) strcpy(p,element.filename); else strcpy(p,""); p += strlen(p) + 1;
	*p++ = '\r'; *p = '\0';
	return rsize;
}

/*
 * Update an items internal pointers to point to the internal record buffer
 */
 
void Space::UpdateItemRecord(Item &element,const char *p)
{
	p += 2;
	p += strlen(p) + 1;
	p += strlen(p) + 1;
	p += strlen(p) + 1;
	element.image = p;
	p += strlen(p) + 1;
	element.filename = p;
}

/*
 * Insert an Exit into a space's Exit Vector
 */
 
void Space::InsertExit (const Exit& source)
{
	exitList.InsertExit(source);
	UpdateRecord();
}

/*
 * Delete an Exit in a space's Exit Vector near the specified point
 */
 
void Space::DeleteExitNear(double x,double y)
{
	exitList.DeleteNear(x,y);
	UpdateRecord();
}

/*
 * Delete an Exit in a space's Exit Vector by index
 */
 
void Space::DeleteExitAtIndex(int index)
{
	exitList.DeleteAtIndex(index);
	UpdateRecord();
}

/*
 * Insert an Item into a space's Item Vector
 */
 
void Space::InsertItem (const Item& source)
{
	itemList.InsertItem(source);
	UpdateRecord();
}

/*
 * Delete an Item in a space's Item Vector near the specified point
 */
 
void Space::DeleteItemNear(double x,double y)
{
	itemList.DeleteNear(x,y);
	UpdateRecord();
}

/*
 * Delete an Item in a space's Item Vector by index
 */
 
void Space::DeleteItemAtIndex(int index)
{
	itemList.DeleteAtIndex(index);
	UpdateRecord();
}
