/* -*- Mode: C; c-file-style: "k&r"; -*-*/

/* parser.c
 *    file parser
 *
 * $Id: parser.c,v 1.15 2001/04/07 12:49:03 antoine Exp $
 *
 * Copyright (C) 2000,
 *     Antoine Lefebvre <antoine.lefebvre@polymtl.ca>
 *     Remi Lefebvre <remi@step.polymtl.ca>
 *
 * gpcp is free software; you can redistribute them and/or modify them
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "parser.h"
#include "token.h"
#include "err.h"

int parse_file(FILE *fd, Data *data)
{
     /* this variable changes sublevel as we parse */
     GPCP_State cursor;

     unsigned char buffer[MAX_BUF];

     int line = 1; /* keep track of the cursor position */
     
     /* state variable */
     int parser_state;
     int token;
     int insub = 0; /* true or false */

     Types keytype;

     /* temporary values for cursor */
     Data *tmpdata;
     
     /* initialisation */
     parser_state = PARSER_MAIN; 
     cursor.level = 0;
     cursor.data  = data;

     while (1)
     {
          /* get a token for treatment */
          token = get_token(fd, buffer, sizeof(buffer), &line);

          /* check for EOF */
          if (token == EOF)
          {
               if ((cursor.level != 0) || (parser_state != PARSER_MAIN))
               {
                    return error (PREMATURE_EOF, line, "");
               }
               else
               {
                    /* reached EOF normally */
                    return 0;
               }
          }

          switch (parser_state)
          {
          case PARSER_MAIN:      

               switch (token)
               {
               case TOK_KEYWORD:

                    /* we check for validity of the keyword against the
                     * registered list. keytype will be set the the right
                     * type. */

                    if (check_key(buffer, NULL, &keytype) != 0)
                         return error(INVALID_KEYWORD, line, buffer);
                    
                    add_keyword(cursor.data, buffer);

                    parser_state = PARSER_KEY;
                    break;

               default:
                    return error(KEYWORD_EXPECTED, line, "");
                    break;
               }
               break;

          case PARSER_KEY:
               switch (token)
               {
               case TOK_EQUAL:
                    parser_state = PARSER_EQU;
                    break;

               default:
                    return error(EQUAL_EXPECTED, line, "");
                    break;
               }
               break;
          
          case PARSER_EQU:
               if (insub)
                    parser_state = PARSER_INSUB;
               else
                    parser_state = PARSER_MAIN;

               switch (token)
               {
               case TOK_FLOAT:
                    /* add_value() does the following:
                     *  1) check the type
                     *  2) add a member to the keylist pointed too by
                     *     cursor.data
                     */

                    /* NOTE: the float was validated in get_token() */

                    if (add_value(cursor.data, buffer, keytype, FLOAT))
                         return error(INVALID_TYPE, line, "FLOAT");
                                             
                    break;

               case TOK_INTEGER:
                    if (add_value(cursor.data, buffer, keytype, INTEGER))
                         return error(INVALID_TYPE, line, "INTEGER");

                    break;

               case TOK_STRING:
                    if (add_value(cursor.data, buffer, keytype, STRING))
                         return error(INVALID_TYPE, line, "STRING");
                    
                    break;

               case TOK_FUNCTION:
                    if (add_value(cursor.data, buffer, keytype, FUNCTION))
                         return error(INVALID_TYPE, line, "FUNCTION");
                    
                    break;

               case TOK_TRUE:
               case TOK_FALSE:
                    if (add_value(cursor.data, buffer, keytype, BOOLEAN))
                         return error(INVALID_TYPE, line, "BOOLEAN");
                    
                    break;

               case TOK_OPENPAR:
                    /* if the token is an opening parenthesis, the previous
                     * keyword was a parent */
                    parser_state = PARSER_SUB;

                    /* get in one level further */

                    /* NOTE: buffer won't be used */
                    if (add_value(cursor.data, buffer, keytype, PARENT))
                         return error(INVALID_TYPE, line, "PARENT");
                    
                    tmpdata = cursor.data;

                    /* update cursor */
                    cursor.level++;
                    cursor.data = cursor.data->key_list
                         [cursor.data->n_key - 1].val.subdata;

                    /* initialise new (Data *) member */
                    cursor.data->n_key = 0;
                    cursor.data->key_list = NULL;
                    cursor.data->prev = tmpdata;
                    cursor.data->parent = tmpdata->key_list +
                         (tmpdata->n_key - 1);

                    break;

               default:
                    return error(VALUE_EXPECTED, line, "");
                    break;
               }
               break;

          case PARSER_SUB:

               insub = 1;
          
               switch (token)
               {
               case TOK_KEYWORD:
                    if (check_key(buffer, cursor.data->parent->name,
                                  &keytype) != 0)
                         return error(INVALID_KEYWORD, line, buffer);
                    
                    add_keyword(cursor.data, buffer);
                    
                    parser_state = PARSER_KEY;
                    break;
                    
               default:
                    return error(VALUE_EXPECTED, line, "");
                    break;
                
               }
               break;

          case PARSER_INSUB:
               switch (token)
               {
               case TOK_COMMA:
                    parser_state = PARSER_SUB;
                    break;

               case TOK_CLOSEPAR:
                    /* set cursor */
                    cursor.level--;
                    cursor.data = cursor.data->prev;

                    if (cursor.level)
                         parser_state = PARSER_INSUB;
                    else
                         parser_state = PARSER_MAIN;

                    break;
                    
               default:
                    return error(CLOSEPAR_EXPECTED, line, "");
                    break;
               }
          }
     }

     return 0;  
}


int check_key(char *buffer, char *parent, Types *t)
{
     int i = 0;

     /* FIXME: should we use strncmp here ? */
     while (options[i].keyword != NULL)
     {
          if ((parent == NULL) || (options[i].parent == NULL))
          {
               if ((strcmp(options[i].keyword, buffer) == 0) &&
                   (options[i].parent == parent))
               {
                    *t = options[i].type;
                    return 0;
                    break;
               } 
          }
          else
          {
               if ((strcmp(options[i].keyword, buffer) == 0) &&
                   (strcmp(options[i].parent, parent) == 0))
               {
                    *t = options[i].type;
                    return 0;
                    break;
               }
          }
          i++;
     }
     return 1;
}


int add_keyword(Data *data, char *buf)
{    
     /* increment the number of subkey */
     data->n_key++;

     /* allocate space for the new keyword */
     data->key_list = (Keywords *) realloc
          (data->key_list, sizeof(Keywords) * data->n_key);
     
     /* allocate space for the name of the new keyword */
     data->key_list[data->n_key - 1].name = (char *)
          malloc (sizeof(char) * (strlen(buf) + 1));
     
     /* copy the name of the keyword in its structure */
     strcpy(data->key_list[data->n_key - 1].name, buf);

     return 0;
}

/* set .type, allocate .val and fill it */
int add_value(Data *data, char *buf, Types keytype, Types type)
{
     
     if (keytype != type)
          return INVALID_TYPE;

     data->key_list[data->n_key - 1].type = type;

     switch (type)
     {
     case FLOAT:
          data->key_list[data->n_key - 1].val.number = (float *)
               malloc (sizeof(float));
          data->key_list[data->n_key - 1].val.number[0] = (float)
               atof(buf);
          break;

     case INTEGER:
          data->key_list[data->n_key - 1].val.integer = (int *)
               malloc (sizeof(int));
          data->key_list[data->n_key - 1].val.integer[0] = (int)
               atoi(buf);
          break;
          
     case STRING:
          data->key_list[data->n_key - 1].val.string = (char *)
               malloc (sizeof(char) * (strlen(buf) + 1));
          
          strcpy(data->key_list[data->n_key - 1].val.string, buf);
          break;

     case FUNCTION:
          break;
          
     case BOOLEAN:
          data->key_list[data->n_key - 1].val.boolean =
               (short *) malloc (sizeof(short));
          /* CHECKME: does that really work ? */
          if (strncasecmp(buf, "true", sizeof("true")) == 0)
          {
               data->key_list[data->n_key - 1].val.boolean[0] = 1;
          }
          else
          {
               data->key_list[data->n_key - 1].val.boolean[0] = 0;
          }
          break;

     case PARENT:
          data->key_list[data->n_key - 1].val.subdata =
               (Data *) malloc (sizeof(Data));
          break;
          
     default:
          break;
          
     }
     return 0;
}










