/*
 *  Copyright (C) 1999
 *  Robert Lissner
 *  Jay MacDonald
 *  Sam Phillips
 *  Keith Wesolowski. 
 */


#include "includes.h"
#include "globals.h"

static gint16 working_idx;
static filter_info old_filter;
static gboolean _in_add = FALSE;

/* Internals */
static void _filter_do_edit (gint16 filter_index);
static void _filter_do_delete (gint16 filter_index);

static void _filter_menuitem_select (GtkWidget *w, gpointer *data);
static void _filter_selection_made (GtkWidget *w, gint row, gint column,
			     GdkEventButton *e, gpointer data);
static void _filter_action (GtkWidget *w, gpointer data);
static void _filter_select (char mode);
static gboolean _filter_compare_record (gint32 field_index,
					gint32 record_index,
					gchar* criterionx, filter_rule rule,
					gboolean use_nocase);

/***************************** BEGIN FILTERS INTERFACE ***********************/

void filter_show_all (GtkWidget *w, gpointer *data) {
    gint32 i;
    if (check_if_changed ())
      return;
    big_draw_start ();
    for (i=0; i < front->last_row; i++)
	gtk_sheet_row_set_visibility (GTK_SHEET (front->sheet), i, TRUE);
    big_draw_end ();
    front->filter_ptr = NULL;
}

void filter_apply (GtkWidget *w, gpointer *data) {
    /* We must first ask which filter to apply, then call filter_do_apply
     * with its index */
    if (check_if_changed ())
	return;
    _in_add = FALSE;
    _filter_select ('A');
}

/* The full monty. */
void filter_do_apply (gint16 filter_index) {
    gint i,j;
    gboolean flag, selected = FALSE;
    GtkSheetRange *x = (GtkSheetRange *) malloc (sizeof (GtkSheetRange));
    filter_info *cf = &front->filters[filter_index];
    front->filter_ptr = cf;

    cancel_level1 (dialog1_win, NULL, NULL);

    if (filter_index < 0) {
	filter_show_all (NULL, NULL);
	return;
    }

    big_draw_start ();
    gtk_sheet_unselect_range (GTK_SHEET (front->sheet),
			      &(GTK_SHEET (front->sheet)->range));
    front->sel_type = 0;
    for (i=0; i < front->last_row; i++) {
	gtk_sheet_row_set_visibility (GTK_SHEET (front->sheet), i, FALSE);
	if (cf->by_and) {
	    flag = TRUE;
	    for (j=0; j < cf->line_ct; j++) {
		if (cf->line[j].field >= 0) {
		    if (!_filter_compare_record (cf->line[j].field, i,
						 cf->line[j].compare,
						 cf->line[j].type,
						 cf->use_nocase))
			flag = FALSE;
		}
	    }
	} else {
	    flag = FALSE;
	    for (j=0; j < cf->line_ct; j++) {
		if (cf->line[j].field >= 0) {
		    if (_filter_compare_record (cf->line[j].field, i,
						cf->line[j].compare,
						cf->line[j].type,
						cf->use_nocase))
			flag = TRUE;
		}
	    }
	}
	gtk_sheet_row_set_visibility (GTK_SHEET (front->sheet), i, flag);
	if (!selected && flag) {
	    x->row0 = i;
	    x->col0 = 0;
	    x->rowi = i;
	    x->coli = 0;
	    selected = TRUE;
	}
    }
    gtk_sheet_select_range (GTK_SHEET (front->sheet), x);
    /* I don't believe we should free x because I'm fairly sure that sheet
     * will use it as-is. Am I wrong?
     */
    big_draw_end ();
}

void filter_add (GtkWidget *w, gpointer *data) {
    int j;
    
    /* Create a new filter and then edit it */
    if (check_if_changed ())
	return;
    if (front->filter_ct >= MAX_FILTERS) {
	level2_error ("Unexpected filter count overflow in filter_add",
		      "I'll report this");
	return;
    }

    _in_add = TRUE;
    
    working_idx = front->filter_ct++;
    /* Allocate and clear a new entry */
    memset (&front->filters[working_idx], 0, sizeof (filter_info));
    strncpy (front->filters[working_idx].name, "Untitled filter",
	     MAX_FIELD_NAME);
    front->filters[working_idx].line_ct = MAX_FILTER_NESTING;
    front->filters[working_idx].by_and = FALSE;
    for (j=0; j < MAX_FILTER_NESTING; j++)
	front->filters[working_idx].line[j].field = -1;
    front->filter_ptr = front->filters + working_idx;
    _filter_do_edit (working_idx);
}

void filter_edit (GtkWidget *w, gpointer *data) {    
    if (check_if_changed ())
	return;
    _in_add = FALSE;
    _filter_select ('E');
}

void filter_delete (GtkWidget *w, gpointer *data) {
    /* Simply removes the selected filter from memory */
    if (check_if_changed ())
	return;
    _in_add = FALSE;
    _filter_select ('D');
}

/******************************* END INTERFACE ******************************/

/******************************* ACTION HANDLERS ****************************/

/* Handle the selection of a menu item. Since the only menuitems we work with
 * are in the add/edit field select and rule select operations, we just do
 * all of them right here.
 */
void _filter_menuitem_select (GtkWidget *w, gpointer *data) {
    gchar *name = gtk_widget_get_name (w);
    gint16 line = ((gint16)(int)(data) & (gint16) 0xf000) >> 12;
    if (!strcmp (name, "field")) {
	gint16 field = (gint16)(int)(data) & (gint16) 0x0fff;
	front->filters[working_idx].line[line].field = field - 1;
    }
    if (!strcmp (name, "rule")) {
	gint16 rule = (gint16)(int)(data) & (gint16) 0x0fff;
	front->filters[working_idx].line[line].type = rule;
    }
}

/* Handles clist filter selections */
void _filter_selection_made (GtkWidget *widget, gint row, gint column,
                             GdkEventButton *event, gpointer data) {

    if (event->type != GDK_2BUTTON_PRESS)
	return;

    working_idx = row;
    switch ((char)(int)(data)) {
    case 'A':
	filter_do_apply (working_idx);
	break;
    case 'E':
	_filter_do_edit (working_idx);
	break;
    case 'D':
	_filter_do_delete (working_idx);
	break;
    default:
	level2_error (_("Unexpected mode in _filter_selection_made."),
		      _("I'll report this"));
	break;
    }
}

void _filter_boolean_handler (GtkWidget *w, gpointer data) {
    if (!strcmp (gtk_widget_get_name (w), "bool"))
	front->filters[(gint)data].by_and
	    = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
    else
	front->filters[(gint)data].use_nocase
	    = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
}

void _filter_entry_handler (GtkWidget *w, gpointer data) {
    gint filter_idx = (gint)data;

    if (filter_idx < 0 || !gtk_widget_get_name (w)) {
	level2_error (_("** PANIC ** in _filter_entry_handler"), _("Bummer"));
	return;
    }
    if (!strcmp (gtk_widget_get_name (w), "filter")) {
	strncpy (front->filters[filter_idx].name,
		 gtk_entry_get_text (GTK_ENTRY (w)),
		 sizeof (front->filters[filter_idx].name));
    } else {
	gint line_idx = atoi (gtk_widget_get_name (w));
	strncpy (front->filters[filter_idx].line[line_idx].compare,
		 gtk_entry_get_text (GTK_ENTRY (w)), sizeof
		 (front->filters[filter_idx].line[line_idx].compare));
    }
}

/* Handles miscellaneous actions, typically from Edit */
void _filter_action (GtkWidget *w, gpointer data) {
    if (GTK_IS_DIALOG (w) || !strcmp (gtk_widget_get_name (w), "cancel")) {
	/* The user has requested that we cancel this operation. So first
	 * find out what the operation is, then don't do it. :) For now, only
	 * _filter_do_edit uses this.
	 */
	if (_in_add) {
	    _filter_do_delete ((gint16)working_idx);
	} else {
	    front->filters[working_idx] = old_filter;
	    cancel_level1 (dialog1_win, NULL, NULL);
	}
	_in_add = FALSE;
	return;
    }
    /* The user clicked the OK Button. Since we've already done everything,
     * just return.
     */
    filter_do_apply (working_idx);
    _in_add = FALSE;
    cancel_level1 (dialog1_win, NULL, NULL);
    dim_list_filter_menu ();
    front_is_changed ();
}

/****************************** END ACTION HANDLERS **************************/

/**************************** BEGIN INTERNAL GUI *****************************/

/***
 * These three functions actually do the work after a user has selected a
 * sort. Note that although the argument is usually stored in a global
 * (working_idx), these all accept arguments so that other modules may make
 * use of them. For now this isn't such a good idea since I haven't thought
 * through all the ramifications, so these remain _ functions.
 ***/

/* Removes the selected filter. */
void _filter_do_delete (gint16 filter_index) {
    /* This is kind of slow, but it's simple and it works. I expect all this
     * stuff will eventually be replaced by a GList of filter structures and
     * then this kind of crap will go away.
     */
    int reportx;

    if (dialog1_win)
	cancel_level1 (dialog1_win, NULL, NULL);
    front->filter_ct--;

    dim_list_filter_menu ();
    front_is_changed ();
    
    /* Handle report modifications */
    for (reportx = 0; reportx < front->report_ct; reportx++) {
	if (front->reports [reportx].filter == filter_index)
	    front->reports [reportx].filter = -1;
	else if (front->reports [reportx].filter > filter_index)
	    front->reports [reportx].filter--;
    }

    if (front->filter_ptr == &front->filters[filter_index])
	front->filter_ptr = NULL;
    if (filter_index == front->filter_ct)
	return;
    if (front->filter_ptr == &front->filters[front->filter_ct])
	front->filter_ptr = &front->filters[filter_index];
    front->filters[filter_index] = front->filters[front->filter_ct];
}

/* Lets the user modify the selected filter */
void _filter_do_edit (gint16 filter_index) {
    GtkWidget *edit_dialog,
	*ok_button,
	*cancel_button,
	*field_selector,
	*field_selector_menu,
	*field_selector_menu_item,
	*rule_selector,
	*rule_selector_menu,
	*rule_selector_menu_item,
	*entry[MAX_FILTER_NESTING],
	*line_hbox[MAX_FILTER_NESTING],
	*filter_name,
	*filter_name_label,
	*name_hbox,
	*case_check,
	*boolean_check,
	*boolean_hbox;

    gint16 line_index;
    gint16 field_index;
    gint rule_index;
    
    static const char *rule_names[] = { "contains",
					"does not contain",
					"is exactly",
					"is not",
					"is less than",
					"is more than",
					"is blank",
					"is not blank",
					NULL
    };
    
    old_filter = front->filters[filter_index];
    
    make_basic_dialog1 ();
    edit_dialog = dialog1_win;

    gtk_window_set_title (GTK_WINDOW (edit_dialog), "Filter edit");
    
    ok_button = gtk_button_new_with_label (_("  OK  "));
    gtk_widget_set_name (ok_button, "ok");
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			GTK_SIGNAL_FUNC (_filter_action), NULL);
    gtk_widget_show (ok_button);
    
    cancel_button = gtk_button_new_with_label (_("  Cancel  "));
    gtk_widget_set_name (cancel_button, "cancel");
    gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
			GTK_SIGNAL_FUNC (_filter_action),
			(gpointer)(gint)filter_index);
    gtk_widget_show (cancel_button);

    gtk_signal_connect (GTK_OBJECT (edit_dialog), "delete_event",
			GTK_SIGNAL_FUNC (_filter_action),
			(gpointer)(gint)filter_index);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->action_area),
			ok_button, FALSE, FALSE, 5);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->action_area),
			cancel_button, FALSE, FALSE, 5);
    
    filter_name = gtk_entry_new ();
    gtk_widget_set_name (filter_name, "filter");
    gtk_signal_connect (GTK_OBJECT (filter_name), "changed",
			GTK_SIGNAL_FUNC (_filter_entry_handler),
			(gpointer)(gint)filter_index);
    filter_name_label = gtk_label_new (_("Filter name"));
    boolean_check = gtk_check_button_new_with_label
	(_("All of these must be true"));
    case_check = gtk_check_button_new_with_label
	(_("Text is case-insensitive"));
    gtk_widget_set_name (boolean_check, "bool");
    gtk_widget_set_name (case_check, "case");
    
    if (front->filters[filter_index].name != NULL)
	gtk_entry_set_text (GTK_ENTRY (filter_name),
			    front->filters[filter_index].name);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (boolean_check),
				  front->filters[filter_index].by_and);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (case_check),
				  front->filters[filter_index].use_nocase);
    gtk_signal_connect (GTK_OBJECT (boolean_check), "toggled",
			GTK_SIGNAL_FUNC (_filter_boolean_handler),
			(gpointer)(gint)filter_index);
    gtk_signal_connect (GTK_OBJECT (case_check), "toggled",
			GTK_SIGNAL_FUNC (_filter_boolean_handler),
			(gpointer)(gint)filter_index);
    
    gtk_widget_show (filter_name);
    gtk_widget_show (filter_name_label);
    gtk_widget_show (boolean_check);
    gtk_widget_show (case_check);
    
    name_hbox = gtk_hbox_new (TRUE, 5);
    boolean_hbox = gtk_hbox_new (TRUE, 5);
    
    gtk_box_pack_start (GTK_BOX (name_hbox), filter_name_label, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX (name_hbox), filter_name, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox), name_hbox,
			TRUE, TRUE, 5);
    gtk_widget_show (name_hbox);
    gtk_box_pack_start (GTK_BOX (boolean_hbox), boolean_check, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX (boolean_hbox), case_check, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox), boolean_hbox,
			TRUE, TRUE, 5);
    gtk_widget_show (boolean_hbox);

    for (line_index = 0; line_index < MAX_FILTER_NESTING; line_index++) {

	char nbuf[32];
	
	line_hbox[line_index] = gtk_hbox_new (TRUE, 5);

	entry[line_index] = gtk_entry_new ();
	g_snprintf (nbuf, 32, "%d", line_index);
	gtk_widget_set_name (entry[line_index], nbuf);
	gtk_entry_set_text (GTK_ENTRY (entry[line_index]),
			    front->filters[filter_index]
			    .line[line_index].compare);
	gtk_signal_connect (GTK_OBJECT (entry[line_index]), "changed",
			    GTK_SIGNAL_FUNC (_filter_entry_handler),
			    (gpointer)(gint)filter_index);
	field_selector = gtk_option_menu_new ();
	field_selector_menu = gtk_menu_new ();

	/* The attachment mechanism assumes that a pointer can store 16 bits
	 * but not necessarily 32, since some inferior platforms only do 16.
	 * So I use 4 bits for line and 12 for field. This allows only 16
	 * lines; the current implementation is for only 6. When this is
	 * replaced with GLists of filter lines, this mechanism will have to be
	 * re-thought. Probably just go ahead and assume sizeof (void*) >= 4.
	 */
	for (field_index=0; field_index <= front->last_field+1; field_index++) {
	    if (!field_index)
		field_selector_menu_item = gtk_menu_item_new_with_label
		    (_("--- Unused ---"));
	    else
		field_selector_menu_item = gtk_menu_item_new_with_label
		    (front->fields[field_index-1].name);
	    
	    gtk_widget_set_name (field_selector_menu_item, "field");
	    gtk_signal_connect (GTK_OBJECT (field_selector_menu_item),
				"activate",
				GTK_SIGNAL_FUNC (_filter_menuitem_select),
				(gpointer) ((line_index << 12) | field_index));
	    gtk_menu_append (GTK_MENU (field_selector_menu),
			     field_selector_menu_item);
	    if (front->filters[filter_index].line[line_index].field + 1
		== field_index)
		gtk_menu_set_active (GTK_MENU (field_selector_menu),
					       field_index);
	    gtk_widget_show (field_selector_menu_item);
	}
	gtk_option_menu_set_menu ((GtkOptionMenu*) field_selector,
				  field_selector_menu);
	gtk_widget_show (field_selector);
	
	gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			    field_selector, TRUE, TRUE, 5);

	rule_selector = gtk_option_menu_new ();
	rule_selector_menu = gtk_menu_new ();

	for (rule_index = FILTER_CONTAINS; rule_names[rule_index] != NULL;
	     rule_index++) {
	    rule_selector_menu_item
		= gtk_menu_item_new_with_label (_(rule_names[rule_index]));
	    gtk_widget_set_name (rule_selector_menu_item, "rule");
	    gtk_signal_connect (GTK_OBJECT (rule_selector_menu_item),
				"activate",
				GTK_SIGNAL_FUNC (_filter_menuitem_select),
				(gpointer) ((line_index << 12)
					    | rule_index));
	    gtk_menu_append (GTK_MENU (rule_selector_menu),
			     rule_selector_menu_item);
	    if (front->filters[filter_index].line[line_index].type
		== rule_index)
		gtk_menu_set_active (GTK_MENU (rule_selector_menu),
					       rule_index);
	    gtk_widget_show (rule_selector_menu_item);
	}
	gtk_option_menu_set_menu ((GtkOptionMenu*) rule_selector,
				  rule_selector_menu);
	gtk_widget_show (rule_selector);

	gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			    rule_selector, TRUE, TRUE, 5);
	gtk_widget_show (line_hbox[line_index]);
	
	gtk_widget_show (entry[line_index]);
	gtk_box_pack_start (GTK_BOX (line_hbox[line_index]),
			    entry[line_index], TRUE, TRUE, 5);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (edit_dialog)->vbox),
			    line_hbox[line_index], TRUE, TRUE, 5);
    }
    gtk_widget_show (edit_dialog);
}

/* This is roughly adapted from the sort selection dialog in sort.c. */
void _filter_select (char mode) {
    /* Selects a filter to work with and places its index in working_idx */
    GtkWidget *fsel_dialog,
	*cancel_button,
	*scroll_w,
	*select_list,
	*clist_hbox;
    gchar *select_titles [1] = { "Double-click to select a filter" };
    gchar *select_entries[2] = { "", NULL };

    /* Whereas sort.c uses 180, we do 240 because although 95% of all
     * __Linux__ users have 75dpi servers, I and many non-Linux users have
     * 100dpi servers instead. And I like to actually see my text.
     */
    static const gint SELECT_LIST_WIDTH = 240;
    static const gint SELECT_LIST_HEIGHT = 240;
    gint16 idx;
    
    make_basic_dialog1 ();
    fsel_dialog = dialog1_win;
    
    gtk_window_set_title (GTK_WINDOW (fsel_dialog), _("Select a filter"));
    gtk_signal_connect (GTK_OBJECT  (fsel_dialog), "delete_event",
			GTK_SIGNAL_FUNC (cancel_level1), NULL);
    
    cancel_button = gtk_button_new_with_label (_("  Cancel  "));
    gtk_signal_connect (GTK_OBJECT(cancel_button),"clicked", 
			GTK_SIGNAL_FUNC (cancel_level1), NULL);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (fsel_dialog)->action_area),
			cancel_button, FALSE, FALSE, 0);
    gtk_widget_show (cancel_button);
    
    gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (fsel_dialog)->vbox), 10);
    
    clist_hbox = gtk_hbox_new (FALSE, 5);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (fsel_dialog)->vbox), clist_hbox,
			TRUE, TRUE, 0);
    
    select_list = gtk_clist_new_with_titles (1, select_titles);
    gtk_clist_set_column_width (GTK_CLIST (select_list), 0,
				SELECT_LIST_WIDTH - 30);  
    gtk_widget_set_usize (select_list, SELECT_LIST_WIDTH, SELECT_LIST_HEIGHT);
    gtk_signal_connect (GTK_OBJECT (select_list), "select_row",
			GTK_SIGNAL_FUNC (_filter_selection_made),
			(gpointer)(int)mode);
    gtk_clist_column_titles_passive (GTK_CLIST (select_list));

    scroll_w = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_add (GTK_CONTAINER (scroll_w), select_list);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_w),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
    gtk_container_set_border_width (GTK_CONTAINER (scroll_w), 5);
    
    gtk_box_pack_start (GTK_BOX (clist_hbox), scroll_w, TRUE, TRUE, 0);
    gtk_widget_show (clist_hbox);
    gtk_widget_show (select_list);
    gtk_widget_show (scroll_w);
    
    /* now populate the clist with filter names */
    for (idx = 0; idx < front->filter_ct; idx++) {
	select_entries [0] = front->filters[idx].name;
	gtk_clist_append (GTK_CLIST (select_list), select_entries);
    }
    gtk_widget_show (fsel_dialog);
}

/******************************* END INTERNAL GUI ****************************/

/***
 * This function tells whether record record_index matches the criterion
 * and rule for field field_index. This is a subset of applying a filter,
 * which may call this function many times, at least once for each record.
 * Note that this function only tests one field per call.
 ***/

#define RETURN_AND_FREE(x) {\
    int z = (x);\
    if (use_nocase) g_free (entry);\
    if (use_nocase) g_free (criterion);\
    return z;\
}
gboolean _filter_compare_record (gint32 field_index, gint32 record_index,
				 gchar *criterionx, filter_rule rule,
				 gboolean use_nocase) {
    gint32 column = front->fields[field_index].sheet_column;
    gchar *entryx = (gchar*) gtk_sheet_cell_get_text (GTK_SHEET (front->sheet),
						     record_index, column);
    gchar *entry = entryx;
    gchar *criterion = criterionx;
    
    /* First handle the case of blank or nonblank criteria */
    if (!entry && rule != FILTER_ISBLANK)
	return FALSE;
    if (rule == FILTER_ISBLANK && (!entry || strlen (entry) == 0))
	return TRUE;
    switch (front->fields[field_index].type) {
	/* We are required to handle string/text types ourselves */
    case FIELD_TYPE_TEXT:
    {
	/* We'll allow nocase comparison for strings as we do with match/find.
	 * By default, filters are case-sensitive. That's just the Unix way.
	 * It also happens to be the RIGHT way; if you want things to be
	 * considered the same, type them the same.
	 */
	if (use_nocase) {
	    entry = g_strdup (entryx);
	    g_strdown (entry);
	    criterion = g_strdup (criterionx);
	    g_strdown (criterion);
	} else {
	    entry = entryx;
	    criterion = criterionx;
	}
	switch (rule) {
	case FILTER_CONTAINS:
	    if (strstr (entry, criterion))
		RETURN_AND_FREE (1);
	    RETURN_AND_FREE (0);
	    break;
	case FILTER_CONTAINSNO:
	    RETURN_AND_FREE (!strstr (entry, criterion));
	    break;
	case FILTER_IS:
	    RETURN_AND_FREE (!strcmp (entry, criterion));
	    break;
	case FILTER_ISNOT:
	    RETURN_AND_FREE (!!strcmp (entry, criterion));
	    break;
	case FILTER_LESS:
	    RETURN_AND_FREE ((strcmp (entry, criterion) > 0));
	    break;
	case FILTER_MORE:
	    RETURN_AND_FREE ((strcmp (entry, criterion) < 0));
	    break;
	case FILTER_ISBLANK:
	    RETURN_AND_FREE (FALSE);
	    break;
	case FILTER_ISNOTBLANK:
	    RETURN_AND_FREE (TRUE);
	    break;
	}
    }
    break;

#undef RETURN_AND_FREE

    /* We must still handle all of the following for the other three types:
     * CONTAINS, CONTAINSNO, ISBLANK, ISNOTBLANK.
     */
    case FIELD_TYPE_NUMERIC:
    case FIELD_TYPE_DATE:
    case FIELD_TYPE_TIME:
    {
	gdouble d_entry = qls2d (entry, front->fields[field_index].type,
				  front->fields[field_index].formatting);
	gdouble d_criterion = qls2d (criterion,
				      front->fields[field_index].type,
				      front->fields[field_index].formatting);
	switch (rule) {
	case FILTER_CONTAINS:
	    /* For contains/does not contain, we ignore the conversion and do
	     * text comparison.
	     */
	    if (strstr (entry, criterion))
		return 1;
	    return 0;
	    break;
	case FILTER_CONTAINSNO:
	    if (strstr (entry, criterion))
		return 0;
	    return 1;
	    break;
	case FILTER_IS:
	    return (d_entry == d_criterion);
	    break;
	case FILTER_ISNOT:
	    return (d_entry != d_criterion);
	    break;
	case FILTER_LESS:
	    return (d_entry < d_criterion);
	    break;
	case FILTER_MORE:
	    return (d_entry > d_criterion);
	    break;
	case FILTER_ISBLANK:
	    return FALSE;
	    break;
	case FILTER_ISNOTBLANK:
	    return TRUE;
	    break;
	}
    }
    break;
    default:
    {
	level2_error ("Internal error in tools_compare_record:\n"
		      "\tInvalid field type.", _("  Bummer  "));
	return 0;
    }
    break;
    }
    return 0; /* Hello, sgi cc! */
}


