

/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/
/*! \file bisection.cpp
    \brief bisection 1-D solver

    \fullpath
    ql/Solvers1D/%bisection.cpp
*/

// $Id: bisection.cpp,v 1.5 2002/01/16 14:41:16 nando Exp $

/* The implementation of the algorithm was inspired by
 * "Numerical Recipes in C", 2nd edition, Press, Teukolsky, Vetterling, Flannery
 * Chapter 9
 */


#include <ql/Solvers1D/bisection.hpp>

namespace QuantLib {

    namespace Solvers1D {

        double Bisection::solve_(const ObjectiveFunction& f,
                                 double xAccuracy) const {

            double dx, xMid, fMid;

            // Orient the search so that f>0 lies at root_+dx
            if (fxMin_ < 0.0) {
                dx = xMax_-xMin_;
                root_ = xMin_;
            } else {
                dx = xMin_-xMax_;
                root_ = xMax_;
            }

            while (evaluationNumber_<=maxEvaluations_) {
                dx /= 2.0;
                xMid=root_+dx;
                fMid=f(xMid);
                evaluationNumber_++;
                    if (fMid <= 0.0)
                        root_=xMid;
                    if (QL_FABS(dx) < xAccuracy || fMid == 0.0) {
                        return root_;
                    }
            }
            throw Error("Bisection: maximum number of function evaluations (" +
                            IntegerFormatter::toString(maxEvaluations_) +
                            ") exceeded");

            QL_DUMMY_RETURN(0.0);
        }

    }

}
