#! /usr/bin/env python

# setup.py --- Setup script for PyXMMS
# Copyright (c) 2002, 2003, 2005 Florent Rougon
#
# This file is part of PyXMMS.
#
# PyXMMS is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 dated June, 1991.
#
# PyXMMS is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING. If not, write to the
# Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
# Boston, MA  02110-1301 USA.

import os, string, sys, distutils.dist
from distutils.core import setup, Extension

PACKAGE = "pyxmms"
VERSION = "2.06"
GLIB_CONFIG = "glib-config"


def get_glib_config():
    """Get the compilation and link parameters for glib."""
    glib_opts = {}
    for option in ("cflags", "libs"):
        glib_config_pipe = os.popen("%s --%s" % (GLIB_CONFIG, option), 'r')
        # strip the trailing newline
        glib_opts[option] = glib_config_pipe.read()[:-1]
        if glib_config_pipe.close() is not None:
            sys.exit("%s returned a non-zero exit status. Aborting."
                     % GLIB_CONFIG)

    # Separate the -I options from the other options because to feed to
    # distutils.Extension as include_dirs and extra_compile_args.
    # This is not perfect since an argument starting with -I might not be a -I
    # option, I think, but Distutils' extra_compile_args go to the end of the
    # compiler command line, so we have to use something else for include
    # directories...
    compile_args = []
    include_dirs = []
    for opt in string.split(glib_opts["cflags"]):
        if len(opt) >= 3 and opt[:2] == "-I":
            include_dirs.append(opt[2:])
        else:
            compile_args.append(opt)
    
    # Make a list of the link arguments to feed to distutils.Extension as
    # extra_link_args
    link_args = string.split(glib_opts["libs"])

    return (include_dirs, compile_args, link_args)


def setup_args():
    """Craft appropriate arguments for distutils.setup."""
    (glib_include_dirs, glib_compile_args, glib_link_args) = get_glib_config()
    
    # Modules built whatever the version of the running Python
    ext_modules = [Extension("xmms._xmmscontrol",
                             ["src/_xmmscontrolmodule.c"],
                             include_dirs=glib_include_dirs,
                             extra_compile_args=glib_compile_args,
                             libraries=["xmms"],
                             extra_link_args=glib_link_args),
                   Extension("xmms._xmmsconfig",
                             ["src/_xmmsconfigmodule.c"],
                             include_dirs=glib_include_dirs,
                             extra_compile_args=glib_compile_args,
                             libraries=["xmms"],
                             extra_link_args=glib_link_args)]

    # Trove classifiers picked up from the list at
    # http://www.python.org/pypi?:action=list_classifiers
    trove_classifiers = [
        "Development Status :: 6 - Mature",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: GNU General Public License (GPL)",
        "Operating System :: OS Independent",
        "Programming Language :: Python",
        "Topic :: Multimedia :: Sound/Audio :: CD Audio :: CD Playing",
        "Topic :: Multimedia :: Sound/Audio :: Players",
        "Topic :: Multimedia :: Sound/Audio :: Players :: MP3",
        "Topic :: Software Development :: Libraries :: Python Modules"]

    # The metadata used below is mainly intended for PyPI... and the PyPI
    # tutorial at http://www.python.org/~jeremy/weblog/030924.html recommends
    # giving the URL of the license, instead of its name (PEP 241 requires the
    # name). So let's go with the URL...
    setup_args = {
        "name": PACKAGE,
        "version": VERSION,
        "author": "Florent Rougon",
        "author_email": "flo@via.ecp.fr",
        "maintainer": "Florent Rougon",
        "maintainer_email": "flo@via.ecp.fr",
        "url": "http://people.via.ecp.fr/~flo/",
        "download_url": "http://people.via.ecp.fr/~flo/2002/PyXMMS/dist/",
        "license": "http://www.gnu.org/licenses/gpl.html",
        "platforms": ["any"],
        "description": "A Python interface to XMMS",
        "long_description":  """\
PyXMMS is a Python package allowing full control of XMMS as well as
management of its configuration file. XMMS is a multimedia player
written for the X Window System.""",
        "keywords": ["xmms"],
        "classifiers": trove_classifiers,
        "package_dir": {"xmms": "src"},
        "ext_modules": ext_modules,
        "packages": ["xmms"]
        }
    return setup_args


def main():
    if sys.hexversion < 0x02020000:
        sys.stderr.write(
            "Python 2.2 or later is needed to run PyXMMS. Sorry.\n")
        sys.exit(1)
    
    # Patch distutils if it can't cope with the "classifiers" or
    # "download_url" keywords
    if sys.hexversion < 0x02020300:
        from distutils.dist import DistributionMetadata
        DistributionMetadata.classifiers = None
        DistributionMetadata.download_url = None

    setup(**setup_args())


if __name__ == "__main__": main()
