// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "xgl2.h"
#include "exceptions.h"
#include <iostream>
#include <stdexcept>
#include <gtkmm/main.h>
#include <boost/scoped_array.hpp>

#include <pango/pangoft2.h>

namespace {
	std::size_t 
	next_power_of_two( std::size_t s) 
	{
		// TODO: Maybe there is a faster means other than linear search?
		std::size_t ctr = 1;
		std::size_t power = 2;
		const std::size_t max_digits = std::numeric_limits<std::size_t>::digits;
		while (ctr <= max_digits) {
			if (s < power)
				return ctr;
			ctr++;
			power <<= 1;
		}
		// Failure - s is > 2 ** max_digits;
		return 0;
	}
}

namespace visual {

#if USE_NEWFONT
xgl2Font::layout::layout( 
	std::string str, 
	std::size_t font_size, 
	Gtk::Widget* widget, 
	Glib::RefPtr<Pango::Context> ft2_context)
	: pixel_extent_width(0), pixel_extent_height(0),
	texture_id( 0), tex_width(0), tex_height(0)
{
	// Get the right font
	Glib::RefPtr<Pango::Context> widget_context = widget->get_pango_context();
	Pango::FontDescription font_desc = widget_context->get_font_description();
	font_desc.set_size( font_size * PANGO_SCALE);
	ft2_context->set_font_description( font_desc);
	
	// Compute the layout
	Glib::RefPtr<Pango::Layout> layout = Pango::Layout::create( ft2_context);
	layout->set_alignment( Pango::ALIGN_LEFT);
	layout->set_text( str);
	
	// Get the device-context extents for the layout.
	Pango::Rectangle pango_extents = layout->get_pixel_logical_extents();
	pixel_extent_width = pango_extents.get_width();
	pixel_extent_height = pango_extents.get_height();
		
	// Allocate userspace memory for rendering the data.
	FT_Bitmap bitmap;
	bitmap.rows = pixel_extent_width;
	bitmap.width = pixel_extent_height;
	bitmap.pitch = bitmap.width;
	boost::scoped_array<unsigned char> backing_store( 
		new unsigned char[pixel_extent_width*pixel_extent_height]);
	memset( backing_store.get(), 0, pixel_extent_width * pixel_extent_height);
	bitmap.buffer = backing_store.get();
	bitmap.num_grays = 256;
	bitmap.pixel_mode = ft_pixel_mode_grays;
	
	// Render the text to backing store
	pango_ft2_render_layout( &bitmap, layout->gobj(), -pango_extents.get_x(), 0);
	
	// Allocate a texture object and texture memory
	tex_width = next_power_of_two( std::size_t( pixel_extent_width));
	tex_height = next_power_of_two( std::size_t( pixel_extent_height));
	boost::scoped_array<unsigned char> dummy_data( 
		new unsigned char[tex_width*tex_height]);
	glEnable( GL_TEXTURE_2D);
	glGenTextures( 1, &texture_id);
	glBindTexture( GL_TEXTURE_2D, texture_id);
	// Configure this texture
	glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexImage2D( GL_TEXTURE_2D, 0, GL_ALPHA8, 
		tex_width, tex_height, 0, 
		GL_ALPHA, GL_UNSIGNED_BYTE, dummy_data.get());
	
	// Dump the rendered text run into OpenGL memory
	glTexSubImage2D( GL_TEXTURE_2D, 0, 
		0, 0, pixel_extent_width, pixel_extent_height, 
		GL_ALPHA, GL_UNSIGNED_BYTE, backing_store.get());
	glDisable( GL_TEXTURE_2D);
}

xgl2Font::layout::~layout()
{
	// TODO: Needs a "pending deletion" system for this deallocation
	if (texture_id != 0) {
		glDeleteTextures( 1, &texture_id);
	}
}

void
xgl2Font::layout::gl_render() const
{
	float right = float( tex_width) / float( pixel_extent_width);
	float top = float( tex_height) / float( pixel_extent_height);
	glEnable( GL_BLEND);
	glEnable( GL_TEXTURE_2D);
	glBindTexture( GL_TEXTURE_2D, texture_id);
	glBegin( GL_QUADS);
		glTexCoord2f( 0.0f, 0.0f);
		glVertex2f( -pixel_extent_width * 0.5f, -pixel_extent_height * 0.5f);
		glTexCoord2f( right, 0.0f);
		glVertex2f( pixel_extent_width * 0.5f, -pixel_extent_height * 0.5f);
		glTexCoord2f( right, top);
		glVertex2f( pixel_extent_width * 0.5f, pixel_extent_height * 0.5f);
		glTexCoord2f( 0, top);
		glVertex2f( -pixel_extent_width * 0.5f, pixel_extent_height * 0.5f);
	glEnd();
	glDisable( GL_TEXTURE_2D);
	glDisable( GL_BLEND);
}

vector 
xgl2Font::layout::get_extent() const
{
	return vector( pixel_extent_width, pixel_extent_height);
}

#if 0
xgl2Font::xgl2Font( 
	boost::shared_ptr<Gtk::Widget> parent_widget, 
	Glib::RefPtr<Pango::Context> ft2_context, std::size_t font_size)
{
	// TODO: Finish the implementation of this function
	ft2_font_map = Glib::wrap( pango_ft2_font_map_new());
	// TODO: Determine the actual screen resolution (in DPI) at runtime.
	pango_ft2_font_map_set_resolution( (PangoFT2FontMap*)ft2_font_map.gobj(), 72, 72);
	
}

boost::shared_ptr<layout>
xgl2Font::lay_out( const std::string& str)
{
	Glib::RefPtr<Pango::Context> ft2_context = Glib::wrap(
		pango_ft2_font_map_create_context( (PangoFT2FontMap*)ft2_font_map.gobj()));
	boost::shared_ptr<layout> ret( 
		new layout( str, font_size, parent_widget, ft2_context));
	return ret;
}
#endif

#else // !0

/****************  xgl2 Font implementation    ********************/

xgl2Font::xgl2Font( struct glContext& _cx, const char* name, double size) 
	: cx(_cx), refcount(1), listBase(0)
{
	
	// Attempt to create a user-defined font.
	Pango::FontDescription f_descr( "courier 12"); //  = Glib::wrap(gtk_style_new())->get_font();
	// std::cout << f_descr.to_string() << std::endl;
	// Allocate and fill a glList with our font glyphs.
	cx.makeCurrent();
	listBase = glGenLists( 256);
	font = Gdk::GL::Font::use_pango_font( f_descr, 0, 128, listBase);
	
	// Attempt to provide a safe fallback font.
	if (font.is_null()) {
		std::cerr << __FILE__ << ": " << __LINE__ << ": Invalid font description: " << name << std::endl 
		          << "  Falling back to: courier 12" << std::endl;
		font = Gdk::GL::Font::use_pango_font( Pango::FontDescription( "courier 12"), 
		                                      0, 128, listBase);
	}
	
	// And gracefully crash, otherwise.
	if (font.is_null()) {
		std::ostringstream err;
		err << __FILE__ << ": " << __LINE__ << ": Failed to load any font." << std::ends;
		const char* msg = err.str().c_str();
		throw std::runtime_error( msg);
	}
	cx.makeNotCurrent();
	
	metrics = font->get_metrics();
}

xgl2Font::~xgl2Font()
{
}

void 
xgl2Font::draw( const char *c) 
{
	if (font) {
		glListBase(listBase);
		glCallLists(strlen(c), GL_UNSIGNED_BYTE, c);
	}
}

// TODO: pass these things as genuine std::strings.
double 
xgl2Font::getWidth( const char *c) 
{
	int ret = strlen( c) * metrics.get_approximate_char_width() * Pango::SCALE;
	return double( ret*2 / cx.width());
}

double 
xgl2Font::ascent() 
{
	return ( double(metrics.get_ascent()) / cx.height());
}

double 
xgl2Font::descent() 
{
	return double(metrics.get_descent()) / cx.height();
}

void 
xgl2Font::release() 
{
	refcount--;
	if (!refcount) {
		cx.add_pending_glDeleteList( listBase, 256);
		delete(this);
	}
}
#endif

/*****************  xgl2Context Implementation  *******************/

// Only sets state variables to zero/null

xgl2Context::xgl2Context()
	: buttonState( 0),
	  buttonsChanged( 0),
	  extKeyState( 0),
	  m_window( 0)
{
}

xgl2Context::~xgl2Context()
{
	cleanup();
}

void
xgl2Context::cleanup()
{
}

/* title: a null-terminated string for the window's title
 * x, y: the position for the window on the screen
 * width, height: the width and height of the window, in pixels.
 */
bool
xgl2Context::initWindow( const char* title, int x, int y, int width, int height, int flags)
{
	cleanup();
	// Gtk::GL::init_check( NULL, NULL);
	
	// Just to be sure...
	if (!title) 
		title = "Visual Python";
	
	
	int attributes[] = {
		Gdk::GL::RGBA,
		Gdk::GL::DOUBLEBUFFER,
		Gdk::GL::DEPTH_SIZE, 12,
		Gdk::GL::ATTRIB_LIST_NONE, // May set to Gdk::GL::STEREO
		Gdk::GL::ATTRIB_LIST_NONE
	};
	
	if (flags & glContext::QB_STEREO) {
		// Try to set stereo
		attributes[4] = Gdk::GL::STEREO;
	}
	
	Glib::RefPtr<Gdk::GL::Config> glconfig = Gdk::GL::Config::create( attributes);
	if (!glconfig && flags & glContext::QB_STEREO) {
		// You failed, try to use simple doublebuffering.
		attributes[4] = Gdk::GL::ATTRIB_LIST_NONE;
		glconfig = Gdk::GL::Config::create( attributes);
	}
	
	if (!glconfig) {
		std::cerr << __FILE__ << ": " << __LINE__ << ": Unable to create GL configuration." << std::endl;
		return false;
	}
	
	// Create and initialize a drawing area widget.	
	m_area.reset(  new Gtk::GL::DrawingArea( glconfig));
	// m_context = m_area->get_gl_context();

	m_window.reset( new Gtk::Window);
	if (flags & glContext::FULLSCREEN)
		m_window->fullscreen();
	
	// Attach the widget to the window.
	m_window->add( *m_area);
	
	// Set the window's properties.
	m_window->set_title( title);
	if (width && height) {
		m_window->set_default_size( width, height);
	}
	if (x || y)
		m_window->move( x, y);
	
	// Enable the signal handlers
	m_area->set_events( Gdk::EXPOSURE_MASK
	                   | Gdk::BUTTON_PRESS_MASK
	                   | Gdk::BUTTON_RELEASE_MASK
	                   | Gdk::POINTER_MOTION_MASK );
	// Connect the signal handlers
	m_window->signal_configure_event().connect( SigC::slot( *this, &xgl2Context::on_configure_event));
 	m_area->signal_motion_notify_event().connect( SigC::slot( *this, &xgl2Context::on_mouse_motion_event));
	m_window->signal_delete_event().connect( SigC::slot( *this, &xgl2Context::on_delete_event));
	m_area->signal_button_press_event().connect( SigC::slot( *this, &xgl2Context::on_button_press_event));
	m_area->signal_button_release_event().connect( SigC::slot( *this, &xgl2Context::on_button_release_event));
	m_window->signal_key_press_event().connect( SigC::slot( *this, &xgl2Context::on_key_press_event)); 
	
	
	// Show our widgets
	m_window->show_all();
	return true;
}

void 
xgl2Context::lockMouse() 
{
}
	
void 
xgl2Context::unlockMouse() 
{ 
}
	
void 
xgl2Context::showMouse() 
{
	std::cout << "xgl2.cpp: showMouse() is not implemented" << std::endl; 
}
	
void 
xgl2Context::hideMouse() 
{ 
	std::cout << "xgl2.cpp: hideMouse() is not implemented" << std::endl; 
}

int 
xgl2Context::getMouseButtons()
{
	return buttonState;
}
	
int 
xgl2Context::getMouseButtonsChanged() 
{
	return buttonsChanged;
}

// Change the properties of the window.
// See also initWindow() for the parameter list.
// TODO: Verify that true is the correct return value.
bool
xgl2Context::changeWindow( const char* title, int x, int y, int width, int height, int flags)
{
	if (title) 
		m_window->set_title( title);
	if (x || y) 
		m_window->move( x, y);
	if (width || height) {
		m_window->set_default_size( width, height);
	}
	return true;
}

bool 
xgl2Context::isOpen() 
{ 
	if (m_window) 
		return m_window->is_visible(); 
	else 
		return false; 
}

std::string
xgl2Context::getKeys()
{
	std::string ret;
	if (!key_queue.empty()) {
		ret = key_queue.front();
		key_queue.pop();
	} 
	else ret = "";
	return ret;
}

int 
xgl2Context::getShiftKey()
{
	if (extKeyState & (1 << 3)) 
		return 1;
	else 
		return 0;
}

int 
xgl2Context::getAltKey()
{
	if (extKeyState & (1 << 4)) 
		return 1;
	else 
		return 0;
}

int 
xgl2Context::getCtrlKey()
{
	if (extKeyState & (1 << 5)) 
		return 1;
	else 
		return 0;
}


vector 
xgl2Context::getMouseDelta() 
{ 
	vector ret = mousePos - oldMousePos;
	// Ensures that subsequent checks for mouse movement return 0 if no
	// mouse movement signals have been recieved.
	oldMousePos = mousePos;
	return ret;
}

vector
xgl2Context::getMousePos()
{
	return vector( mousePos.x/m_area->get_width(), mousePos.y/m_area->get_height(), 0);
}


void
xgl2Context::makeCurrent()
{
	if (!m_area->get_gl_drawable())
		return;
	m_area->get_gl_drawable()->make_current( m_area->get_gl_context() );
	delete_pending_lists();
}
	
void
xgl2Context::makeNotCurrent()
{
	if (!m_area->get_gl_drawable())
		return;
	m_area->get_gl_drawable()->wait_gl();
}
	
void
xgl2Context::swapBuffers() 
{
	if (!m_area->get_gl_drawable())
		return;
	m_area->get_gl_drawable()->swap_buffers();
}


// Returns the top-left location of the window on the screen.
// Beware of window managers with gravity other than NW.
vector 
xgl2Context::origin()
{
	if (m_window->get_gravity() != Gdk::GRAVITY_NORTH_WEST) {
		std::cerr << __FILE__ << ": " << __LINE__ << ": WARNING: origin is unreliable.\n";
	}
	int x=0;
	int y=0;
	m_window->get_position( x, y);
	return vector( x, y, 0);
}

// Returns the bottom-right corner of the window on the screen.
vector 
xgl2Context::corner()
{
	if (m_window->get_gravity() != Gdk::GRAVITY_NORTH_WEST){
		std::cerr << __FILE__ << ": " << __LINE__ << ": WARNING: corner is unreliable.\n";
	}
	int x=0;
	int y=0;
	m_window->get_position( x, y);
	return vector( x+m_area->get_width(), y+m_area->get_height(), 0);
}

int 
xgl2Context::width()
{
	return m_area->get_width();
}
	
int 
xgl2Context::height() 
{
	return m_area->get_height();
}

std::string 
xgl2Context::lastError()
{
	return error_message;
}

#if !USE_NEWFONT
// Make sure to call ->release() on the returned pointer.
glFont* 
xgl2Context::getFont(const char* description, double size)
{
	return new xgl2Font( *this, description, size);
}

#else
boost::shared_ptr<glFont>
xgl2Context::getFont( double size)
{
	if (ft2_context.is_null()) {
		ft2_context = Glib::wrap( pango_ft2_get_context( 72, 72));
	}
	return boost::shared_ptr<glFont>( new xgl2Font( m_area, ft2_context, size));
}
#endif

/******************** Signal Handlers ******************************
 * All handlers pass the signal to the Window class's default handler, by returning false
 * These signals handle X events directly, rather than widget-specific events.
 */

// Grab the new mouse position, maintain it in local state.
bool 
xgl2Context::on_mouse_motion_event( GdkEventMotion* event)
{
	oldMousePos = mousePos;
	mousePos.set_x( event->x);
	mousePos.set_y( event->y);
	return false;
}

// The window is being killed by the user via the window manager.
// TODO: Exit the application when this signal is recieved.
bool 
xgl2Context::on_delete_event( GdkEventAny* event)
{
	if (event->type == GDK_DELETE) {
		cleanup();
	}
	return false;
}

// See header documentation for valid buttonState and buttonsChanged values.
bool
xgl2Context::on_button_press_event( GdkEventButton* event)
{
	// event.state may be ctrl, alt, or shift
	// event.button may be 1-5. 1=left, 2=middle, 3=right (normally)
	if (event->button == 1) {
		buttonState |= 1;
		buttonsChanged |= 1;
	}
	else if (event->button == 2) {
		buttonState |= (1 << 2);
		buttonsChanged |= (1 << 2);
	}
	else if (event->button == 3) {
		buttonState |= (1 << 1);
		buttonsChanged |= (1 << 1);
	}
	
	// Only check for the 3 primary extended key types, ignore everything else.
	if (event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK)) {
		if (event->state & GDK_SHIFT_MASK) {
			extKeyState |= (1 << 3);
		}
		if (event->state & GDK_CONTROL_MASK) {
			extKeyState |= (1 << 4);
		}
		if (event->state & GDK_MOD1_MASK) {
			// May vary, but usually represents the alt key.
			extKeyState |= (1 << 5);
		}
	}
	return true;
}

// Clear state flags when a button is released.  
// Perform bitwise AND with the bitwise NOT of the appropriate flag to clear it.
// See also on_button_press().
bool 
xgl2Context::on_button_release_event( GdkEventButton* event)
{
	if (event->button == 1) {
		buttonState &= ~1;
		buttonsChanged |= 1;
	}
	else if (event->button == 2) {
		buttonState &= ~(1 << 2);
		buttonsChanged |= (1 << 2);
	}
	else if (event->button == 3) {
		buttonState &= ~(1 << 1);
		buttonsChanged |= (1 << 1);
	}
	
	if (event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK)) {
		if (event->state & GDK_SHIFT_MASK) {
			extKeyState |= (1 << 3);
		}
		if (event->state & GDK_CONTROL_MASK) {
			extKeyState |= (1 << 4);
		}
		if (event->state & GDK_MOD1_MASK) {
			// May vary, but usually represents the alt key.
			extKeyState |= (1 << 5);
		}
	}
	return true;
}

// Capture all keyboard press events and store them for sequential retrieval.
bool
xgl2Context::on_key_press_event( GdkEventKey* key)
{
  key_queue.push( std::string( key->string)); 
  return true;
}

// Someone resized the window: update the local state variables.
bool
xgl2Context::on_configure_event( GdkEventConfigure* event)
{
	return true;
}

} // !namespace visual
