// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include <boost/python/detail/wrap_python.hpp>

#include "platform.h"
#include <queue>
#include <cstdlib>

#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>

#ifdef HAVE_GTK_GL_EXT_MM
# include <gtkmm/main.h>
# include <gtkmm/gl/init.h>
# include <boost/scoped_ptr.hpp>

boost::scoped_ptr<Gtk::Main> kit;
#else // Must be using Gtk 1.2.
# include <gtk/gtk.h>
#endif

#include "display.h"

namespace visual {

void init_platform()
{
#ifdef HAVE_GTK_GL_EXT_MM
	kit.reset( new Gtk::Main( 0, 0));
	Gtk::GL::init( 0, 0);
	gdk_threads_init();	
#endif
}

void threaded_exit ( int status)
{
	gdk_threads_enter();
	#if HAVE_GTK_GL_EXT_MM
	kit->quit();
	#else
	gtk_main_quit();
	#endif
	gdk_threads_leave();
}

// 2003-02-16 Added explicit casts to unsigned int vice implicit casts from the compiler.
// TODO: Convert time values to integral types at the Python boundary.
void threaded_sleep( double seconds)
{
	  Py_BEGIN_ALLOW_THREADS
	  usleep( (unsigned int)(seconds * 1.0e6));
	  Py_END_ALLOW_THREADS
}

// 2003-02-16 Added explicit casts to unsigned int vice implicit casts from the compiler.
// TODO: Convert time values to integral types at the Python boundary.
void nothread_sleep( double seconds)
{
	usleep( (unsigned int)(seconds * 1.0e6));
}

double sclock()
{
	struct timeval tv;
	gettimeofday( &tv, NULL);

	return (tv.tv_sec + tv.tv_usec / 1.0e6);
}

/******** mutex implementation ********/

mutex::mutex( int spincount, int _count)
	 : count(_count)
{
	if (!g_thread_supported())
		g_thread_init( NULL);
	pthread_mutex_init( &mtx, NULL);
}

mutex::~mutex()
{
	pthread_mutex_destroy( &mtx);
}

/*********** event handling not needed ********/

bool handle_event( PyObject*)
{
	return false;
}

/*********** threaded_timer implementation *************/

int quit( void *)
{
	// Py_Exit() should work, but for some reason it frequently just hangs.
	// So, we force it.
	Py_Exit( 0);
}

void* mainloop ( void* interp)
{
	#ifdef HAVE_GTK_GL_EXT_MM
	kit->run();
	#else
	gtk_init( NULL, NULL);
		#if GTK_VERSION_MAJOR >= 2
		gdk_threads_init();
		#endif
	gtk_main();
	#endif
	PyGILState_STATE state = PyGILState_Ensure();
	Py_AddPendingCall( &quit, 0);
	PyGILState_Release( state);
	pthread_exit(0);
}

// 2003-02-16 Added explicit casts to unsigned int vice implicit casts from the
// compiler.
void _threaded_timer( double seconds, bool (*callback)(void*), void *data)
{
	static pthread_t thread;
	// Note, gcc reports that thread_init is an unused variable.  Strictly speaking
	// it is since it is never read, but it is used to ensure that pthread_create()
	// is called only once.
	static int thread_init __attribute__((__unused__)) 
		= pthread_create( &thread, NULL, mainloop, 0);

	if (callback) {
		gdk_threads_enter();
		gtk_timeout_add( (unsigned int)(seconds * 1000.0), (gint(*)(void*))callback, data);
		gdk_threads_leave();
	}
}

} // !namespace visual
