// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "light.h"
#include <boost/python/extract.hpp>

namespace visual {
	
lighting::lighting() 
{
	ambient = 0.2;
	n_lights = 2;
	L[0] = vector(0.25, 0.5, 1.0).norm() * 0.8;
	L[1] = vector(-1.0,-0.25, -0.5).norm() * 0.3;
}

lighting::lighting(const lighting& copy, const tmatrix& frame) 
{
	ambient = copy.ambient;
	n_lights = copy.n_lights;
	for(int l=0;l<n_lights;l++) {
		L[l] = frame.times_v( copy.L[l] );
	}
}

lighting::lighting( const lighting& old, boost::python::list lights)
{
	ambient = old.ambient;
	n_lights = boost::python::extract<int>(lights.attr("__len__")());
	if (n_lights > 8)
		n_lights = 8;
	for (int i = 0; i < n_lights; i++) {
		L[i] = vector(lights[i]);
	}
}

double 
lighting::illuminate( double nx, double ny, double nz) 
{
	double d;
	double i = ambient;
	for(int l=0;l<n_lights;l++) {
		d = L[l].x*nx + L[l].y*ny + L[l].z*nz;
		if (d>0) i+=d;
	}
	return i;
}

double 
lighting::illuminate( const vector& normal) 
{
	double d;
	double i = ambient;
	for(int l=0; l<n_lights; l++) {
		d = L[l].dot(normal);
		if (d>0) i+=d;
	}
	return i;
}

boost::python::list
lighting::as_list()
{
	boost::python::list ret;
	for ( int i = 0; i < n_lights; i++) {
		ret.append( L[i]);
	}
	return ret;
}

} // !namespace visual
