/* KInterbasDB Python Package - Implementation of Dynamic Type Translation
**
** Version 3.1
**
** The following contributors hold Copyright (C) over their respective
** portions of code (see license.txt for details):
**
** [Original Author (maintained through version 2.0-0.3.1):]
**   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
** [Maintainers (after version 2.0-0.3.1):]
**   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
**   2002-2004 [dsr]   David Rushby          <woodsplitter@rocketmail.com>
** [Contributors:]
**   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
**   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
*/

static PyObject *dynamically_type_convert_input_obj_if_necessary(
    PyObject *py_input,
    boolean is_array_element,
    unsigned short dialect,
    short data_type, short data_subtype, short scale,
    PyObject *converter
  );


/* Infinitely peristent global variables that are "private" to this source file: */
PyObject *_type_names_all_supported;

PyObject *_cached_type_name_TEXT;
PyObject *_cached_type_name_TEXT_UNICODE;
PyObject *_cached_type_name_BLOB;

PyObject *_cached_type_name_INTEGER;
PyObject *_cached_type_name_FIXED;
PyObject *_cached_type_name_FLOATING;

PyObject *_cached_type_name_TIMESTAMP;
PyObject *_cached_type_name_DATE;
PyObject *_cached_type_name_TIME;


#define IS_UNICODE_CHAR_OR_VARCHAR(data_type, data_subtype) \
  (((data_type) == SQL_VARYING || (data_type) == SQL_TEXT) && (data_subtype) > 2)

#define CACHED_TYPE_NAME_TEXT_OR_TEXT_UNICODE(data_subtype) \
  (((data_subtype) <= 2) ? _cached_type_name_TEXT : _cached_type_name_TEXT_UNICODE)


/* INIT_CONST_TYPE_NAME: */
#define _ICTN(ptr, name) \
  if ( ( ptr = PyString_FromString(name) ) == NULL ) { \
    goto INIT_TYPE_TRANSLATION_FAILED; \
  } \
  if ( PyList_Append(_type_names_all_supported, ptr) != 0 ) { \
    goto INIT_TYPE_TRANSLATION_FAILED; \
  }

static int init_kidb_type_translation(void) {
  _type_names_all_supported = PyList_New(0);
  if (_type_names_all_supported == NULL) {
    goto INIT_TYPE_TRANSLATION_FAILED;
  }

  _ICTN( _cached_type_name_TEXT,           "TEXT"                            );
  _ICTN( _cached_type_name_TEXT_UNICODE,   "TEXT_UNICODE"                    );
  _ICTN( _cached_type_name_BLOB,           "BLOB"                            );

  _ICTN( _cached_type_name_INTEGER,        "INTEGER"                         );
  _ICTN( _cached_type_name_FIXED,          "FIXED"                           );
  _ICTN( _cached_type_name_FLOATING,       "FLOATING"                        );

  _ICTN( _cached_type_name_TIMESTAMP,      "TIMESTAMP"                       );
  _ICTN( _cached_type_name_DATE,           "DATE"                            );
  _ICTN( _cached_type_name_TIME,           "TIME"                            );

  return 0;

 INIT_TYPE_TRANSLATION_FAILED:
  return -1;
} /* init_kidb_type_translation */


static PyObject *_get_cached_type_name_for_conventional_code(
    unsigned short dialect, short data_type, short data_subtype, short scale
  )
{
  switch (data_type) {
    case SQL_TEXT:
    case SQL_VARYING:
      return CACHED_TYPE_NAME_TEXT_OR_TEXT_UNICODE(data_subtype);

    case SQL_BLOB:
      return _cached_type_name_BLOB;

    case SQL_SHORT:
    case SQL_LONG:
    #ifdef INTERBASE6_OR_LATER
    case SQL_INT64:
    #endif /* INTERBASE6_OR_LATER */
      return
            IS_FIXED_POINT__CONVENTIONAL(dialect, data_type, data_subtype, scale)
          ? _cached_type_name_FIXED
          : _cached_type_name_INTEGER
        ;

    case SQL_FLOAT:
    case SQL_DOUBLE:
    case SQL_D_FLOAT:
      /* 2005.07.11:  Internal floating point value can "logically" represent
      ** fixed-point value in dialect < 3 databases. */
      if (dialect < 3 && scale != 0) {
        return _cached_type_name_FIXED;
      } else {
        return _cached_type_name_FLOATING;
      }

    case SQL_TIMESTAMP:
      return _cached_type_name_TIMESTAMP;
    #ifdef INTERBASE6_OR_LATER
    case SQL_TYPE_DATE:
      return _cached_type_name_DATE;
    case SQL_TYPE_TIME:
      return _cached_type_name_TIME;
    #endif /* INTERBASE6_OR_LATER */

    default:
      #if PYTHON_2_2_OR_LATER
      {
        PyObject *err_msg = PyString_FromFormat(
            "Unable to determine type name from these parameters:"
            "  dialect: %d, data_type: %d, data_subtype: %d, scale: %d",
            (int) dialect, (int) data_type, (int) data_subtype,  (int) scale
          );
        if (err_msg != NULL) {
          raise_exception(InternalError, PyString_AS_STRING(err_msg));
          Py_DECREF(err_msg);
        }
      }
      #else /* not PYTHON_2_2_OR_LATER */
        raise_exception(InternalError,
            "_get_cached_type_name_for_conventional_code: unknown type"
          );
      #endif

      return NULL;
  }
} /* _get_cached_type_name_for_conventional_code */


static PyObject *_get_cached_type_name_for_array_code(
    unsigned short dialect, short data_type, short data_subtype, short scale
  )
{
  switch (data_type) {
    case blr_text:
    case blr_text2:
    case blr_varying:
    case blr_varying2:
    case blr_cstring:
    case blr_cstring2:
      return CACHED_TYPE_NAME_TEXT_OR_TEXT_UNICODE(data_subtype);

    case blr_quad:
      /* ISC_QUAD structure; since the DB engine doesn't support arrays of
      ** arrays, assume that this item refers to a blob id. */
    case blr_blob:
    case blr_blob_id:
      return _cached_type_name_BLOB;

    case blr_short:
    case blr_long:
    #ifdef INTERBASE6_OR_LATER
    case blr_int64:
    #endif /* INTERBASE6_OR_LATER */
      return
            IS_FIXED_POINT__ARRAY_EL(dialect, data_type, data_subtype, scale)
          ? _cached_type_name_FIXED
          : _cached_type_name_INTEGER
        ;

    case blr_float:
    case blr_double:
    case blr_d_float:
      /* 2005.07.11:  Internal floating point value can "logically" represent
      ** fixed-point value in dialect < 3 databases. */
      if (dialect < 3 && scale != 0) {
        return _cached_type_name_FIXED;
      } else {
        return _cached_type_name_FLOATING;
      }

    case blr_timestamp:
      return _cached_type_name_TIMESTAMP;
    #ifdef INTERBASE6_OR_LATER
    case blr_sql_date:
      return _cached_type_name_DATE;
    case blr_sql_time:
      return _cached_type_name_TIME;
    #endif /* INTERBASE6_OR_LATER */

    default:
      raise_exception(InternalError,
          "_get_cached_type_name_for_array_code: unknown type"
        );
      return NULL;
  }
} /* _get_cached_type_name_for_array_code */


static PyObject *_get_converter(
    PyObject *trans_dict,
    unsigned short dialect,
    short data_type, short data_subtype, short scale,
    boolean is_array_field
  )
{
  /* Returns a borrowed reference to the converter if one is registered in
  ** $trans_dict for the specified type  (the registered converter might be
  ** Py_None to mandate "naked" translation).
  ** If no converter was present, returns NULL.
  ** Also returns NULL on error, so use PyErr_Occurred() to determine whether
  ** the NULL return value indicates "missing" or "error." */
  PyObject *converter = NULL;

  if (trans_dict != NULL) {
    PyObject *type_name = (
        is_array_field ?
            _get_cached_type_name_for_array_code(dialect,
                data_type, data_subtype, scale
              )
          : _get_cached_type_name_for_conventional_code(dialect,
                data_type, data_subtype, scale
              )
      );
    if (type_name == NULL) { goto fail; }
    converter = PyDict_GetItem(trans_dict, type_name);
    if (converter != NULL) {
      return converter;
    }
  }

  /* Converter couldn't be found (not an error). */
  assert (converter == NULL);
  return NULL;

  fail:
    /* An exception should've already been set: */
    assert (PyErr_Occurred());
    return NULL;
} /* _get_converter */


#define _make__connection_get_DIRECTION_converter(direction) \
  static PyObject *connection_get_ ## direction ## _converter( \
      ConnectionObject *conn, \
      short data_type, short data_subtype, short scale, \
      boolean is_array_field \
    ) \
  { \
    PyObject *converter = _get_converter(conn->type_trans_ ## direction, \
        conn->dialect, \
        data_type, data_subtype, scale, is_array_field \
      ); \
    if (converter == NULL && !PyErr_Occurred()) { \
      /* No converter was present; return borrowed ref to None. */ \
      converter = Py_None; \
    } \
    return converter; \
  }

_make__connection_get_DIRECTION_converter(out)
_make__connection_get_DIRECTION_converter(in)


#define _make__cursor_get_DIRECTION_converter(direction) \
  static PyObject *cursor_get_ ## direction ## _converter( \
      CursorObject *cursor, \
      short data_type, short data_subtype, short scale, \
      boolean is_array_field \
    ) \
  { \
    PyObject *trans_dict = cursor->type_trans_ ## direction; \
    PyObject *converter = _get_converter(trans_dict, \
        cursor->connection->dialect, \
        data_type, data_subtype, scale, is_array_field \
      ); \
    \
    if (converter != NULL || PyErr_Occurred()) { \
      return converter; \
    } \
    \
    /* Fall back on the connection's translation dictionary, if any. */ \
    return connection_get_ ## direction ## _converter(cursor->connection, \
        data_type, data_subtype, scale, is_array_field \
      ); \
  }

_make__cursor_get_DIRECTION_converter(in)
_make__cursor_get_DIRECTION_converter(out)


/* 2003.10.16: */
static PyObject *connection_get_translator_output_type(
    ConnectionObject *con, PyObject *translator_key
  )
{
  /* Helper function for cursor_get_translator_output_type. */
  assert (PyString_Check(translator_key));
{
  PyObject *output_type_dict = con->output_type_trans_return_type_dict;
  if (output_type_dict != NULL) {
    return PyDict_GetItem(output_type_dict, translator_key);
  }
  return NULL;
}
} /* connection_get_translator_output_type */


/* cursor_get_translator_output_type's search might "bubble" to its connection
** in a manner similar to the "bubble" in cursor_get_out_converter. */
static PyObject *cursor_get_translator_output_type(
    CursorObject *cursor, PyObject *translator_key
  )
{
  /* If a record of the return type of the output translator registered under
  ** $translator_key is found, returns a borrowed reference to that type.
  ** Otherwise, returns NULL (which simply means "not found"--it doesn't mean
  ** there was an error). */
  assert (PyString_Check(translator_key));
{
  PyObject *output_type_dict = cursor->output_type_trans_return_type_dict;
  if (output_type_dict != NULL) {
    PyObject *output_type = PyDict_GetItem(output_type_dict, translator_key);
    if (output_type != NULL) {
      return output_type; /* Return borrowed reference. */
    } /* Else, fall through and search cursor->connection's output type dict. */
  }

  return connection_get_translator_output_type(cursor->connection, translator_key);
}
} /* cursor_get_translator_output_type */


#define TYPE_NAMES_ALL_VALID             1
#define TYPE_NAMES_INVALID               0
#define TYPE_NAMES_VALIDATION_PROBLEM   -1

static int _validate_type_names(PyObject *trans_dict) {
  /* Returns:
  ** TYPE_NAMES_ALL_VALID if all type names are valid,
  ** TYPE_NAMES_INVALID if at least one type name is invalid,
  ** TYPE_NAMES_VALIDATION_PROBLEM upon error in validation process. */
  int status = TYPE_NAMES_VALIDATION_PROBLEM; /* Guilty until proven innocent. */
  PyObject *keys = PyDict_Keys(trans_dict);
  int key_count;
  int i;

  if (keys == NULL) {
    goto _VALIDATE_TYPE_NAMES_FAIL;
  }

  key_count = PyList_GET_SIZE(keys);
  for (i = 0; i < key_count; i++) {
    /* PyList_GET_ITEM "returns" a borrowed ref, and can't fail as long as keys
    ** is of the correct type, which it certainly is here. */
    PyObject *k = PyList_GET_ITEM(keys, i);

    int contains = PySequence_Contains(_type_names_all_supported, k);
    if (contains == -1) { /* error in PySequence_Contains */
      goto _VALIDATE_TYPE_NAMES_FAIL;
    } else if (contains == 0) {
      /* k was not in the master list of supported type names. */
      #if PYTHON_2_2_OR_LATER
        PyObject *str_all_supported = NULL;
        PyObject *str_k = PyObject_Str(k);
        if (str_k == NULL) {
          goto _VALIDATE_TYPE_NAMES_FAIL;
        }

        str_all_supported = PyObject_Str(_type_names_all_supported);
        if (str_all_supported == NULL) {
          Py_DECREF(str_k);
          goto _VALIDATE_TYPE_NAMES_FAIL;
        }

        {
          PyObject *form_msg = PyString_FromFormat(
              "Cannot translate type '%s'. Type must be one of %s.",
              PyString_AS_STRING(str_k), PyString_AS_STRING(str_all_supported)
            );
          Py_DECREF(str_k);
          Py_DECREF(str_all_supported);
          if (form_msg == NULL) {
            goto _VALIDATE_TYPE_NAMES_FAIL;
          }
          raise_exception(ProgrammingError, PyString_AS_STRING(form_msg));
          Py_DECREF(form_msg);
        }
      #else /* not PYTHON_2_2_OR_LATER */
        raise_exception(ProgrammingError, "Cannot translate this type.");
      #endif /* PYTHON_2_2_OR_LATER */
      status = TYPE_NAMES_INVALID;
      goto _VALIDATE_TYPE_NAMES_FAIL;
    } /* else, contains == 1, which is the 'success' condition. */
  }
  status = TYPE_NAMES_ALL_VALID;
  goto _VALIDATE_TYPE_NAMES_CLEANUP;

 _VALIDATE_TYPE_NAMES_FAIL:
    assert (PyErr_Occurred());
    /* Fall through to cleanup. */

 _VALIDATE_TYPE_NAMES_CLEANUP:
  Py_XDECREF(keys);
  return status;
} /* _validate_type_names */


#define DICT_IS_NONE_OR_EMPTY(d) \
  ((d) == Py_None || PyDict_Size(d) == 0)

/* Generic programming the ugly way: */
#define _CREATE_TYPE_TRANS_SETTER(func_name, member_name, type_name, type_infra_name) \
  static PyObject *func_name( PyObject *self, PyObject *args ) { \
    type_name *target; \
    PyObject *trans_dict; \
    PyObject *output_type_trans_return_type_dict = NULL; \
    \
    if ( !PyArg_ParseTuple( args, "O!O!|O!", \
            &type_infra_name, &target, &PyDict_Type, &trans_dict, \
            &PyDict_Type, &output_type_trans_return_type_dict \
       )) \
    { return NULL; } \
    if (_validate_type_names(trans_dict) != TYPE_NAMES_ALL_VALID) \
      { return NULL; } \
    \
    /* If the target's output_type_trans_return_type_dict is to be replaced, \
    ** the caller will have supplied a non-NULL value for it (NULL indicates \
    ** that the optional argument was not specified). */ \
    if (output_type_trans_return_type_dict != NULL) { \
      Py_XDECREF(target->output_type_trans_return_type_dict); \
      /* If the new output_type_trans_return_type_dict is None or empty, \
      ** set the target's corresponding member to NULL rather than recording \
      ** the useless incoming value. \
      ** Note that output_type_trans_return_type_dict might be empty when \
      ** trans_dict is *not* empty, because when a translator is set to None \
      ** (which indicates that it should return the kinterbasdb's internal \
      ** representation of the value), output_type_trans_return_type_dict \
      ** will not contain an entry for that translation key (instead, \
      ** XSQLDA2Description supplies a default type). */ \
      if ( DICT_IS_NONE_OR_EMPTY(output_type_trans_return_type_dict) ) { \
        target->output_type_trans_return_type_dict = NULL; \
      } else { \
        Py_INCREF(output_type_trans_return_type_dict); \
        target->output_type_trans_return_type_dict = output_type_trans_return_type_dict; \
      } \
    } \
    \
    Py_XDECREF(target->member_name); /* Free old translation dict, if any. */ \
    if ( DICT_IS_NONE_OR_EMPTY(trans_dict) ) { \
      target->member_name = NULL; \
    } else { \
      /* Corresponding DECREF is in delete_[type] or the XDECREF above. */ \
      Py_INCREF(trans_dict); \
      target->member_name = trans_dict; \
    } \
    \
    Py_INCREF(Py_None); \
    return Py_None; \
  }
/* end of _CREATE_TYPE_TRANS_SETTER */

#define _CREATE_TYPE_TRANS_GETTER(func_name, member_name, type_name, type_infra_name) \
  static PyObject *func_name( PyObject *self, PyObject *args ) { \
    type_name *target; \
    \
    if ( !PyArg_ParseTuple( args, "O!", &type_infra_name, &target ) ) { return NULL; } \
    \
    if (target->member_name != NULL) { \
      /* Copy the dict so that the type translation settings can't be modified \
      ** except via a set_type_trans_* method. */ \
      return PyDict_Copy(target->member_name); \
    } else { \
      Py_INCREF(Py_None); \
      return Py_None; \
    } \
  }
/* end of _CREATE_TYPE_TRANS_GETTER */

/* Getters/setters for ConnectionObject: */
/* Out: */
_CREATE_TYPE_TRANS_SETTER(
    pyob_con_set_type_trans_out, type_trans_out, ConnectionObject, ConnectionType
  )
_CREATE_TYPE_TRANS_GETTER(
    pyob_con_get_type_trans_out, type_trans_out, ConnectionObject, ConnectionType
  )
/* In: */
_CREATE_TYPE_TRANS_SETTER(
    pyob_con_set_type_trans_in, type_trans_in, ConnectionObject, ConnectionType
  )
_CREATE_TYPE_TRANS_GETTER(
    pyob_con_get_type_trans_in, type_trans_in, ConnectionObject, ConnectionType
  )

/* Getters/setters for CursorObject: */
/* Out: */
_CREATE_TYPE_TRANS_SETTER(
    pyob_cur_set_type_trans_out, type_trans_out, CursorObject, CursorType
  )
_CREATE_TYPE_TRANS_GETTER(
    pyob_cur_get_type_trans_out, type_trans_out, CursorObject, CursorType
  )
/* In: */
_CREATE_TYPE_TRANS_SETTER(
    pyob_cur_set_type_trans_in, type_trans_in, CursorObject, CursorType
  )
_CREATE_TYPE_TRANS_GETTER(
    pyob_cur_get_type_trans_in, type_trans_in, CursorObject, CursorType
  )


/* 2003.03.30: */
static PyObject *dynamically_type_convert_input_obj_if_necessary(
    PyObject *py_input,
    boolean is_array_element,
    unsigned short dialect,
    short data_type, short data_subtype, short scale,
    PyObject *converter
  )
{
  /* if $converter is None, returns:
  **   a new reference to the original value
  ** else:
  **   the return value of the converter (which is a new reference) */
  if (converter == Py_None) {
    Py_INCREF(py_input);
    return py_input;
  }{
  boolean is_fixed_point;
  PyObject *py_converted = NULL;
  PyObject *py_argument_to_converter;

  PyObject *argz = PyTuple_New(1);
  if (argz == NULL) { return PyErr_NoMemory(); }

  /* Next, set py_argument_to_converter, the single argument that the converter
  ** will receive (though it's only one argument, it might be a sequence). */

  /* Special case for fixed point fields:  pass the original input object
  ** and the scale figure in a 2-tuple, rather than just the original input
  ** object, as with all other field types. */
  is_fixed_point = (
        is_array_element
      ? IS_FIXED_POINT__ARRAY_EL(dialect, data_type, data_subtype, scale)
      : IS_FIXED_POINT__CONVENTIONAL(dialect, data_type, data_subtype, scale)
    );
  if (is_fixed_point) {
    /* Reference ownership of this new 2-tuple is passed to argz via
    ** PyTuple_SET_ITEM.  argz will then delete this new 2-tuple when argz
    ** itself is deleted.  The refcount of py_input is INCd when it enters
    ** the new 2-tuple; DECd when the 2-tuple is deleted. */
    py_argument_to_converter = Py_BuildValue("(Oi)", py_input, scale);
  } else if (IS_UNICODE_CHAR_OR_VARCHAR(data_type, data_subtype)) {
    py_argument_to_converter = Py_BuildValue("(Oi)", py_input, data_subtype);
  } else {
    /* We currently hold only a borrowed reference to py_input, since it's
    ** an input parameter rather than a newly created object.
    ** Therefore, we must now artificially INCREF py_input so that
    ** PyTuple_SET_ITEM(argz, ...) can "steal" ownership of a reference to
    ** py_input and then discard that reference when argz is destroyed. */
    Py_INCREF(py_input);
    py_argument_to_converter = py_input;
  }
  if (py_argument_to_converter == NULL) {
    Py_DECREF(argz);
    return PyErr_NoMemory();
  }
  PyTuple_SET_ITEM(argz, 0, py_argument_to_converter);

  py_converted = PyObject_CallObject(converter, argz); /* The MEAT. */
  if (py_converted == NULL) { goto fail; }

  /* Special case for fixed-point values with precision 10-18 in dialects < 3:
  ** The value returned by the converter is a scaled Python int; we need to
  ** convert it to a non-scaled Python float, because in dialects < 3,
  ** logically fixed-point fields with precisions 10-18 are actually stored as
  ** floating point.  Those with precisions 1-9 are stored internally as
  ** integers, similar to the way they're stored in dialect 3; the clause
  **   && data_subtype == SUBTYPE_NONE
  ** in the condition below prevents this special case from applying to the
  ** precision 1-9 fields. */
  if (    dialect < 3 && is_fixed_point && scale != 0
       && data_subtype == SUBTYPE_NONE
       && py_converted != Py_None
     )
  {
    PyObject *py_conv_unscaled;
    PyObject *py_conv_as_py_float = PyNumber_Float(py_converted);
    if (py_conv_as_py_float == NULL) { goto fail; }
    py_conv_unscaled = PyFloat_FromDouble(
        PyFloat_AS_DOUBLE(py_conv_as_py_float) / pow(10.0f, (double) -scale)
      );
    Py_DECREF(py_conv_as_py_float);
    if (py_conv_unscaled == NULL) { goto fail; }
    /* Replace py_converted with py_conv_unscaled: */
    Py_DECREF(py_converted);
    py_converted = py_conv_unscaled;
  }

  goto cleanup;

  fail:
    assert (PyErr_Occurred());

    Py_XDECREF(py_converted);
    /* Fall through to cleanup. */

  cleanup:
    /* Notice that we do not DECREF py_argument_to_converter, because ownership
    ** of the reference to py_argument_to_converter will have been pass to
    ** the container argz. */
    Py_XDECREF(argz);

    return py_converted;
}} /* dynamically_type_convert_input_obj_if_necessary */


static PyObject *dynamically_type_convert_output_obj_if_necessary(
    PyObject *db_plain_output, PyObject *converter,
    short data_type, short data_subtype
  )
{
  /* Unlike dynamically_type_convert_input_obj_if_necessary, this function
  ** does NOT return a new reference.
  ** if $converter is None:
  **   returns the passed reference to the original value db_plain_output
  ** else:
  **   returns the return value of the converter (which is a new reference),
  **   BUT ALSO deletes the passed reference to db_plain_output, in effect
  **   "replacing" db_plain_output with py_converted.
  **   The passed reference to db_plain_output is deleted EVEN if this function
  **   encounters an error.
  */
  if (converter == Py_None) {
    return db_plain_output;
  }{
  PyObject *py_converted;
  boolean is_unicode_char_or_varchar = IS_UNICODE_CHAR_OR_VARCHAR(data_type, data_subtype);
  PyObject *argz = PyTuple_New(1);
  if (argz == NULL) {
    /* Yes, DECing db_plain_output here is correct (see comment at start): */
    Py_DECREF(db_plain_output);
    return PyErr_NoMemory();
  }

  if (!is_unicode_char_or_varchar) {
    /* The following statement "steals" the ref to db_plain_output, which is
    ** appropriate behavior in this situation. */
    PyTuple_SET_ITEM(argz, 0, db_plain_output);
  } else {
    /* If it's a unicode CHAR or VARCHAR, create a 2-tuple containing:
    ** (
    **    the raw (encoded) string,
    **    the database engine's internal character set code
    ** ). */
    PyObject *tuple_of_raw_string_and_charset_code = PyTuple_New(2);
    PyObject *db_charset_code = PyInt_FromLong(data_subtype);
    if (tuple_of_raw_string_and_charset_code == NULL || db_charset_code == NULL) {
      /* Yes, DECing db_plain_output here is correct (see comment at start): */
      Py_DECREF(db_plain_output);
      Py_DECREF(argz);
      return PyErr_NoMemory();
    }
    /* The following statements "steal" the refs to the element values, which
    ** is appropriate behavior in this situation.  Reference ownership of
    ** db_plain_output and db_charset_code is handed off to the container
    ** tuple_of_raw_string_and_charset_code; in turn, reference ownership of
    ** tuple_of_raw_string_and_charset_code is handed off to the container argz.
    ** When argz is released at the end of this function, the release
    ** "cascades", releasing the three other references mentioned above. */
    PyTuple_SET_ITEM(tuple_of_raw_string_and_charset_code, 0, db_plain_output);
    PyTuple_SET_ITEM(tuple_of_raw_string_and_charset_code, 1, db_charset_code);

    PyTuple_SET_ITEM(argz, 0, tuple_of_raw_string_and_charset_code);
  }

  py_converted = PyObject_CallObject(converter, argz); /* The MEAT. */

  Py_DECREF(argz);

  return py_converted;
}} /* dynamically_type_convert_output_obj_if_necessary */
