"""
Test module for compatibility with plain HDF files
==================================================

:Author:   Ivan Vilata i Balaguer
:Contact:  ivilata@carabos.com
:Created:  2005-09-29
:License:  BSD
:Revision: $Id: test_hdf5compat.py 1569 2006-04-28 17:00:57Z faltet $
"""

import unittest
import tempfile

import numarray

import tables
import common
from common import verbose, allequal


class HDF5CompatibilityTestCase(common.PyTablesTestCase):

    """
    Base class for HDF5 compatibility tests.

    Test cases deriving from this class must define an ``h5fname``
    attribute with the name of the file to be opened, and a ``_test()``
    method with checks on the opened file.
    """

    def setUp(self):
        self.h5file = None


    def tearDown(self):
        if self.h5file is not None:
            self.h5file.close()
        self.h5file = None


    def test(self):
        # There should be a warning telling the user
        # this is a plain HDF5 file, but not a PyTables one.
        self.h5file = self.assertWarns(
            UserWarning, tables.openFile, self.h5fname)
        self._test()



class EnumTestCase(HDF5CompatibilityTestCase):

    """
    Test for enumerated datatype.

    See ftp://ftp.ncsa.uiuc.edu/HDF/files/hdf5/samples/enum.c.
    """

    h5fname = common.testFilename('smpl_enum.h5')

    def _test(self):
        self.assert_('/EnumTest' in self.h5file)

        arr = self.h5file.getNode('/EnumTest')
        self.assert_(isinstance(arr, tables.Array))

        enum = arr.getEnum()
        expectedEnum = tables.Enum(['RED', 'GREEN', 'BLUE', 'WHITE', 'BLACK'])
        self.assertEqual(enum, expectedEnum)

        data = list(arr.read())
        expectedData = [
            enum[name] for name in
            ['RED', 'GREEN', 'BLUE', 'WHITE', 'BLACK',
             'RED', 'GREEN', 'BLUE', 'WHITE', 'BLACK']]
        self.assertEqual(data, expectedData)



class NumericTestCase(HDF5CompatibilityTestCase):

    """
    Test for several numeric datatypes.

    See ftp://ftp.ncsa.uiuc.edu/HDF/files/hdf5/samples/[fiu]l?{8,16,32,64}{be,le}.c.
    """

    def _test(self):
        self.assert_('/TestArray' in self.h5file)

        arr = self.h5file.getNode('/TestArray')
        self.assert_(isinstance(arr, tables.Array))

        self.assertEqual(arr.stype, self.stype)
        self.assertEqual(arr.byteorder, self.byteorder)
        self.assertEqual(arr.shape, (6, 5))

        data = arr.read()
        expectedData = numarray.array([
            [0, 1, 2, 3, 4],
            [1, 2, 3, 4, 5],
            [2, 3, 4, 5, 6],
            [3, 4, 5, 6, 7],
            [4, 5, 6, 7, 8],
            [5, 6, 7, 8, 9]], type=self.stype)
        self.assert_(common.areArraysEqual(data, expectedData))



class F64BETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_f64be.h5')
    stype = 'Float64'
    byteorder = 'big'

class F64LETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_f64le.h5')
    stype = 'Float64'
    byteorder = 'little'

class I64BETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_i64be.h5')
    stype = 'Int64'
    byteorder = 'big'

class I64LETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_i64le.h5')
    stype = 'Int64'
    byteorder = 'little'

class I32BETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_i32be.h5')
    stype = 'Int32'
    byteorder = 'big'

class I32LETestCase(NumericTestCase):
    h5fname = common.testFilename('smpl_i32le.h5')
    stype = 'Int32'
    byteorder = 'little'



class ChunkedCompoundTestCase(HDF5CompatibilityTestCase):

    """
    Test for a more complex and chunked compound structure.

    This is generated by a chunked version of the example in
    ftp://ftp.ncsa.uiuc.edu/HDF/files/hdf5/samples/compound2.c.
    """

    h5fname = common.testFilename('smpl_compound_chunked.h5')

    def _test(self):
        self.assert_('/CompoundChunked' in self.h5file)

        tbl = self.h5file.getNode('/CompoundChunked')
        self.assert_(isinstance(tbl, tables.Table))

        self.assertEqual(
            tbl.colnames,
            ('a_name', 'c_name', 'd_name', 'e_name', 'f_name', 'g_name'))

        self.assertEqual(tbl.colstypes['a_name'], 'Int32')
        self.assertEqual(tbl.colshapes['a_name'], 1)

        self.assertEqual(tbl.colstypes['c_name'], 'CharType')
        self.assertEqual(tbl.colshapes['c_name'], 1)

        self.assertEqual(tbl.colstypes['d_name'], 'Int16')
        self.assertEqual(tbl.colshapes['d_name'], (5, 10))

        self.assertEqual(tbl.colstypes['e_name'], 'Float32')
        self.assertEqual(tbl.colshapes['e_name'], 1)

        self.assertEqual(tbl.colstypes['f_name'], 'Float64')
        self.assertEqual(tbl.colshapes['f_name'], (10,))

        self.assertEqual(tbl.colstypes['g_name'], 'UInt8')
        self.assertEqual(tbl.colshapes['g_name'], 1)

        for m in range(len(tbl)):
            row = tbl[m]
        # This version of the loop seems to fail because of ``iterrows()``.
        ##for (m, row) in enumerate(tbl):
            self.assertEqual(row['a_name'], m)
            self.assertEqual(row['c_name'], "Hello!")
            dRow = row['d_name']
            for n in range(5):
                for o in range(10):
                    self.assertEqual(dRow[n][o], m + n + o)
            self.assertAlmostEqual(row['e_name'], m * 0.96, places=6)
            fRow = row['f_name']
            for n in range(10):
                self.assertAlmostEqual(fRow[n], m * 1024.9637)
            self.assertEqual(row['g_name'], ord('m'))


class ContiguousCompoundTestCase(HDF5CompatibilityTestCase):

    """
    Test for support of native contiguous compound datasets.

    This example has been provided by Dav Clark.
    """

    h5fname = common.testFilename('non-chunked-table.h5')

    def _test(self):
        self.assert_('/test_var/structure variable' in self.h5file)

        tbl = self.h5file.getNode('/test_var/structure variable')
        self.assert_(isinstance(tbl, tables.Table))

        self.assertEqual(
            tbl.colnames,
            ('a', 'b', 'c', 'd'))

        self.assertEqual(tbl.colstypes['a'], 'Float64')
        self.assertEqual(tbl.colshapes['a'], 1)

        self.assertEqual(tbl.colstypes['b'], 'Float64')
        self.assertEqual(tbl.colshapes['b'], 1)

        self.assertEqual(tbl.colstypes['c'], 'Float64')
        self.assertEqual(tbl.colshapes['c'], (2,))

        self.assertEqual(tbl.colstypes['d'], 'CharType')
        self.assertEqual(tbl.colshapes['d'], 1)

        for row in tbl.iterrows():
            self.assertEqual(row['a'], 3.0)
            self.assertEqual(row['b'], 4.0)
            self.assert_(allequal(row['c'], numarray.array([2.0, 3.0],
                                                           type="Float64")))
            self.assertEqual(row['d'], "d")

        self.h5file.close()
        self.h5file = tables.openFile(self.h5fname,'a')
        tbl = self.h5file.getNode('/test_var/structure variable')
        # Tests for appending
        self.assertRaises(tables.HDF5ExtError, tbl.append,
                          [(4.0,5.0,[2.0,3.0],'d')])
        # Appending using the Row interface
        self.assertRaises(tables.HDF5ExtError, tbl.row.append)


class ExtendibleTestCase(HDF5CompatibilityTestCase):

    """
    Test for extendible datasets.

    See the example programs in the Introduction to HDF5.
    """

    h5fname = common.testFilename('smpl_SDSextendible.h5')

    def _test(self):
        self.assert_('/ExtendibleArray' in self.h5file)

        arr = self.h5file.getNode('/ExtendibleArray')
        self.assert_(isinstance(arr, tables.EArray))

        self.assertEqual(arr.byteorder, 'big')
        self.assertEqual(arr.atom.stype, 'Int32')
        self.assertEqual(arr.atom.shape, (0, 5))
        self.assertEqual(len(arr), 10)

        data = arr.read()
        expectedData = numarray.array([
            [1, 1, 1, 3, 3],
            [1, 1, 1, 3, 3],
            [1, 1, 1, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0],
            [2, 0, 0, 0, 0]], type=arr.atom.stype)
        self.assert_(common.areArraysEqual(data, expectedData))



def suite():
    """Return a test suite consisting of all the test cases in the module."""

    theSuite = unittest.TestSuite()
    niter = 1

    for i in range(niter):
        theSuite.addTest(unittest.makeSuite(EnumTestCase))

        theSuite.addTest(unittest.makeSuite(F64BETestCase))
        theSuite.addTest(unittest.makeSuite(F64LETestCase))
        theSuite.addTest(unittest.makeSuite(I64BETestCase))
        theSuite.addTest(unittest.makeSuite(I64LETestCase))
        theSuite.addTest(unittest.makeSuite(I32BETestCase))
        theSuite.addTest(unittest.makeSuite(I32LETestCase))

        theSuite.addTest(unittest.makeSuite(ChunkedCompoundTestCase))
        theSuite.addTest(unittest.makeSuite(ContiguousCompoundTestCase))

        theSuite.addTest(unittest.makeSuite(ExtendibleTestCase))

    return theSuite



if __name__ == '__main__':
    unittest.main(defaultTest='suite')



## Local Variables:
## mode: python
## py-indent-offset: 4
## tab-width: 4
## fill-column: 72
## End:
