# File: message.py
# Purpose: a message instance

import string
import os
import os.path
import GtkExtra
import time
import StringIO
import binascii
import tempfile
import pynei18n
import msgeditbox
import rfc822
import copy
from pyneheaders import *

def parse_2_body_and_headers(head_and_body):
	"""
	Parse a message header+body to headers (as a
	dictionary) and body text
	"""
	# Headers are terminated by a double newline.
	endhead = string.find(head_and_body, "\n\n")
	
	# Header and upper case version
	header_text = head_and_body[0:endhead+1]
	body_text   = head_and_body[endhead+2:]
	# Parse headers to a dictionary 
	header_dict = headers_2_dict( StringIO.StringIO(header_text) )

	return (header_dict, body_text)


def headers_2_dict(f):
	"""
	Takes an open file object of message headers and parses the
	headers to a dictionary.
	"""
	keys = {}
	lastkey = None

	while 1:
		s = f.readline()

		if s == "":
			break

		s = string.replace(s, "\r", "")
		s = string.replace(s, "\n", "")
		s = string.replace(s, "\t", " ")

		keyname = string.lower( string.split(s, ":")[0] )

		# if we found
		if len(string.split(s, ":")) > 1 and s[:1] != " ":
			keys[keyname] = string.join(string.split(s,":")[1:], ":")
			# remove leading space
			if keys[keyname][:1] == " ":
				keys[keyname] = keys[keyname][1:]
			lastkey = keyname
		elif lastkey != None:
			keys[lastkey] = keys[lastkey] + s
	return keys

class pynemsg:
	"""
	An email or news article
	"""
	def __init__(self):
		self.headers = {}
		# Format (year,month,day,hour,min,sec,0,0,0)
		self.date = (0, 0, 0, 0, 0, 0, 0, 0, 0)
		# to decide when to expire news articles
		self.date_received = (0, 0, 0, 0, 0, 0, 0, 0, 0)
		# Message text (inc headers)
		self.body = ""
		self.opts = 0
		# messages waiting in the outbox will additionally have the
		# uid of the box they came from in:
		self.senduid = None
		# Message parts. This will contain strings of encoded attachments
		# including the attachment header thingy
		self.parts_text = []
		self.parts_header = []

	def make_source(self, presend=0):
		"""
		Take a message and it's contents (attachments and such like)
		and create a single body with all the headers required to
		post.
		Set presend=1 before actual posting to include full attachments.
		"""
		import pyne # for pyne.ver_string

		num_parts = len(self.parts_text)
		############# HEADERS
		if self.headers.has_key("from"):
			body =        "From: "+self.headers["from"]+"\n"
		if self.headers.has_key("reply-to"):
			body = body + "Reply-To: "+self.headers["reply-to"]+"\n"
		if self.headers.has_key("organization"):
			body = body + "Organization: "+self.headers["organization"]+"\n"
		if self.headers.has_key("to"):
			body = body + "To: "+self.headers["to"]+"\n"
		if self.headers.has_key("subject"):
			body = body + "Subject: "+self.headers["subject"]+"\n"
		# date stamp of when last edited
		body = body + time.strftime("Date: %a, %d %b %Y %H:%M:%S +0000\n", self.date)
		#body = body + time.strftime("Date: %a, %d %b %Y %H:%M:%S %Z\n", self.date)
		if self.headers.has_key("references"):
			body = body + "References: "+self.headers["references"]+"\n"
		if self.headers.has_key("newsgroups"):
			body = body + "X-Newsreader: "+pyne.ver_string+"\n"
		else:
			body = body + "X-Mailer: "+pyne.ver_string+"\n"
		# Content type
		if num_parts == 1:
			body = body + "Content-Type: text/plain\n"
		else:
			body = body + "Content-Type: multipart/mixed; boundary=\""+multi_part_boundary+"\"\n"
		body = body + "MIME-Version: 1.0\n"
		# Confirm delivery
		if self.headers.has_key("return-receipt-to"):
			body = body + "Return-Receipt-To: "+self.headers["from"]+"\n"
		# Optional Cc
		if self.headers.has_key("cc"):
			if self.headers["cc"] != "":
				body = body + "Cc: "+self.headers["cc"]+"\n"
		if self.headers.has_key("bcc"):
			if self.headers["bcc"] != "":
				body = body + "Bcc: "+self.headers["bcc"]+"\n"
		# news stuff:
		if presend == 0:
			body = body + "Message-ID: "+self.headers["message-id"]+"\n"
		if self.headers.has_key("newsgroups"):
			body = body + "Lines: "+str(len(string.split(self.parts_text[0], "\n")))+"\n"
			body = body + "Newsgroups: "+self.headers["newsgroups"]+"\n"

		# simple single part messages may simply have the body grafted
		# on and that's all.
		if num_parts == 1:
			body = body + "\n" + self.parts_text[0]
			self.body = body
			return
		else:
			# It's a multi-part message.
			# Add the plain text bit first.
			body = body + "\nThis is a multi-part message in MIME format.\n"
			body = body + "\n--" + multi_part_boundary + "\n"
			body = body + "Content-Type: text/plain\n"
			body = body + "Content-Transfer-Encoding: 8bit\n\n"
			body = body + self.parts_text[0] + "\n"

			# Then add the other parts
			for x in range(1, num_parts):
				body = body + "--" + multi_part_boundary + "\n"
				body = body + "Content-Type: "+self.parts_header[x]["content-type"]+"\n"
				body = body + "Content-Transfer-Encoding: "+self.parts_header[x]["content-transfer-encoding"]+"\n"
				body = body + "Content-Disposition: "+self.parts_header[x]["content-disposition"]+"\n\n"
				body = body + self.parts_text[x]

			# And terminate
			body = body + "--" + multi_part_boundary + "--\n"

			self.body = body
		return

	def edit(self, folder, user, is_new_msg=0):
		"""
		Create a message composing window and let it do the hard work.
		"""
		# If the message has already been sent then
		# we want to make a copy of it in the outbox
		# and edit that.
		if self.opts & MSG_SENT:
			outbox = user.get_folder_by_uid("outbox")
			# Copy message, but change message-id
			msg = self
			msg.headers = copy.copy(self.headers)
			del msg.headers["message-id"]
			msg.opts = msg.opts & ~(MSG_SENT)
			# This should give it a message id
			outbox.io.save_article(msg)
			outbox.messages.append(msg.headers["message-id"])
			outbox.changed = 1

			user.update()
			msgeditbox.msgeditbox(msg, outbox, user, 1)
		else:
			msgeditbox.msgeditbox(self, folder, user, is_new_msg)

	def parseheaders(self, user, headers_only=0):
		"""
		The body now contains headers and body text.
		Extract sender, subject and date and split up
		if it's multi-part.
		Pass headers_only=1 if you don't need bodies to be
		parsed (multipart decoded, etc..)
		"""
		# Convert and "\r\n"s to "\n"
		self.body = string.replace(self.body, "\r\n", "\n")

		# Wipe the contents. We are going to get that
		self.parts_text = []
		self.parts_header = []

		part_header, part_text = parse_2_body_and_headers(self.body)
		self.parts_text.append(part_text)
		self.parts_header.append(part_header)
		self.headers = part_header

		# We *need* some header fields
		if not self.headers.has_key("subject"):
			self.headers["subject"] = ""
		if not self.headers.has_key("from"):
			self.headers["from"] = ""

		# Clean up dodgy references
		if self.headers.has_key("references"):
			s = self.headers["references"]
			l = []
			while string.find(s, "<") != -1:
				x = string.find(s, "<")
				y = string.find(s, ">")
				l.append(s[x:y+1])
				s = s[y+1:]
				if y == -1:
					break
			self.headers["references"] = string.join(l, " ")
	
		# Get a nice gmtime format time from the one in the header
		if self.headers.has_key("date"):
			self.date = rfc822.parsedate(self.headers["date"])
			# rfc822.parsedate is really anal and fails on small
			# padding errors in the date line. XXX XXX
			if self.date == None:
				self.date = time.gmtime(time.time())
			# for mentally retarded mailers, 2 digit years
			elif self.date[0] < 1000: # date[0] is year
				year = 2000 + self.date[0]
				self.date = (year,) + self.date[1:]
		else:
			# Just set current time...
			self.date = time.gmtime(time.time())

		# We are parsing a message with headers only
		if headers_only == 1:
			self.opts = self.opts | MSG_NO_BODY
			return
		# we do have a body >:-(
		self.opts = self.opts & (~MSG_NO_BODY)
	
		# Break multipart messages up
		i = 0
		while i < len(self.parts_text):
			headers = self.parts_header[i]
			body = self.parts_text[i]
			if not headers.has_key("content-type"):
				# not a multi-part section	
				i = i+1
				continue
			content_type = string.lower(headers["content-type"])
			if string.find(content_type, "boundary=") == -1:
				# not a multi-part section
				i = i+1
				continue
			else:
				# get boundary string
				start_boundary = 9 + string.find(content_type, "boundary=")
				#end_boundary = start_boundary + string.find(content_type[start_boundary:], "\"")
				boundary = headers["content-type"][start_boundary:]#end_boundary]
				if boundary[0] == "\"":
					boundary = boundary[1:]
				if boundary[-1] == "\"":
					boundary = boundary[:-1]
				# get chunks of body between this boundary
				offset = b = 0
				subtexts = []
				subheads = []
				while 1:
					# find start of a boundary
					b = string.find(body[offset:], "--"+boundary)
					if b == -1:
						# no attachment
						break
					if b == string.find(body[offset:], "--"+boundary+"--"):
						# end of attachments
						break
					# line after start boundary
					b = offset + b + len(boundary) + 3 # 3==len("--"+"\n")
					# stupid messages with no terminating boundary
					if string.find(body[b:], "--"+boundary) == -1:
						offset = len(body)
					else:
						# end boundary
						offset = string.find(body[b:], "--"+boundary) + b
					# append section
					head, text = parse_2_body_and_headers(body[b:offset])
					subheads.append(head)
					subtexts.append(text)
				# dump new bits on
				# remove this part, since it isn't a single part
				if len(subheads):
					del self.parts_header[i]
					del self.parts_text[i]
					# now add the seperate parts it was composed of
					for x in range(0, len(subheads)):
						self.parts_header.insert(i+x, subheads[x])
						self.parts_text.insert(i+x, subtexts[x])
				else:
					i = i+1
		# Mails with base64 encoded body must be decoded
		if self.parts_header[0].has_key("content-transfer-encoding"):
			if string.lower(self.parts_header[0]["content-transfer-encoding"]) == "base64":
				import base64
				try:
					self.parts_text[0] = base64.decodestring(self.parts_text[0])
				except binascii.Error:
					GtkExtra.message_box("Error", "Error while decoding attachment:\nIncorrect Padding.", ("Ok",))
				self.parts_text[0] = string.replace(self.parts_text[0], "\r\n", "\n")
		# Parse html bodies to plain text
		if self.parts_header[0].has_key("content-type"):
			if string.find(string.lower(self.parts_header[0]["content-type"]), "text/html") != -1:
				# Get a temporary filename
				tempfilename = tempfile.mktemp(".html")
				# Open it and write the html to it
				temp = open(tempfilename, "w")
				temp.write(self.parts_text[0])
				temp.close()
				# get output from user's html parsy proggy
				f = os.popen(user.html_parser+" %s" % tempfilename)
				body = f.read()
				os.remove(tempfilename)
				f.close()
				if body != "":
					self.parts_text[0] = body

	def save_attachment(self, index, filename):
		"""
		Save attachment parts[index] to file 'filename'
		"""
		# Get attachment string
		content_transfer_encoding = self.get_attachment_info(index)[3]

		import mimetools
		import os

		# Get a temporary filename
		tempfilename = tempfile.mktemp()

		# Open it and write the encoded data to it
		temp = open(tempfilename, "w")
		temp.write(self.parts_text[index])
		temp.close()

		# Re-open in read mode
		temp = open(tempfilename, "r")
	
		# Open the output file
		f = open(filename, "w")

		# Decode and write to disk
		if content_transfer_encoding == "base64":
			### BASE64
			try:
				mimetools.decode(temp, f, "base64")
			except:
				f.close()
				temp.close()
				os.remove(tempfilename)
				os.remove(filename)
				GtkExtra.message_box(_("Error"), _("Cannot decode attachment."), (_("Ok"),))
				return
		elif content_transfer_encoding == "uuencode":
			### UUENCODE
			try:
				mimetools.decode(temp, f, "uuencode")
			except:
				f.close()
				temp.close()
				os.remove(tempfilename)
				os.remove(filename)
				GtkExtra.message_box(_("Error"), _("Cannot decode attachment."), (_("Ok"),))
				return
		elif content_transfer_encoding == "quoted-printable":
			### QUOTED-PRINTABLE
			try:
				mimetools.decode(temp, f, "quoted-printable")
			except:
				f.close()
				temp.close()
				os.remove(tempfilename)
				os.remove(filename)
				GtkExtra.message_box(_("Error"), _("Cannot decode attachment."), (_("Ok"),))
				return
		else:
			### Assume 7/8bit text
			f.write(temp.read())

		f.close()
		temp.close()

		# Remove temporary file
		os.remove(tempfilename)

	def add_attachment(self, filename):
		"""
		Add a base64 encoded attachment of file 'filename'
		onto the message.
		"""
		import base64

		# Read the file to be attached
		try:
			f = open(filename, "r")
		except IOError:
			return
		unencoded = f.read()
		f.close()

		# base64 encode it
		encoded = base64.encodestring(unencoded)

		# Truncate filenames like '/home/yourname/file.txt'
		# to 'file.txt'
		smallname = os.path.basename(filename)

		part_header = {}
		part_text = encoded + "\n"
		# Create headers
		part_header["content-type"] = "application/octet-stream; name=\""+smallname+"\""
		part_header["content-transfer-encoding"] = "base64"
		part_header["content-disposition"] = "attachment; filename=\""+smallname+"\""

		# Add to message
		self.parts_text.append(part_text)
		self.parts_header.append(part_header)

	def get_attachment_info(self, index):
		"""
		Returns list of attachment data in the form:
		[ content-type, filename, size, content-transfer-encoding ]
		"""
		part_header = self.parts_header[index]
		part_text = self.parts_text[index]
		# Get content-type
		if part_header.has_key("content-type"):
			content_type = part_header["content-type"]
			x = string.find(content_type, ";")
			if x != -1:
				content_type = content_type[:x]
		else:
			content_type = "unknown"
		# Get filename
		if part_header.has_key("content-id"):
			filename = os.path.basename( part_header["content-id"] )
		elif part_header.has_key("content-disposition"):
			content_disposition = part_header["content-disposition"]
			x = string.find(content_disposition, "filename=\"")
			if x==-1:
				filename = "noname"
			else:
				y = string.find(content_disposition[x+10:], "\"")
				filename = content_disposition[x+10:x+10+y]
		elif part_header.has_key("content-type"):
			content_disposition = part_header["content-type"]
			x = string.find(content_disposition, "name=\"")
			if x==-1:
				filename = "noname"
			else:
				y = string.find(content_disposition[x+6:], "\"")
				filename = content_disposition[x+6:x+6+y]
		else:
			filename = "noname"
		# Get content-transfer-type
		if part_header.has_key("content-transfer-encoding"):
			content_transfer_encoding = string.lower(part_header["content-transfer-encoding"])
		else:
			content_transfer_encoding = "unknown"
		# Get size. assume base64
		# Note. We do not ignore padding but WTF...
		length = str(int(len(part_text)*0.75))
	
		return [ content_type, filename, length, content_transfer_encoding ]

