/* ************************************************************************
  Module:        lexwidgets.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000

  Description:
                 Some widgets.

  Changes:

****************************************************************************
                 COPYRIGHT (C) 1999, 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "lexgui.h"

/* -------------------------------------------------------------------------
        hand_CB()  Callback for the mouse movement with the hand scroller.
                   This works for scrolling wheel mouses too. Note this is
                   an event rather than a callback, so it has an extra
                   parameter (GdkEventMotion *event). This function is 
                   shared by the scrolling window and the scrolled text box.
   ------------------------------------------------------------------------- */
void hand_CB(GtkWidget *widget, GdkEventMotion *event, aj_struct *sws)
{
  gint skip;
  gint x, y;
  gint xdistance, ydistance;
  gfloat newxval, newyval;
  GdkModifierType state;

  if(sws->tba == 1) /* If hand scroll mode not active */
      return;

  /* If a text box */
  if(GTK_IS_TEXT(sws->text))
    GTK_TEXT(sws->text)->button = 0;/*Setting to 0 effectively disables the
                                      built-in selection & select-scrolling.*/

  /* --- If it's a hint... (combining several events) --- */
  if (event->is_hint) 
  {
    /* --- Get new position --- */
    gdk_window_get_pointer (event->window, &x, &y, &state);
  }
  else
  {
    /* --- Get new position --- */
    x = event->x;
    y = event->y;
    state = (GdkModifierType)(event->state);
  }

  /* --- If the mouse button is down --- */
  if (state & GDK_BUTTON1_MASK)
  {
    xdistance = x - sws->prevx;
    ydistance = y - sws->prevy;

    /* set prevx and prevy to current x & y. If determined that we will
       actually scroll (below), prevx and prevy gets reset to reflect
       the additional window movement amount. */
    sws->prevx = x;
    sws->prevy = y;

    /* Note the hand drags the window with the hand, opposite of the
       scroll bar movement. Also note the skip value for x and y amount
       scrolled is set to the x and y distance the mouse travels,
       and therefore is proportional to the speed the mouse is moved. */

    /* prevx and prevy are set to current x & y; but on a scrolled window the
       x & y also move when the window is scrolled, so subtract or add
       the skip amount back in (see below). On a text box, the x & y remain
       static, so don't subtract or add the skip back in. This becomes
       evident if x & y are printed at the beginning of this function. If
       x & y move while scrolling with the hand (scrolled window), the numbers 
       will be the same as you scroll (because they move with the hand);
       If the x & y remain static, new values will be reflected while
       scrolling (because the hand moves past them). */

     skip = abs(xdistance);
    if(xdistance > 0)
    {
      newxval = (sws->hadj)->value - skip; /* +x so -horiz scroll */
      if(newxval < 0.0) /* subtracting, so test for min scroll */
        newxval = 0.0; /* don't scroll; prevx remains the same */
      else
        if(!GTK_IS_TEXT(sws->text)) /* if not a text box */
          sws->prevx -= skip; /* do scroll, so subtract skip */
    }
    else
    {
      newxval = (sws->hadj)->value + skip; /* -x so +horiz scroll */
      if(newxval > (sws->hadj)->upper - (sws->hadj)->page_size)/*adding, so
                                                                 test for max*/
        newxval = (sws->hadj)->upper - (sws->hadj)->page_size; /*no scroll; 
                                                                 prevx remains*/
      else
        if(!GTK_IS_TEXT(sws->text)) /* if not a text box */
          sws->prevx += skip; /* do scroll, so add skip */
    }
     skip = abs(ydistance);
    if(ydistance > 0)
    {
      newyval = (sws->vadj)->value - skip; /* +y so -vert scroll */
      if(newyval < 0.0) /* subtracting, so test for min scroll */
        newyval = 0.0; /* don't scroll; prevy remains the same */
      else
        if(!GTK_IS_TEXT(sws->text)) /* if not a text box */
           sws->prevy -= skip; /* do scroll, so subtract skip */
    }
    else
    {
      newyval = (sws->vadj)->value + skip; /* -y so +vert scroll */
      if(newyval > (sws->vadj)->upper - (sws->vadj)->page_size)/*adding, so
                                                                 test for max*/
        newyval = (sws->vadj)->upper - (sws->vadj)->page_size;/*no scroll; 
                                                                prevy remains*/
      else
        if(!GTK_IS_TEXT(sws->text)) /* if not a text box */
          sws->prevy += skip; /* do scroll, so add skip */
    }
    gtk_adjustment_set_value(sws->hadj, newxval);
    gtk_adjustment_set_value(sws->vadj, newyval);
  }
  else
  {
  /* When button not pressed (and thus not scrolling),
     set prevx and prevy to current x & y */
    sws->prevx = x;
    sws->prevy = y;
  }
} 
/* -------------------------------------------------------------------------
        mousepress_CB() When mouse is pressed or released in the hand scroll
                        mode, for changing the cursors.
   ------------------------------------------------------------------------- */
void mousepress_CB(GtkWidget *widget, GdkEventButton *event, cur_struct *c) 
{
  if(*(c->tba_ptr) == 0) /* If hand scroll mode active */
    if(event->button == 1) /* first mouse button */
      gdk_window_set_cursor (event->window, c->cursor);

      /* The above call is used instead of the following, because it
         works for either the scrolled window or the scrolled text box.
         Otherwise for the scrolled window it would be:
      gdk_window_set_cursor(GTK_WIDGET(c->widget)->window, c->cursor);
         and for the text box it would be:
      gdk_window_set_cursor(GTK_TEXT(c->widget)->text_area, c->cursor); */
}
/* -------------------------------------------------------------------------
        sc_win_setup() Creates the cursers, top window, top hbox, & top vbox
                       for scrolled regular windows or scrolled text windows.
   ------------------------------------------------------------------------- */
void sc_win_setup(GtkWidget **topwin, GtkWidget **topvbox,
                  char *title, int width, int height)
{
  *topwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(*topwin), title);
  gtk_container_border_width(GTK_CONTAINER(*topwin), 5);
  /* Tried gtk_window_set_default_size() but the window was very small the
     third time it was called; the below 2 lines don't have this problem. */
  gtk_widget_set_usize(*topwin, width, height);
  gtk_window_set_policy(GTK_WINDOW(*topwin), TRUE, TRUE, FALSE);

  *topvbox = gtk_vbox_new(FALSE, 0); /* contains sc_win or text box &
                                       close button */
  gtk_container_add(GTK_CONTAINER(*topwin), *topvbox);
  gtk_widget_show(*topvbox);
}
/* -------------------------------------------------------------------------
        make_sc_win() Makes a scrolled window with the hand cursor. Send
                      it a **static** sw_struct pointer to set up. This
                      window will scroll anything placed in swptr->sc_vbox,
                      including text (via a label). For a scrolled text box,
                      which handles its own scrolling, see make_tx_win().

                        swptr->aj.text For make_tx_win(). Set to NULL here.
                        swptr->aj.tba 0 = hand scroll active. 1 = disabled. 
                               ** Set tba before calling this routine.
                        swptr->aj.hadj is sc_win's hadj.
                        swptr->aj.vadj is sc_win's vadj.
                        swptr->aj.prevx stores X position.
                        swptr->aj.prevy stores Y position.
                        swptr->topwin is the top window of this popup.
                        swptr->topvbox top verticle box.
                        swptr->tophbox top horizontal box.
                        swptr->sc_win is the scrolled window.
                        swptr->sc_fixedcon is what's scrolled.
                        swptr->sc_vbox is used to put things in to scroll. 
                        swptr->info_only_flag set to 1 when an infowin.
                           ** set info_only_flag before calling this routine.
   ------------------------------------------------------------------------- */
void make_sc_win(sw_struct *swptr, char *title, int width, int height)
{
  sc_win_setup(&(swptr->topwin), &(swptr->topvbox), title, width, height);

  swptr->aj.text = NULL; /*Used for scrolled text box only- see make_tx_win()*/
  swptr->aj.prevx = 0;
  swptr->aj.prevy = 0;

  swptr->tophbox = gtk_hbox_new(FALSE, 7);
  gtk_container_set_border_width (GTK_CONTAINER(swptr->tophbox), 10);
  gtk_box_pack_start(GTK_BOX(swptr->topvbox), swptr->tophbox, TRUE, TRUE, 0);
  gtk_widget_show(swptr->tophbox);

  /* used fixed container as a way to set the bkgnd color inside sc_win */
  swptr->sc_fixedcon = gtk_fixed_new(); /* sc_fixedcon is in scrolled window
                                           and is what's scrolled. */
  set_color(&(swptr->sc_fixedcon), WHITE, BG, NORMAL);
  gtk_widget_show(swptr->sc_fixedcon);

  /* sc_vbox is in sc_fixedcon in scrolled window. It is what contains the text
     that is scrolled. The text is set in the calling function afterwards */
  swptr->sc_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(swptr->sc_vbox), 10);
  gtk_container_add(GTK_CONTAINER(swptr->sc_fixedcon), swptr->sc_vbox);
  gtk_widget_show(swptr->sc_vbox);
  
  swptr->sc_win = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swptr->sc_win),
                                  GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swptr->sc_win),
                                        swptr->sc_fixedcon);
  gtk_box_pack_start(GTK_BOX(swptr->tophbox), swptr->sc_win, TRUE, TRUE, 0);

  gtk_widget_show(swptr->sc_win);  

  /*-----------------------------------------------------------------
         Set up the "hand scroller" (I wanted to make it fancy!)
    -----------------------------------------------------------------*/
  /* Get the adjustments created by the scrolled window sc_win.
     These are used by the hand scroller. There's a callback for an
     adjustment in lexalignpop.c if I want to do one for hadj and vadj*/

  swptr->aj.hadj = 
     gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(swptr->sc_win));
  swptr->aj.vadj = 
     gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(swptr->sc_win));

  /* Don't need an event box like in the tutorial because the
     fixed container defined above includes a window (need an X window
     to receive events). */

  gtk_widget_set_events (swptr->sc_fixedcon, GDK_BUTTON_PRESS_MASK
                                | GDK_BUTTON_RELEASE_MASK
                                | GDK_POINTER_MOTION_MASK
                                | GDK_POINTER_MOTION_HINT_MASK);
  gtk_signal_connect (GTK_OBJECT(swptr->sc_fixedcon), "motion_notify_event",
                      GTK_SIGNAL_FUNC (hand_CB), (gpointer)&(swptr->aj));

    /* The info window does not tie up the top or top msg box so there's
       no need to reset or sensitize the top. However, this function is also
       used to make a scrolled window for setting the printer defaults
       dynamically, so its close button does need to reset the top and
       clear the message boxes. */
  if(swptr->info_only_flag) /* the infowin has this set */
    close_ibutton(&(swptr->topvbox), &(swptr->topwin)); /* info window */
  /* else  upon returning a different close button will be made */

  /* Callback for the X kill */
  if(swptr->info_only_flag) /* the infowin has this set */
  {
    gtk_signal_connect(GTK_OBJECT(swptr->topwin), "delete_event",
                       GTK_SIGNAL_FUNC(close_iwindowX),
                       (gpointer)&(swptr->topwin));
    show_sc_win(swptr);
  }
  /* else upon returning other things will be built, a different
     delete_event will be called, and then show_sc_win() will be called. */
}
/* -------------------------------------------------------------------------
        show_sc_win() shows the top scrolled-win window, then builds things
                      that can't be built until the top win is shown.
   ------------------------------------------------------------------------- */
void show_sc_win(sw_struct *swptr)
{
  static int init = 0;
  static cur_struct cur, cur1;

  if(!init)
  {
    init = 1;
    cur.cursor = create_hand_cursor();    /* Use Eric Harlow's hand cursor */
    cur1.cursor = create_hand_g_cursor(); /* Modified for grabbing */
  } 

  gtk_widget_show(swptr->topwin);

  /* The following must come after the parent is shown, otherwise
     at runtime you get:
    '(gdk_window_set_cursor): assertion `window != NULL' failed.'
     because the window is not yet created. If you desire
     to do this before the parent is shown, you would use
     gtk_widget_realize on the fixed container, and then set the
     cursor. This may present other problems, like in my case the
     fixed container did not get set to the size of the label it
     contains, making the entire window very small. This was very
     confusing to figure out. The tutorial uses gtk_widget_realize
     and Harlow's book sets the cursors after the parent's
     gtk_widget_show, as done here. */
  
  if(swptr->aj.tba == 0) /* If hand scroll mode active */
  {
   /* Set cursor to hand grabbing when mouse pressed; set to regular
      hand when mouse released. */
    cur.tba_ptr = &(swptr->aj.tba);
    cur.widget = swptr->sc_fixedcon;
    cur1.tba_ptr = &(swptr->aj.tba);
    cur1.widget = swptr->sc_fixedcon;

    /* Set initial cursor */
    if(swptr->aj.tba == 0) /* If hand scroll mode active */
      gdk_window_set_cursor ((swptr->sc_fixedcon)->window, cur.cursor);

    gtk_signal_connect (GTK_OBJECT(swptr->sc_fixedcon), "button_press_event",
                        GTK_SIGNAL_FUNC (mousepress_CB), (gpointer)&cur1);
    gtk_signal_connect (GTK_OBJECT(swptr->sc_fixedcon), "button_release_event",
                        GTK_SIGNAL_FUNC (mousepress_CB), (gpointer)&cur);
  }

  /* The following sets the cursor on the window of the scrollbar of
     the scrolled window. The GDK_HAND2 is provided by gdk. */
  gdk_window_set_cursor
    (
      (GTK_SCROLLED_WINDOW(swptr->sc_win)->hscrollbar)->window,
      gdk_cursor_new (GDK_HAND2)
    );

  gdk_window_set_cursor
    (
      (GTK_SCROLLED_WINDOW(swptr->sc_win)->vscrollbar)->window,
      gdk_cursor_new (GDK_HAND2)
    );
}
/* -------------------------------------------------------------------------
      mode_CB() Callback for toggle-like mode buttons. 
                Also see mode_b_CB().
   ------------------------------------------------------------------------- */
void mode_CB(GtkWidget *widget, tx_struct *tx) 
{
  if(widget == tx->tb1) /* If text select mode button */
  {
    tx->aj.tba = 1; /* cur.tba_ptr, cur1.tba_ptr, and cur2.tba_ptr are set to 
                       the address so they automatically get the new value. */
    gtk_widget_set_style(tx->f1, tx->active_style);
    gtk_widget_set_style(tx->f0, tx->inactive_style);
  }
  else
  {
    tx->aj.tba = 0;  /* cur.tba_ptr, cur1.tba_ptr, and cur2.tba_ptr are set to
                       the address so they automatically get the new value. */
    gtk_widget_set_style(tx->f0, tx->active_style);
    gtk_widget_set_style(tx->f1, tx->inactive_style);
  }
} 
/* -------------------------------------------------------------------------
      mode_b_CB() Callback for toggle-like mode buttons.
   ------------------------------------------------------------------------- */
void mode_b_CB(GtkWidget *widget, cur_struct *c)
{
  gdk_window_set_cursor(GTK_TEXT(c->widget)->text_area, c->cursor);
}
/* -------------------------------------------------------------------------
        make_tx_win() Makes a scrolled text window with the hand cursor. Send
                      it a **static** tx_struct pointer to set up. Could
                      not use make_sc_win() because a text box has its own
                      viewport-like clipping property.
                        txptr->top is a pointer to the pup top window.
                        txptr->aj.text is the text box widget.
                        txptr->aj.tba designates hand scroll or text select.
                        txptr->aj.hadj is text box's hadj.
                        txptr->aj.vadj is text box's vadj.
                        txptr->aj.prevx stores X position.
                        txptr->aj.prevy stores Y position.
                        txptr->topwin is the top window of this popup.
                        txptr->topvbox top verticle box.
                        txptr->hbox contains fixed cont. for the mode buttons.
                        txptr->tb0 is the hand scroll mode button.
                        txptr->f0 is fixed container (for color outline).
                        txptr->tb1 is the text select mode button.
                        txptr->f1 is fixed container (for color outline).
                        txptr->inactive_style is default style for tb0 & tb1.
                        txptr->active_style is style with color changed.
                        txptr->io is an io_struct used for querying. This is
                               done after returning from this function, but
                               txptr->io.aj_ptr needs to be set to
                               &(txptr->aj).
   ------------------------------------------------------------------------- */
void make_tx_win(tx_struct *txptr, char *title, int width, int height)
{
  GtkWidget *table, *vscroll;
  GtkWidget *sel_pixmapwidget, *scr_pixmapwidget;
  static int init = 0;
  static cur_struct cur, cur1, cur2;
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new();
  set_tool_color(&tooltips);

  sc_win_setup(&(txptr->topwin), &(txptr->topvbox), title, width, height);

  txptr->aj.prevx = 0;
  txptr->aj.prevy = 0;

  txptr->hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(txptr->topvbox), txptr->hbox, FALSE, FALSE, 0);
  gtk_widget_show(txptr->hbox);

  txptr->f0 = gtk_fixed_new();
  gtk_widget_set_usize(txptr->f0, 30, 30);
  gtk_box_pack_start(GTK_BOX(txptr->hbox), txptr->f0, FALSE, FALSE, 15);
  txptr->inactive_style =
                   gtk_style_copy(gtk_widget_get_style(GTK_WIDGET(txptr->f0)));
  txptr->active_style = 
                   gtk_style_copy(gtk_widget_get_style(GTK_WIDGET(txptr->f0)));
  (txptr->active_style)->bg[0] = get_color(YELLOW);
  gtk_widget_show(txptr->f0);

  txptr->f1 = gtk_fixed_new();
  gtk_widget_set_usize(txptr->f1, 30, 30);
  gtk_box_pack_start(GTK_BOX(txptr->hbox), txptr->f1, FALSE, FALSE, 0);
  gtk_widget_show(txptr->f1);

  /* tb0 and tb1 are made to act like toggle buttons */
  txptr->tb0 = gtk_button_new(); /*Hand scroll activater button*/
  gtk_container_set_border_width(GTK_CONTAINER(txptr->tb0), 2);
  txptr->aj.tba = 0;
  gtk_widget_set_usize(txptr->tb0, 30, 30);
  gtk_container_add(GTK_CONTAINER(txptr->f0), txptr->tb0);
  gtk_tooltips_set_tip(tooltips, txptr->tb0, "Hand Scroller Mode", NULL);
  gtk_widget_show(txptr->tb0);

  txptr->tb1 = gtk_button_new(); /*Select text activater button*/
  gtk_container_set_border_width(GTK_CONTAINER(txptr->tb1), 2);
  gtk_widget_set_usize(txptr->tb1, 30, 30);
  gtk_container_add(GTK_CONTAINER(txptr->f1), txptr->tb1);
  gtk_tooltips_set_tip(tooltips, txptr->tb1, "Copy Selected Text Mode", NULL);
  gtk_widget_show(txptr->tb1);

  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER(table), 15);
  gtk_table_set_row_spacing (GTK_TABLE (table), 0, 2);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 2);
  gtk_box_pack_start (GTK_BOX (txptr->topvbox), table, TRUE, TRUE, 0);
  gtk_widget_show(table);

  txptr->aj.text = gtk_text_new(NULL, NULL);
  GTK_TEXT(txptr->aj.text)->button = 0;
  gtk_text_set_editable(GTK_TEXT(txptr->aj.text), FALSE);
  gtk_table_attach (GTK_TABLE (table), txptr->aj.text, 0, 1, 0, 1,
        (GtkAttachOptions)(GTK_EXPAND | GTK_SHRINK | GTK_FILL),
        (GtkAttachOptions)(GTK_EXPAND | GTK_SHRINK | GTK_FILL), 0, 0);
  gtk_widget_show(txptr->aj.text);
  
  if(!init)
  {
    init = 1;
    cur.cursor = create_hand_cursor();    /* Use Eric Harlow's hand cursor */
    cur1.cursor = create_hand_g_cursor(); /* Modified for grabbing */
    cur2.cursor = gdk_cursor_new(GDK_XTERM);
  } 
  /* Selected color of 3rd mouse button (takes same color as "active").
     Hide by making it the color of the background. */
  set_color(&(txptr->aj.text), WHITE, BG, ACTIVE);

  vscroll = gtk_vscrollbar_new(GTK_TEXT(txptr->aj.text)->vadj);
  gtk_table_attach (GTK_TABLE (table), vscroll, 1, 2, 0, 1,
        (GtkAttachOptions)GTK_FILL, 
        (GtkAttachOptions)(GTK_EXPAND | GTK_SHRINK | GTK_FILL), 0, 0);
  gtk_widget_show(vscroll);

  /* Did not do a hscroll because that's not available for a text box. */

  /* Set up the "hand scroller". The hadj won't do anything in hand_CB(). */
  txptr->aj.hadj = GTK_TEXT(txptr->aj.text)->hadj;
  txptr->aj.vadj = GTK_TEXT(txptr->aj.text)->vadj;

  gtk_widget_set_events(txptr->aj.text, GDK_BUTTON_PRESS_MASK
                                | GDK_BUTTON_RELEASE_MASK
                                | GDK_POINTER_MOTION_MASK
                                | GDK_POINTER_MOTION_HINT_MASK);
  gtk_signal_connect (GTK_OBJECT(txptr->aj.text), "motion_notify_event",
                      GTK_SIGNAL_FUNC (hand_CB), (gpointer)&(txptr->aj));

  close_twbutton(txptr);

  /* Set the address of this tx_struct's aj_struct to pointer reference in
     this tx_struct's io_struct, so it can be accessed by functions that
     only get an io_struct passed in (like put_text()). */
  txptr->io.aj_ptr = &(txptr->aj);

  /* Callback for the X kill */
  gtk_signal_connect(GTK_OBJECT(txptr->topwin), "delete_event",
                     GTK_SIGNAL_FUNC(deleteSTWX), (gpointer)txptr);
  gtk_widget_show(txptr->topwin);

  /* The following must come after the topwin is shown */

  scr_pixmapwidget = create_pix(&(txptr->topwin), 0);
  gtk_container_add(GTK_CONTAINER(txptr->tb0), scr_pixmapwidget);
  sel_pixmapwidget = create_pix(&(txptr->topwin), 1);
  gtk_container_add(GTK_CONTAINER(txptr->tb1), sel_pixmapwidget);

  cur.tba_ptr = &(txptr->aj.tba);
  cur.widget = txptr->aj.text;
  cur1.tba_ptr = &(txptr->aj.tba);
  cur1.widget = txptr->aj.text;
  cur2.tba_ptr = &(txptr->aj.tba);
  cur2.widget = txptr->aj.text;

  /* Set initial cursor & style */
  if(txptr->aj.tba == 0)
  {
    gdk_window_set_cursor(GTK_TEXT(txptr->aj.text)->text_area, cur.cursor);
    gtk_widget_set_style(txptr->f0, txptr->active_style);
  }
  else
  {
    gdk_window_set_cursor(GTK_TEXT(txptr->aj.text)->text_area, cur2.cursor);
    gtk_widget_set_style(txptr->f1, txptr->active_style);
  }

  gtk_signal_connect(GTK_OBJECT(txptr->tb0), "clicked",
                     GTK_SIGNAL_FUNC(mode_CB), (gpointer)txptr);
  gtk_signal_connect(GTK_OBJECT(txptr->tb1), "clicked",
                     GTK_SIGNAL_FUNC(mode_CB), (gpointer)txptr);
  gtk_signal_connect(GTK_OBJECT(txptr->tb0), "clicked",
                     GTK_SIGNAL_FUNC(mode_b_CB), (gpointer)&cur);
  gtk_signal_connect(GTK_OBJECT(txptr->tb1), "clicked",
                     GTK_SIGNAL_FUNC(mode_b_CB), (gpointer)&cur2);
  
  gtk_signal_connect (GTK_OBJECT(txptr->aj.text), "button_press_event",
                      GTK_SIGNAL_FUNC (mousepress_CB), (gpointer)&cur1);
  gtk_signal_connect (GTK_OBJECT(txptr->aj.text), "button_release_event",
                      GTK_SIGNAL_FUNC (mousepress_CB), (gpointer)&cur);

  /* Set cursor on scroll bar */
  gdk_window_set_cursor(GTK_WIDGET(vscroll)->window, gdk_cursor_new(GDK_HAND2));
}
