/*
 * Pspell interface header file
 * Copyright 2000 by Kevin Atkinson under the terms of the LGPL
 *
 */

#ifndef PSPELL__MANAGER
#define PSPELL__MANAGER

#include "error.h"
#include "config.h"
#include "wordlist.h"

#ifndef __cplusplus
  typedef struct PspellManager PspellManager;
#else
  class PspellManager;
#endif

#ifdef __cplusplus
extern "C" {
#endif

  PspellCanHaveError * new_pspell_manager(PspellConfig * config);
  /* returns a new manager class based on the settings in config  
   * It is allocated with new however you SHOULD NOT just delete
   * it as that will not probally release the shared library
   * linked in
   */

  PspellManager * to_pspell_manager(PspellCanHaveError *);
  /* casts a PspellCanHaveError to a PspellManager.  Do not cast
   * directly in C as it might not do it correctly.
   */

  void delete_pspell_manager(PspellManager *);
  /* deletes a PspellManager class and releases the shared library */

  int pspell_manager_error_number(const PspellManager *);
  
  const char * pspell_manager_error_message(const PspellManager *);

  const PspellErrorExtraInfo * 
    pspell_manager_error_extra_info(const PspellManager *);


  PspellConfig * pspell_manager_config(PspellManager *);

  /*
   * Strings from this point on are expected to be in the 
   * encoding specified by encoding()
   *
   * A size of -1 means the string is null terminted.
   * The size is the length of the word in bytes even if the 
   * encoding is "machine unsigned 16" or "machine unsigned 32"
   *
   */
  
  int pspell_manager_check(PspellManager *, const char *, int size);
  /* 
   * returns  0 if it is not in the dictionary;
   *          1 if it is;
   *         -1 on error;
   */

  
  int pspell_manager_add_to_personal(PspellManager *, const char *, int size);
  int pspell_manager_add_to_session(PspellManager *, const char *, int size);
  /* Returns 0 if the operation fails.  1 otherwise. */

  /*
   * because the word lists may potently have to convert from one
   * encoding to another the pointer returned by the emulation is only
   * valid to the next call.
   */

  const PspellWordList * pspell_manager_master_word_list(PspellManager *);   
  const PspellWordList * pspell_manager_personal_word_list(PspellManager *); 
  const PspellWordList * pspell_manager_session_word_list(PspellManager *); 
  
  int pspell_manager_save_all_word_lists(PspellManager *); 
  
  int pspell_manager_clear_session(PspellManager *);

  const PspellWordList * pspell_manager_suggest(PspellManager *, 
						const char *, int size);
  /* return null on error
   * the word list returned by suggest is only valid until the next
   * call to suggest
   */
  
  int pspell_manager_store_replacement(PspellManager *, 
				       const char * mis, int mis_size, 
				       const char * cor, int cor_size);
  
#ifdef __cplusplus
}
#endif

/* This is here for backwards compatibility for programs which use the *
 * old manager functions that don't have a size parameter. To use them *
 * simply define USE_ORIGNAL_MANAGER_FUNC before pspell.h is included  */

#if defined(USE_ORIGINAL_MANAGER_FUNCS)

#define pspell_manager_check(m,str)\
  pspell_manager_check(m,str,-1)
#define pspell_manager_add_to_personal(m,str)\
  pspell_manager_add_to_personal(m,str,-1)
#define pspell_manager_add_to_session(m,str)\
  pspell_manager_add_to_session(m,str,-1)
#define pspell_manager_suggest(m,str)\
  pspell_manager_suggest(m,str,-1)
#define pspell_manager_store_replacement(m,mis,cor)\
  pspell_manager_store_replacement(m,mis,-1,cor,-1)

#endif

#ifdef __cplusplus

class PspellManager : public PspellCanHaveError
{
 public:

  virtual PspellConfig & config() = 0;

  ////////////////////////////////////////////////////////////////
  // 
  // Strings from this point on are expected to be in the 
  // encoding specified by encoding()
  //

  virtual int check(const char *, int size) = 0;
  int check(const char * str) {return check(str,-1);}
  // returns  0 if it is not in the dictionary;
  //          1 if it is;
  //         -1 on error;
  
  virtual bool add_to_personal(const char *, int size) = 0;
  virtual bool add_to_session(const char *, int size)  = 0;
  bool add_to_personal(const char * str)
    {return add_to_personal(str,-1);}
  bool add_to_session (const char * str)
    {return add_to_session (str,-1);}
  // returns false if the operation fails
    
  // because the word lists may potently have to convert from one
  // encoding to another the pointer returned by the emulation is only
  // valid to the next call.

  virtual const PspellWordList * master_word_list()   = 0;
  virtual const PspellWordList * personal_word_list() = 0;
  virtual const PspellWordList * session_word_list()  = 0;
  // returns null on error
  
  virtual bool save_all_word_lists() = 0;
  
  virtual bool clear_session() = 0;

  virtual const PspellWordList * suggest(const char *, int size) = 0;
  const PspellWordList * suggest(const char * str) {return suggest(str, -1);}
  // return null on error
  // the word list returned by suggest is only valid until the next
  // call to suggest
  
  virtual bool store_replacement(const char * mis, int, 
				 const char * cor, int) = 0;
  bool store_replacement(const char * mis, const char * cor) 
    {return store_replacement(mis,-1,cor,-1);}
  
  virtual ~PspellManager() {}
};

#endif

#endif /* PSPELL__MANAGER */
