#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <config.h>
#include <support.h>
#include <xcio.h>
#include <sysmsg.h>

#include "option.h"
#include "frame.h"
#include "timer.h"
#include "ccp.h"
#include "log.h"
#include "env.h"
#include "mbuf.h"
#include "dev/device.h"

struct mbuf *queueHead, *queueTail;
extern int devFd;

static struct {
    struct timer_s timer;
    bool_t in;
    bool_t out;
} idleOpt;

extern time_t CpDecodeFrame(), LqrDecodeFrame();
extern time_t PapDecodeFrame(), ChapDecodeFrame();
static time_t CompDecodeFrame();
extern time_t IpInput(), VjInput(), IpEnqueue();
extern bool_t IpTrigger();
extern void IpRebuild();

extern void LcpSetup(), IpcpSetup(), CcpSetup(), LqrSetup();
extern void LcpInit(), IpcpStart(), CcpInit(), IpUp(), LqrStart();

static struct {
    const char *name;
    void (*setup)();
    void (*start)();
    time_t (*decode)();
    void (*rebuild)();
    bool_t (*trigger)();
    time_t (*enqueue)();
    u_int16_t nbo_type;		/* Network byte order */
    u_int8_t
	f_enable:1,
	f_setup:1,
	f_reject:1,
	f_dummy:5;
} frameProto[]={
    {
	"IP",
	NULL, IpUp, IpInput, IpRebuild, IpTrigger, IpEnqueue,
	NBO_PROTO_IP, 1, 0, 0, 0
    }, {
	"VJCOMP",
	NULL, NULL, VjInput, NULL, NULL, NULL,
	NBO_PROTO_VJC, 1, 0, 0, 0
    }, {
	"VJUNCOMP",
	NULL, NULL, VjInput, NULL, NULL, NULL,
	NBO_PROTO_VJUC, 1, 0, 0, 0
    }, {
	"ICOMP",
	NULL, NULL, NULL, NULL, NULL, NULL,
	0x00fb, 0, 0, 0, 0
    }, {
	"COMP",
	NULL, NULL, CompDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_COMP, 1, 0, 0, 0
    }, {
	"IPCP",
	IpcpSetup, IpcpStart, CpDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_IPCP, 1, 0, 0, 0
    }, {
	"ACP",
	NULL, NULL, NULL, NULL, NULL, NULL,
	0x8029, 0, 0, 0, 0
    }, {
	"IPXCP",
	NULL, NULL, NULL, NULL, NULL, NULL,
	NBO_PROTO_IPXCP, 0, 0, 0, 0
    }, {
	"NBCP",
	NULL, NULL, NULL, NULL, NULL, NULL,
	NBO_PROTO_NBCP, 0, 0, 0, 0
    }, {
	"ICCP",
	NULL, NULL, NULL, NULL, NULL, NULL,
	0x80fb, 0, 0, 0, 0
    }, {
	"CCP",
	CcpSetup, CcpInit, CpDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_CCP, 0, 0, 0, 0
    }, {
	"LQR",
	LqrSetup, LqrStart, LqrDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_LQR, 0, 0, 0, 0
    }, {
	"LCP",
	LcpSetup, LcpInit, CpDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_LCP, 0, 0, 0, 0
    }, {
	"PAP",
	NULL, NULL, PapDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_PAP, 1, 0, 0, 0
    }, {
	"CHAP",
	NULL, NULL, ChapDecodeFrame, NULL, NULL, NULL,
	NBO_PROTO_CHAP, 1, 0, 0, 0
    },
};

#define	NUM_FRAMES	(sizeof(frameProto)/sizeof(frameProto[0]))

char *
TostrProto(u_int16_t *nbo_proto)
{
    static char name[7];
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++) {
	if (frameProto[i].nbo_type == *nbo_proto)
	    return((char *)frameProto[i].name);
    }
    sprintf(name, "0x%04x", ntohs(*nbo_proto));
    return(name);
}

u_int16_t
StrtoProto(char *name)
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++) {
	if (!strcasecmp(frameProto[i].name, name))
	    return(ntohs(frameProto[i].nbo_type));
    }
    return(0);
}

/* Idle Timer */

static void
IdleTimerHandler()
{
    extern int CmdDisconnect();

    CmdDisconnect(0, NULL);
}

inline static void
KeepIdleTimer(u_int32_t idle)
{
    if (*idleOpt.timer.restp < idle) *idleOpt.timer.restp = idle;
}

static bool_t
EnvIdleTime(int argc, char **argv, char *outs)
{
    time_t idle;

    if (!argc) {
	idle = *idleOpt.timer.restp;
	sprintf(outs, "%02d:%02d:%02d",
		idle / 3600, (idle % 3600) / 60, idle % 60);
	return FALSE;
    }

    if (sscanf(argv[1], "%d", &idle) == 1 && idle > 0) {
	*idleOpt.timer.restp = idle;
	return TRUE;
    }
    return FALSE;
}

static bool_t
EnvIdleMode(int argc, char **argv, char *outs)
{
    int len;

    if (!argc) {
	char *name;

	if (idleOpt.in && idleOpt.out)
	    name = "bidirectional";
	else if (idleOpt.in)
	    name = "input";
	else if (idleOpt.out)
	    name = "output";
	else
	    name = "pause";

	strcpy(outs, name);
	return FALSE;
    }

    len = strlen(argv[1]);
    if (!strncasecmp("pause", argv[1], len)) {
	idleOpt.timer.st_paused = 1;
	idleOpt.in = idleOpt.out = FALSE;
    } else {
	idleOpt.timer.st_paused = 0;
	idleOpt.in = idleOpt.out = TRUE;
	if (!strncasecmp("input", argv[1], len))
	    idleOpt.out = FALSE;
	if (!strncasecmp("output", argv[1], len))
	    idleOpt.in = FALSE;
    }
    return TRUE;
}

void
IdleTimerSetup()
{
    static struct env_s envidle[]={
	{"INIT", {&pppOpt.i_to}, ENV_INT32, 0, 0, 0666},
	{"TIME", {EnvIdleTime}, ENV_SET, 0, 0, 0666},
	{"MODE", {EnvIdleMode}, ENV_SET, 0, 0, 0666},
	{NULL}
    };

    RegisterEnvs(envidle, "IDLE", NULL);
    idleOpt.timer.name = "IDLE";
    idleOpt.in = idleOpt.out = TRUE;
    idleOpt.timer.restp = &pppInfo.idle;
    idleOpt.timer.callback = IdleTimerHandler;
    idleOpt.timer.st_paused = 0;
}

/* Frame */

int
FrameEnable(u_int16_t nbo_type, int sw)
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++) {
	if (frameProto[i].nbo_type == nbo_type) {
	    if (sw >= 0) frameProto[i].f_enable = sw ? 1: 0;
	    return(frameProto[i].f_enable);
	}
    }
    return(FALSE);
}

int
FrameReject(u_int16_t nbo_proto, int sw)
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++) {
	if (frameProto[i].nbo_type == nbo_proto) {
	    if (sw >= 0) frameProto[i].f_reject = sw ? 1: 0;
	    return(frameProto[i].f_reject);
	}
    }
    return(-1);
}

bool_t
EnvFrame(int argc, char **argv, char *outs)
{
    unsigned int i;

    if (!argc) {
	char *p;

	p = outs;
	for (i = 0; i < NUM_FRAMES; i ++) {
	    if (frameProto[i].f_enable)
		p += SprintF(p, frameProto[i].f_reject ? "(%s) ": "%s ",
			     frameProto[i].name);
	}
	return FALSE;
    }
    return FALSE;
}

void
FrameResetReject()
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++)
	frameProto[i].f_reject = frameProto[i].f_enable ? 0: 1;
}

void
FrameDump(const char *name, u_char *buf, int len)
{
    int i;

    Logf(LOG_DUMP, "%s(%d)\n", name, len);
    for (i = 0; i < len; i ++) {
	Logf(LOG_DUMP, "%02x%c", buf[i],
	     ((i+1) % 8) ? ' ': ((i+1) % 16) ?  '|': '\n');
    }
    if (i % 16) Logf(LOG_DUMP, "\n");
}

int
FrameEnqueue(u_char *buf, int len, u_int16_t nbo_proto, priority_t pri)
{
    struct mbuf *qp, *qt;
#ifdef	QUEUE_ID
    static u_int8_t queueId;
#endif
    extern fd_set orgWriteFds;

    if (devFd < 0) {
	int status=CmdConnect(0, NULL);

	if (status < 0) {
	    if (status != OPEN_AGAIN) {
		char *argv[]={"auto", "off"};

		CmdAuto(2, argv);
		ConsoleMsg(MS_E_AUTO_FAIL, NULL);
	    }
	    return(0);
	}
    }

    if (ISLOG(LOG_DUMP)) {
	char str[32];

	sprintf(str, "FrameEnqueue@%s/%d", TostrProto(&nbo_proto), pri);
	FrameDump(str, buf, len);
    }
    qp = TALLOC(struct mbuf);
    qp->m_data = qp->m_off = (u_char *)Malloc(len);
    qp->m_len = len;
    qp->m_proto = nbo_proto;
    qp->m_pri = pri;
#ifdef	QUEUE_ID
    qp->id = queueId ++;
#endif
#ifdef	QUEUE_ID
    if (ISLOG(LOG_DUMP))
	Logf(LOG_DUMP, "enqueue qid=%d/%d\n", qp->id, qp->m_pri);
#endif
    memcpy(qp->m_data, buf, qp->m_len);
    /*
      qp: new queue
      qt: qt >= qp && qt->m_next < qp
                                      qp
                                     x(1)
       queueHead                      V     queueTail
          a(2) b(2)  c(1)  d(1)  e(1)  f(0)  g(0)
                         qt->m_prev qt qt->m_next

          a(2) b(2)  c(1)  d(1)  e(1)  x(1)  f(0)  g(0)
    */
    if (!queueHead) {
	/* queue list is empty */
	queueTail = queueHead = qp;
	qp->m_next = qp->m_prev = NULL;
    } else {
	qt = queueTail;
	/* search lower queue */
	while (qt) {
	    if (qt->m_pri >= pri) break;
	    qt = qt->m_prev;
	}
	if (!qt) {
	    /* i.e. new queue is the highest */
	    qp->m_prev = NULL;
	    qp->m_next = queueHead;
	    queueHead->m_prev = qp;
	    queueHead = qp;
	} else {
	    qp->m_next = qt->m_next;
	    if (qt->m_next) qt->m_next->m_prev = qp;
	    qt->m_next = qp;
	    qp->m_prev = qt;
	    if (qt == queueTail) queueTail = qp;
	}
    }
    FD_SET(devFd, &orgWriteFds);
    return(0);
}

void
RebuildQueue(u_int16_t nbo_proto)
{
    struct mbuf *qp;
    unsigned int i;

    if (queueHead) {
	extern fd_set orgWriteFds;

	FD_SET(devFd, &orgWriteFds);
    } else return;
    for (qp = queueHead; qp != NULL; qp = qp->m_next) {
	if (qp->m_proto != nbo_proto) continue;

	for (i = 0; i < NUM_FRAMES; i ++) {
	    if (frameProto[i].nbo_type == nbo_proto) {
#ifdef	QUEUE_ID
		if (ISLOG(LOG_DUMP))
		    Logf(LOG_DUMP, "rebuild qid=%d/%d\n",
			 qp->id, qp->m_pri);
#endif
		if (frameProto[i].rebuild)
		    frameProto[i].rebuild(qp->m_data, qp->m_len);
		break;
	    }
	}
    }
}

int
FrameDequeue(struct mbuf qbuf)
{
    struct mbuf *qp;

    if (!(qp = queueHead)) return(-1);
    memcpy(&qbuf, qp, sizeof(qbuf));
    if ((queueHead = qp->m_next) == NULL)
	queueTail = NULL;
    else
	queueHead->m_prev = NULL;
    Free(qp);
    return(0);
}

int
FrameRead(u_char *buf, int len, u_int16_t nbo_proto)
{
    unsigned int i;
    time_t idle;

/*
   if (ISLOG(LOG_DUMP)) {
	char str[20];
	u_int16_t nbo_proto;

	nbo_proto = htons(hbo_proto);
	sprintf(str, "FrameRead: %s", TostrProto(&nbo_proto));
	FrameDump(str, buf, len);
    }
*/
    pppInfo.r.count ++;
    for (i = 0; i < NUM_FRAMES; i ++) {
	if (frameProto[i].nbo_type == nbo_proto) {
	    if (frameProto[i].f_reject) LcpSPJ(nbo_proto);
	    else if (frameProto[i].decode) {
		idle = frameProto[i].decode(buf, len,
					    frameProto[i].nbo_type);
		if (idleOpt.in) KeepIdleTimer(idle);
	    }
	    break;
	}
    }
    return(0);
}

void
CompAllocBuffer(u_char **src, u_char **dst)
{
    static u_char *srcExtractBuf, *dstExtractBuf;

    if (!srcExtractBuf) srcExtractBuf = Malloc(MAX_FRAMESIZ);
    if (!dstExtractBuf) dstExtractBuf = Malloc(MAX_FRAMESIZ);
    *src = srcExtractBuf + MAX_HEADERGAP;
    *dst = dstExtractBuf + MAX_HEADERGAP;
}

static time_t
CompDecodeFrame(u_char *buf, int len, long arg)
{
    u_int16_t nbo_proto;
    u_char *bp;

    if (compUC) {
	bp = compUC(buf, &len, &nbo_proto);
	if (bp) FrameRead(bp, len, nbo_proto);
	else CcpSRR();
    }
    return(0);
}

void
FrameSetup()
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++)
	if (frameProto[i].setup) {
	    frameProto[i].f_setup = 1;
	    frameProto[i].setup();
	}
}

void
FrameStartLayer(u_int16_t nbo_mask)
{
    unsigned int i;

    for (i = 0; i < NUM_FRAMES; i ++)
	if (frameProto[i].start) {
	    if ((frameProto[i].nbo_type & NBO_PROTO_MASK) == nbo_mask)
		frameProto[i].start();
	}
}

void
NetworkStart()
{
    if (pppOpt.a_server != AUTH_SERVER_NONE) {
	LastLogWrite(TRUE);
    }

    FrameStartLayer(NBO_PROTO_NCPMASK);
    if (pppOpt.i_to) {
	*idleOpt.timer.restp = pppOpt.i_to;
    }
    if (*idleOpt.timer.restp <= 0) {
	idleOpt.timer.st_paused = 1;
    }
    TimerAdd(&idleOpt.timer);
}

int
CmdCpUp(int argc, char *argv[])
{
    unsigned int i;

    if (argc < 2 || pppInfo.phase < PS_NETWORK) return(-1);
    for (i = 0; i < NUM_FRAMES; i ++)
	if (frameProto[i].start
	    && (frameProto[i].nbo_type & NBO_PROTO_MASK)
	    == NBO_PROTO_NCPMASK
	    && !strcasecmp(frameProto[i].name, argv[1])) {
	    frameProto[i].start();
	}
    return(0);
}

int
IfRead(int fd)
{
    char *buf;
    int n;
    time_t idle;
    u_int16_t proto;

    if ((n = SysIfRead(fd, &buf, &proto)) > 0) {
	unsigned i;

	for (i = 0; i < NUM_FRAMES; i ++) {
	    if (frameProto[i].nbo_type == proto) {
		if (devFd < 0) {
		    bool_t tri=FALSE;

		    if (frameProto[i].trigger)
			tri = frameProto[i].trigger(buf, n);
		    if (!tri) break;
		}
		if (frameProto[i].enqueue) {
		    idle = frameProto[i].enqueue(buf, n);
		    pppInfo.s.nsize += n;
		    if (idleOpt.out) KeepIdleTimer(idle);
		}
		break;
	    }
	}
    }
    return(n);
}
