/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *            Copyright (C) 2005 SUSE Linux Products GmbH                  *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "powerlib.h"

#ifndef POWERSAVE_HAL_H
#define POWERSAVE_HAL_H

#define DBUS_API_SUBJECT_TO_CHANGE 1
#include <hal/libhal.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** @brief error for hal connect error */
#define HAL_ERROR -7

/** @brief inits the access to hal 
 *
 * if hal context is already initialized, does nothing.
 *
 * @retval true if already initialized connection could be set up
 * @retval false false if connection could not be established
*/
int ps_hal_init(void);

/** @brief frees the access to hal
 *
 * the free includes any dbus connection, dbus errors or hal contexts call
 * this inside your application as soon as the hal service disappears
 */
void ps_hal_free(void);

/** brief get integer property through hal
 *
 * fetches one interger value through hal. Throws a dbus error with error
 * message if needed.
 *
 * @param device the device from which should be fetched
 * @param property the property which should be fetched
 *
 * @return -1 on error
 */
int ps_hal_get_property_int(char *device, char *property);

/** brief get bool property through hal
 *
 * fetches one bool value through hal. Throws a dbus error with error
 * message if needed.
 *
 * @param device the device from which should be fetched
 * @param property the property which should be fetched
 *
 * @return -1 on error
 */
int ps_hal_get_property_bool(char *device, char *property);

/** brief get string property through hal
 *
 * fetches one string value through hal. Throws a dbus error with error
 * message if needed.
 *
 * @param device the device from which should be fetched
 * @param property the property which should be fetched
 *
 * @return requested value or NULL on error (string must be freed)
 */
char* ps_hal_get_property_string(char *device, char *property);

/** brief get stringlist property through hal
 *
 * fetches a string list through hal. Throws a dbus error with error
 * message if needed.
 *
 * @param device the device from which should be fetched
 * @param property the property which should be fetched
 *
 * @return array of strings or NULL on error (string must be freed with
 *         libhal_free_string_array)
 */
char **ps_hal_get_property_strlist(char *device, char *property);

/** @brief query a capability of a device
 *
 * @param device the device usi
 * @param capability the capability to query for
 *
 * @return < 0 on error
 */
int ps_hal_query_capability(char *device, char *capability);

/** @brief get the hal context used
 *
 * @return the hal context
 */
LibHalContext* ps_hal_context(void);

/** @brief get the dbus error used
 *
 * @return the dbus error
 */
DBusError* ps_hal_dbus_error(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif				/* POWERSAVE_HAL_H */
