/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>

#include <sstream>

#include "acpi.h"
#include "stringutil.h"
#include "dbus_server.h"
#include "event_management.h"
#include "globals.h"

using namespace Powersave::Globals;

string ACPI_Interface::ACPI_EVENTFILE = "/proc/acpi/event";

ACPI_Interface::ACPI_Interface():PM_Interface()
{
	pDebug(DBG_DEBUG, "Constructor ACPI_Interface");

	_last_button_occurence = getTimestamp();

	if (setBatteryAlarm_() < 0) {
		pDebug(DBG_DIAG, "could not set battery alarm");
	}

}

ACPI_Interface::~ACPI_Interface()
{
	/* reset thermal trip points to BIOS settings (general_conf object
	 * stored BIOS settings) */
	if (_thermal_trip_points_supported 
	    && (config_obj->ENABLE_THERMAL_MANAGEMENT == KERNEL
		|| config_obj->ENABLE_THERMAL_MANAGEMENT == KERNEL_PASSIVE)) {
		for (int x = 0;
		     x < MAX_THERMAL_ZONES && config_obj->thermal_zones[x].present;
		     x++) {
			// ignore thermal zones without a passive trip point in
			// KERNEL_PASSIVE mode
			if (config_obj->ENABLE_THERMAL_MANAGEMENT == KERNEL_PASSIVE 
			    && config_obj->thermal_zones[x].passive < 1)
				continue;
			setThermalTrippoints(x, config_obj->thermal_zones[x]);
		}
	}
}

int ACPI_Interface::openHWEventFD()
{
	int fd;

	char discard[MAX_LINE_SIZE];
	struct stat file_stats;

	if (stat(ACPI_EVENTFILE.c_str(), &file_stats)) {
		pDebug(DBG_DIAG, "Cannot stat %s", ACPI_EVENTFILE.c_str());
		return -1;
	}
	if (S_ISSOCK(file_stats.st_mode)) {
		if ((fd = openAcpidSocket()) < 0) {
			pDebug(DBG_DIAG, "Cannot connect to acpid socket %s",
			       ACPI_EVENTFILE.c_str());
			return -1;
		}
	} else if (S_ISREG(file_stats.st_mode)) {
		fd = open(ACPI_EVENTFILE.c_str(), O_RDONLY | O_NONBLOCK);
		if (fd < 0) {
			pDebug(DBG_WARN, "Cannot open %s: %s",
			       ACPI_EVENTFILE.c_str(), strerror(errno));
			return -1;
		}
	} else {
		pDebug(DBG_DIAG, "No valid acpi event file: %s", ACPI_EVENTFILE.c_str());
		return -1;
	}
	// discard any events happened before
	while (read(fd, discard, sizeof(discard)) > 0) {
		pDebug(DBG_INFO, "Event '%s' discarded", discard);
	}
	// set errno back: could be EAGAIN if no data is available
	errno = 0;
	return fd;
}

int ACPI_Interface::handleHWEventRequest(int fd)
{
	char buf[2] = "";
	int x = 0, y;
	std::string line = "";
	std::string type;
	std::string dev_name;
	std::string port;
	std::string count;
	long time_in_millisecs = 0;

	/* this may come from acpid socket or from /proc/acpi/event */
	while ((y = read(fd, buf, 1)) > 0) {
		line += buf;
		x++;
		if (*buf == '\n')
			break;
	}
	// end of file ? - maybe acpid and/or socket missing ?
	if (x == 0) {
		pDebug(DBG_DIAG, "hwEvent is empty or not present, fd: %d", fd);
		return -2;
	}
	// error 
	else if (y < 0) {
		int tmp = DBG_WARN;
		if (errno == EAGAIN)
			tmp = DBG_DIAG;
		pDebug(tmp, "Could not read from hwEvent fd %d: %s",
		       fd, strerror(errno));
		// this seems to be a real error; we can also get EAGAIN which
		// does not have to be an error (but should be handled, TODO!)
		if (line.empty())
			return -2;
	}
	// just to make sure
	// this also removes the trailing '\n'
	line = stripLeadingWS(line);
	line = stripTrailingWS(line);

	// empty event?
	if (line.empty())
		return 0;

	pDebug(DBG_DIAG, "ACPI Event: '%s' ignore buttons: %s",
	       line.c_str(), (_sleep_triggered ? "yes" : "no"));

	std::istringstream eStream(line.c_str());

	eStream >> type >> dev_name >> port >> count;
	// check if any of obove sections are empty
	if (!type.length() || !dev_name.length() || !port.length() || !count.length()) {
		pDebug(DBG_WARN, "Empty or incomplete event. Ignoring...");
		// we should not get here since we already tested for empty events above.
		return 1;
	}
	pDebug(DBG_DEBUG, "type: %s, dev_name: %s, port: %s, count: %s",
	       type.c_str(), dev_name.c_str(), port.c_str(), count.c_str());

	// notify the clients about an acpi event
	DBus_Server::emitSignal("AcpiEvent", line.c_str());

	if (type == "button/sleep") {
		if (_sleep_triggered) {
			pDebug(DBG_DIAG, "Sleep button ignored while resuming.");
			return 0;
		}

		// workaround for double button occurencies
		time_in_millisecs = getTimestamp();
		if ((time_in_millisecs - _last_button_occurence) 
		    < _button_time_window) {
			pDebug(DBG_DIAG, 
			       "button.sleep event occured twice in %dms, ignored", _button_time_window);
			_last_button_occurence = time_in_millisecs;
			return 0;
		} else {
			_last_button_occurence = time_in_millisecs;
			pDebug(DBG_INFO, "button.sleep event occured");
			_eM->executeEvent("button.sleep", line.c_str());
			return 1;
		}
	} else if (type == "button/power") {
		if (_sleep_triggered) {
			pDebug(DBG_DIAG, "Power button ignored while resuming.");
			return 0;
		}
		// workaround for double button occurencies
		time_in_millisecs = getTimestamp();
		if ((time_in_millisecs - _last_button_occurence)
		    < _button_time_window) {
			pDebug(DBG_DIAG, 
			       "button.power event occured twice in %dms, ignored", _button_time_window);
			_last_button_occurence = time_in_millisecs;
			return 0;
		} else {
			// wait a bit, maybe the machine should be cut off power
			if (config_obj->current_scheme->POWER_BUTTON_DELAY > 0)
				sleep(config_obj->current_scheme->POWER_BUTTON_DELAY);
			// get new timestamp because of sleep.
			_last_button_occurence = getTimestamp();
			pDebug(DBG_INFO, "button.power event occured");
			_eM->executeEvent("button.power", line.c_str());
			return 1;
		}
	} else if (type == "button/lid") {
		if (_sleep_triggered) {
			pDebug(DBG_DIAG, "Lid button ignored while resuming.");
			return 0;
		}
		switch (getLidState()) {
		case LID_OPEN:
			pDebug(DBG_INFO, "lid open event occured");
			_eM->executeEvent("button.lid.open", line.c_str());
			return 1;
		case LID_CLOSED:
			pDebug(DBG_INFO, "lid closed event occured");
			_eM->executeEvent("button.lid.closed", line.c_str());
			return 1;
		default:
			pDebug(DBG_ERR, "Could not determine lid state");
			return -1;
		}
	} else if (type == "battery") {
		// This is to avoid polling of battery.
		// set /proc/acpi/battery/*/alarm and wait for events in /proc/acpi/event

		// adjust battery alarm here, maybe we have a false alarm!
		if (setBatteryAlarm_() < 0) {
			pDebug(DBG_DIAG, "could not set battery alarm");
		}
		return 1;
	}
	// break if else to execute a battery event if ac_offline 
	// and battery state is not normal
	else if (type == "ac_adapter") {
		if (checkACStateChanges() < 0) {
			pDebug(DBG_WARN, "Could not read out AC_Adapter state.");
			return -1;
		}
		return 1;
	} else if (type == "thermal_zone") {
		pDebug(DBG_INFO, "Thermal event happend.");
		checkTemperatureStateChanges();
		return 1;
	} else if (type == "processor") {
		pDebug(DBG_INFO, "Processor event happend -> ignoring.");
		return 1;
	} else {
		pDebug(DBG_DIAG, "unknown HW event, using [other]."
		       " type '%s' dev_name '%s' port '%s' count '%s'",
		       type.c_str(), dev_name.c_str(), port.c_str(), count.c_str());
		_eM->executeEvent("other", line.c_str());
		return 1;
	}
}

int ACPI_Interface::setBatteryAlarm_()
{
	int ret = 1;
	static bool supported = true;

	if (!supported)
		return 1;


	switch (_battery.batteryState()) {
	case BAT_NORM_STATE:
		if (setBatteryAlarm(config_obj->current_scheme->BAT_WARN_LIMIT) < 0)
			ret = -1;
		pDebug(DBG_INFO, "Alarm set to %d%%",
		       config_obj->current_scheme->BAT_WARN_LIMIT);
		break;
	case BAT_WARN_STATE:
		if (setBatteryAlarm(config_obj->current_scheme->BAT_LOW_LIMIT) < 0)
			ret = -1;
		pDebug(DBG_INFO, "Alarm set to %d%%",
		       config_obj->current_scheme->BAT_LOW_LIMIT);
		break;
	case BAT_LOW_STATE:
		if (setBatteryAlarm(config_obj->current_scheme->BAT_CRIT_LIMIT) < 0)
			ret = -1;
		pDebug(DBG_INFO, "Alarm set to %d%%",
		       config_obj->current_scheme->BAT_CRIT_LIMIT);
		break;
	case BAT_CRIT_STATE:
		pDebug(DBG_INFO, "State is already critical, alarm not set");
		// normally not needed but checked anyway ...
		break;
	case BAT_NONE_STATE:
		// should never happen ...
		ret = -1;
		break;
	}

	if (ret < 0)
		supported = false;

	return ret;
}

void ACPI_Interface::activateSettings()
{
	PM_Interface::activateSettings();
	int x;
	/* override BIOS trip points in /proc/acpi/thermal_zone/../trip_points */
	if (_thermal_trip_points_supported && config_obj->ENABLE_THERMAL_MANAGEMENT == KERNEL) {
		for (x = 0;
		     x < MAX_THERMAL_ZONES && config_obj->current_scheme->thermal_zones[x].present;
		     x++) {

			setThermalTrippoints(x, config_obj->current_scheme->thermal_zones[x]);
		}
		/* trip points could not be set -> deactivating */
		if (x == 0) {
			_thermal_trip_points_supported = false;
			pDebug(DBG_DIAG, "No trip point support");
		}
	}
	if (_cooling_mode_supported) {
		for (x = 0; x < MAX_THERMAL_ZONES
			     && setCoolingMode(x, config_obj->current_scheme->COOLING_POLICY) > 0;
		     x++) ;
		if (x == 0) {
			_cooling_mode_supported = false;
			pDebug(DBG_DIAG, "BIOS has no cooling mode support.");
		}
	}
	/* throttle CPU if configured  */
	if (config_obj->current_scheme->ALWAYS_THROTTLE) {
		_throttleInterface.throttle(config_obj->current_scheme->MAX_CPU_THROTTLING);
	} else
		_throttleInterface.dethrottle();
}

int ACPI_Interface::suspend_to_ram()
{
	if (!((_supported_sleeping_states & ACPI_S3)
	      || (_supported_sleeping_states & ACPI_S3_BIOS))) {
		pDebug(DBG_WARN, "ACPI S3 state not available");
		return -1;
	}

	pDebug(DBG_INFO, "Set machine into suspend2ram (S3) mode");

	executeScript(ACPI_SLEEP_SCRIPT, "suspend2ram");

	return 1;
}

int ACPI_Interface::standby()
{
	if (!(_supported_sleeping_states & ACPI_S1)) {
		pDebug(DBG_WARN, "ACPI S1 state not available");
		return -1;
	}

	pDebug(DBG_INFO, "Set machine into standby (S1) mode");

	executeScript(ACPI_SLEEP_SCRIPT, "standby");

	return 1;
}

unsigned long ACPI_Interface::getTimestamp()
{
	timeval t;
	gettimeofday(&t, NULL);
	return (t.tv_sec * 1000 + t.tv_usec / 1000);
}

int ACPI_Interface::openAcpidSocket()
{
	int fd;
	int r;
	struct sockaddr_un addr;
	int fl;

	fd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (fd < 0) {
		return fd;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_UNIX;
	sprintf(addr.sun_path, "%s", ACPI_EVENTFILE.c_str());

	r = connect(fd, (struct sockaddr *)&addr, sizeof(addr));
	if (r < 0) {
		close(fd);
		return r;
	}
	fl = fcntl(fd, F_GETFL);
	fcntl(fd, F_SETFL, fl | O_NONBLOCK);
	return fd;
}
