/* Change  _("string") to translation[nnn] on C sources.
   Based on:
     GNU gettext - internationalization aids
     Copyright (C) 1995, 1996, 1997, 1998 Free Software Foundation, Inc.
     Written by Peter Miller <millerp@canb.auug.org.au>
   Modifications are:
     Copyright (C) 1999, Enrique Zanardi <ezanard@debian.org>
     Written by Enrique Zanardi <ezanard@debian.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "gettext.h"
#include "domain.h"
#include "config.h"

#define _(s) s
#define PARAMS(s) s

extern struct loaded_domain *load_domain (char *domain_file);

/* The ANSI C standard defines several phases of translation:

   1. Terminate line by \n, regardless of the external representation
      of a text line.  Stdio does this for us.

   2. Convert trigraphs to their single character equivalents.
      (We don't support trigraphs).

   3. Concatenate each line ending in backslash (\) with the following
      line.

   4. Replace each comment with a space character.

   5. Parse each resulting logical line as preprocessing tokens a
      white space.

   6. Recognize and carry out directives (it also expands macros on
      non-directive lines, which we do not do here).
      (We may safely ignore this step here).

   7. Replaces escape sequences within character strings with their
      single character equivalents (we do this in step 5, because we
      don't have to worry about the #include argument).

   8. Concatenates adjacent string literals to form single string
      literals (because we don't expand macros, there are a few things
      we will miss).

   9. Converts the remaining preprocessing tokens to C tokens and
      discards any white space from the translation unit.

   This lexer implements the above, and presents the scanner
   with a stream of C tokens. */

enum token_type_ty
{
  token_type_character_constant,
  token_type_eof,
  token_type_eoln,
  token_type_hash,
  token_type_lp,
  token_type_comma,
  token_type_name,
  token_type_number,
  token_type_string_literal,
  token_type_symbol,
  token_type_white_space
};
typedef enum token_type_ty token_type_ty;

typedef struct token_ty token_ty;
struct token_ty
{
  token_type_ty type;
  char *string;
  long number;
  char *buffer;
};

/* Global Variables */
static FILE *fp;
static int cplusplus_comments;
static char *pubbuffer;
static int pubbufpos;
struct loaded_domain *domain;
static char *programname;
static char *mofile;

/* Prototypes for local functions.  */
static int phase1_getc PARAMS ((void));
static void phase1_ungetc PARAMS ((int __c));
static int phase3_getc PARAMS ((void));
static void phase3_ungetc PARAMS ((int __c));
static int phase4_getc PARAMS ((void));
static void phase4_ungetc PARAMS ((int __c));
static int phase7_getc PARAMS ((void));
static void phase7_ungetc PARAMS ((int __c));
static void phase5_get PARAMS ((token_ty *__tp));
static void phase5_unget PARAMS ((token_ty *__tp));
static void phase8_get PARAMS ((token_ty *__tp));

static void flushbuffer (token_ty tp);
static void bufferunget (void);
static void bufferget (int __c);
static char *fetchbuffer (void);
static void reset_buffer (int free);
void *xrealloc (void *p, size_t n);
char *xstrdup (char *string);
int getidx (char *msgid);


/* 1. Terminate line by \n, regardless of the external representation of
   a text line.  Stdio does this for us, we just need to check that
   there are no I/O errors, and cope with potentially 2 characters of
   pushback, not just the one that ungetc can cope with.  */

/* Maximum used guaranteed to be < 4.  */
static unsigned char phase1_pushback[4];
static int phase1_pushback_length;


static int
phase1_getc ()
{
  int c;

  if (phase1_pushback_length)
    {
      c = phase1_pushback[--phase1_pushback_length];
      bufferget(c);
      return c;
    }
  while (1)
    {
      c = getc (fp);
      bufferget(c);
      switch (c)
	{
	case EOF:
	  if (ferror (fp))
	    {
	    bomb:
	      fprintf(stderr,_("error while reading stdin"));
	      exit (1);
	    }
	  return EOF;

	case '\\':
	  c = getc (fp);
          bufferget(c);
	  if (c == EOF)
	    {
	      if (ferror (fp))
		goto bomb;
	      return '\\';
	    }
	  if (c != '\n')
	    {
	      ungetc (c, fp);
              bufferunget();
	      return '\\';
	    }
	  break;

	default:
	  return c;
	}
    }
}


static void
phase1_ungetc (c)
     int c;
{
  switch (c)
    {
    case EOF:
      break;

    default:
      phase1_pushback[phase1_pushback_length++] = c;
      bufferunget();
      break;
    }
}


/* 3. Concatenate each line ending in backslash (\) with the following
   line.  Basically, all you need to do is elide "\\\n" sequences from
   the input.  */

/* Maximum used guaranteed to be < 4.  */
static unsigned char phase3_pushback[4];
static int phase3_pushback_length;


static int
phase3_getc ()
{
  int c;
  if (phase3_pushback_length) {
    c = phase3_pushback[--phase3_pushback_length];
    bufferget(c);
    return c;
  }
  for (;;)
    {
      c = phase1_getc ();
      if (c != '\\')
	return c;
      c = phase1_getc ();
      if (c != '\n')
	{
	  phase1_ungetc (c);
	  return '\\';
	}
    }
}

static void
phase3_ungetc (c)
     int c;
{
  if (c != EOF) {
    phase3_pushback[phase3_pushback_length++] = c;
    bufferunget();
  }
}


/* 4. Replace each comment that is not inside a character constant or
   string literal with a space character.
   We also optionally understand C++ comments.  */

static int
phase4_getc ()
{
  int c;
  int state;

  c = phase3_getc ();
  if (c != '/')
    return c;
  c = phase3_getc ();
  switch (c)
    {
    default:
      phase3_ungetc (c);
      return '/';

    case '*':
      /* C comment.  */
      state = 0;
      while (1)
	{
	  c = phase3_getc ();
	  if (c == EOF)
	    break;
	  switch (c)
	    {
	    case '\n':
	      state = 0;
	      continue;

	    case '*':
	      state = 1;
	      continue;

	    case '/':
	      if (state == 1)
		  break;
	      /* FALLTHROUGH */

	    default:
	      state = 0;
	      continue;
	    }
	  break;
	}
      return ' ';

    case '/':
      /* C++ comment.  */
      if (!cplusplus_comments)
	{
	  phase3_ungetc ('/');
	  return '/';
	}
      while (1)
	{
	  c = phase3_getc ();
	  if (c == '\n' || c == EOF)
	    break;
	}
      return '\n';
    }
}


static void
phase4_ungetc (c)
     int c;
{
  phase3_ungetc (c);
}


/* 7. Replace escape sequences within character strings with their
   single character equivalents.  This is called from phase 5, because
   we don't have to worry about the #include argument.  There are
   pathological cases which could bite us (like the DOS directory
   separator), but just pretend it can't happen.  */

#define P7_QUOTES (1000 + '"')
#define P7_QUOTE (1000 + '\'')
#define P7_NEWLINE (1000 + '\n')

static int
phase7_getc ()
{
  int c, n, j;

  /* Use phase 3, because phase 4 elides comments.  */
  c = phase3_getc ();

  /* Return a magic newline indicator, so that we can distinguish
     between the user requesting a newline in the string (e.g. using
     "\n" or "\15") from the user failing to terminate the string or
     character constant.  The ANSI C standard says: 3.1.3.4 Character
     Constants contain ``any character except single quote, backslash or
     newline; or an escape sequence'' and 3.1.4 String Literals contain
     ``any character except double quote, backslash or newline; or an
     escape sequence''.

     Most compilers give a fatal error in this case, however gcc is
     stupidly silent, even though this is a very common typo.  OK, so
     gcc --pedantic will tell me, but that gripes about too much other
     stuff.  Could I have a ``gcc -Wnewline-in-string'' option, or
     better yet a ``gcc -fno-newline-in-string'' option, please?  Gcc is
     also inconsistent between string literals and character constants:
     you may not embed newlines in character constants; try it, you get
     a useful diagnostic.  --PMiller  */
  if (c == '\n')
    return P7_NEWLINE;

  if (c == '"')
    return P7_QUOTES;
  if (c == '\'')
    return P7_QUOTE;
  if (c != '\\')
    return c;
  c = phase3_getc ();
  switch (c)
    {
    default:
      /* Unknown escape sequences really should be an error, but just
	 ignore them, and let the real compiler complain.  */
      phase3_ungetc (c);
      return '\\';

    case '"':
    case '\'':
    case '?':
    case '\\':
      return c;

      /* The \a and \v escapes were added by the ANSI C Standard.
	 Prior to the Standard, most compilers did not have them.
	 Because we need the same program on all platforms we don't
	 provide support for them here.

	 The gcc sources comment that \a is commonly available in
	 pre-ANSI compilers.  --PMiller  */

    case 'b':
      return '\b';

      /* The \e escape is preculiar to gcc, and assumes an ASCII
         character set (or superset).  We don't provide support for it
         here.  */

    case 'f':
      return '\f';
    case 'n':
      return '\n';
    case 'r':
      return '\r';
    case 't':
      return '\t';

    case 'x':
      c = phase3_getc ();
      switch (c)
	{
	default:
	  phase3_ungetc (c);
	  phase3_ungetc ('x');
	  return '\\';

	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
	case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
	  break;
	}
      n = 0;
      for (;;)
	{
	  switch (c)
	    {
	    default:
	      phase3_ungetc (c);
	      return n;
	      break;

	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
	      n = n * 16 + c - '0';
	      break;;

	    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
	      n = n * 16 + 10 + c - 'A';
	      break;

	    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
	      n = n * 16 + 10 + c - 'a';
	      break;
	    }
	  c = phase3_getc ();
	}
      return n;

    case '0': case '1': case '2': case '3':
    case '4': case '5': case '6': case '7':
      n = 0;
      for (j = 0; j < 3; ++j)
	{
	  n = n * 8 + c - '0';
	  c = phase3_getc ();
	  switch (c)
	    {
	    default:
	      break;

	    case '0': case '1': case '2': case '3':
	    case '4': case '5': case '6': case '7':
	      continue;
	    }
	  break;
	}
      phase3_ungetc (c);
      return n;
    }
}


static void
phase7_ungetc (c)
     int c;
{
  phase3_ungetc (c);
}


/* 5. Parse each resulting logical line as preprocessing tokens and
   white space.  Preprocessing tokens and C tokens don't always match.  */

/* Maximum used guaranteed to be < 4.  */
static token_ty phase5_pushback[4];
static int phase5_pushback_length;


static void
phase5_get (tp)
     token_ty *tp;
{
  static char *buffer;
  static int bufmax;
  int bufpos;
  int c;

  if (phase5_pushback_length)
    {
      *tp = phase5_pushback[--phase5_pushback_length];
      return;
    }
  tp->string = 0;
  tp->number = 0;
  tp->buffer = 0;
  c = phase4_getc ();
  switch (c)
    {
    case EOF:
      tp->type = token_type_eof;
      tp->buffer = fetchbuffer();
      return;

    case '\n':
      tp->type = token_type_eoln;
      tp->buffer = fetchbuffer();
      return;

    case ' ':
    case '\f':
    case '\t':
      for (;;)
	{
	  c = phase4_getc ();
	  switch (c)
	    {
	    case ' ':
	    case '\f':
	    case '\t':
	      continue;

	    default:
	      phase4_ungetc (c);
	      break;
	    }
	  break;
	}
      tp->type = token_type_white_space;
      tp->buffer = fetchbuffer();
      return;

    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G':
    case 'H': case 'I': case 'J': case 'K': case 'L': case 'M': case 'N':
    case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U':
    case 'V': case 'W': case 'X': case 'Y': case 'Z':
    case '_':
    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case 'g':
    case 'h': case 'i': case 'j': case 'k': case 'l': case 'm': case 'n':
    case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u':
    case 'v': case 'w': case 'x': case 'y': case 'z':
      bufpos = 0;
      for (;;)
	{
	  if (bufpos >= bufmax-1)
	    {
	      bufmax += 100;
	      buffer = xrealloc (buffer, bufmax);
	    }
	  buffer[bufpos++] = c;
	  c = phase4_getc ();
	  switch (c)
	    {
	    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
	    case 'G': case 'H': case 'I': case 'J': case 'K': case 'L':
	    case 'M': case 'N': case 'O': case 'P': case 'Q': case 'R':
	    case 'S': case 'T': case 'U': case 'V': case 'W': case 'X':
	    case 'Y': case 'Z':
	    case '_':
	    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
	    case 'g': case 'h': case 'i': case 'j': case 'k': case 'l':
	    case 'm': case 'n': case 'o': case 'p': case 'q': case 'r':
	    case 's': case 't': case 'u': case 'v': case 'w': case 'x':
	    case 'y': case 'z':
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
	      continue;

	    default:
	      phase4_ungetc (c);
	      break;
	    }
	  break;
	}
      if (bufpos >= bufmax-1)
	{
	  bufmax += 100;
	  buffer = xrealloc (buffer, bufmax);
	}
      buffer[bufpos] = 0;
      tp->string = xstrdup (buffer);
      tp->type = token_type_name;
      tp->buffer = fetchbuffer();
      return;

    case '.':
      c = phase4_getc ();
      phase4_ungetc (c);
      switch (c)
	{
	default:
	  tp->type = token_type_symbol;
          tp->buffer = fetchbuffer();
	  return;

	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	  c = '.';
	  break;
	}
      /* FALLTHROUGH */

    case '0': case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8': case '9':
      /* The preprocessing number token is more "generous" than the C
	 number tokens.  This is mostly due to token pasting (another
	 thing we can ignore here).  */
      bufpos = 0;
      while (1)
	{
	  if (bufpos >= bufmax-1)
	    {
	      bufmax += 100;
	      buffer = xrealloc (buffer, bufmax);
	    }
	  buffer[bufpos++] = c;
	  c = phase4_getc ();
	  switch (c)
	    {
	    case 'e':
	    case 'E':
	      if (bufpos >= bufmax-1)
		{
		  bufmax += 100;
		  buffer = xrealloc (buffer, bufmax);
		}
	      buffer[bufpos++] = c;
	      c = phase4_getc ();
	      if (c != '+' || c != '-')
		{
		  phase4_ungetc (c);
		  break;
		}
	      continue;

	    case 'A': case 'B': case 'C': case 'D':           case 'F':
	    case 'G': case 'H': case 'I': case 'J': case 'K': case 'L':
	    case 'M': case 'N': case 'O': case 'P': case 'Q': case 'R':
	    case 'S': case 'T': case 'U': case 'V': case 'W': case 'X':
	    case 'Y': case 'Z':
	    case 'a': case 'b': case 'c': case 'd':           case 'f':
	    case 'g': case 'h': case 'i': case 'j': case 'k': case 'l':
	    case 'm': case 'n': case 'o': case 'p': case 'q': case 'r':
	    case 's': case 't': case 'u': case 'v': case 'w': case 'x':
	    case 'y': case 'z':
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
	    case '.':
	      continue;

	    default:
	      phase4_ungetc (c);
	      break;
	    }
	  break;
	}
      if (bufpos >= bufmax-1)
	{
	  bufmax += 100;
	  buffer = xrealloc (buffer, bufmax);
	}
      buffer[bufpos] = 0;
      tp->type = token_type_number;
      tp->number = atol (buffer);
      tp->buffer = fetchbuffer();
      return;

    case '\'':
      /* We could worry about the 'L' before wide character constants,
	 but ignoring it has no effect unless one of the keywords is
	 "L".  Just pretend it won't happen.  Also, we don't need to
	 remember the character constant.  */
      while (1)
	{
	  c = phase7_getc ();
	  if (c == P7_NEWLINE)
	    {
	      fprintf(stderr, _("warning: unterminated character constant"));
	      phase7_ungetc ('\n');
	      break;
	    }
	  if (c == EOF || c == P7_QUOTE)
	    break;
	}
      tp->type = token_type_character_constant;
      tp->buffer = fetchbuffer();
      return;

    case '"':
      /* We could worry about the 'L' before wide string constants,
	 but since gettext's argument is not a wide character string,
	 let the compiler complain about the argument not matching the
	 prototype.  Just pretend it won't happen.  */
      bufpos = 0;
      while (1)
	{
	  c = phase7_getc ();
	  if (c == P7_NEWLINE)
	    {
	      fprintf(stderr, _("warning: unterminated string literal"));
	      phase7_ungetc ('\n');
	      break;
	    }
	  if (c == EOF || c == P7_QUOTES)
	    break;
	  if (c == P7_QUOTE)
	    c = '\'';
	  if (bufpos >= bufmax-1)
	    {
	      bufmax += 100;
	      buffer = xrealloc (buffer, bufmax);
	    }
	  buffer[bufpos++] = c;
	}
      if (bufpos >= bufmax-1)
	{
	  bufmax += 100;
	  buffer = xrealloc (buffer, bufmax);
	}
      buffer[bufpos] = 0;
      tp->type = token_type_string_literal;
      tp->string = xstrdup (buffer);
      tp->buffer = fetchbuffer();
      return;

    case '(':
      tp->type = token_type_lp;
      tp->buffer = fetchbuffer();
      return;

    default:
      /* We could carefully recognize each of the 2 and 3 character
        operators, but it is not necessary, as we only need to recognize
        gettext invocations.  Don't bother.  */
      tp->type = token_type_symbol;
      tp->buffer = fetchbuffer();
      return;
    }
}


static void
phase5_unget (tp)
     token_ty *tp;
{
  if (tp->type != token_type_eof)
    phase5_pushback[phase5_pushback_length++] = *tp;
}


/* 8. Concatenate adjacent string literals to form single string
   literals (because we don't expand macros, there are a few things we
   will miss).  */

static void
phase8_get (tp)
     token_ty *tp;
{
  phase5_get (tp);
  if (tp->type != token_type_string_literal)
    return;
  while (1)
    {
      token_ty tmp;
      size_t len;

      phase5_get (&tmp);
      switch (tmp.type) 
      {
	case token_type_string_literal:
          len = strlen (tp->string);
          tp->string = xrealloc (tp->string, len + strlen (tmp.string) + 1);
          strcpy (tp->string + len, tmp.string);
          free (tmp.string);
	  /* FALLTHROUGH */

	case token_type_white_space:
	case token_type_eoln:
          len = strlen (tp->buffer);
          tp->buffer = xrealloc (tp->buffer, len + strlen (tmp.buffer) + 1);
          strcpy (tp->buffer + len, tmp.buffer);
          free (tmp.buffer);
	  break;

	default:
	  phase5_unget (&tmp);
	  return;
      }
    }
}

/* 9. Convert the remaining preprocessing tokens to C tokens and
   discards any white space from the translation unit.  */

void my_lex (token_ty *tp) {
  while (1)
    {
      token_ty token;

      phase8_get (&token);
      switch (token.type)
	{
        case token_type_eof:
	/* Remove the EOF character from the buffer */
	  token.buffer[strlen(token.buffer)-1]='\0';
	  flushbuffer(token);
          tp->type = token_type_eof;
          return;

	case token_type_white_space:
	case token_type_eoln:
	  flushbuffer(token);
	  break;

	case token_type_name:
	  if ( (strcmp (token.string, "_") == 0
		  || strcmp (token.string, "N_") == 0) )
	  {
	    free (token.buffer);
	    tp->type = token_type_name;
	  }
	  else
	  {
	    flushbuffer(token);
	    tp->type = token_type_symbol;
	  }
	  free (token.string);
	  return;

	case token_type_lp:
	  tp->buffer = token.buffer;
	  tp->type = token_type_lp;
	  return;

	case token_type_string_literal:
	  tp->buffer = token.buffer;
	  tp->type = token_type_string_literal;
	  tp->string = token.string;
	  return;

	default:
	  flushbuffer(token);
	  tp->type = token_type_symbol;
	  return;
	}
    }
}

void scan_file(void) {
  int state;

  /* The file is broken into tokens.  Scan the token stream, looking for
     a keyword, followed by a left paren, followed by a string.  When we
     see this sequence, we have something to remember.  We assume we are
     looking at a valid C or C++ program, and leave the complaints about
     the grammar to the compiler.  */

  /* Start state is 0.  */
  state = 0;

  while (1)
   {
     token_ty token;

     /* A simple state machine is used to do the recognising:
        State 0 = waiting for something to happen
        State 1 = seen one of our keywords with string in first parameter
        State 2 = was in state 1 and now saw a left paren */
     my_lex (&token);
     switch (token.type)
       {
       case token_type_name:
	 state = 1;
	 continue;

       case token_type_lp:
	 switch (state)
	   {
	   case 1:
	     free (token.buffer);
	     state = 2;
	     break;
	   default:
	     flushbuffer(token);
	     state = 0;
	   }
	 continue;

       case token_type_string_literal:
	 if (state == 2)
	 {
	   printf ("("VARNAME"[%d]",getidx(token.string));
	   free (token.buffer);
	   free (token.string);
	 }
	 else
	 {
	   flushbuffer(token);
	   free (token.string);
	 }
	 state = 0;
	 continue;

       default:
	 state = 0;
	 continue;

       case token_type_eof:
	 break;
       }
     break;
   }
}

static void flushbuffer(token_ty tp) {
  printf("%s",tp.buffer);
  free (tp.buffer);
}

static void bufferunget() {
  if (pubbufpos > 0) pubbufpos--;
}

static void bufferget(int __c) {
  static int bufmax;

  if (pubbuffer == NULL)
  {
     bufmax = 0;
     pubbufpos = 0;
  }
  if (pubbufpos >= bufmax-1) 
  {
     bufmax += 100;
     pubbuffer = xrealloc (pubbuffer, bufmax);
  }
  pubbuffer[pubbufpos++]=__c;
}

static void reset_buffer(int freebuf) {
  if (freebuf) free(pubbuffer);
  pubbufpos = 0;
  pubbuffer=NULL;
}

static char *fetchbuffer(void) {
  char *tmp;
  if (pubbuffer) {
	  pubbuffer[pubbufpos]='\0';
	  tmp=strdup(pubbuffer);
	  reset_buffer(1);
	  return tmp;
  }
  return pubbuffer;
}

/* Some functions from gettext/lib/ , slightly modified. */

/* Change the size of an allocated block of memory P to N bytes.
   If P is NULL, allocate N bytes of memory dynamically. 
   (With error checking).
 */

void *
xrealloc (p, n)
     void *p;
     size_t n;
{
  if (p == NULL)
    p = malloc (n);
  else
    p = realloc (p, n);
  if (p == 0)
  {
    fprintf(stderr,_("Memory exhausted"));
    exit (1);
  }
  return p;
}

/* Return a newly allocated copy of STRING.  */

char *
xstrdup (string)
     char *string;
{
  return strcpy (xrealloc (NULL, strlen (string) + 1), string);
}

int getidx(char *msgid) {
  struct string_desc *dscOrig;
  int i;

  if (domain == NULL)
    domain = load_domain(mofile);
  if (domain == NULL) {
    fprintf(stderr,"Error loading %s file\n",mofile);
    exit(-1);
  }
  dscOrig=domain->orig_tab;
  for (i=0; i<domain->nstrings; i++, dscOrig++) {
    if (strcmp(msgid,(char *)domain->data + dscOrig->offset) == 0) {
      return(i);
    }
  }
  fprintf(stderr,"String \"%s\" not found.\n",msgid);
  exit(-1);
}

void usage(int exitcode) {
  printf("Usage: %s [-m MO-file] < filename.c > filename.tmp.c",programname);
  exit(exitcode);
}

int main(int argc, char **argv) {
  int optchar;

  /* Set default name for MO-file */
  mofile="messages.mo";

  programname = argv[0];

  while ((optchar = getopt (argc, argv, "+m:h")) != EOF)
    switch (optchar)
    {
    case 'm':
      mofile = optarg;
      break;
    case 'h':
      usage (0);
    default:
      usage (-1);
    }
			
  /* Inform scanner whether we have C++ files or not.  */
  cplusplus_comments = 0;

  /* Initialize messages catalog */
  domain = NULL;
  /* Initialize public buffer */
  reset_buffer(0);
  /* Read the file from standard input */
  fp = stdin;

  scan_file();
  exit(0);
}
