# Author:    Lisandro Dalcin
# Contact:   dalcinl@users.sourceforge.net
# Copyright: This module has been placed in the public domain.
# Id: $Id$

"""
Logging Facilities
==================

Access to logging facilities.
"""

# --------------------------------------------------------------------

__date__     = '$Date$'
__version__  = '$Version$'
__revision__ = '$Revision$'

__docformat__ = 'reStructuredText'

# --------------------------------------------------------------------

__all__ = ['Log']


# --------------------------------------------------------------------

from petsc4py.lib import _petsc

# --------------------------------------------------------------------

class Log(object):

    """
    PETSc Logging.
    """

    @staticmethod
    def begin():
        """
        """
        _petsc.PetscLogBegin()

    def beginAll():
        """
        """
        _petsc.PetscLogAllBegin()

    @staticmethod
    def logActions(flag):
        """
        """
        _petsc.PetscLogActions(flag)

    @staticmethod
    def logObjects(flag):
        """
        """
        _petsc.PetscLogObjects(flag)
    
    @staticmethod
    def printSummary(filename='stdout', comm=None):
        """
        """
        _petsc.PetscLogPrintSummary(comm, filename)

    # Stages
    # ------
    
    @staticmethod
    def stageRegister(name):
        """
        """
        return _petsc.PetscLogStageRegister(name)

    @staticmethod
    def stageGetVisible(stage):
        """
        """
        return bool(_petsc.PetscLogStageGetVisible(stage))

    @staticmethod
    def stageSetVisible(stage, flag):
        """
        """
        _petsc.PetscLogStageSetVisible(stage, flag)

    @staticmethod
    def stageGetActive(stage):
        """
        """
        return bool(_petsc.PetscLogStageGetActive(stage))

    @staticmethod
    def stageSetActive(stage, flag):
        """
        """
        _petsc.PetscLogStageSetActive(stage, flag)

    @staticmethod
    def stageActivate(stage):
        """
        """
        _petsc.PetscLogStageSetActive(stage, True)

    @staticmethod
    def stageDeactivate(stage):
        """
        """
        _petsc.PetscLogStageSetActive(stage, False)

    @staticmethod
    def stagePush(stage):
        """
        """
        _petsc.PetscLogStagePush(stage)

    @staticmethod
    def stagePop():
        """
        """
        _petsc.PetscLogStagePop()

    # Classes
    # -------
    
    @staticmethod
    def classRegister(name):
        """
        """
        return _petsc.PetscLogClassRegister(name)

    @staticmethod
    def classSetActive(cookie, flag):
        """
        """
        if flag:
            _petsc.PetscLogEventActivateClass(cookie)
        else:
            _petsc.PetscLogEventDeactivateClass(cookie)

    @staticmethod
    def classActivate(cookie):
        """
        """
        _petsc.PetscLogEventActivateClass(cookie)

    @staticmethod
    def classDeactivate(cookie):
        """
        """
        _petsc.PetscLogEventDeactivateClass(cookie)

    # Events
    # ------
    
    @staticmethod
    def eventRegister(name, cookie=None):
        """
        """
        if cookie is None:
            return _petsc.PetscLogEventRegister(name)
        else:
            return _petsc.PetscLogEventRegister(name, cookie)

    @staticmethod
    def eventSetActive(event, flag):
        """
        """
        if flag:
            _petsc.PetscLogEventActivate(event)
        else:
            _petsc.PetscLogEventDeactivate(event)

    @staticmethod
    def eventActivate(event):
        """
        """
        _petsc.PetscLogEventActivate(event)

    @staticmethod
    def eventDeactivate(event):
        """
        """
        _petsc.PetscLogEventDeactivate(event)

    @staticmethod
    def eventBegin(event, *objs):
        """
        """
        _petsc.PetscLogEventBegin(event, *objs[:4])

    @staticmethod
    def eventEnd(event, *objs):
        """
        """
        _petsc.PetscLogEventEnd(event, *objs[:4])

    # Flops
    # -----
    
    @staticmethod
    def logFlops(flops):
        """
        """
        _petsc.PetscLogFlops(flops)

    # Time
    # ----
    
    @staticmethod
    def getTime():
        """
        """
        return _petsc.PetscGetTime()

    @staticmethod
    def getCPUTime():
        """
        """
        return _petsc.PetscGetCPUTime()

# --------------------------------------------------------------------

class Stage(int):

    """
    """

    def __new__(cls, stage):
        if isinstance(stage, str):
            _stage = Log.stageRegister(stage)
        else:
            _stage = int(stage)
        return int.__new__(cls, _stage)

    def push(self):
        """
        """
        Log.stagePush(self)

    def pop(self):
        """
        """
        Log.stagePop()

    active  = property(fget=Log.stageGetActive,
                       fset=Log.stageSetActive)

    visible = property(fget=Log.stageGetVisible,
                       fset=Log.stageSetVisible)


class Class(int):

    """
    """

    def __new__(cls, cookie):
        if isinstance(cookie, str):
            _cookie = Log.classRegister(cookie)
        else:
            _cookie = int(cookie)
        return int.__new__(cls, _cookie)

    active  = property(fget=None,
                       fset=Log.classSetActive)


class Event(int):

    """
    """

    def __new__(cls, event, cookie=None):
        if isinstance(event, str):
            _event = Log.eventRegister(event, cookie)
        else:
            _event = int(event)
        return int.__new__(cls, _event)

    def begin(self, *objs):
        """
        """
        Log.eventBegin(self, *objs)

    def end(self, *objs):
        """
        """
        Log.eventEnd(self, *objs)

    active  = property(fget=None,
                       fset=Log.eventSetActive)

# --------------------------------------------------------------------

Log.Stage = Stage
Log.Class = Class
Log.Event = Event

# --------------------------------------------------------------------
