/*
 * mcu.h
 *
 * H323 routines for a simple MCU
 *
 * Copyright (C) 1993-1998 Equivalence Pty. Ltd.
 * Copyright (C) 2006 Post Increment
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of ths code were written by by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Stonevoice, slc. http://www.stonevoice.com
 *
 * Portions of this code were written by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Citron Networks (http://www.citron.com.tw)
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 Craig Southeren (craig@postincrement.com)
 *
 * $Log: h323.h,v $
 * Revision 1.4  2009/02/24 11:47:51  willamowius
 * compatibility with older H323Plus versions
 *
 * Revision 1.3  2008/05/23 11:18:15  willamowius
 * switch BOOL to PBoolean to be able to compile with Ptlib 2.2.x
 *
 * Revision 1.2  2008/02/19 06:36:11  shorne
 * Added ability to have 16CIF & 4CIF size video. Thx Marcos Jardini for the initial 4CIF patch
 *
 * Revision 1.1  2007/10/17 19:44:30  shorne
 * Initial Commit
 *
 * Revision 2.6.2.1  2007/08/16 20:29:01  shorne
 * Better Gatekeeper anf Gateway prefix support
 *
 * Revision 2.6  2006/08/02 06:24:53  csoutheren
 * Add provision for recording input audio
 *
 * Revision 2.5  2006/07/21 08:01:40  csoutheren
 * Fixed conference member detect
 * Re-factored video mixer code slightly
 *
 * Revision 2.4  2006/07/21 05:52:51  csoutheren
 * Add flag to indicate if remote endpoint is an MCU
 *
 * Revision 2.3  2006/07/21 05:08:03  csoutheren
 * Stability fixes and more inline documentation
 * Thanks to Paolo Amadini of Stonevoice
 *
 * Revision 2.2  2006/06/21 06:11:36  csoutheren
 * Fixes for latest pwlib
 *
 * Revision 2.1  2006/06/09 04:39:59  csoutheren
 * Migrated VideoBranch to main trunk
 *
 * Revision 1.1.2.6  2006/05/31 08:48:02  csoutheren
 * Fixed crash on second call when entry/exit files do not exist
 * Fix crash when Cisco HOLD used. Thanks to Christian Bongiovanni of Stonevoice
 *
 * Revision 1.1.2.5  2006/04/26 13:09:08  csoutheren
 * Fix problem when connecting file not available
 * Add optional time limit for rooms
 *
 * Revision 1.1.2.4  2006/04/06 08:20:29  csoutheren
 * Retyped conference member identifier to epxlicit type
 * Added support for H.245 terminal added and terminal left
 *
 * Revision 1.1.2.3  2006/04/06 01:11:16  csoutheren
 * Latest sources include
 *   - premedia blanking and optional image display
 *   - ablity to defer conference join for authentication if required
 *   - more bulletproofing on conference join
 *   - new video copy/fill functions
 *
 * Revision 1.3  2006/04/04 08:18:11  craigs
 * Latest version with titling and fixed PINs
 *
 * Revision 1.2  2006/04/03 12:52:54  craigs
 * Add GUI switch for custom mixer
 * Add fix for QCIF pre-media frame
 * Add start of code for room switch
 *
 * Revision 1.1  2006/03/31 07:36:12  craigs
 * New version with PINs and premedia blanking
 *
 * Revision 1.1.2.1  2006/03/28 05:13:38  csoutheren
 * Normalised file headers
 * Fixed problem with QCIF video
 * Seperated H.323 and MCU process functions into seperate files
 *
 */

#ifndef _OpenMCU_H323_H
#define _OpenMCU_H323_H

#ifdef _WIN32
#pragma warning(disable:4786)
#endif

#include <ptlib.h>
#include <ptlib/videoio.h>
#include <opalwavfile.h>

#include "config.h"
class OpenMCUH323EndPoint : public H323EndPoint
{
  PCLASSINFO(OpenMCUH323EndPoint, H323EndPoint);

  public:
    OpenMCUH323EndPoint(ConferenceManager & conferenceManager, H323Capability::CapabilityFrameSize _size);
    
    PString IncomingConferenceRequest(H323Connection & connection, 
                                      const H323SignalPDU & setupPDU);

    PBoolean OutgoingConferenceRequest(const PString & room);

    // overrides from H323EndPoint
    virtual H323Connection * CreateConnection(unsigned callReference,void * userData,H323Transport * transport,H323SignalPDU * setupPDU);
    virtual void TranslateTCPAddress(PIPSocket::Address &localAddr, const PIPSocket::Address &remoteAddr);

    PBoolean behind_masq;
    PIPSocket::Address *masqAddressPtr;

    // new functions
    void Initialise(PConfig & cfg, PConfigPage * rsrc);

    PString GetRoomStatus(const PString & block);
    PString GetMonitorText();

    ConferenceManager & GetConferenceManager()
    { return conferenceManager; }

#if OPENMCU_VIDEO
    unsigned GetVideoTxQuality() const
    { return videoTxQuality; }

    unsigned GetVideoFrameRate() const
    { return videoRate; }
#endif

  protected:
#if OPENMCU_VIDEO
    PBoolean enableVideo;
    unsigned videoRate;
    unsigned videoTxQuality;
#endif
	
	PString gkAlias;
	PStringList PrefixList;
	PStringList AliasList;


  protected:
    ConferenceManager & conferenceManager;
	PBoolean Initialised;
	H323Capability::CapabilityFrameSize MaxFrameSize;
};

////////////////////////////////////////////////////

class OpenMCUH323Connection;

class OutgoingAudio : public PChannel
{
  PCLASSINFO(OutgoingAudio, PChannel);

  public:
    OutgoingAudio(H323EndPoint & ep, OpenMCUH323Connection & conn);
    
    PBoolean Read(void * buffer, PINDEX amount);
    PBoolean Close();

  protected:
    void CreateSilence(void * buffer, PINDEX amount);

    H323EndPoint & ep;
    OpenMCUH323Connection & conn;

    PAdaptiveDelay delay;
    PMutex audioChanMutex;
};

////////////////////////////////////////////////////

class IncomingAudio : public PChannel
{
  PCLASSINFO(IncomingAudio, PChannel);

  public:
    IncomingAudio(H323EndPoint & ep, OpenMCUH323Connection & conn);

    PBoolean Write(const void * buffer, PINDEX amount);
    PBoolean Close();

  protected:
    H323EndPoint & ep;
    OpenMCUH323Connection & conn;
    PMutex audioChanMutex;
    PAdaptiveDelay delay;
};

////////////////////////////////////////////////////

class NotifyH245Thread : public PThread
{
  public:
    NotifyH245Thread(Conference & conference, PBoolean _join, ConferenceMember * _memberToIgnore);
    void Main();

  protected:
    PBoolean join;
    ConferenceMember * memberToIgnore;
    PStringArray tokens;
    int mcuNumber;
    int terminalIdToSend;
};

////////////////////////////////////////////////////

class H323Connection_ConferenceMember;
class PVideoInputDevice_OpenMCU;
class PVideoOutputDevice_OpenMCU;

class OpenMCUH323Connection : public H323Connection
{
  PCLASSINFO(OpenMCUH323Connection, H323Connection);

  public:
    OpenMCUH323Connection(OpenMCUH323EndPoint &, unsigned, void *);
    ~OpenMCUH323Connection();

    virtual void LeaveConference();

    // overrides from H323Connection
    virtual PBoolean OpenAudioChannel(PBoolean, unsigned,   H323AudioCodec & codec);
#if OPENMCU_VIDEO
    virtual PBoolean OpenVideoChannel(PBoolean isEncoding, H323VideoCodec & codec);
#endif
		virtual void CleanUpOnCallEnd();
    virtual void OnEstablished();
    virtual AnswerCallResponse OnAnswerCall(const PString &, const H323SignalPDU &, H323SignalPDU &);
    virtual void OnUserInputString(const PString & value);
    virtual PBoolean OnReceivedSignalSetup(const H323SignalPDU & setupPDU);
    virtual PBoolean OnReceivedCallProceeding(const H323SignalPDU & proceedingPDU);

    virtual PBoolean OnIncomingAudio(const void * buffer, PINDEX amount);
    virtual PBoolean OnOutgoingAudio(void * buffer, PINDEX amount);
    virtual PString GetAudioTransmitCodecName() const { return audioTransmitCodecName; }
    virtual PString GetAudioReceiveCodecName() const  { return audioReceiveCodecName; }
    virtual PString GetRemoteName() const             { return remoteName; }

#if OPENMCU_VIDEO
    virtual PBoolean OnIncomingVideo(const void * buffer, int width, int height, PINDEX amount);
    virtual PBoolean OnOutgoingVideo(void * buffer, int width, int height, PINDEX & amount);
    virtual PString GetVideoTransmitCodecName() const { return videoTransmitCodecName; }
    virtual PString GetVideoReceiveCodecName() const  { return videoReceiveCodecName; }
    virtual PBoolean GetPreMediaFrame(void * buffer, int width, int height, PINDEX & amount);
#endif

  protected:
    virtual void LogCall(const PBoolean accepted = TRUE);
    virtual void JoinConference(const PString & room);

#if OPENMCU_VIDEO
    virtual PBoolean InitGrabber(PVideoInputDevice  * grabber, int frameWidth, int frameHeight);
#endif

    PMutex connMutex;
    OpenMCUH323EndPoint & ep;

    // Name of the room to join when the welcome procedure ends.
    //  This is usually initialized when the call is answered,
    //  before the welcome procedure begins.
    PString requestedRoom;

    // Room the connection is joined to. It is NULL before the
    //  welcome procedure ends, or after the member is disconnected
    //  from the conference.
    Conference * conference;

    // Object used to treat the connection as a conference member.
    //  It is NULL before the connection is joined to the conference,
    //  or after the member is disconnected from the conference.
    H323Connection_ConferenceMember * conferenceMember;

    // Valid states for the welcome procedure. Note that new states may
    //  be added because the procedure can be customized by subclassing.
    enum WelcomeStates {
      // Reserved, never switch to this state
      NotStartedYet,

      // First state, by default jumps to PlayingConnecting
      PlayingWelcome,

      // Plays connecting wave file, then jumps to CompleteConnection
      PlayingConnecting,

      // Jumping here causes the connection to be joined to requestedRoom
      CompleteConnection,

      // The system jumps here if the join to the conference fails
      JoinFailed,

      // The system jumps here when the member is disconnected
      ConferenceEnded,

      // Use this as the starting value for custom sates in subclasses
      CustomWelcome_FirstUsableValue
    };

    // This is the current state of the welcome procedure. It is not
    //  declared using type WelcomeStates to allow the use of custom
    //  enumerated values.
    int welcomeState;

    // Use this function to change the current state. This call fires
    //  the OnWelcomeStateChanged callback immediately.
    void ChangeWelcomeState(int newState);

    // Call this function during the OnWelcomeStateChanged callback
    //  to play a new wave file during the welcome procedure. If
    //  useTheFile is FALSE, fileToPlay is empty or the file does
    //  not exist, the currently playing wave file is stopped and
    //  the OnWelcomeWaveEnded callback will be called anyway (as
    //  if an empty wave file has been played).
    void PlayWelcomeFile(PBoolean useTheFile, PFilePath & fileToPlay);

    // Called whenever the current state changes. Use welcomeState
    //  to know the current (new) state.
    virtual void OnWelcomeStateChanged();

    // Called by OnOutgoingAudio to process the current welcome
    //  state, even if the member is already joined to a conference.
    virtual void OnWelcomeProcessing();

    // Called when the currently playing wave file has ended. This
    //  is not called if the state changed before the wave file
    //  stopped. Use welcomeState to know the current state.
    virtual void OnWelcomeWaveEnded();

    PString remoteName;

    // Wave file played during the welcome procedure.
    OpalWAVFile playFile;

    // optional record file used during the welcome procedure
    OpalWAVFile recordFile;
    unsigned int recordSilenceCount;
    unsigned int recordSilenceThreshold;
    unsigned int recordDuration;
    unsigned int recordLimit;

    void StartRecording(
      const PFilePath & filename, 
      unsigned int recordLimit = 5, 
      unsigned int recordSilenceThreshold = 1
    );
    virtual void OnFinishRecording();

    //PAdaptiveDelay playDelay;

    // True if the state has not changed since the wave file started.
    PBoolean wavePlayingInSameState;

    PString audioTransmitCodecName;
    PString audioReceiveCodecName;

    PBoolean isMCU;

#if OPENMCU_VIDEO
    PVideoInputDevice_OpenMCU * videoGrabber;
    PVideoOutputDevice_OpenMCU * videoDisplay;
    PString videoTransmitCodecName;
    PString videoReceiveCodecName;
#endif
};

////////////////////////////////////////////////////

class H323Connection_ConferenceConnection : public ConferenceConnection
{
  PCLASSINFO(H323Connection_ConferenceConnection, ConferenceConnection);
  public:
    H323Connection_ConferenceConnection(void * id)
      : ConferenceConnection(id)
    { conn = (OpenMCUH323Connection *)id; }

    virtual PString GetName() const
    { return conn->GetCallToken(); }

  protected:
    OpenMCUH323Connection * conn;
};

class H323Connection_ConferenceMember : public ConferenceMember
{
  PCLASSINFO(H323Connection_ConferenceMember, ConferenceMember);
  public:
    H323Connection_ConferenceMember(Conference * _conference, OpenMCUH323EndPoint & _ep, const PString & _h323Token, ConferenceMemberId _id, PBoolean isMCU);
    ~H323Connection_ConferenceMember();

    virtual ConferenceConnection * CreateConnection()
    { 
      H323Connection * conn = ep.FindConnectionWithLock(h323Token);
      if (conn == NULL)
        return NULL;

      H323Connection_ConferenceConnection * confConn = new H323Connection_ConferenceConnection(conn); 
      conn->Unlock();
      return confConn;
    }

    virtual void Close();

    virtual PString GetH323Token() const
    { return h323Token; }

    virtual PString GetTitle() const;

    PString GetMonitorInfo(const PString & hdr);

    void OnReceivedUserInputIndication(const PString & str)
    { 
      H323Connection * conn = ep.FindConnectionWithLock(h323Token);
      if (conn != NULL)
        conn->SendUserInput(str); 
      conn->Unlock();
    }

  protected:
    OpenMCUH323EndPoint & ep;
    PString h323Token;
};

////////////////////////////////////////////////////

#if OPENMCU_VIDEO

#include <ptlib/vconvert.h>

class PVideoInputDevice_OpenMCU : public PVideoInputDevice
{
 PCLASSINFO(PVideoInputDevice_OpenMCU, PVideoInputDevice);
 public:
  /** Create a new MCU video input device.
   */
    PVideoInputDevice_OpenMCU(OpenMCUH323Connection & conn);

    /**Open the device given the device name.
      */
    PBoolean Open(
      const PString & title,        /// Device name to open
      PBoolean startImmediate = TRUE    /// Immediately start device
    );

    /**Determine of the device is currently open.
      */
    PBoolean IsOpen() ;

    /**Close the device.
      */
    PBoolean Close();

    /**Start the video device I/O.
      */
    PBoolean Start();

    /**Stop the video device I/O capture.
      */
    PBoolean Stop();

    /**Determine if the video device I/O capture is in progress.
      */
    PBoolean IsCapturing();

    /**Get a list of all of the drivers available.
      */
    static PStringArray GetInputDeviceNames();

#if PTLIB_MAJOR <= 2 && PTLIB_MINOR < 3
    virtual PStringList GetDeviceNames() const
#else
    virtual PStringArray GetDeviceNames() const
#endif
      { return GetInputDeviceNames(); }

    /**Get the maximum frame size in bytes.

       Note a particular device may be able to provide variable length
       frames (eg motion JPEG) so will be the maximum size of all frames.
      */
    virtual PINDEX GetMaxFrameBytes();

    /**Grab a frame. 

       There will be a delay in returning, as specified by frame rate.
      */
    virtual PBoolean GetFrameData(
      BYTE * buffer,                 /// Buffer to receive frame
      PINDEX * bytesReturned = NULL  /// Optional bytes returned.
    );

    /**Grab a frame.

       Do not delay according to the current frame rate.
      */
    virtual PBoolean GetFrameDataNoDelay(
      BYTE * buffer,                 /// Buffer to receive frame
      PINDEX * bytesReturned = NULL  /// OPtional bytes returned.
    );


    /**Generate a static image, containing a constant field of grey.
     */
    void GrabBlankImage(BYTE *resFrame);

    /**Set the video format to be used.

       Default behaviour sets the value of the videoFormat variable and then
       returns the IsOpen() status.
    */
    virtual PBoolean SetVideoFormat(
      VideoFormat videoFormat   /// New video format
    );

    /**Get the number of video channels available on the device.

       Default behaviour returns 1.
    */
    virtual int GetNumChannels() ;

    /**Set the video channel to be used on the device. 
    */
    virtual PBoolean SetChannel(
         int channelNumber  /// New channel number for device.
    );
    
    /**Set the colour format to be used.

       Default behaviour sets the value of the colourFormat variable and then
       returns the IsOpen() status.
    */
    virtual PBoolean SetColourFormat(
      const PString & colourFormat   // New colour format for device.
    );
    
    /**Set the video frame rate to be used on the device.

       Default behaviour sets the value of the frameRate variable and then
       return the IsOpen() status.
    */
    virtual PBoolean SetFrameRate(
      unsigned rate  /// Frames per second
    );
         
    /**Get the minimum & maximum size of a frame on the device.

       Default behaviour returns the value 1 to UINT_MAX for both and returns
       FALSE.
    */
    virtual PBoolean GetFrameSizeLimits(
      unsigned & minWidth,   /// Variable to receive minimum width
      unsigned & minHeight,  /// Variable to receive minimum height
      unsigned & maxWidth,   /// Variable to receive maximum width
      unsigned & maxHeight   /// Variable to receive maximum height
    ) ;

    /**Set the frame size to be used.

       Default behaviour sets the frameWidth and frameHeight variables and
       returns the IsOpen() status.
    */
    virtual PBoolean SetFrameSize(
      unsigned width,   /// New width of frame
      unsigned height   /// New height of frame
    );
         
    void ClearMapping() { return ; }

    /**Try all known video formats & see which ones are accepted by the video driver
     */
    virtual PBoolean TestAllFormats()
      { return TRUE; }

  protected:
    OpenMCUH323Connection & mcuConnection;
    unsigned grabCount;
    PINDEX   videoFrameSize;
    PINDEX   scanLineWidth;
    PAdaptiveDelay grabDelay;
};


class PVideoOutputDevice_OpenMCU : public PVideoOutputDevice
{
  PCLASSINFO(PVideoOutputDevice_OpenMCU, PVideoOutputDevice);

  public:
    /** Create a new video output device.
     */
    PVideoOutputDevice_OpenMCU(OpenMCUH323Connection & _mcuConnection);

    /**Get a list of all of the drivers available.
      */
    static PStringArray GetOutputDeviceNames();

#if PTLIB_MAJOR <= 2 && PTLIB_MINOR < 3
    virtual PStringList GetDeviceNames() const
#else
    virtual PStringArray GetDeviceNames() const
#endif
      { return GetOutputDeviceNames(); }

    /**Open the device given the device name.
      */
    virtual PBoolean Open(
      const PString & deviceName,   /// Device name to open
      PBoolean startImmediate = TRUE    /// Immediately start device
    );

    /**Start the video device I/O.
      */
    PBoolean Start();

    /**Stop the video device I/O capture.
      */
    PBoolean Stop();

    /**Close the device.
      */
    virtual PBoolean Close();

    /**Determine if the device is currently open.
      */
    virtual PBoolean IsOpen();

    /**Get the maximum frame size in bytes.

       Note a particular device may be able to provide variable length
       frames (eg motion JPEG) so will be the maximum size of all frames.
      */
    virtual PINDEX GetMaxFrameBytes();

    /**Set a section of the output frame buffer.
      */
    virtual PBoolean SetFrameData(
      unsigned x,
      unsigned y,
      unsigned width,
      unsigned height,
      const BYTE * data,
      PBoolean endFrame = TRUE
    );

    /**Indicate frame may be displayed.
      */
    virtual PBoolean EndFrame();

  protected:  
    OpenMCUH323Connection & mcuConnection;
};

#endif // OPENMCU_VIDEO

#endif //_OpenMCU_H323_H

