/*
 * video.cxx
 *
 * Video conferencing functions for a simple MCU
 *
 * Copyright (c) 2000 Equivalence Pty. Ltd.
 * Copyright (c) 2004 Post Increment
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of ths code were written by by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Stonevoice, slc. http://www.stonevoice.com
 *
 * Portions of this code were written by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Citron Networks (http://www.citron.com.tw)
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 Craig Southeren (craig@postincrement.com)
 *
 * $Log: video.cxx,v $
 * Revision 2.6  2006/07/21 08:01:40  csoutheren
 * Fixed conference member detect
 * Re-factored video mixer code slightly
 *
 * Revision 1.10  2006/07/21 06:07:08  craigs
 * Backports from open source version
 *
 * Revision 2.5  2006/07/21 05:08:03  csoutheren
 * Stability fixes and more inline documentation
 * Thanks to Paolo Amadini of Stonevoice
 *
 * Revision 2.4  2006/06/21 06:11:36  csoutheren
 * Fixes for latest pwlib
 *
 * Revision 2.3  2006/06/09 04:39:59  csoutheren
 * Migrated VideoBranch to main trunk
 *
 * Revision 2.2.2.16  2006/06/06 08:24:16  csoutheren
 * Added support for echo test room
 * Fix problem with using high frame rates
 *
 * Revision 2.2.2.15  2006/04/18 03:05:09  csoutheren
 * Fix video mix problem with > 3 members
 * Add test room
 *
 * Revision 2.2.2.14  2006/04/12 04:54:18  csoutheren
 * Fix more problems with audio-only members
 *
 * Revision 2.2.2.13  2006/04/12 01:46:13  csoutheren
 * Added defaut image when audio-only member enters video conference
 * Added image to Web gui
 *
 * Revision 2.2.2.12  2006/04/06 08:20:29  csoutheren
 * Retyped conference member identifier to epxlicit type
 * Added support for H.245 terminal added and terminal left
 *
 * Revision 2.2.2.11  2006/04/06 01:11:16  csoutheren
 * Latest sources include
 *   - premedia blanking and optional image display
 *   - ablity to defer conference join for authentication if required
 *   - more bulletproofing on conference join
 *   - new video copy/fill functions
 *
 * Revision 2.2.2.10  2006/04/06 00:50:30  csoutheren
 * Latest changes (more to come)
 *
 * Revision 2.2.2.8  2006/03/28 05:13:38  csoutheren
 * Normalised file headers
 * Fixed problem with QCIF video
 * Seperated H.323 and MCU process functions into seperate files
 *
 * Revision 2.2.2.7  2006/03/27 12:53:14  csoutheren
 * More fixes for QCIF video (not yet verified)
 *
 * Revision 2.2.2.6  2006/03/27 09:14:15  csoutheren
 * Fixed Unix compile warnings and errors
 * Added VideoTxQuality value
 *
 * Revision 2.2.2.5  2006/03/24 22:49:24  csoutheren
 * Video now working
 *
 * Revision 2.2.2.4  2006/03/21 14:02:49  csoutheren
 * More video mixiing.
 * Still some crashes in video mixer code when members exit conference
 *
 * Revision 2.2.2.3  2006/03/17 07:00:24  csoutheren
 * More video implementation
 * Video mixing now working except copying subimage UV components causes overrun
 * Y is OK, which is why the images are grey. Also need to work out flipping requirements
 *
 * Revision 2.2.2.2  2006/03/14 08:02:50  csoutheren
 * More implemenrtation of video
 * Video mixing infrastructure implemented but not fully working
 *
 * Revision 2.2.2.1  2006/03/06 05:45:30  csoutheren
 * Start of implementation for video. Still in progress
 *
 * Revision 2.2  2004/03/23 11:40:06  csoutheren
 * Fixed problem where deleting map element in-place causes crash at end of call
 * Fixed problem where referencing map by iterator rather than ID
 * Fixed code formatting problems
 *
 * Revision 2.1  2004/03/11 20:49:44  csoutheren
 * Removed warnings
 *
 * Revision 2.0  2004/03/08 02:06:24  csoutheren
 * Totally rewritten to use new connection locking mecahnism
 * Added ability to monitor conferences
 * Added initial support for H.323 MCU messages
 * Thanks to Citron Networks for supporting this work
 *
 */


#include <ptlib.h>

#ifdef _WIN32
#pragma warning(disable:4786)
#endif

#include "config.h"

#if OPENMCU_VIDEO

#include "mcu.h"
#include "h323.h"
#include <ptlib/vconvert.h>

#define Q3CIF_WIDTH   (CIF_WIDTH / 3)
#define Q3CIF_HEIGHT  (CIF_HEIGHT / 3)
#define Q3CIF_SIZE    (Q3CIF_WIDTH*Q3CIF_HEIGHT*3/2)

#define Q4CIF_WIDTH   (CIF_WIDTH / 4)
#define Q4CIF_HEIGHT  (CIF_HEIGHT / 4)
#define Q4CIF_SIZE    (Q4CIF_WIDTH*Q4CIF_HEIGHT*3/2)

#define MAX_SUBFRAMES   16

///////////////////////////////////////////////////////////////////////////////////////
//
//  declare a video capture (input) device for use with OpenMCU
//

PVideoInputDevice_OpenMCU::PVideoInputDevice_OpenMCU(OpenMCUH323Connection & _mcuConnection)
  : mcuConnection(_mcuConnection)
{
  SetColourFormat("YUV420P");
  channelNumber = 0; 
  grabCount = 0;
  SetFrameRate(25);
}


BOOL PVideoInputDevice_OpenMCU::Open(const PString & devName, BOOL /*startImmediate*/)
{
  //file.SetWidth(frameWidth);
  //file.SetHeight(frameHeight);

  deviceName = devName;

  return TRUE;    
}


BOOL PVideoInputDevice_OpenMCU::IsOpen() 
{
  return TRUE;
}


BOOL PVideoInputDevice_OpenMCU::Close()
{
  return TRUE;
}


BOOL PVideoInputDevice_OpenMCU::Start()
{
  return TRUE;
}


BOOL PVideoInputDevice_OpenMCU::Stop()
{
  return TRUE;
}


BOOL PVideoInputDevice_OpenMCU::IsCapturing()
{
  return IsOpen();
}


PStringList PVideoInputDevice_OpenMCU::GetInputDeviceNames()
{
  PStringList list;
  list.AppendString("openmcu");
  return list;
}


BOOL PVideoInputDevice_OpenMCU::SetVideoFormat(VideoFormat newFormat)
{
  return PVideoDevice::SetVideoFormat(newFormat);
}


int PVideoInputDevice_OpenMCU::GetNumChannels() 
{
  return 0;
}


BOOL PVideoInputDevice_OpenMCU::SetChannel(int newChannel)
{
  return PVideoDevice::SetChannel(newChannel);
}

BOOL PVideoInputDevice_OpenMCU::SetColourFormat(const PString & newFormat)
{
  if (!(newFormat *= "YUV420P"))
    return FALSE;

  if (!PVideoDevice::SetColourFormat(newFormat))
    return FALSE;

  return SetFrameSize(frameWidth, frameHeight);
}


BOOL PVideoInputDevice_OpenMCU::SetFrameRate(unsigned rate)
{
  if (rate < 1)
    rate = 1;
  else if (rate > 50)
    rate = 50;

  return PVideoDevice::SetFrameRate(rate);
}


BOOL PVideoInputDevice_OpenMCU::GetFrameSizeLimits(unsigned & minWidth,
                                           unsigned & minHeight,
                                           unsigned & maxWidth,
                                           unsigned & maxHeight) 
{
  maxWidth  = CIF_WIDTH;
  maxHeight = CIF_HEIGHT;
  minWidth  = QCIF_WIDTH;
  minHeight = QCIF_HEIGHT;

  return TRUE;
}


BOOL PVideoInputDevice_OpenMCU::SetFrameSize(unsigned width, unsigned height)
{
  if (!PVideoDevice::SetFrameSize(width, height))
    return FALSE;

  videoFrameSize = CalculateFrameBytes(frameWidth, frameHeight, colourFormat);
  scanLineWidth = videoFrameSize/frameHeight;
  return videoFrameSize > 0;
}


PINDEX PVideoInputDevice_OpenMCU::GetMaxFrameBytes()
{
  return GetMaxFrameBytesConverted(videoFrameSize);
}


BOOL PVideoInputDevice_OpenMCU::GetFrameData(BYTE * buffer, PINDEX * bytesReturned)
{    
  grabDelay.Delay(msBetweenFrames);

  return GetFrameDataNoDelay(buffer, bytesReturned);
}

 
BOOL PVideoInputDevice_OpenMCU::GetFrameDataNoDelay(BYTE *destFrame, PINDEX * bytesReturned)
{
  grabCount++;

  if (!mcuConnection.OnOutgoingVideo(destFrame, frameWidth, frameHeight, *bytesReturned))
    return FALSE;

  if (converter != NULL) {
    if (!converter->Convert(destFrame, destFrame, bytesReturned))
      return FALSE;
  }

  if (bytesReturned != NULL)
    *bytesReturned = videoFrameSize;

  return TRUE;
}

///////////////////////////////////////////////////////////////////////////////////////
//
//  declare a video display (output) device for use with OpenMCU
//

PVideoOutputDevice_OpenMCU::PVideoOutputDevice_OpenMCU(OpenMCUH323Connection & _mcuConnection)
  : mcuConnection(_mcuConnection)
{
}


BOOL PVideoOutputDevice_OpenMCU::Open(const PString & _deviceName, BOOL /*startImmediate*/)
{
  deviceName = _deviceName;
  return TRUE;
}

BOOL PVideoOutputDevice_OpenMCU::Close()
{
  return TRUE;
}

BOOL PVideoOutputDevice_OpenMCU::Start()
{
  return TRUE;
}

BOOL PVideoOutputDevice_OpenMCU::Stop()
{
  return TRUE;
}

BOOL PVideoOutputDevice_OpenMCU::IsOpen()
{
  return TRUE;
}


PStringList PVideoOutputDevice_OpenMCU::GetOutputDeviceNames()
{
  PStringList list;
  return list;
}


PINDEX PVideoOutputDevice_OpenMCU::GetMaxFrameBytes()
{
  return GetMaxFrameBytesConverted(CalculateFrameBytes(frameWidth, frameHeight, colourFormat));
}


BOOL PVideoOutputDevice_OpenMCU::SetFrameData(unsigned x, unsigned y,
                                              unsigned width, unsigned height,
                                              const BYTE * data,
                                              BOOL /*endFrame*/)
{
  if (x != 0 || y != 0 || width != frameWidth || height != frameHeight) {
    PTRACE(1, "YUVFile output device only supports full frame writes");
    return FALSE;
  }

  return mcuConnection.OnIncomingVideo(data, width, height, width*height*3/2);
}


BOOL PVideoOutputDevice_OpenMCU::EndFrame()
{
  return TRUE;
}


///////////////////////////////////////////////////////////////////////////////////////

VideoFrameStoreList::~VideoFrameStoreList()
{
  while (videoFrameStoreList.begin() != videoFrameStoreList.end()) {
    FrameStore * vf = videoFrameStoreList.begin()->second;
    delete vf;
    videoFrameStoreList.erase(videoFrameStoreList.begin());
  }
}

VideoFrameStoreList::FrameStore & VideoFrameStoreList::AddFrameStore(int width, int height)
{ 
  VideoFrameStoreListMapType::iterator r = videoFrameStoreList.find(WidthHeightToKey(width, height));
  if (r != videoFrameStoreList.end())
    return *(r->second);
  FrameStore * vf = new FrameStore(width, height);
  videoFrameStoreList.insert(VideoFrameStoreListMapType::value_type(WidthHeightToKey(width, height), vf)); 
  return *vf;
}

VideoFrameStoreList::FrameStore & VideoFrameStoreList::GetFrameStore(int width, int height) 
{
  VideoFrameStoreListMapType::iterator r = videoFrameStoreList.find(WidthHeightToKey(width, height));
  if (r != videoFrameStoreList.end())
    return *(r->second);

  FrameStore * vf = new FrameStore(width, height);
  videoFrameStoreList.insert(VideoFrameStoreListMapType::value_type(WidthHeightToKey(width, height), vf)); 
  return *vf;
}

void VideoFrameStoreList::InvalidateExcept(int w, int h)
{
  VideoFrameStoreListMapType::iterator r;
  for (r = videoFrameStoreList.begin(); r != videoFrameStoreList.end(); ++r) {
    unsigned int key = r->first;
    int kw, kh; KeyToWidthHeight(key, kw, kh);
    r->second->valid = (w == kw) && (h == kh);
  }
}

VideoFrameStoreList::FrameStore & VideoFrameStoreList::GetNearestFrameStore(int width, int height, BOOL & found)
{
  // see if exact match, and valid
  VideoFrameStoreListMapType::iterator r = videoFrameStoreList.find(WidthHeightToKey(width, height));
  if ((r != videoFrameStoreList.end()) && r->second->valid) {
    found = TRUE;
    return *(r->second);
  }

  // return the first valid framestore
  for (r = videoFrameStoreList.begin(); r != videoFrameStoreList.end(); ++r) {
    if (r->second->valid) {
      found = TRUE;
      return *(r->second);
    }
  }

  // return not found
  found = FALSE;
  return *(videoFrameStoreList.end()->second);
}

///////////////////////////////////////////////////////////////////////////////////////

static inline int ABS(int v)
{  return (v >= 0) ? v : -v; }

MCUVideoMixer::VideoMixPosition::VideoMixPosition(ConferenceMemberId _id,  ConferenceMember & /*mbr*/, int _x, int _y, int _w, int _h)
  : id(_id), xpos(_x), ypos(_y), width(_w), height(_h)
{ 
  audioLevel = 0; 
}

void MCUVideoMixer::ConvertRGBToYUV(BYTE R, BYTE G, BYTE B, BYTE & Y, BYTE & U, BYTE & V)
{
  Y = (BYTE)PMIN(ABS(R *  2104 + G *  4130 + B *  802 + 4096 +  131072) / 8192, 235);
  U = (BYTE)PMIN(ABS(R * -1214 + G * -2384 + B * 3598 + 4096 + 1048576) / 8192, 240);
  V = (BYTE)PMIN(ABS(R *  3598 + G * -3013 + B * -585 + 4096 + 1048576) / 8192, 240);
}

void MCUVideoMixer::FillYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B, int w, int h)
{
  BYTE Y, U, V;
  ConvertRGBToYUV(R, G, B, Y, U, V);

  const int ysize = w*h;
  const int usize = w*h/4;
  const int vsize = w*h/4;

  memset((BYTE *)buffer + 0,             Y, ysize);
  memset((BYTE *)buffer + ysize,         U, usize);
  memset((BYTE *)buffer + ysize + usize, V, vsize);
}

void MCUVideoMixer::FillCIFYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B)
{
  FillYUVFrame(buffer, R, G, B, CIF_WIDTH, CIF_HEIGHT);
}

void MCUVideoMixer::FillQCIFYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B)
{
  FillYUVFrame(buffer, R, G, B, QCIF_WIDTH, QCIF_HEIGHT);
}

void MCUVideoMixer::FillCIFYUVRect(void * frame, BYTE R, BYTE G, BYTE B, int xPos, int yPos, int rectWidth, int rectHeight)
{
  FillYUVRect(frame, CIF_WIDTH, CIF_HEIGHT, R, G, B, xPos, yPos, rectWidth, rectHeight);
}

void MCUVideoMixer::FillYUVRect(void * frame, int frameWidth, int frameHeight, BYTE R, BYTE G, BYTE B, int xPos, int yPos, int rectWidth, int rectHeight)
{
  //This routine fills a region of the video image with data. It is used as the central
  //point because one only has to add other image formats here.

  int offset       = ( yPos * frameWidth ) + xPos;
  int colourOffset = ( (yPos * frameWidth) >> 2) + (xPos >> 1);

  BYTE Y, U, V;
  ConvertRGBToYUV(R, G, B, Y, U, V);

  BYTE * Yptr = (BYTE*)frame + offset;
  BYTE * UPtr = (BYTE*)frame + (frameWidth * frameHeight) + colourOffset;
  BYTE * VPtr = (BYTE*)frame + (frameWidth * frameHeight) + (frameWidth * frameHeight/4)  + colourOffset;

  int rr ;
  int halfRectWidth = rectWidth >> 1;
  int halfWidth     = frameWidth >> 1;
  
  for (rr = 0; rr < rectHeight;rr+=2) {
    memset(Yptr, Y, rectWidth);
    Yptr += frameWidth;
    memset(Yptr, Y, rectWidth);
    Yptr += frameWidth;

    memset(UPtr, U, halfRectWidth);
    memset(VPtr, V, halfRectWidth);

    UPtr += halfWidth;
    VPtr += halfWidth;
  }
}

void MCUVideoMixer::CopyRectIntoCIF(const void * _src, void * _dst, int xpos, int ypos, int width, int height)
{
  BYTE * src = (BYTE *)_src;
  BYTE * dst = (BYTE *)_dst + (ypos * CIF_WIDTH) + xpos;

  BYTE * dstEnd = dst + CIF_SIZE;
  int y;

  // copy Y
  for (y = 0; y < height; ++y) {
    PAssert(dst + width < dstEnd, "Y write overflow");
    memcpy(dst, src, width);
    src += width;
    dst += CIF_WIDTH;
  }

  // copy U
  dst = (BYTE *)_dst + (CIF_WIDTH * CIF_HEIGHT) + (ypos * CIF_WIDTH/4) + xpos / 2;
  for (y = 0; y < height/2; ++y) {
    PAssert(dst + width/2 <= dstEnd, "U write overflow");
    memcpy(dst, src, width/2);
    src += width/2;
    dst += CIF_WIDTH/2;
  }

  // copy V
  dst = (BYTE *)_dst + (CIF_WIDTH * CIF_HEIGHT) + (CIF_WIDTH * CIF_HEIGHT) / 4 + (ypos * CIF_WIDTH/4) + xpos / 2;
  for (y = 0; y < height/2; ++y) {
    PAssert(dst + width/2 <= dstEnd, "V write overflow");
    memcpy(dst, src, width/2);
    src += width/2;
    dst += CIF_WIDTH/2;
  }
}

void MCUVideoMixer::CopyRectIntoQCIF(const void * _src, void * _dst, int xpos, int ypos, int width, int height)
{
  BYTE * src = (BYTE *)_src;
  BYTE * dst = (BYTE *)_dst + (ypos * QCIF_WIDTH) + xpos;

  BYTE * dstEnd = dst + QCIF_SIZE;
  int y;

  // copy Y
  for (y = 0; y < height; ++y) {
    PAssert(dst + width < dstEnd, "Y write overflow");
    memcpy(dst, src, width);
    src += width;
    dst += QCIF_WIDTH;
  }

  // copy U
  dst = (BYTE *)_dst + (QCIF_WIDTH * QCIF_HEIGHT) + (ypos * CIF_WIDTH/4) + xpos / 2;
  for (y = 0; y < height/2; ++y) {
    PAssert(dst + width/2 <= dstEnd, "U write overflow");
    memcpy(dst, src, width/2);
    src += width/2;
    dst += QCIF_WIDTH/2;
  }

  // copy V
  dst = (BYTE *)_dst + (QCIF_WIDTH * QCIF_HEIGHT) + (QCIF_WIDTH * QCIF_HEIGHT) / 4 + (ypos * QCIF_WIDTH/4) + xpos / 2;
  for (y = 0; y < height/2; ++y) {
    PAssert(dst + width/2 <= dstEnd, "V write overflow");
    memcpy(dst, src, width/2);
    src += width/2;
    dst += QCIF_WIDTH/2;
  }
}

void MCUVideoMixer::ConvertQCIFToCIF(const void * _src, void * _dst)
{
  BYTE * src = (BYTE *)_src;
  BYTE * dst = (BYTE *)_dst;

  //BYTE * dstEnd = dst + CIF_SIZE;
  int y, x;
  BYTE * srcRow;

  // copy Y
  for (y = 0; y < QCIF_HEIGHT; y++) {
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    src += QCIF_WIDTH;
  }

  // copy U
  for (y = 0; y < QCIF_HEIGHT/2; y++) {
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH/2; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH/2; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    src += QCIF_WIDTH/2;
  }

  // copy V
  for (y = 0; y < QCIF_HEIGHT/2; y++) {
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH/2; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    srcRow = src;
    for (x = 0; x < QCIF_WIDTH/2; x++) {
      dst[0] = dst[1] = *srcRow++;
      dst += 2;
    }
    src += QCIF_WIDTH/2;
  }
}



///////////////////////////////////////////////////////////////////////////////////////

MCUSimpleVideoMixer::MCUSimpleVideoMixer(BOOL _forceScreenSplit)
  : forceScreenSplit(_forceScreenSplit)
{
  frameStores.AddFrameStore(CIF_WIDTH, CIF_HEIGHT);
  imageStore.SetSize(CIF_SIZE);

  converter = PColourConverter::Create("YUV420P", "YUV420P", CIF_WIDTH, CIF_HEIGHT);
}

BOOL MCUSimpleVideoMixer::ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount)
{
  PWaitAndSignal m(mutex);

  // special case of one member means fill with black
  if (rows == 0) {
    VideoFrameStoreList::FrameStore & fs = frameStores.GetFrameStore(width, height);
    if (!fs.valid) {
      if (!OpenMCU::Current().GetEmptyMediaFrame(fs.data.GetPointer(), width, height, amount))
        FillYUVFrame(fs.data.GetPointer(), 0, 0, 0, width, height);
      fs.valid = TRUE;
    }
    memcpy(buffer, fs.data.GetPointer(), amount);
  }

  // special case of two members means we do nothing, and tell caller to look for full screen version of other video
  if (rows == 1) 
    return FALSE;

  return ReadMixedFrame(buffer, width, height, amount);
}

BOOL MCUSimpleVideoMixer::ReadSrcFrame(VideoFrameStoreList & srcFrameStores, void * buffer, int width, int height, PINDEX & amount)
{
  PWaitAndSignal m(mutex);

  VideoFrameStoreList::FrameStore & cifFs = srcFrameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT);

  // get the mixed frame
  if ((width == CIF_WIDTH) && (height == CIF_HEIGHT)) {
    if (!cifFs.valid) {
      if (!OpenMCU::Current().GetEmptyMediaFrame(cifFs.data.GetPointer(), width, height, amount))
        MCUVideoMixer::FillYUVFrame(cifFs.data.GetPointer(), 0, 0, 0, CIF_WIDTH, CIF_HEIGHT);
      cifFs.valid = TRUE;
    }
    memcpy(buffer, cifFs.data.GetPointer(), amount);
  }

  else if (width == QCIF_WIDTH && height == QCIF_HEIGHT) {
    VideoFrameStoreList::FrameStore & qcifFs = srcFrameStores.GetFrameStore(width, height);
    if (!qcifFs.valid) {
      if (!cifFs.valid) {
        if (!OpenMCU::Current().GetEmptyMediaFrame(qcifFs.data.GetPointer(), width, height, amount))
          MCUVideoMixer::FillYUVFrame(qcifFs.data.GetPointer(), 0, 0, 0, width, height);
      } else {
        converter->SetSrcFrameSize(CIF_WIDTH, CIF_HEIGHT);
        converter->SetDstFrameSize(width, height, TRUE);
        PINDEX amount = CIF_SIZE;
        PINDEX bytesReturned = QCIF_SIZE;
        converter->Convert(cifFs.data.GetPointer(), qcifFs.data.GetPointer(), amount, &bytesReturned);
      }
      qcifFs.valid = TRUE;
    }
    memcpy(buffer, qcifFs.data.GetPointer(), amount);
  }

  return TRUE;
}



BOOL MCUSimpleVideoMixer::ReadMixedFrame(void * buffer, int width, int height, PINDEX & amount)
{
  return ReadSrcFrame(frameStores, buffer, width, height, amount);
}


BOOL MCUSimpleVideoMixer::WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount)
{
  PWaitAndSignal m(mutex);

  // special case of one member means we do nothing, and don't bother looking for other user to copy from
  if (rows == 0) 
    return TRUE;

  // special case of two members means we do nothing, and tell caller to look for another frame to write to
  if (rows == 1) 
    return FALSE;

  // write data into sub frame of mixed image
  VideoMixPositionMap::const_iterator r = videoPositions.find(id);
  if (r == videoPositions.end())
    return TRUE;

  r->second->WriteSubFrame(*this, buffer, width, height, amount);

  return TRUE;
}

void MCUSimpleVideoMixer::SetAudioLevel(ConferenceMemberId id, unsigned audioLevel)
{
  PWaitAndSignal m(mutex);

  // write data into sub frame of mixed image
  VideoMixPositionMap::const_iterator r = videoPositions.find(id);
  if (r != videoPositions.end()) {
    r->second->audioLevel = audioLevel;
    SetSubFrameLevel(*r->second, audioLevel);
  }
}


void MCUSimpleVideoMixer::SetSubFrameLevel(VideoMixPosition & /*vmp*/, unsigned /*audioLevel*/)
{
}

void MCUSimpleVideoMixer::CalcVideoSplitSize(unsigned int imageCount, int & subImageWidth, int & subImageHeight, int & cols, int & rows)
{
  if (!forceScreenSplit && imageCount < 2) {
    subImageWidth  = CIF_WIDTH;
    subImageHeight = CIF_HEIGHT;
    cols           = 0;
    rows           = 0;
  }
  else
  if (!forceScreenSplit && imageCount == 2) {
    subImageWidth  = CIF_WIDTH;
    subImageHeight = CIF_HEIGHT;
    cols           = 1;
    rows           = 1;

  }
  else
  if (imageCount <= 4) {
    subImageWidth  = QCIF_WIDTH;
    subImageHeight = QCIF_HEIGHT;
    cols           = 2;
    rows           = 2;
  }
  else
  if (imageCount <= 9) {
    subImageWidth  = Q3CIF_WIDTH;
    subImageHeight = Q3CIF_HEIGHT;
    cols           = 3;
    rows           = 3;
  }
  else if (imageCount <= 16) {
    subImageWidth  = Q4CIF_WIDTH;
    subImageHeight = Q4CIF_HEIGHT;
    cols           = 4;
    rows           = 4;
  }
}

void MCUSimpleVideoMixer::ReallocatePositions()
{
  FillCIFYUVFrame(frameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT).data.GetPointer(), 0, 0, 0);
  frameStores.InvalidateExcept(CIF_WIDTH, CIF_HEIGHT);

  VideoMixPositionMap::iterator r;

  if (cols == 0 || cols == 1) {
    r = videoPositions.begin();
    if (r != videoPositions.end()) {
      VideoMixPosition & vmp = *(r->second);
      vmp.xpos   = 0;
      vmp.ypos   = 0;
      vmp.width  = subImageWidth;
      vmp.height = subImageHeight;
    }
    return;
  }

  // reallocate display positions, but always use the top left corner last)
  unsigned int start = 1;
  unsigned int i = 0;
  for (r = videoPositions.begin(); r != videoPositions.end(); ++r) {
    unsigned int x = (i + start) % cols;
    unsigned int y = ((i + start) / cols) % rows;
    VideoMixPosition & vmp = *(r->second);
    vmp.xpos   = x * subImageWidth;
    vmp.ypos   = y * subImageHeight;
    vmp.width  = subImageWidth;
    vmp.height = subImageHeight;
    ++i;
  }
}

BOOL MCUSimpleVideoMixer::AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr)
{
  PWaitAndSignal m(mutex);

  // make sure this source is not already in the list
  VideoMixPositionMap::const_iterator r = videoPositions.find(id);
  if (r != videoPositions.end()) 
    return TRUE;

  if (videoPositions.size() == MAX_SUBFRAMES)
    return FALSE;

  // calculate the kind of video split we need to include the new source
  int newSubImageWidth, newSubImageHeight, newCols, newRows; 
  CalcVideoSplitSize((unsigned int)(videoPositions.size()+1), newSubImageWidth, newSubImageHeight, newCols, newRows);

  VideoMixPosition * newPosition = NULL;

  // if the subimage size has changed, rearrange everything
  if ((newSubImageWidth != subImageWidth) || (newSubImageHeight != subImageHeight) || (cols != newCols) || (rows != newRows)) {
    rows           = newRows;
    cols           = newCols;
    subImageWidth  = newSubImageWidth;;
    subImageHeight = newSubImageHeight;

    newPosition = CreateVideoMixPosition(id, mbr, 0, 0, subImageWidth, subImageHeight);
    videoPositions.insert(VideoMixPositionMap::value_type(id, newPosition));

    ReallocatePositions();
  }

  // otherwise find an empty position
  else if (rows == 0) {
    newPosition = CreateVideoMixPosition(id, mbr, 0, 0,subImageWidth, subImageHeight);
    videoPositions.insert(VideoMixPositionMap::value_type(id, newPosition));
  }
  else {
    BOOL found = FALSE;
    int i = 0;
    int start = 1;
    VideoMixPositionMap::iterator r = videoPositions.end();
    while (!found && i < (rows*cols)) {
      int x = (i + start) % cols;
      int y = ((i + start) / cols) % rows;
      r = videoPositions.begin(); 
      while (r != videoPositions.end()) {
        VideoMixPosition & vmp = *(r->second);
        if (vmp.xpos == (x * subImageWidth) && vmp.ypos == (y * subImageHeight))
          break;
        ++r;
      }
      if (r == videoPositions.end())
        break;
      ++i;
    }

    PAssert(r == videoPositions.end(), "could not find free video position");
    unsigned int x = (i + start) % cols;
    unsigned int y = ((i + start) / cols) % rows;

    newPosition = CreateVideoMixPosition(id, mbr, x * subImageWidth, y * subImageHeight, subImageWidth, subImageHeight);
    videoPositions.insert(VideoMixPositionMap::value_type(id, newPosition));
  }

  if (newPosition != NULL) {
    PBYTEArray fs(CIF_SIZE);
    int amount = CIF_SIZE;
    if (!OpenMCU::Current().GetEmptyMediaFrame(fs.GetPointer(), CIF_WIDTH, CIF_HEIGHT, amount))
      FillYUVFrame(fs.GetPointer(), 0, 0, 0, CIF_WIDTH, CIF_HEIGHT);
    WriteSubFrame(*newPosition, fs.GetPointer(), CIF_WIDTH, CIF_HEIGHT, amount);
  }

  return TRUE;
}

void MCUSimpleVideoMixer::RemoveVideoSource(ConferenceMemberId id, ConferenceMember & mbr)
{
  PWaitAndSignal m(mutex);

  // make sure this source is in the list
  {
    VideoMixPositionMap::iterator r = videoPositions.find(id);
    if (r == videoPositions.end()) 
      return;

    // clear the position where the frame was
    VideoMixPosition & vmp = *(r->second);
    if (videoPositions.size() == 1)
      FillCIFYUVFrame(frameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT).data.GetPointer(), 0, 0, 0);
    else
      FillCIFYUVRect(frameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT).data.GetPointer(), 0, 0, 0, vmp.xpos, vmp.ypos, vmp.width, vmp.height);
    frameStores.InvalidateExcept(CIF_WIDTH, CIF_HEIGHT);

    // erase the video position information
    delete r->second;

    // remove the source from the list
    videoPositions.erase(r);
  }

  // calculate the kind of video split we need to include the new source
  int newSubImageWidth, newSubImageHeight, newCols, newRows; 
  CalcVideoSplitSize((unsigned int)videoPositions.size(), newSubImageWidth, newSubImageHeight, newCols, newRows);

  // if the subimage size has changed, rearrange everything
  if (newSubImageWidth != subImageWidth || newSubImageHeight != subImageHeight || newRows != rows || newCols != cols) {
    rows           = newRows;
    cols           = newCols;
    subImageWidth  = newSubImageWidth;
    subImageHeight = newSubImageHeight;

    ReallocatePositions();
  }
}


BOOL MCUSimpleVideoMixer::WriteSubFrame(VideoMixPosition & vmp, const void * buffer, int width, int height, PINDEX amount)
{
  PWaitAndSignal m(mutex);

  VideoFrameStoreList::FrameStore & cifFs = frameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT);

  if (width == vmp.width && height == vmp.height) {
    CopyRectIntoCIF(buffer, cifFs.data.GetPointer(), vmp.xpos, vmp.ypos, vmp.width, vmp.height);
    frameStores.InvalidateExcept(CIF_WIDTH, CIF_HEIGHT);
  }

  else {
    converter->SetSrcFrameSize(width, height);
    PINDEX bytesReturned = (vmp.width * vmp.height * 3) / 2;
    converter->SetDstFrameSize(vmp.width, vmp.height, TRUE);
    converter->Convert((const BYTE *)buffer, imageStore.GetPointer((vmp.width * vmp.height * 3) / 2), amount, &bytesReturned);
    CopyRectIntoCIF(imageStore.GetPointer(), cifFs.data.GetPointer(), vmp.xpos, vmp.ypos, vmp.width, vmp.height);
    frameStores.InvalidateExcept(CIF_WIDTH, CIF_HEIGHT);
  }
  return TRUE;
};

///////////////////////////////////////////////////////////////////////////////////////

#if ENABLE_TEST_ROOMS

TestVideoMixer::TestVideoMixer(unsigned _frames)
  : frames(_frames), allocated(FALSE)
{
}

BOOL TestVideoMixer::AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr)
{
  PWaitAndSignal m(mutex);

  if (allocated)
    return FALSE;

  allocated = TRUE;

  unsigned i;
  for (i = 0; i < frames; ++i) {

    // calculate the kind of video split we need to include the new source
    CalcVideoSplitSize(i+1, subImageWidth, subImageHeight, cols, rows);

    VideoMixPosition * newPosition = CreateVideoMixPosition(id, mbr, 0, 0, subImageWidth, subImageHeight);
    videoPositions.insert(VideoMixPositionMap::value_type((ConferenceMemberId)(i+(BYTE *)id), newPosition));

    ReallocatePositions();
  }

  return TRUE;
}

BOOL TestVideoMixer::ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount)
{
  PWaitAndSignal m(mutex);

  // not allocated means fill with black
  if (!allocated) {
    VideoFrameStoreList::FrameStore & fs = frameStores.GetFrameStore(width, height);
    if (!fs.valid) {
      if (!OpenMCU::Current().GetEmptyMediaFrame(fs.data.GetPointer(), width, height, amount))
        FillYUVFrame(fs.data.GetPointer(), 0, 0, 0, width, height);
      fs.valid = TRUE;
    }
    memcpy(buffer, fs.data.GetPointer(), amount);
    return TRUE;
  }

  return ReadMixedFrame(buffer, width, height, amount);
}


BOOL TestVideoMixer::WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount)
{
  PWaitAndSignal m(mutex);

  unsigned i;
  for (i = 0; i < frames; ++i) {

    // write data into sub frame of mixed image
    VideoMixPositionMap::const_iterator r;
    for (r = videoPositions.begin(); r != videoPositions.end(); ++r) {
      r->second->WriteSubFrame(*this, buffer, width, height, amount);
    }
  }

  return TRUE;
}

#endif // ENABLE_TEST_ROOMS


#if ENABLE_ECHO_MIXER

EchoVideoMixer::EchoVideoMixer()
  : MCUSimpleVideoMixer()
{
}

BOOL EchoVideoMixer::AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr)
{
  PWaitAndSignal m(mutex);

  CalcVideoSplitSize(1, subImageWidth, subImageHeight, cols, rows);
  VideoMixPosition * newPosition = CreateVideoMixPosition(id, mbr, 0, 0, subImageWidth, subImageHeight);
  videoPositions.insert(VideoMixPositionMap::value_type(id, newPosition));
  ReallocatePositions();

  return TRUE;
}

BOOL EchoVideoMixer::WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount)
{
  PWaitAndSignal m(mutex);

  // write data into sub frame of mixed image
  VideoMixPositionMap::const_iterator r = videoPositions.begin();
  if (r != videoPositions.end())
    r->second->WriteSubFrame(*this, buffer, width, height, amount);

  return TRUE;
}

BOOL EchoVideoMixer::ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount)
{
  return ReadMixedFrame(buffer, width, height, amount);
}


#endif // ENABLE_ECHO_MIXER

#endif // OPENMCU_VIDEO
