/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef REQUEST_H
#define REQUEST_H

#include <cstdlib>
#include <string>
#include <lo/lo.h>

using std::string;


#include <iostream>
using std::cerr; using std::endl;

namespace Om {

#define NOBODY NULL

/** Represents an OSC request from some client.
 *
 * These are created by the OSCReceiver and sent along with Events so the
 * Event can send responses back to the client on success/fail.
 */
class Request
{
public:
	/** Construct a request.
	 *
	 * The lo_address must be duplicated here, as the address passed from
	 * lo_message_get_source is not the client's responsibility, and liblo
	 * does nasty things to it once the callback is completed.
	 *
	 * It would be nice to avoid this, it adds overhead to every single OSC
	 * message received
	 */
	Request(lo_address source, int id)
	: m_source(lo_address_new_from_url(lo_address_get_url(source))),
	  m_id(id)
	{
	}

	~Request()
	{
		lo_address_free(m_source);
	}
	
	void respond_ok();
	void respond_error(const string& msg);
	
	lo_address source() { return m_source; }
	int        id()     { return m_id; }

private:
	// Prevent copies
	Request(const Request& copy)            { exit(EXIT_FAILURE); }
	Request& operator=(const Request& copy) { exit(EXIT_FAILURE); }

	lo_address m_source;
	int        m_id;
};


} // namespace Om

#endif // REQUEST_H

