/* NVClock 0.5.0 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "backend.h"
#include "io.h"

#define SDR 0
#define DDR 1

static int base_freq;
static int factor = 1;

typedef enum
{
    GETMEMSPEED = 0,
    GETNVSPEED = 1,
    SETMEMSPEED = 2,
    SETNVSPEED = 3,
} Action;


float CalcSpeed(int m, int n, int p)
{
    return (float)((int)((float)n / (float)m * base_freq) >> p);
}

int GetSpeed(float *memclk, float *nvclk, int number, int debug)
{
    SetSpeed(0, memclk, number, debug, GETMEMSPEED);
    SetSpeed(0, nvclk, number, debug, GETNVSPEED);
}

int ChangeSpeeds(float memclk, float nvclk, int number, int debug)
{
    SetSpeed( (int)(memclk*1000), 0, number, debug, SETMEMSPEED);
    SetSpeed( (int)(nvclk*1000), 0, number, debug, SETNVSPEED);
}

/* Calculate the requested speed. */
static int ClockSelect(int clockIn, int *PLL, Action action)
{
    float m, n, p, speed, bestm, bestn;
    int diff, diff2, mlow, mhigh, nlow, nhigh;
    p = 0;
    speed = (float)clockIn;
    diff2 = (int)speed;

    /* Only use this for the memory */
    if(clockIn > 380000 && action == SETMEMSPEED)
    {
	mlow = 1;
//	mhigh = 4; /* Enable to get more speeds, but it is untested. */
	mhigh = 2;
	nlow = 14;
	nhigh = 93;
    }
    
    else
    {
	mlow = 7;
	mhigh = 14;
	nlow = 14;
	nhigh = 255;
    }
    
    /*
    Calculate the m and n values. There are a lot of values which give the same speed;
    We choose the speed for which the difference with the request speed is as small as possible.
    */
    for(m = mlow; m <= mhigh; m++)
    {
	for(n = nlow; n <= nhigh; n++)
	{
	    diff = (int)(speed - (n/m * base_freq));
	    
	    if(abs(diff) < abs(diff2))
	    {
		diff2 = diff;
		bestm = m;
		bestn = n;
	    }	    
	}
    }

    *PLL = ((int)p << 16) + ((int)bestn << 8) + bestm;
}

/* Overclock the memory or core. */
int SetSpeed(unsigned int clk, float *clkout, int number, int debug, Action action)
{
    unsigned long base;
    int m, n, p, newPLL, fd, mem_type, offset;
    riva_hw nv_card;
    
    base = (long)card[number].mem;

    if( (fd = openDevMem(number)) == -1 )
    {
	printf("Can't open /dev/mem or /dev/nvidia%d\n", number);
    	return -1;
    }
    
    /* Set the correct offset for the memory/core */
    if(action == SETMEMSPEED || action == GETMEMSPEED)
    {
	offset = 0x504;
    }

    else
    {
	offset = 0x500;
    }

    /* Map the registers of the nVidia chip */
    nv_card.PEXTDEV = mapDevMem(fd, base + 0x101000, 0x1000);
    nv_card.PFB     = mapDevMem(fd, base + 0x100000, 0x1000);
    nv_card.PRAMDAC = mapDevMem(fd, base + 0x680000, 0x2000);
    

    /* Check if memory is SDR or DDR, needed for some problematic GF2MX/GF4MX cards */
    mem_type = (nv_card.PFB[0x200/4] & 0x01) ? DDR : SDR;

    /* find out the reference frequency */    
    if(card[number].device_id > 0x29)
    {
	/* The Geforce 4 and Quadro 4 use a base frequency of 27MHz */
	base_freq = (nv_card.PEXTDEV[0x0000/4] &(1<<6) ) ? 14318 : (nv_card.PEXTDEV[0x0000/4] & (1<<22)) ? 27000 : 13500;
    }

    else
    {
	base_freq = (nv_card.PEXTDEV[0x0000/4] & 0x40) ? 14318 : 13500;
    }

    /* Only overclock if we really want to overclock */
    if(action == SETMEMSPEED || action == SETNVSPEED)
    {
	/* This is a workaround meant for some Geforce2 MX/Geforce4 MX cards
	*  using SDR memory. Gf2MX/Gf4MX cards use 4x16 SDR memory report
	*  twice as high clockspeeds. I call that "fake ddr".
	*  By detecting the memory type, pci id and clockspeed we check
	*  if this occurs. It is a workaround.
	*/
	if(action == SETMEMSPEED && mem_type == SDR && ( card[number].device_id == 0x110 || 
	    card[number].device_id == 0x111 || card[number].device_id == 0x172))
	{
	    float memclk, nvclk;
	    card[number].device_id = 0x110;
	    GetSpeed(&memclk, &nvclk, number, 0);
	    if((memclk*1000*factor) > 280000)	clk *= 2;
	}

	/* HERE the new clocks are selected (in KHz). */
	ClockSelect(clk, &newPLL, action);

	/* Overclock */
	nv_card.PRAMDAC[offset/4] = newPLL;
    }

    m = nv_card.PRAMDAC[offset/4] & 0xff;
    n = (nv_card.PRAMDAC[offset/4] >> 8) & 0xff;
    p = (nv_card.PRAMDAC[offset/4] >> 16) & 0x0f;

	    
    /* This is a workaround meant for some Geforce2 MX/Geforce4 MX cards
    *  using SDR memory. Gf2MX/Gf4MX cards use 4x16 SDR memory report
    *  twice as high clockspeeds. I call that "fake ddr".
    *  By detecting the memory type, pci id and clockspeed we check
    *  if this occurs. It is a workaround. We divide the memclk later by 2.
    */
    if((action == SETMEMSPEED || action == GETMEMSPEED) && mem_type == SDR && ( card[number].device_id == 0x110 || 
	card[number].device_id == 0x111 || card[number].device_id == 0x172))
    {
	int memclk = CalcSpeed(m, n, p);
	if(memclk > 280000)
	{
	    factor = 2;
	}
    }


    if(debug == 1)
    {
	/* Print the memory speed */
	if(action == SETMEMSPEED || action == GETMEMSPEED)
	{
	    int memclk = CalcSpeed(m, n, p);
	    printf("MPLL=  %i    m=%i n=%i p=%i\n", memclk, 
	    nv_card.PRAMDAC[0x504/4] & 0xff, ( nv_card.PRAMDAC[0x504/4] >> 8) & 0xff, (nv_card.PRAMDAC[0x504/4] >> 16) & 0x0f);
    	    printf("PLL_COEFF_SELECT = %08lX\n\n", nv_card.PRAMDAC[0x50c/4] );
	}
	
	else
	{
	    int nvclk = CalcSpeed(m, n, p);
	    printf("NVPLL=  %i    m=%i n=%i p=%i\n", nvclk, 
	    nv_card.PRAMDAC[0x500/4] & 0xff, ( nv_card.PRAMDAC[0x500/4] >> 8) & 0xff, (nv_card.PRAMDAC[0x500/4] >> 16) & 0x0f);
    	    printf("PLL_COEFF_SELECT = %08lX\n\n", nv_card.PRAMDAC[0x50c/4] );
	}
    }
    
    /* Return the speed if this function was used for getting the speeds */
    if(action == GETMEMSPEED || action == GETNVSPEED)
    {
	int factor2 = 1;
	if(action == GETMEMSPEED) factor2 = factor;
    
	*clkout = ( CalcSpeed(m,n,p) / factor2 ) / 1000;
    }

    return 0;
}
