/* mgeups.c - model specific routines for MGE units
   Copyright (C) 2001 Ryan Murray <rmurray@cyberhqz.com>

   general outline based on:
	   ipt-anzen.c - model specific routines for IPT Anzen units
	   Copyright (C) 2000 Travis Tabbal <bigboss@xmission.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <sys/ioctl.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "shared.h"
#include "config.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define UPSDELAY 5
#define MAXTRIES 10
#define INFOMAX 16
#define ENDCHAR 13	/* replies end with CR */

#define SS_BATTERY_LOW 4
#define SS_POWER_FAIL 5
#define SS_OVERLOAD 6

#define BS_POWER_FAIL 0
#define BS_POWER_OK 1
#define BS_BAT 2
#define BS_PWS 11
#define BS_TDP 12

#define LS_PWRON 9
#define LS_AUTORESTART 13

	int	shmok = 1, cap_upstemp = 0;
	float	lowvolt = 0, voltrange;
	int	lownorm, highnorm;
	int	sddelay = 90;	/* wait 90 seconds for shutdown */
static	int	loadok = 1, batlevok = 1;
extern	char	*pidfn;

void initinfo (void)
{
	create_info(INFOMAX, shmok);

	/* setup variables that should always exist */

	addinfo (INFO_MFR, "Merlin Gerin", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_LOWXFER, "", 0, 0);
	addinfo (INFO_HIGHXFER, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
}

/* Send a whole string, I hate sending a character at a time */

void xsend (char *str) 
{
	int i, len;

	len = strlen (str);

	for (i = 0; i < len; i++) 
		upssendchar (str[i]);

	upssendchar (13);
}
	
void ups_sync()
{
        char    buf[256];
        int     tries = 0;
	int i;

        printf ("Syncing with UPS: ");
        fflush (stdout);

        for (;;) {
                tries++;
                if (tries > MAXTRIES) {
                        printf ("\nFailed - giving up...\n");
                        exit (1);
                }
                xsend ("Z\r");
                upsrecv (buf, sizeof(buf), ENDCHAR, "");
                printf (".");
                fflush (stdout);
		xsend ("Ax 1\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		xsend ("Si\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		i = atoi( buf );
		xsend ("Bl ?\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		if (buf[0] == '?') {
			batlevok = 0;
		}
		xsend ("Iv ?\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		xsend ("Bs\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		switch ( i ) {
			case 1000:
			case 2000:
			case 3000:
				addinfo (INFO_MODEL, "Pulsar", 0, 0);
				break;
			case 4000:
				addinfo (INFO_MODEL, "Comet", 0, 0);
				break;
			case 5000:
				addinfo (INFO_MODEL, "Galaxy", 0, 0);
				break;
			default:
				fatalx( "Unknown UPS type" );
				break;
		}
		xsend ("Si 1\r");
		upsrecv(buf, sizeof(buf), ENDCHAR, "");
		if ( buf[0] != '?' ) {
			addinfo( INFO_MODEL, buf, 0, 0 );
		}
		break; /* suspicious? */
        }

        printf (" done\n");
 }

void getbaseinfo (char *port)
{
	ups_sync ();
}

/* normal idle loop - keep up with the current state of the UPS */

void updateinfo (void)
{
	char	temp[256],
		sstat[16], battstat[24];

	temp[0] = '\0';

	xsend ("Ss\r");
	upsrecv (sstat, sizeof(sstat), ENDCHAR, "");
	if ( strlen(sstat) >= 8 ) {
		if (sstat[4] == '1') {
			strcat( temp, "LB " );
		}
	}

	xsend ("Bs\r");
	upsrecv (battstat, sizeof(battstat), ENDCHAR, "");
	if ( strlen(battstat) >= 17 ) {
		if (battstat[1] == '1') {
			strcat( temp, "OL " );
		}
		if (battstat[0] == '1') {
			strcat( temp, "OB " );
		}
		/* battstat[2] is charging or on battery */

		if (temp[strlen(temp)-1] == ' ')
			temp[strlen(temp)-1] = 0;

		setinfo(INFO_STATUS, "%s", temp);
	}

	if ( batlevok ) {
		xsend ("Bl\r");
		upsrecv (temp, sizeof(temp), ENDCHAR, "");
		printf ("Battery percent: \"%s\"\n", temp);
	}

	if ( loadok ) {
		xsend ("Ll\r");
		upsrecv (temp, sizeof(temp), ENDCHAR, "");
		if ( temp[0] == '?' ) {
			loadok = 0;
		}
		printf ("Load: \"%s\"\n", temp);
	}
	writeinfo();
}

/* power down the attached load immediately */

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		default:
			upslogx(LOG_INFO, "instcmd: unknown/unsupported type 0x%04x",
		        	auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers()
{
	upsh.instcmd = instcmd;
	/* TODO: future */
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i;
	int RTS = TIOCM_RTS;

	printf ("Network UPS Tools - MGEUPS Pulsar driver 0.01 (%s)\n",UPS_VERSION);
	openlog ("mgeups", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	while ((i = getopt(argc, argv, "+hd:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot(); 

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/mgeups-%s", STATEPATH,
	          portname);

	open_serial (argv[0], B2400); 
	ioctl( upsfd, TIOCMBIC, &RTS );

	initinfo();

	createmsgq();	/* try to create IPC message queue */

	getbaseinfo(argv[0]);

	setuphandlers();

	pidfn = xmalloc(SMALLBUF);
	snprintf (pidfn, sizeof(pidfn), "%s/mgeups-%s.pid", STATEPATH,
	          portname);
	background(); 
	writepid(pidfn);

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
