/* engetron.c - model specific routines for Engetron Jr model

   Copyright (C) 2000  Adriano Roberto de Lima <arlima@bigfoot.com>
                       Marcio Argollo F. de Menezes <marcio@if.uff.br>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX  16
#define UPSDELAY 5
#define MAXTRIES 10

int	shmok = 1;
char	*va;
float	lowvolt = 0, highvolt = 0, voltrange = 0, 
  vn,   /* nominal tension - model dependent */ 
  il;   /* maximum load current - model dependent */



void 
usage (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS1\n", prog);
	exit (1);
}

void 
help (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-h       - display this help\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}

void initinfo (void)
{
	create_info (INFOMAX, shmok);
  
	addinfo(INFO_MFR, "", 0, 0);
	addinfo(INFO_MODEL, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);
	addinfo(INFO_UPSTEMP, "", 0, 0);
}


void 
setup_serial(void)
{	
  struct termios tio;
  
  tio.c_iflag = IXON | IXOFF;
  tio.c_oflag = 0;
  tio.c_cflag = (B9600 | CS8 | CREAD | HUPCL | CLOCAL);
  tio.c_lflag = 0;
  tio.c_cc[VMIN] = 1;
  tio.c_cc[VTIME] = 0;
  
  if (tcsetattr (upsfd, TCSANOW, &tio) == -1)
    fatal("tcsetattr");
}

char 
readchar (void)
{
  char	c;
  int	ret;
  
  ret = read (upsfd, &c, 1);
  
  if (ret > 0)
    return (c & 0x7f);
  else
    return 0;
}

int 
xrecv (char *buf, int max_count)
{
  char	ch, *p;
  int counter;

  p = buf;
  *p = '\0';
  counter=0;

  while ((ch = readchar ())) {
    if (ch == '\n') {
      counter++;
      if(counter==max_count){
	*p = '\0';
	return 0;
      }
    }
    
    *p++ = ch;
  }
  
  *p = '\0';
  return 0;
}

void 
ups_ident(void)
{
  char	
    buf[512], 
    *ptr, 
    *com;
  int	
    tries = 0;

  tcflush(upsfd,TCIOFLUSH);	
  printf ("Identifying UPS: ");
  fflush (stdout);
  
  for (;;) {
    tries++;
    if (tries > MAXTRIES) {
      printf ("\nFailed - giving up...\n");
      exit (1);
    }
    upssend("%s", "Identidade\r");
    printf (".");
    fflush (stdout);
    sleep (UPSDELAY);
    printf (".");
    fflush (stdout);
    xrecv (buf,9);
    printf (".");
    fflush (stdout);
    if ((buf[0] != '\0') && (buf[0] != 'I'))
      break;
  }
  tcflush(upsfd,TCIOFLUSH);	
  
  printf (" done\n");
  
  va = NULL;
  
  ptr = strchr(buf, 'N');

  com = strchr (ptr, 'V') - 1;
  if (com)
    *com = '\0';
  
	setinfo(INFO_MFR, "%s", "ENGETRON");
	setinfo(INFO_MODEL, "%s", ptr);
  
  ptr=com+1;
  
  com = strchr (ptr, '\n');
  if (com)
    *com = '\0';

  va = xstrdup (ptr);
 
  
  ptr=com+1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf (ptr, "%f %*s %f", &lowvolt, &highvolt);
  voltrange = highvolt - lowvolt;
}

void ups_update (void)
{
	char *ptr, *com, onoff, vr[128], buf[300], buf2[300];
	int mask, onbattery, status;
	int i;
	float bvoltp, info_eng[24];

  vn = 12.0;
  il = 2.8;

  upssend("%s", "Status\r");     
  sleep (UPSDELAY);
  xrecv (buf,7);
  
  if ((strlen(buf) < 271) || (buf[0] != 'S'))
    return;
  
  ptr = strchr(buf, ':') + 1;
  
  com = strchr(ptr, 'V') - 1;
  if (com)
    *com = '\0';
  
  ptr=com+1;
  
  sscanf (ptr, "%*s %s", vr);

  for (i = 1; i < 16; i++) {
    com = strchr (ptr, ':');
    ptr = com + 1;
    sscanf(ptr, "%f", &info_eng[i]);
  }
  
  com = strchr (ptr, ':');
  ptr = com + 1;
  if(info_eng[8]!=0.0){
    sscanf(ptr, "%f %*s %f", &info_eng[16], &info_eng[17]);
  }
  
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f", &info_eng[18]);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%x", &status);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f %f %f", &info_eng[19], &info_eng[20], &info_eng[21]);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f %c", &info_eng[22], &onoff);

	setinfo(INFO_UTILITY, "%3.1f", info_eng[2]);


	strcpy (buf2, "");
  
  mask = 1;
  onbattery=0;
  for(i=0;i<16;i++){
    if((status&mask)!=0){
      switch (i) {
      case 0: strcat (buf2, "RB ");
	break;
      case 2: strcat (buf2, "OVER ");
	break;
      case 10: strcat (buf2, "LB ");
	break;
      case 11: strcat (buf2, "TRIM ");
	break;
      case 12:
	onbattery=1;
	strcat (buf2, "OB ");
	break;
      default:
	break;
      }
    }
    mask <<= 1;
  }
  
  if(onoff=='D')
    strcat (buf2, "OFF ");
  else{
    if(onbattery==0)
      strcat (buf2, "OL ");
  }
  
	/* lose trailing space if present */
	if (buf2[strlen(buf2)-1] == ' ')
		buf2[strlen(buf2)-1] = 0;

	setinfo(INFO_STATUS, "%s", buf2);

  /*********************** modificar tudo ******************/
  if(onbattery){
    bvoltp = ((6060 / vn) * info_eng[11])/10 - 567; 
  } 
  else {

    /*    bvoltp = (80.0 * (info_eng[11] - vn) / (1.15 * vn) + 20.0 * (il - info_eng[12]) / (0.6 * il));*/

    bvoltp = (87.0 * info_eng[11])  / vn - (2.0 * info_eng[12]) / il;
  }
  if (bvoltp < 0.0) {
    bvoltp = 0;
  }
  else {
    if (bvoltp > 100.0) {
      bvoltp = 100.0;
    }
  }
  
	setinfo(INFO_BATTPCT, "%3.1f", bvoltp);

	/*********************** ate aqui ******************/
 
	setinfo(INFO_ACFREQ, "%2.2f", info_eng[5]);
	setinfo(INFO_LOADPCT, "%3.1f", 100.0*info_eng[8]/info_eng[1]);
	setinfo(INFO_UPSTEMP, "%3.1f", info_eng[18]);

	writeinfo();
}



int 
main (int argc, char **argv)
{
  char	
    *prog, 
    *portname;
  int
    i;

  printf ("Network UPS Tools - Engetron Jr. UPS driver 0.10 (%s)\n", UPS_VERSION);
  openlog ("engetron", LOG_PID, LOG_FACILITY);

  prog = argv[0];
  while ((i = getopt(argc, argv, "+hk:")) != EOF) {
    switch (i) {
    case 'h':
      help(prog);
      break;
    default:
      usage(prog);
      break;
    }
  }
  
  argc -= optind;
  argv += optind;

  if (argc < 1) {
	printf ("Error: no device specified!\n");
	usage (prog);
  }
  
  droproot();
  
  portname = NULL;
  for (i = strlen(argv[0]); i >= 0; i--)
    if (argv[0][i] == '/') {
      portname = &argv[0][i+1];
      break;
    }
  
  if (portname == NULL) {
    printf ("Unable to abbreviate %s\n", argv[0]);
    exit (1);
  }

  snprintf (statefn, sizeof(statefn), "%s/engetron-%s", STATEPATH,
	    portname);

  upsfd = open(argv[0], O_RDWR | O_NDELAY);
  if (upsfd == -1)
    fatal("open %s", argv[0]);
  
  setup_serial();
  initinfo();
  ups_ident();
  createmsgq();	/* try to create IPC message queue */
  
  printf ("Detected %s %s on %s\n", getdata(INFO_MFR), getdata(INFO_MODEL), argv[0]);
  
  background();
  
  for (;;)
    ups_update();

  return 0;
}
