/* 
   bestfort.c - model specific routines for Best Power Fortress (some
   don't speak to bestups, which does some entirely different protocol
   than this one that my Fortress LI 1020 speaks).  I think I've got a
   "traditional" Fortress as opposed to one of the other dozen-odd
   lines Best Power has made or acquired over the years.  The protocol
   here is documented in Best's document FTS502b.  Just ask and
   they'll send it to you as a PDF.

   This module is a 75% rewritten mangle of the bestups module by
   Russell.  It lost the "test battery" command since my fortress does
   this automagically each week at 2am.  It gained a -D aka debugging
   option to not background and summarize stuff from each poll.

   Warning: I have not tested the -k function.

   Copyright (C) 2000  Grant Taylor <gtaylor@picante.com>
   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>
#include <assert.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX  16

int debugging = 0;

	int	shmok = 1;

/* Model ID strings.  The Fortress returns an enumerated integer value
   for model type. */
static const int num_models = 5;
static char *models[] = {"Unknown",
			 "Patriot/SPS",
			 "Fortress",
			 "FERRUPS",
			 "UNITY/I single-phase" };

/* Blob of UPS configuration data from the formatconfig string */
struct {
  int valid;			/* set to 1 when this is filled in */

  double idealbvolts;		/* various interestin battery voltages */
  double fullvolts;
  double emptyvolts;
  int va;			/* capacity of UPS in Volt-Amps */
  int watts;			/* capacity of UPS in watts */
  int model;			/* enumerated model type */
} fc;


/* Forward decls */
void setup_serial(void);
int readupsline (char *buf);
int execute_int(char *cmd, char *response);

/* Set up all the funky shared memory stuff used to communicate with upsd */
void initinfo (void)
{
	create_info (INFOMAX, shmok);

	/* now set up room for all future variables that are supported */
	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
}

/* TODO: adapt to the standard upscommon open/send/recv calls */

/* This seems unlikely, as the current ones don't handle the
   unterminated line containing the prompt that you get back after
   each command.  Perhaps the thing is to make our own or something;
   the standard calls are nice in that they use alarms to do timeouts
   and such.  OTOH, the Fortress *always* babbles back at you giving
   at least a prompt...   */

int open_ups(char *filename)
{
  chdir("/dev");

  upsfd = open(filename, O_RDWR | O_NDELAY);
  if (upsfd == -1)
    fatal("open");
  
  setup_serial();
  
  return upsfd;
}

void setup_serial(void)
{	
  struct	termios	tio;

  /* Zero out the structure to prevent confusion.  It took me forever
     to figure out why my serial reading routines read everything from
     the UPS perfectly, except for the infuriating detail that all the
     `0's were missing: since it didn't wipe the c_cc part clean, the
     VSTART character got set to `0', so that got eaten.  The (horrid)
     example code from Best Power was immune to this problem since
     everything is a (n implicitly initialized) global.  */

  memset(&tio, 0, sizeof(tio));
  
  tio.c_iflag = IXON | IXOFF;
  tio.c_oflag = 0;
  tio.c_cflag = (B1200 | CS8 | CREAD | HUPCL | CLOCAL);
  tio.c_lflag = 0;
  tio.c_cc[VMIN] = 1;
  tio.c_cc[VTIME] = 0;
  tio.c_cc[VSTART] = 17;	/* CTRL('q'); is not portable */

  if (tcsetattr (upsfd, TCSANOW, &tio) == -1)
    fatal("tcsetattr");
}

char readchar (void)
{
  char	c='z';
  int	ret;


  /* It's not obvious that O_NDELAY was useful.  We should probably do
     selects with timeouts or set a timer against blocking i/o or
     something.  Luckily UPS's aren't prone to goign missing.  */

 readspot:
  ret = read (upsfd, &c, 1);

  if (ret > 0) {
    return (c & 0x7f);
  } else {
    if (errno == EAGAIN) {
      sleep(1);
      goto readspot;
    }

    perror("read error in readchar()");
    return 0;
  }
}

/* Run a command.  Put the response, or empty string if none, into
   buffer response.  God help you if the buffer is too short.  It
   would be best to not execute a continuous statement like
   `contstatus' ;) */

int execute(char *cmd, char *response)
{
  char junkbuf[256];

  /* First synchronize with a control-c and backspace.  Then we'll be
     at a "clean" prompt and ready to go. */

  return execute_int("\x08\x03", junkbuf) || execute_int(cmd,response);
}

int execute_int(char *cmd, char *response)
{
  int ret, prompt;
  char *ptr, linebuf[256];
  
  /* First send command string plus the \r */
  ptr = cmd;
  while (*ptr) {
  writespot1:
    ret = write(upsfd, ptr++, 1);
    if (ret == -1) {
      if (errno == EAGAIN) 
	goto writespot1;

      perror("execute(): write() failed");
      return -1;
    }
  }

 writespot2:
  ret = write(upsfd, "\r", 1);	/* carriage return */
  if (ret == -1) {
    if (errno==EAGAIN) 
      goto writespot2;

    perror("execute(): write() failed");
    return -1;
  }

  /* OK, now we read.  We expect to see what we just wrote, followed
     by some or no output, followed by a prompt ending in "=> " */

  if (strlen(cmd) && cmd[0] != '\x08') {
    int lineoffset = 0;
    do {			/* read until we see our command */
      prompt = readupsline(linebuf);
      lineoffset = strlen(cmd) - strlen(linebuf);
    } while (0 != strcmp(cmd,linebuf+lineoffset) && prompt != -1);
  }

  response[0] = '\0';
  do {
    prompt = readupsline(linebuf);

    if (prompt) {
      /* the next prompt */
    } else if (strlen(linebuf) > 1 && linebuf[0] != '\r') {
      /* interesting output */
      strcat(response, linebuf);
      strcat(response, "\r");
    }
  } while (prompt == 0);

  return ((prompt == -1) ? -1 : 0);
}

/* 
   read a line from the UPS, terminated by either a '\r' or by a =>
   (the prompt).

   return -1 for error
   return 0 for normal lines
   return 1 for prompt lines
 */
int readupsline (char *buf)
{
  char	ch, *p;
  int l;
  
  p = buf;
  *p = '\0';
  l=0;
  
  while ((ch = readchar ())) {
    if (ch == '\n')
      continue;		/* absorb duplicate nl/crs */
    
    l++;
    
    /* done on carriage return/newline, don't include cr */
    if (ch == '\n' ||  ch == '\r') {
      *p = '\0';
      return 0;
    }
    
    /* include this character */
    *p++ = ch;
    
    /* done if last two chars are '=>' */
    if (l >= 2) {
      char *prompt = p-2;
      if (0 == memcmp(prompt, "=>", 2)) {
	*p = '\0';
	return 1;		/* one for prompt line */
      }
    }
  }
  
  *p = '\0';
  return -1;
}

void ups_sync(void)
{
  char	buf[256];

  printf ("Syncing: ");
  fflush (stdout);

  /* A bit better sanity might be good here.  As is, we expect the
     human to observe the time being totally not a time. */

  if (0 == execute("time", buf)) {
    fprintf(stderr, "UPS Time: %s\n", buf);
  } else {
    fprintf(stderr, "Error connecting to UPS.\n");
    exit(1);
  }
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
  char	temp[256];

  upslogx(LOG_INFO, "Initiating UPS shutdown\n");
  printf ("Initiating forced UPS shutdown!\n");
  
  open_ups(port);
  
  /* shutdown in 5 seconds, autostart later */
  execute("shutdown autostart 5", temp);
  
  printf ("Waiting for poweroff...\n");
  sleep (90);
  printf ("Hmm, did the shutdown fail?  Oh well...\n");
  exit (1);                               
}

void usage (char *prog)
{
  printf ("usage: %s [-h] [-k] [-D] <device>\n", prog);
  printf ("Example: %s /dev/ttyS0\n", prog);
  exit (1);
}

void help (char *prog)
{
  printf ("usage: %s [-h] [-k] <device>\n", prog);
  printf ("\n");
  printf ("-h       - display this help\n");
  printf ("-k       - force shutdown\n");
  printf ("-D       - run in foreground for debugging/testing\n");
  printf ("<device> - /dev entry corresponding to UPS port\n");
}



void ups_ident (void)
{
  char	temp[256], fcstring[512];

  /* Obtain Model */
  if (0 != execute("fc", fcstring)) {
    fprintf(stderr, "Failed execute in ups_ident()\n");
    exit(1);
  }
  
  /* response is a  one-line packed string starting with $ */
  if (fcstring[0] != '$') {
    fprintf(stderr,
	    "Bad response from formatconfig command in ups_ident()\n");
    exit(1);
  }

  if (debugging)
    fprintf(stderr, "fc: %s\n", fcstring);
  
  /* chars 4:2  are a two-digit ascii hex enumerated model code */
  memcpy(temp, fcstring+3, 2);
  temp[2] = '\0';
  sscanf(temp, "%x", &fc.model);

  if (fc.model >= num_models) {
    fprintf(stderr,
	    "Unknown model 0x%x in ups_ident()\n",
	    fc.model);
    exit(1);
  }
  
  /* chars 12:5 are five-digit ascii decimal VA rating */
  memcpy(temp, fcstring+11, 5);
  temp[5] = '\0';
  fc.va = atoi(temp);
  
  /* chars 17:5 are 5-dig ascii deciaml watts rating */
  memcpy(temp, fcstring+16, 5);
  temp[5] = '\0';
  fc.watts = atoi(temp);
  
  /* chars 46:4 are ascii decimal xxx.x nominal battery volt */
  memcpy(temp, fcstring+45, 4);
  temp[4] = '\0';
  fc.idealbvolts = atof(temp) / 10.0;
  
  /* chars 50:4 are full charge battery voltage */
  memcpy(temp, fcstring+49, 4);
  temp[4] = '\0';
  fc.fullvolts = atof(temp) / 10.0;
  
  /* chars 58:4 are empty battery voltage */
  memcpy(temp, fcstring+57, 4);
  temp[4] = '\0';
  fc.emptyvolts = atof(temp) / 10.0;
  
  setinfo(INFO_MFR, "%s", "Best Power");
  setinfo(INFO_MODEL, "%s %d", models[fc.model], fc.va);
  
  fc.valid = 1;
  
  printf("Best Power %s detected\n", getdata(INFO_MODEL));
  printf("Battery voltages %5.1f nominal, %5.1f full, %5.1f empty\n", 
	 fc.idealbvolts,
	 fc.fullvolts,
	 fc.emptyvolts);
  
  return;
}

void ups_update (void)
{
  char fstring[512];

  if (! fc.valid) {
    fprintf(stderr, 
	    "upsupdate run before ups_ident() read ups config\n");
    assert(0);
  }

  if (0==execute("format",fstring)) {
    int inverter=0, charger=0, 
      vin=0, btimeleft=0,
      linestat=0, alstat=0, vaout=0;
    double ampsout=0.0, vbatt=0.0, battpercent=0.0, loadpercent=0.0;
    char tmp[16], statstr[128];

    /* Inverter status.  0=off 1=on */
    memcpy(tmp, fstring+16, 2);
    tmp[2] = '\0';
    inverter = atoi(tmp);

    /* Charger status.  0=off 1=on */
    memcpy(tmp, fstring+18, 2);
    tmp[2] = '\0';
    charger = atoi(tmp);
    
    /* Input Voltage. integer number */
    memcpy(tmp, fstring+24, 4);
    tmp[4] = '\0';
    vin = atoi(tmp);

    /* Iout.  int times 10 */
    memcpy(tmp, fstring+36, 4);
    tmp[4] = '\0';
    ampsout = ((double)(atoi(tmp)) / 10.0);

    /* Battery voltage.  int times 10 */
    memcpy(tmp, fstring+50, 4);
    tmp[4] = '\0';
    vbatt = ((double)(atoi(tmp)) / 10.0);

    /* Volt-amps out.  int  */
    memcpy(tmp, fstring+40, 6);
    tmp[6] = '\0';
    vaout = atoi(tmp);

    /* Line status.  Bitmask */
    memcpy(tmp, fstring+72, 2);
    tmp[2] = '\0';
    linestat = atoi(tmp);

    /* Alarm status reg 1.  Bitmask */
    memcpy(tmp, fstring+20, 2);
    tmp[2] = '\0';
    alstat = atoi(tmp);

    /* Alarm status reg 2.  Bitmask */
    memcpy(tmp, fstring+22, 2);
    tmp[2] = '\0';
    alstat = alstat | (atoi(tmp) << 8);


    /* Get various assorted parameters */

    /* Runtime */
    if (0 == execute("p 9", fstring)) {
      sscanf(fstring, "09 Runtime %dm", &btimeleft);
    }

    /* Percent Load */
    if (0 == execute("p 16", fstring)) {
      int l;
      sscanf(fstring, "16 FullLoad%% %d", &l);
      loadpercent = (double) l;
    }
    
    /* Compute battery percent left based on battery voltages. */
    battpercent = ((vbatt - fc.emptyvolts) 
		   / (fc.fullvolts - fc.emptyvolts) * 100.0);
    if (battpercent < 0.0) 
      battpercent = 0.0;
    else if (battpercent > 100.0)
      battpercent = 100.0;
    
    /* Compute status string */
    {
      int lowbatt, overload, replacebatt, boosting, trimming;

      lowbatt = alstat & (1<<1);
      overload = alstat & (1<<6);
      replacebatt = alstat & (1<<10);
      boosting = inverter && (linestat & (1<<2)) && (vin < 115);
      trimming = inverter && (linestat & (1<<2)) && (vin > 115);

      strcpy(tmp, inverter ? "OB" : "OL");
      if (lowbatt) strcat (tmp, " LB");
      if (trimming) strcat (tmp, " TRIM");
      if (boosting) strcat (tmp, " BOOST");
      if (replacebatt) strcat (tmp, " RB");
      if (overload) strcat (tmp, " OVER");

      strlcpy(statstr, tmp, sizeof(statstr));

    }


    if (debugging) {
      fprintf(stderr,
	      "Poll: inverter %d charger %d vin %d vaout %d btimeleft %d\n",
	      inverter, charger, vin, vaout, btimeleft);
      fprintf(stderr,
	      "      ampsout %5.1f vbatt %5.1f batpcnt %5.1f loadpcnt %5.1f\n",
	      ampsout, vbatt, battpercent, loadpercent);
      fprintf(stderr,
	      "      STATUS '%s'\n", statstr);

    }



    /* Stuff information into info structures */

    setinfo(INFO_STATUS, "%s", statstr);

    setinfo(INFO_BATTPCT, "%02.1f", battpercent);

    setinfo(INFO_LOADPCT, "%03.1f", loadpercent);

    setinfo(INFO_UTILITY, "%05.1f", (double)vin);

    writeinfo();


  }

  return;
}

int main (int argc, char **argv)
{
	char	*prog, *portname;
	int	i;

	printf ("Network UPS Tools - Best Fortress driver 0.1 (%s)\n", 
		UPS_VERSION);
	openlog ("bestfort", LOG_PID, LOG_FACILITY);
	prog = argv[0];

	while ((i = getopt(argc, argv, "+hk:D")) != EOF) {
		switch (i) {
			case 'k':
				forceshutdown(optarg);
				break;
			case 'D':
			  debugging = 1;
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/bestfort-%s", STATEPATH,
	          portname);

	open_ups(portname);

	ups_sync();		/* get a prompt */

	initinfo();
	createmsgq();

#if 0
	setuphandlers();	/* seems uneeded? */
#endif

	ups_ident();		/* run formatconfig command and
                                   extract fixed information into
                                   global fc. struct */

#if 0
	/* Fortress does automagical testing monthly */
	addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
	addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
#endif

	printf ("Detected %s %s on %s\n", 
		getdata(INFO_MFR), 
	        getdata(INFO_MODEL), 
		argv[0]);

	if (!debugging)
	  background();

	for (;;) {
	  ups_update();
	  
	  if (getupsmsg(2))       /* TODO: remove debug scaffolding */
	    upslogx(LOG_INFO, "Received a message from upsd\n");
	  
	}

	return 0;
}


#if 0
void instcmd (int auxcmd, int dlen, char *data)
{
  assert(0);

	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			break;
		case CMD_BTEST1:	/* start battery test */
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x\n",
			        auxcmd);
	}
}

void setuphandlers()
{
  assert(0);
  /*	upsh.instcmd = instcmd; */
}
#endif /* 0 */

