// file kernel/n/ppc32/div_n2.S: O(n^2) division of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                         Division quadratique                          |
 |                                                                       |
 +-----------------------------------------------------------------------*/


                               ; +-------------+
                               ; |  Inversion  |
                               ; +-------------+

; chiffre xn(inv)(chiffre a)
;
; entre :
;    a = entier naturel tel que a >= BASE/2
;    x,y = scratch
;
; sortie :
;    u <- floor((BASE^2-1)/a), compris entre BASE et 2*BASE-1
;    retourne u - BASE
;
; algorithme : itration de Newton pour la fonction
;
;               u -> u + u*(BASE - (a*u)/BASE)/BASE
;
; en tronquant les quotients par dfaut et en partant de
; u0 = sqrt(BASE)*floor((BASE-1)/(a1+1)) o a1 = floor(a/sqrt(BASE)).
; La suite (u_n) calcule est strictement croissante tant que
;
;            u_n < 2*BASE  et  a*u_n < BASE^2
;
; La condition u_n < 2*BASE sera toujours satisfaite si a > BASE/2,
; dans ce cas on sort lorsque a*u_n >= BASE^2 et le quotient cherch
; est u_n - 1 ou u_n - 2.
; Pour a = BASE/2, on court-circuite les calculs et on sort directement
; le rsultat : u = 2*BASE-1.

#define INVERSE(a,u,x,y) \
	slwi.  u,  a,  1   /* si a = BASE/2, u <- BASE-1                 */ @\
	li     u,  -1                                                       @\
	beq    9f                                                           @\
        srwi   u,  a, 16   /* u <- a1+1                                  */ @\
        addi   u,  u,  1                                                    @\
        li     x, -1       /* u <- floor((BASE-1)/(a1+1))                */ @\
        divwu  u,  x,  u                                                    @\
        srwi   x,  u, 16   /* si < sqrt(BASE), ajoute 1                  */ @\
        xori   x,  x,  1                                                    @\
        add    u,  x,  u                                                    @\
        slwi   u,  u, 16   /* u <- u0 - BASE                             */ @\
7:                                                                          @\
        mulhwu x,  a,  u   /* x <- floor(a*u/BASE)                       */ @\
        add.   x,  a,  x                                                    @\
        bge    8f          /* x >= 0 ssi a*u >= BASE^2                   */ @\
        neg    x,  x       /* y <- floor(u*(BASE - floor(a*u/BASE))/BASE)*/ @\
        mulhwu y,  x,  u                                                    @\
        add    y,  x,  y                                                    @\
        add    u,  u,  y   /* u <- valeur suivante                       */ @\
        b      7b                                                           @\
8:                                                                          @\
        /* ici a*u >= BASE^2 : le quotient cherch est u-1 ou u-2        */ @\
        mullw  y,  a,  u   /* y <- (a*u) mod BASE                        */ @\
        subfc  y,  a,  y   /* CA <- 0 ssi c est u-1                      */ @\
        subfe  y,  y,  y   /* r3 <- CA + 1                               */ @\
        addi   y,  y,  2                                                    @\
        subf   u,  y,  u   /* u <- rsultat final                        */ @\
9:


                       ; +---------------------------+
                       ; |  Division 64 bits par 32  |
                       ; +---------------------------+

; entre :
;    a = entier naturel tel que a >= BASE/2
;    b = floor((BASE^2-1)/a) - BASE
;    d:c = entier naturel tel que d <= a
;    x = scratch
;
; sortie :
;    q   <- max(floor(d:c/a), BASE-1)
;    d:c <- d:c - q*a

#define DIV(c,d,a,b,q,x) \
	mulhwu   q,  b,  d   /* q <- quotient sous estim d au plus 3 */ @\
	add      q,  q,  d                                               @\
	mullw    x,  q,  a   /* d:c <- d:c - q*a                      */ @\
	subfc    c,  x,  c                                               @\
	mulhwu   x,  q,  a                                               @\
	subfe    d,  x,  d                                               @\
                                                                         @\
	/* corrige la division tant que d:c >= a et q < BASE-1        */ @\
8:                                                                       @\
	addic.   q,  q,  1                                               @\
	beq      9f                                                      @\
	subfc    c,  a,  c                                               @\
	addme.   d,  d                                                   @\
	bge      8b                                                      @\
                                                                         @\
	/* restaure le reste et dcrmente q */                          @\
	addc     c,  a,  c                                               @\
	addze    d,  d                                                   @\
9:                                                                       @\
	subi     q,  q,  1

                         ; +-------------------------+
                         ; |  Division  un chiffre  |
                         ; +-------------------------+


; unsigned long xn(div_1)(chiffre *a, long la, unsigned long b, chiffre *c)
;
; entre :
; a = naturel de longueur la >= 0
; b = long > 0
; c = naturel de longueur la, peut tre confondu avec a
;
; sortie :
; c <- floor(a/b)
; retourne a mod b
        
#ifdef assembly_sn_div_1
#define L(x) .Lsn_div_1_##x
.globl _sn_div_1
_sn_div_1:

	; variables locales
	#define _a_  r3
	#define _b_  r5
	#define _c_  r6
	#define _ah_ r7   /* chiffre de poids fort de a << s */
	#define _al_ r8   /* chiffre  suivant de a << s      */
	#define _bi_ r9   /* floor((BASE^2-1)/b) - BASE      */
	#define _s_  r10  /* dcalage pour avoir b >= BASE/2 */
	#define _t_  r11  /* dcalage complmentaire         */
	#define _q_  r12  /* chiffre  de poids fort de c     */
	#define _x_  r2   /* scratch                         */
	#define _y_  r4   /* scratch                         */

	mtctr r4		; ctr <- la
	slwi  _x_,  r4,  2
	add   _a_,  _x_,  _a_	; a <- &a[la]
	add   _c_,  _x_,  _c_	; c <- &c[la]

        ; dcale b pour avoir b >= BASE/2, puis l inverse
	cntlzw _s_, _b_
	li    _t_,  32
	subf  _t_,  _s_,  _t_
	slw   _b_,  _b_,  _s_
        INVERSE(_b_,_bi_,_x_,_y_)

	; boucle sur les chiffres de a
	li    _al_, 0		; init retenue
L(loop):
	and.  _y_,  _s_,  _s_	; ah:al <- chiffres de poids fort de a << s
	lwzu  _x_,  -4(_a_)
	beq   1f
	srw   _y_,  _x_,  _t_
1:
	add   _ah_, _al_, _y_
	slw   _al_, _x_,  _s_
        DIV(_al_,_ah_,_b_,_bi_,_q_,_x_)
	stwu  _q_, -4(_c_)
	bdnz  L(loop)

	; dcale le dernier reste
	srw   r3, _al_, _s_
	blr

	#undef _a_
	#undef _b_
	#undef _c_
	#undef _ah_
	#undef _al_
	#undef _bi_
	#undef _s_
	#undef _t_
	#undef _q_
	#undef _x_
	#undef _y_
	
#undef L
#endif /* assembly_sn_div_1 */

; unsigned long xn(mod_1)(chiffre *a, long la, unsigned long b)
;
; entre :
; a = naturel de longueur la >= 0
; b = long > 0
;
; sortie :
; retourne a mod b
        
#ifdef assembly_sn_mod_1
#define L(x) .Lsn_mod_1_##x
.globl _sn_mod_1
_sn_mod_1:

	; variables locales
	#define _a_  r3
	#define _b_  r5
	#define _ah_ r7   /* chiffre de poids fort de a << s */
	#define _al_ r8   /* chiffre  suivant de a << s      */
	#define _bi_ r9   /* floor((BASE^2-1)/b) - BASE      */
	#define _s_  r10  /* dcalage pour avoir b >= BASE/2 */
	#define _t_  r11  /* dcalage complmentaire         */
	#define _q_  r12  /* chiffre  de poids fort de c     */
	#define _x_  r2   /* scratch                         */
	#define _y_  r4   /* scratch                         */

	mtctr r4		; ctr <- la
	slwi  _x_,  r4,  2
	add   _a_,  _x_,  _a_	; a <- &a[la]

        ; dcale b pour avoir b >= BASE/2, puis l inverse
	cntlzw _s_, _b_
	li    _t_,  32
	subf  _t_,  _s_,  _t_
	slw   _b_,  _b_,  _s_
        INVERSE(_b_,_bi_,_x_,_y_)

	; boucle sur les chiffres de a
	li    _al_, 0		; init retenue
L(loop):
	and.  _y_,  _s_,  _s_	; ah:al <- chiffres de poids fort de a << s
	lwzu  _x_,  -4(_a_)
	beq   1f
	srw   _y_,  _x_,  _t_
1:
	add   _ah_, _al_, _y_
	slw   _al_, _x_,  _s_
        DIV(_al_,_ah_,_b_,_bi_,_q_,_x_)
	bdnz  L(loop)

	; dcale le dernier reste
	srw   r3, _al_, _s_
	blr

	#undef _a_
	#undef _b_
	#undef _ah_
	#undef _al_
	#undef _bi_
	#undef _s_
	#undef _t_
	#undef _q_
	#undef _x_
	#undef _y_
	
#undef L
#endif /* assembly_sn_mod_1 */

                         ; +------------------------+
                         ; |  Division quadratique  |
                         ; +------------------------+

; void xn(div_n2)(chiffre *a, long lc, chiffre *b, long lb, chiffre *c)
;
; entre :
; a = naturel de longueur lc+lb
; b = naturel de longueur lb
; c = naturel de longueur lc
;
; contraintes : 
; lb >= 2, lc > 0, le bit de poids fort de b est non nul,
; a < BASE^lc*b
; a,b,c non confondus
;
; sortie :
; a <- a mod b
; c <- floor(a/b)

#ifdef assembly_sn_div_n2
#define L(x) .Lsn_div_n2_##x
#ifdef debug_div_n2
.globl _sn_div_n2_buggy
_sn_div_n2_buggy:
#else
.globl _sn_div_n2
_sn_div_n2:
Lsn_div_n2:
#endif

	; paramtres aprs relogement
	#define _a_     r12
	#define _b_     r11
	#define _c_     r7
	#define _lb_    r6
	#define _lc_    r8

        ; variables locales
        ; al:ah  = chiffres de poids fort de a
        ; bl     = chiffre de poids fort de b
        ; bi     = floor((BASE^2-1)/bh) - BASE
        ; q      = chiffre de poids fort de c
	; add    = point d entre pour addloop
	; msub   = point d entre pour mulsubloop
	#define _al_    r4
	#define _ah_    r5
	#define _bh_    r31
	#define _bi_    r30
	#define _q_     r9
	#define _add_   r29
	#define _msub_  r28

	; mise en place des paramtres
	mflr   r0
	stmw   r28,  4(r1)
	subi   _lc_, r4,   1
	slwi   r2,   _lc_, 2
	add    _a_,  r2,   r3	; a <- &a[lc-1]
	add    _c_,  r2,   r7	; c <- &c[lc-1]
	slwi   r2,   r6,   2
	add    _a_,  r2,   _a_	; a <- &a[lb+lc-1]
	add    _b_,  r2,   r5	; b <- &b[lb]
	lwz    _bh_, -4(_b_)
        INVERSE(_bh_,_bi_,r3,r4)

	; prpare le droulement des boucles internes
	neg    r2,  r6
	addi   _lb_, r6,  31	; lb <- ceil(lb/32)
	srwi   _lb_, _lb_, 5
        clrlslwi r2, r2,   27,2 ; r2 <- 4*((-lb) % 32)
	slwi   _add_, r2,  2	; dcalage pour  addloop    =  4 instructions
	mulli  _msub_,r2,  10	; dcalage pour  mulsubloop = 10 instructions
	bcl    20,31, L(here)   ; ajoute les adresses de base
L(here):
	mflr   r2
	add    _add_,  _add_,  r2
	addi   _add_,  _add_,  lo16(Lsn_addloop    - L(here))
/*	addis  _add_,  _add_,  ha16(Lsn_addloop    - L(here)) */
	add    _msub_, _msub_, r2
	addi   _msub_, _msub_, lo16(Lsn_mulsubloop - L(here))
/*	addis  _msub_, _msub_, ha16(Lsn_mulsubloop - L(here)) */

	; boucle sur les chiffres de a
L(loop):
	lwz    _ah_,  0(_a_)
	lwz    _al_, -4(_a_)
        DIV(_al_,_ah_,_bh_,_bi_,_q_,r3)
	slwi   r2,   _lb_, 7
	subf   _a_,  r2,  _a_	; restaure a
	subf   _b_,  r2,  _b_	; restaure b
	mtlr   _msub_		; a <- a - q*b
	li     r3,   0
	mr     r10,  _q_
	mtctr  _lb_
	blrl
	lwz    _ah_, 0(_a_)
	subfc. _ah_, r3,  _ah_
	
        ; corrige le quotient et le reste si ngatif
        beq    L(q_ok)
L(cor1):
        subi   _q_,  _q_,  1	; q--
	slwi   r2,   _lb_, 7
	subf   _a_,  r2,  _a_	; restaure a
	subf   _b_,  r2,  _b_	; restaure b
	mtlr   _add_		; a <- a + b
	mr     r10,  _a_
	mtctr  _lb_
	blrl
	addze. _ah_, _ah_
        bne    L(cor1)

	; chiffre suivant
L(q_ok):
        stw    _ah_,  0(_a_)
        stw    _q_,   0(_c_)    ; c[lc-1] <- q
        subic. _lc_, _lc_,  1
        subi   _c_,  _c_,   4
        subi   _a_,  _a_,   4
        bge    L(loop)

        ; termin
L(done):
        mtlr   r0
	lmw    r28, 4(r1)
	blr
	
	#undef _a_
	#undef _b_
	#undef _c_
	#undef _lb_
	#undef _lc_
	#undef _al_
	#undef _ah_
	#undef _bh_
	#undef _bi_
	#undef _q_
	#undef _add_
	#undef _msub_
	
#undef L
#endif /* assembly_sn_div_n2 */
#if !defined(assembly_sn_div_n2) || defined(debug_div_n2)
REPLACE(sn_div_n2)
#endif
