/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <gtk/gtk.h>

#include <nd_debug.h>
#include <nd_prefs.h>
#include <nd_recent.h>
#include <nd_macros.h>
#include <interface.h>

#define ND_CONFIG_FILE ".netdude"
#define ND_NO_SAVE_DIR "(none)"


typedef struct nd_preferences
{
  int     trace_win_size;
  char   *tcpdump_path;
  int     show_splash;
  int     show_full_path;
  
  int     show_timestamps;
  int     show_timestamps_absolute;
  gfloat  timestamps_delay;

  char   *load_dir;
  char   *save_dir;

  char   *recent[NUM_RECENT];
}
ND_Preferences;


static GtkWidget *prefs_dialog = NULL;
static ND_Preferences prefs;
static ND_Preferences prefs_backup;

static char *prefs_config_file(void);
static int   prefs_read_config_file(ND_Preferences *p);
static int   prefs_write_config_file(ND_Preferences *p);
static int   prefs_init_defaults(ND_Preferences *p);
static void  prefs_sync_app_to_settings(void);
static void  prefs_sync_settings_to_dialog(void);
static void  prefs_sync_dialog_to_settings(void);

static char *
prefs_config_file(void)
{
  struct stat st;
  static char cfg_file[MAXPATHLEN] = "\0";

  if (cfg_file[0] != 0)
    return cfg_file;

  snprintf(cfg_file, MAXPATHLEN, "%s/%s", getenv("HOME"), ND_CONFIG_FILE);

  if (stat(cfg_file, &st) < 0)
    return NULL;

  return cfg_file;
}


static int
prefs_read_config_file(ND_Preferences *p)
{
  char *cfg_file = NULL;
  char s[MAXPATHLEN];
  FILE *f;

  if (!p)
    return (FALSE);

  if ((cfg_file = prefs_config_file()) == NULL)
    return (FALSE);

  /* Cleanup things properly */
  FREE(p->tcpdump_path);
  FREE(p->load_dir);
  FREE(p->save_dir);

  prefs_init_defaults(p);

  if ((f = fopen(cfg_file, "r")) == NULL)
    {
      return (FALSE);
    }

  for ( ; ; )
    {
      if (fscanf(f, "%s", s) == EOF)
	break;

      if (!strcmp(s, "trace_win_size"))
	{
	  fscanf(f, "%i\n", &(p->trace_win_size));
	}
      else if (!strcmp(s, "tcpdump_path"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->tcpdump_path);
	  p->tcpdump_path = strdup(s);
	}
      else if (!strcmp(s, "show_timestamps"))
	{
	  fscanf(f, "%i\n", &(p->show_timestamps));
	}
      else if (!strcmp(s, "show_timestamps_absolute"))
	{
	  fscanf(f, "%i\n", &(p->show_timestamps_absolute));
	}
      else if (!strcmp(s, "timestamps_delay"))
	{
	  fscanf(f, "%f\n", &(p->timestamps_delay));
	}
      else if (!strcmp(s, "load_dir"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->load_dir);
	  p->load_dir = strdup(s);
	}
      else if (!strcmp(s, "save_dir"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->save_dir);
	  p->save_dir = strdup(s);
	}
      else if (!strcmp(s, "show_splash"))
	{
	  fscanf(f, "%i\n", &(p->show_splash));
	}
      else if (!strcmp(s, "show_full_path"))
	{
	  fscanf(f, "%i\n", &(p->show_full_path));
	}
      else if (!strcmp(s, "recent_0"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->recent[0]);
	  p->recent[0] = strdup(s);
	}
      else if (!strcmp(s, "recent_1"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->recent[1]);
	  p->recent[1] = strdup(s);
	}
      else if (!strcmp(s, "recent_2"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->recent[2]);
	  p->recent[2] = strdup(s);
	}
      else if (!strcmp(s, "recent_3"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->recent[3]);
	  p->recent[3] = strdup(s);
	}
      else if (!strcmp(s, "recent_4"))
	{
	  fscanf(f, "%s\n", s);
	  FREE(p->recent[4]);
	  p->recent[4] = strdup(s);
	}
    }

  fclose(f);
  return (TRUE);
}


static int
prefs_write_config_file(ND_Preferences *p)
{
  int   i;
  FILE *f;

  if (!p)
    return (FALSE);

  if ((f = fopen(prefs_config_file(), "w")) == NULL)
    return (FALSE);

  fprintf(f, "trace_win_size \t\t\t\t %i\n", p->trace_win_size);
  fprintf(f, "tcpdump_path \t\t\t\t %s\n", p->tcpdump_path);
  fprintf(f, "show_splash \t\t\t\t %i\n", p->show_splash);
  fprintf(f, "show_timestamps \t\t\t %i\n", p->show_timestamps);
  fprintf(f, "show_timestamps_absolute \t\t %i\n", p->show_timestamps_absolute);
  fprintf(f, "show_full_path \t\t %i\n", p->show_full_path);
  fprintf(f, "timestamps_delay \t\t\t %f\n", p->timestamps_delay);

  if (!p->load_dir)
    fprintf(f, "load_dir \t\t\t\t %s\n", ND_NO_SAVE_DIR);
  else
    fprintf(f, "load_dir \t\t\t\t %s\n", p->load_dir);

  if (!p->save_dir)
    fprintf(f, "save_dir \t\t\t\t %s\n", ND_NO_SAVE_DIR);
  else
    fprintf(f, "save_dir \t\t\t\t %s\n", p->save_dir);

  for (i = 0; i < NUM_RECENT; i++)
    {
      if (p->recent[i])
	fprintf(f, "recent_%i \t\t\t\t %s\n", i, p->recent[i]);
    }
  
  fclose(f);
  return (TRUE);
}


static int
prefs_init_defaults(ND_Preferences *p)
{
  int  i;
  char s[MAXPATHLEN];

  snprintf(s, MAXPATHLEN, "%s/", getenv("HOME"));
  memset(p, 0, sizeof(ND_Preferences));

  p->trace_win_size = 1000;
  p->tcpdump_path = strdup(TCPDUMP);
  p->show_splash = TRUE;

  p->show_timestamps = TRUE;
  p->show_timestamps_absolute = FALSE;
  p->show_full_path = TRUE;
  p->timestamps_delay = 0.2;

  p->load_dir = strdup(s);
  p->save_dir = NULL;

  for (i = 0; i < NUM_RECENT; i++)
    p->recent[0] = NULL;

  return (TRUE);
}


static void  
prefs_sync_app_to_settings(void)
{
}


static void  
prefs_sync_settings_to_dialog(void)
{
  GtkWidget *w;

  D_ASSERT(prefs_dialog);

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "trace_win_size_spin");
  D_ASSERT(w);
  prefs.trace_win_size = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_tcpdump_loc_entry");
  D_ASSERT(w);
  FREE(prefs.tcpdump_path);
  prefs.tcpdump_path = strdup(gtk_entry_get_text(GTK_ENTRY(w)));

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_timestamps_checkbutton");
  D_ASSERT(w);
  prefs.show_timestamps = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_absolute_checkbutton");
  D_ASSERT(w);
  prefs.show_timestamps_absolute = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_full_path_checkbutton");
  D_ASSERT(w);
  prefs.show_full_path = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_splash_checkbutton");
  D_ASSERT(w);
  prefs.show_splash = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  
  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_timestamps_delay_spinbutton");
  D_ASSERT(w);
  prefs.timestamps_delay = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w));

}


static void
prefs_sync_dialog_to_settings(void)
{
  int        val;
  GtkWidget *w;

  if (!prefs_dialog)
    return;

  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "trace_win_size_spin")))
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), prefs.trace_win_size); 

  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_tcpdump_loc_entry")))
    {
      char tcpdump_path[MAXPATHLEN];

      if (nd_prefs_get_item("tcpdump_path", tcpdump_path))
	gtk_entry_set_text(GTK_ENTRY(w), tcpdump_path);
    }

  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_timestamps_checkbutton")))
    {
      if (nd_prefs_get_item("show_timestamps", &val))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val);
    }

  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_absolute_checkbutton")))
    {
      if (nd_prefs_get_item("show_timestamps_absolute", &val))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val);
    }

  
  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_splash_checkbutton")))
    {
      if (nd_prefs_get_item("show_splash", &val))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val);
    }

  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_show_full_path_checkbutton")))
    {
      if (nd_prefs_get_item("show_full_path", &val))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val);
    }
  
  if ((w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "pref_timestamps_delay_spinbutton")))
    {
      gfloat delay;

      if (nd_prefs_get_item("timestamps_delay", &delay))
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), delay);
    }

}


int
nd_prefs_init(void)
{
  int result = TRUE;
  char *cfg_file = NULL;

  memset(&prefs, 0, sizeof(ND_Preferences));

  if ((cfg_file = prefs_config_file()) != NULL)
    result = prefs_read_config_file(&prefs);
  else
    {
      result = prefs_init_defaults(&prefs);
      prefs_write_config_file(&prefs);
    }

  return (result);
}


int
nd_prefs_save(void)
{
  return (prefs_write_config_file(&prefs));
}


void
nd_prefs_save_general(void)
{
  ND_Preferences p;
  int i;

  /* Load the preferences as currently on disk,
     and overwrite the user-independent settings.
     Then write back to disk.
  */

  memset(&p, 0, sizeof(ND_Preferences));
  prefs_read_config_file(&p);

  FREE(p.load_dir);
  FREE(p.save_dir);
  p.load_dir = prefs.load_dir;
  p.save_dir = prefs.save_dir;
  
  for (i = 0; i < NUM_RECENT; i++)
    p.recent[i] = prefs.recent[i];
;
  /* Needn't free the strings here */

  prefs_write_config_file(&p);
}


int            
nd_prefs_get_item(const char *key, void *data)
{
  int result = TRUE;

  if (!key || key[0] == '\0' || !data)
    return (FALSE);

  if (!strcmp(key, "trace_win_size"))
    {
      int *i = (int*)data;

      *i = prefs.trace_win_size;
    }
  else if (!strcmp(key, "tcpdump_path"))
    {
      char *s = (char*)data;

      if (prefs.tcpdump_path)
	snprintf(s, MAXPATHLEN, prefs.tcpdump_path);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "load_dir"))
    {
      char *s = (char*)data;
      
      if (prefs.load_dir)
	snprintf(s, MAXPATHLEN, prefs.load_dir);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "save_dir"))
    {
      char *s = (char*)data;

      if (prefs.save_dir)
	snprintf(s, MAXPATHLEN, prefs.save_dir);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "show_timestamps"))
    {
      int *i = (int*)data;      
      *i = prefs.show_timestamps;
    }
  else if (!strcmp(key, "show_timestamps_absolute"))
    {
      int *i = (int*)data;      
      *i = prefs.show_timestamps_absolute;
    }
  else if (!strcmp(key, "show_splash"))
    {
      int *i = (int*)data;      
      *i = prefs.show_splash;
    }
  else if (!strcmp(key, "show_full_path"))
    {
      int *i = (int*)data;      
      *i = prefs.show_full_path;
    }
  else if (!strcmp(key, "timestamps_delay"))
    {
      gfloat *d = (gfloat*)data;      
      *d = prefs.timestamps_delay;
    }
  else if (!strcmp(key, "recent_0"))
    {
      char *s = (char*)data;

      if (prefs.recent[0])
	snprintf(s, MAXPATHLEN, prefs.recent[0]);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "recent_1"))
    {
      char *s = (char*)data;

      if (prefs.recent[1])
	snprintf(s, MAXPATHLEN, prefs.recent[1]);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "recent_2"))
    {
      char *s = (char*)data;

      if (prefs.recent[2])
	snprintf(s, MAXPATHLEN, prefs.recent[2]);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "recent_3"))
    {
      char *s = (char*)data;

      if (prefs.recent[3])
	snprintf(s, MAXPATHLEN, prefs.recent[3]);
      else
	result = FALSE;
    }
  else if (!strcmp(key, "recent_4"))
    {
      char *s = (char*)data;

      if (prefs.recent[4])
	snprintf(s, MAXPATHLEN, prefs.recent[4]);
      else
	result = FALSE;
    }
  else
    {
      result = FALSE;
    }

  return (result);
}


int            
nd_prefs_set_item(const char *key, void *data)
{
  int result = TRUE;

  if (!key || key[0] == '\0' || !data)
    return (FALSE);

  if (!strcmp(key, "trace_win_size"))
    {
      int *i = (int*)data;
      prefs.trace_win_size = *i;
    }
  else if (!strcmp(key, "tcpdump_path"))
    {
      char *s = (char*)data;
      FREE(prefs.tcpdump_path);
      prefs.tcpdump_path = strdup(s);
    }
  else if (!strcmp(key, "load_dir"))
    {
      char *s = (char*)data;
      FREE(prefs.load_dir);
      prefs.load_dir = strdup(s);
    }
  else if (!strcmp(key, "save_dir"))
    {
      char *s = (char*)data;
      FREE(prefs.save_dir);
      prefs.save_dir = strdup(s);
    }
  else if (!strcmp(key, "show_timestamps"))
    {
      int *i = (int*)data;
      prefs.show_timestamps = *i;
    }
  else if (!strcmp(key, "show_timestamps_absolute"))
    {
      int *i = (int*)data;
      prefs.show_timestamps_absolute = *i;
    }
  else if (!strcmp(key, "show_splash"))
    {
      int *i = (int*)data;
      prefs.show_splash = *i;
    }
  else if (!strcmp(key, "show_full_path"))
    {
      int *i = (int*)data;
      prefs.show_full_path = *i;
    }
  else if (!strcmp(key, "timestamps_delay"))
    {
      gfloat *d = (gfloat*)data;
      prefs.timestamps_delay = *d;
    }
  else if (!strcmp(key, "recent_0"))
    {
      char *s = (char*)data;
      FREE(prefs.recent[0]);
      prefs.recent[0] = strdup(s);
    }
  else if (!strcmp(key, "recent_1"))
    {
      char *s = (char*)data;
      FREE(prefs.recent[1]);
      prefs.recent[1] = strdup(s);
    }
  else if (!strcmp(key, "recent_2"))
    {
      char *s = (char*)data;
      FREE(prefs.recent[2]);
      prefs.recent[2] = strdup(s);
    }
  else if (!strcmp(key, "recent_3"))
    {
      char *s = (char*)data;
      FREE(prefs.recent[3]);
      prefs.recent[3] = strdup(s);
    }
  else if (!strcmp(key, "recent_4"))
    {
      char *s = (char*)data;
      FREE(prefs.recent[4]);
      prefs.recent[4] = strdup(s);
    }
  else
    {
      result = FALSE;
    }

  prefs_sync_dialog_to_settings();

  return (result);
}

void     
nd_prefs_dialog_show(void)
{
  memcpy(&prefs_backup, &prefs, sizeof(ND_Preferences));

  if (!prefs_dialog)
    prefs_dialog = create_pref_dialog();

  prefs_sync_dialog_to_settings();
  gtk_widget_show(prefs_dialog);
}


void     
nd_prefs_dialog_ok(void)
{
  prefs_sync_settings_to_dialog();
  prefs_sync_app_to_settings();

  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}


void     
nd_prefs_dialog_save(void)
{
  prefs_sync_settings_to_dialog();
  prefs_write_config_file(&prefs);
  gtk_widget_hide(prefs_dialog);
}


void     
nd_prefs_dialog_cancel(void)
{
  memcpy(&prefs, &prefs_backup, sizeof(ND_Preferences));

  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}

