/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <net/ethernet.h>
#include <net/if_arp.h>
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include <gtk/gtk.h>
#include <callbacks.h>
#include <nd.h>
#include <nd_globals.h>
#include <nd_gui.h>
#include <nd_misc.h>
#include <nd_packet.h>
#include <nd_tracefile.h>
#include <support.h>

#define FDDI_HDR_LEN    13
#define SNAP_LEN        8

struct fddi_header {
  u_char fddi_fc;
  u_char fddi_dhost[6];
  u_char fddi_shost[6];
};

struct snap_header
{
  u_char dsap;
  u_char ssap;
  u_char ctrl;
  u_char snap[5];
};


static void
packet_get_linklevel_protocol_string(char *string, int strlen,
				     short unsigned int prot)
{
  switch (prot)
    {
    case ETHERTYPE_PUP:
      snprintf(string, strlen, _("Prot (Xerox PUP)"));
      break;
    case ETHERTYPE_IP:
      snprintf(string, strlen, _("Prot (IP)"));
      break;
    case ETHERTYPE_ARP:
      snprintf(string, strlen, _("Prot (ARP)"));
      break;
    case ETHERTYPE_REVARP:
      snprintf(string, strlen, _("Prot (RARP)"));
      break;
    default:
      snprintf(string, strlen, _("Prot (%hi)"), prot);
    }
}


void 
nd_packet_set_link_level(ND_Packet *p, int upward)
{
  GtkWidget   *w;
  char         s[1024];

  if (!p)
    return;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);

  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_FDDI));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_FDDI);

  switch (p->link_prot)
    {
    case ND_PROT_ETHER:
      {
	struct ether_header *eh;
	
	eh = (struct ether_header*)p->link_data;
	
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "fddi_tab");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(w), _("Ethernet"));

	sprintf(s, _("Dest. Addr. (%x:%x:%x:%x:%x:%x)"),
		eh->ether_dhost[0]&0xff, eh->ether_dhost[1]&0xff, 
		eh->ether_dhost[2]&0xff, eh->ether_dhost[3]&0xff, 
		eh->ether_dhost[4]&0xff, eh->ether_dhost[5]&0xff);
	
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "link_dest_addr");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
	
	sprintf(s, _("Src. Addr. (%x:%x:%x:%x:%x:%x)"),
		eh->ether_shost[0]&0xff, eh->ether_shost[1]&0xff, 
		eh->ether_shost[2]&0xff, eh->ether_shost[3]&0xff, 
		eh->ether_shost[4]&0xff, eh->ether_shost[5]&0xff);
	
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "link_source_addr");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

	packet_get_linklevel_protocol_string(s, 1024, ntohs(eh->ether_type));

	w = gtk_object_get_data(GTK_OBJECT(toplevel), "link_type");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
      }
      break;
    case ND_PROT_FDDI:
      {
	struct fddi_header  *fh;
	struct snap_header  *sh;
	
	fh = (struct fddi_header*)p->link_data;
	sh = (struct snap_header*)(p->link_data + FDDI_HDR_LEN);
	
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "fddi_tab");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), _("FDDI"));
      }
      break;
    case ND_PROT_SLIP:
      {
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "fddi_tab");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), _("SLIP"));
      }
      break;
    case ND_PROT_PPP:
      {
	w = gtk_object_get_data(GTK_OBJECT(toplevel), "fddi_tab");
	D_ASSERT(w);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), _("PPP"));
      }
      break;
    default:
      w = gtk_object_get_data(GTK_OBJECT(toplevel), "fddi_tab");
      D_ASSERT(w);
      gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), _("Unknown"));
    }

  if (upward)
    {
      switch (p->net_prot)
	{
	case ND_PROT_IP:
	  nd_packet_set_ip(p, 1);
	  break;
	case ND_PROT_ARP:
	  nd_packet_set_arp(p);
	  break;
	case ND_PROT_RARP:
	  nd_packet_set_rarp(p);
	  break;
	default:
	}
    }
  else
    nd_trace_update_tcpdump_packet(p);
}


void 
nd_packet_set_ip(ND_Packet *p, int upward)
{
  GtkWidget   *w;
  char         s[1024];
  struct ip   *iphdr;
  int          opts_len, opts_done, o_len;
  u_char      *opts_p;

  /* If you take the red pill, you'll see that this code is boring
     and painful. If you take the blue one, you'll find yourself
     at the end of this function and will never think of it again,
     hopfully :)
  */

  if (!p)
    return;

  iphdr = nd_packet_ip(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1"); 
  D_ASSERT(w);
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_IP));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_IP);
  
  sprintf(s, _("IP Version(%u)"), iphdr->ip_v);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_v");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("Hdr Len (%u)"), 4 * iphdr->ip_hl);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_hl");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  if (iphdr->ip_tos & IPTOS_LOWDELAY)
    sprintf(s, _("ToS (Min. Delay)"));
  else if (iphdr->ip_tos & IPTOS_THROUGHPUT)
    sprintf(s, _("ToS (Max. Throughput)"));
  else if (iphdr->ip_tos & IPTOS_RELIABILITY)
    sprintf(s, _("ToS (Max. Reliability)"));
  else if (iphdr->ip_tos & IPTOS_MINCOST)
    sprintf(s, _("ToS (Min. Cost)"));
  else
    sprintf(s, _("ToS (%u)"), iphdr->ip_tos);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_tos");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("Total Length (%u)"), htons(iphdr->ip_len));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_len");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("IP ID (%u)"), htons(iphdr->ip_id));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_id");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_rf");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_rf_toggled, NULL);
  if (htons(iphdr->ip_off) & IP_RF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_rf_toggled, NULL);
  
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_df");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_df_toggled, NULL);
  if (htons(iphdr->ip_off) & IP_DF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_df_toggled, NULL);
  
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_mf");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_mf_toggled, NULL);
  if (htons(iphdr->ip_off) & IP_MF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_mf_toggled, NULL);
  
  sprintf(s, _("Fragment Offset (%u)"), (ntohs(iphdr->ip_off) & IP_OFFMASK) * 8);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_off");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("Time To Live (%u)"), iphdr->ip_ttl);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_ttl");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  switch (iphdr->ip_p)
    {
    case IPPROTO_ICMP:
      sprintf(s, _("Protocol (ICMP)"));
      break;
    case IPPROTO_IGMP:
      sprintf(s, _("Protocol (IGMP)"));
      break;
    case IPPROTO_IPIP:
      sprintf(s, _("Protocol (IP in IP)"));
      break;
    case IPPROTO_TCP:
      sprintf(s, _("Protocol (TCP)"));
      break;
    case IPPROTO_EGP:
      sprintf(s, _("Protocol (EGP)"));
      break;
    case IPPROTO_UDP:
      sprintf(s, _("Protocol (UDP)"));
      break;
    case IPPROTO_IPV6:
      sprintf(s, _("Protocol (IPv6)"));
      break;
    case IPPROTO_ROUTING:
      sprintf(s, _("Protocol (IPv6 Rtg.)"));
      break;
    case IPPROTO_FRAGMENT:
      sprintf(s, _("Protocol (IPv6 Frg.)"));
      break;
    case IPPROTO_RSVP:
      sprintf(s, _("Protocol (RSVP)"));
      break;
    case IPPROTO_ICMPV6:
      sprintf(s, _("Protocol (ICMPv6)"));
      break;
    case IPPROTO_NONE:
      sprintf(s, _("Protocol (NONE)"));
      break;
    case IPPROTO_RAW:
      sprintf(s, _("Protocol (RAW)"));
      break;
    default:
      sprintf(s, _("Protocol (%u)"), iphdr->ip_p);
      break;
    }
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_p");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("Header Checksum (%u)"), htons(iphdr->ip_sum));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_sum");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  if ((iphdr->ip_sum != nd_misc_ip_checksum(p)) &&
      p->is_complete)
    nd_misc_widget_set_red(w, TRUE);
  else
    nd_misc_widget_set_red(w, FALSE);
  
  sprintf(s, _("Source Address (%s)"), inet_ntoa(iphdr->ip_src));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_src");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  sprintf(s, _("Destination Address (%s)"), inet_ntoa(iphdr->ip_dst));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "ip_dst");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  nd_gui_ip_table_clear_options();
  
  /* Setup options GUI elements manually: */
  
  opts_len = (iphdr->ip_hl*4) - 20;
  opts_done = 0;
  
  if (opts_len > 0)
    {
      opts_p = p->net_data + 20;
      
      while (opts_done < opts_len)
	{
	  switch(*opts_p)
	    {
	    case 0:
	      /* end of option list - RFC791 */
	      opts_done = opts_len;
	      break;
	    case 1:
	      /* No op */
	      opts_p++;
	      opts_done++;
	      nd_gui_ip_table_add_option("N", _("No Option"),  1);	    
	      break;
	    case 130:
	    case 133:
	      /*Security - see RFC1108*/
	      /*we sanity check this, but otherwise pass it normally*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("Sec", _("Security"), o_len);	    
	      break;
	    case 131:
	      /*Loose Source and Record Route - RFC791*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("LS&RR", _("Loose source route / Record route"), o_len);	    
	      break;
	    case 137:
	      /*Strict Source and Record Route - RFC791*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("SS&RR", _("Strict source route / Record route"), o_len);	    
	      break;
	    case 7:
	      /*Record Route - RFC791*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("RR", _("Record Route"), o_len);	    
	      break;
	    case 136:
	      /*Stream ID - RFC791*/
	      /*we sanity check this, but otherwise pass it normally*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("SID", _("Stream ID"), o_len);	    
	      break;
	    case 68:
	      /*Internet timestamp - RFC791*/
	      /*harmless...*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("TS", _("Internet Timestamp"), o_len);	    
	      break;
	    case 148:
	      /*Router Alert - See RFC2113*/
	      /*we sanity check this, but otherwise pass it normally*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("RA", _("Router Alert"), o_len);	    
	      break;
	    default:
	      opts_p++;
	      o_len = *opts_p;
	      opts_p += o_len-1;
	      opts_done += o_len;
	      nd_gui_ip_table_add_option("?", _("Unknown"), o_len);	    
	    }

	  if (opts_done > opts_len)
	    {
	      printf("Warning -- bogus options. ...\n");
	    }
	}
    }      

  /* If requested and this is a first fragment
     (or normal packet), set contents too: */
  if ((ntohs(iphdr->ip_off) & IP_OFFMASK) == 0 &&
      (ntohs(iphdr->ip_off) & IP_MF) == 0      &&
      upward)
    {
      switch (iphdr->ip_p)
	{
	case IPPROTO_ICMP:
	  nd_packet_set_icmp(p);
	  break;
	case IPPROTO_TCP:
	  nd_packet_set_tcp(p, 1);
	  break;
	case IPPROTO_UDP:
	  nd_packet_set_udp(p, 1);
	  break;
	default:
	}
    }
  else
    nd_trace_update_tcpdump_packet(p);
}



/* This function sets the gui items of the IP header in an
   ICMP error message. It does not adjust the other elements
   in the ICMP error tab, that's done by nd_packet_set_icmp()
   below.
*/

void 
nd_packet_set_icmp_error_ip(ND_Packet *p)
{
  GtkWidget      *w;
  char            s[1024];
  struct ip      *iphdr;
  struct udphdr  *udphdr;
  int             opts_len, opts_done, o_len;
  u_char         *opts_p;

  /* If you take the red pill, you'll see that this code is boring
     and painful. If you take the blue one, you'll find yourself
     at the end of this function and will never think of it again,
     hopfully :)  --cK.
  */

  if (!p)
    return;

  iphdr  = nd_packet_icmp_error_ip(p);
  udphdr = nd_packet_icmp_error_udp(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  gtk_widget_hide(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ICMP));
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ICMP_ERROR));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ICMP_ERROR);

  sprintf(s, _("IP Version(%u)"), iphdr->ip_v);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_v");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Hdr Len (%u)"), 4 * iphdr->ip_hl);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_hl");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if (iphdr->ip_tos & IPTOS_LOWDELAY)
    sprintf(s, _("ToS (Min. Delay)"));
  else if (iphdr->ip_tos & IPTOS_THROUGHPUT)
    sprintf(s, _("ToS (Max. Throughput)"));
  else if (iphdr->ip_tos & IPTOS_RELIABILITY)
    sprintf(s, _("ToS (Max. Reliability)"));
  else if (iphdr->ip_tos & IPTOS_MINCOST)
    sprintf(s, _("ToS (Min. Cost)"));
  else
    sprintf(s, _("ToS (%u)"), iphdr->ip_tos);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_tos");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Total Length (%u)"), htons(iphdr->ip_len));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_len");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("IP ID (%u)"), htons(iphdr->ip_id));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_id");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_rf");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_rf_toggled,
				   GINT_TO_POINTER(ND_PROT_ICMP));

  if (htons(iphdr->ip_off) & IP_RF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_rf_toggled,
				     GINT_TO_POINTER(ND_PROT_ICMP));

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_df");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_df_toggled,
				   GINT_TO_POINTER(ND_PROT_ICMP));
  if (htons(iphdr->ip_off) & IP_DF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_df_toggled,
				     GINT_TO_POINTER(ND_PROT_ICMP));

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_mf");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_ip_mf_toggled,
				   GINT_TO_POINTER(ND_PROT_ICMP));
  if (htons(iphdr->ip_off) & IP_MF)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_ip_mf_toggled,
				     GINT_TO_POINTER(ND_PROT_ICMP));

  sprintf(s, _("Fragment Offset (%u)"), (ntohs(iphdr->ip_off) & IP_OFFMASK) * 8);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_off");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Time To Live (%u)"), iphdr->ip_ttl);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_ttl");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  switch (iphdr->ip_p)
    {
    case IPPROTO_ICMP:
      sprintf(s, _("Protocol (ICMP)"));
      break;
    case IPPROTO_IGMP:
      sprintf(s, _("Protocol (IGMP)"));
      break;
    case IPPROTO_IPIP:
      sprintf(s, _("Protocol (IP in IP)"));
      break;
    case IPPROTO_TCP:
      sprintf(s, _("Protocol (TCP)"));
      break;
    case IPPROTO_EGP:
      sprintf(s, _("Protocol (EGP)"));
      break;
    case IPPROTO_UDP:
      sprintf(s, _("Protocol (UDP)"));
      break;
    case IPPROTO_IPV6:
      sprintf(s, _("Protocol (IPv6)"));
      break;
    case IPPROTO_ROUTING:
      sprintf(s, _("Protocol (IPv6 Rtg.)"));
      break;
    case IPPROTO_FRAGMENT:
      sprintf(s, _("Protocol (IPv6 Frg.)"));
      break;
    case IPPROTO_RSVP:
      sprintf(s, _("Protocol (RSVP)"));
      break;
    case IPPROTO_ICMPV6:
      sprintf(s, _("Protocol (ICMPv6)"));
      break;
    case IPPROTO_NONE:
      sprintf(s, _("Protocol (NONE)"));
      break;
    case IPPROTO_RAW:
      sprintf(s, _("Protocol (RAW)"));
      break;
    default:
      sprintf(s, _("Protocol (%u)"), iphdr->ip_p);
      break;
    }
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_p");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Header Checksum (%u)"), htons(iphdr->ip_sum));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_sum");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if ((iphdr->ip_sum != nd_misc_icmp_error_ip_checksum(p)) &&
      p->is_complete)
    nd_misc_widget_set_red(w, TRUE);
  else
    nd_misc_widget_set_red(w, FALSE);

  sprintf(s, _("Source Address (%s)"), inet_ntoa(iphdr->ip_src));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_src");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Destination Address (%s)"), inet_ntoa(iphdr->ip_dst));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_ip_dst");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Source Port (%u)"), ntohs(udphdr->uh_sport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_src_port");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Dest Port (%u)"), ntohs(udphdr->uh_dport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_dst_port");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);  
}


/* This function sets the GUI fields for ICMP messages. It sets them
   both in the tab for ICMP queries and ICMP errors -- when an ICMP
   error is recognized, it calls nd_packet_set_icmp_error_ip() to
   set those fields correctly as well.
*/
void        
nd_packet_set_icmp(ND_Packet *p)
{
  GtkWidget     *w;
  char           s[1024];
  struct icmp   *icmphdr;

  if (!p)
    return;

  icmphdr = nd_packet_icmp(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ICMP));
  gtk_widget_hide(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ICMP_ERROR));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ICMP);

  switch (icmphdr->icmp_type)
    {
    case ICMP_ECHOREPLY:
      sprintf(s, _("Type (Echo Reply)"));
      break;
    case ICMP_REDIRECT:
      sprintf(s, _("Type (Redirect)"));
      break;
    case ICMP_ECHO:
      sprintf(s, _("Type (Echo Request)"));
      break;
    case ICMP_UNREACH:
      sprintf(s, _("Type (Dest. Unreachable)"));
      break;
    case ICMP_SOURCEQUENCH:
      sprintf(s, _("Type (Sourcequench)"));
      break;
    case ICMP_ROUTERADVERT:
      sprintf(s, _("Type (Router Adv.)"));
      break;
    case ICMP_ROUTERSOLICIT:
      sprintf(s, _("Type (Router Sol.)"));
      break;
    case ICMP_TIMXCEED:
      sprintf(s, _("Type (Time Exceeded)"));
      break;
    case ICMP_PARAMPROB:
      sprintf(s, _("Type (Param. Problem)"));
      break;
    case ICMP_TSTAMP:
      sprintf(s, _("Type (Timestamp Req.)"));
      break;
    case ICMP_TSTAMPREPLY:
      sprintf(s, _("Type (Timestamp Reply)"));
      break;
    case ICMP_IREQ:
      sprintf(s, _("Type (Info Req.)"));
      break;
    case ICMP_IREQREPLY:
      sprintf(s, _("Type (Info Reply)"));
      break;
    case ICMP_MASKREQ:
      sprintf(s, _("Type (Addr. Mask Req.)"));
      break;
    case ICMP_MASKREPLY:
      sprintf(s, _("Type (Addr. Mask Reply)"));
      break;
    default:
      sprintf(s, _("Type (%u)"), icmphdr->icmp_type);
    }

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_type");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_type");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  switch (icmphdr->icmp_type)
    {
    case ICMP_UNREACH:

      nd_packet_set_icmp_error_ip(p);

      switch (icmphdr->icmp_code)
	{	  
	case ICMP_UNREACH_NET:
	  sprintf(s, _("Code (Net Unr.)"));
	  break;
	case ICMP_UNREACH_HOST:
	  sprintf(s, _("Code (Host Unr.)"));
	  break;
	case ICMP_UNREACH_PROTOCOL:
	  sprintf(s, _("Code (Protocol Unr.)"));
	  break;
	case ICMP_UNREACH_PORT:
	  sprintf(s, _("Code (Port Unr.)"));
	  break;
	case ICMP_UNREACH_NEEDFRAG:
	  sprintf(s, _("Code (Fragm. with DF)"));
	  break;
	case ICMP_UNREACH_SRCFAIL:
	  sprintf(s, _("Code (Source Route Failed)"));
	  break;
	case ICMP_UNREACH_NET_UNKNOWN:
	  sprintf(s, _("Code (Net Unknown)"));
	  break;
	case ICMP_UNREACH_HOST_UNKNOWN:
	  sprintf(s, _("Code (Host Unknown)"));
	  break;
	case ICMP_UNREACH_ISOLATED:
	  sprintf(s, _("Code (Source Host Isolated)"));
	  break;
	case ICMP_UNREACH_NET_PROHIB:
	  sprintf(s, _("Code (Dest. Net Prohib.)"));
	  break;
	case ICMP_UNREACH_HOST_PROHIB:
	  sprintf(s, _("Code (Dest. Host Prohib.)"));
	  break;
	case ICMP_UNREACH_TOSNET:
	  sprintf(s, _("Code (Net Unr. for ToS)"));
	  break;
	case ICMP_UNREACH_TOSHOST:
	  sprintf(s, _("Code (Host Unr. for ToS)"));
	  break;
	case ICMP_UNREACH_FILTER_PROHIB:
	  sprintf(s, _("Code (Prohib. by Filter)"));
	  break;
	case ICMP_UNREACH_HOST_PRECEDENCE:
	  sprintf(s, _("Code (Host Prec. Violation)"));
	  break;
	case ICMP_UNREACH_PRECEDENCE_CUTOFF:
	  sprintf(s, _("Code (Prec. Cutoff)"));
	  break;
	default:
	  sprintf(s, _("Code (%u)!"), icmphdr->icmp_code);
	}
      break;

    case ICMP_REDIRECT:

      nd_packet_set_icmp_error_ip(p);

      switch (icmphdr->icmp_code)
	{	  
	case ICMP_REDIRECT_NET:
	  sprintf(s, _("Code (Red. Net)"));
	  break;
	case ICMP_REDIRECT_HOST:
	  sprintf(s, _("Code (Red. Host)"));
	  break;
	case ICMP_REDIRECT_TOSNET:
	  sprintf(s, _("Code (Red. Net w/ ToS)"));
	  break;
	case ICMP_REDIRECT_TOSHOST:
	  sprintf(s, _("Code (Red. Host w/ ToS)"));
	  break;
	default:
	  sprintf(s, _("Code (%u)!"), icmphdr->icmp_code);
	}
      break;

    case ICMP_TIMXCEED:

      nd_packet_set_icmp_error_ip(p);

      switch (icmphdr->icmp_code)
	{	  
	case ICMP_TIMXCEED_INTRANS:
	  sprintf(s, _("Code (TTL = 0 in Transit)"));
	  break;
	case ICMP_TIMXCEED_REASS:
	  sprintf(s, _("Code (TTL = 0 in Reassembly)"));
	  break;
	default:
	  sprintf(s, _("Code (%u)!"), icmphdr->icmp_code);
	}
      break;

    case ICMP_PARAMPROB:

      nd_packet_set_icmp_error_ip(p);

      switch (icmphdr->icmp_code)
	{
	case 0:
	  sprintf(s, _("Code (IP Header Bad)"));
	  break;
	case ICMP_PARAMPROB_OPTABSENT:
	  sprintf(s, _("Code (Option Missing)"));
	  break;
	default:
	  sprintf(s, _("Code (%u)!"), icmphdr->icmp_code);
	}
      break;
      
    default:
      sprintf(s, _("Code (%u)"), icmphdr->icmp_code);
    }

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_code");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_code");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Checksum (%u)"), ntohs(icmphdr->icmp_cksum));

  if ((icmphdr->icmp_cksum != nd_misc_icmp_checksum(p)) &&
      p->is_complete)
    {
      w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_cksum");
      D_ASSERT(w);
      nd_misc_widget_set_red(w, TRUE);
      gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
      w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_cksum");
      D_ASSERT(w);
      nd_misc_widget_set_red(w, TRUE);
      gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
    }
  else
    {
      w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_cksum");
      D_ASSERT(w);
      nd_misc_widget_set_red(w, FALSE);
      gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
      w = gtk_object_get_data(GTK_OBJECT(toplevel), "icmp_error_cksum");
      D_ASSERT(w);
      nd_misc_widget_set_red(w, FALSE);
      gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
    }

  nd_trace_update_tcpdump_packet(p);
}


void 
nd_packet_set_tcp(ND_Packet *p, int upward)
{
  GtkWidget      *w;
  char            s[1024];
  struct tcphdr  *tcphdr;
  struct servent *srvnt = NULL;
  int             opts_len, opts_done, o_len;
  u_char         *opts_p;

  if (!p)
    return;

  tcphdr = nd_packet_tcp(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_TCP));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_TCP);

  if ( (srvnt = getservbyport(tcphdr->th_sport, "tcp")))
    sprintf(s, _("Source Port (%u, %s)"), htons(tcphdr->th_sport), srvnt->s_name);
  else
    sprintf(s, _("Source Port (%u)"), htons(tcphdr->th_sport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_sport");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if ( (srvnt = getservbyport(tcphdr->th_dport, "tcp")))
    sprintf(s, _("Destination Port (%u, %s)"), htons(tcphdr->th_dport), srvnt->s_name);
  else
    sprintf(s, _("Destination Port (%u)"), htons(tcphdr->th_dport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_dport");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Seq Number (%u)"), (u_int32_t)htonl(tcphdr->th_seq));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_seq");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Ack Number (%u)"), (u_int32_t)htonl(tcphdr->th_ack));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_ack");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Hdr Len (%u)"), (4 * tcphdr->th_off));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_off");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Reserved (%u)"), (tcphdr->th_x2 << 2) + ((tcphdr->th_flags >> 6) & 0x03));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_x2");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_urg_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_urg_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_URG)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_urg_flag_toggled, NULL);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_ack_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_ack_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_ACK)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_ack_flag_toggled, NULL);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_psh_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_psh_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_PUSH)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_psh_flag_toggled, NULL);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_rst_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_rst_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_RST)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_rst_flag_toggled, NULL);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_syn_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_syn_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_SYN)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_syn_flag_toggled, NULL);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_fin_flag");
  D_ASSERT(w);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_th_fin_flag_toggled, NULL);
  if (tcphdr->th_flags & TH_FIN)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_th_fin_flag_toggled, NULL);

  sprintf(s, _("Window Size (%u)"), htons(tcphdr->th_win));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_win");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Checksum (%u)"), htons(tcphdr->th_sum));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_sum");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if ((tcphdr->th_sum != nd_misc_tcp_checksum(p)) &&
      p->is_complete)
    nd_misc_widget_set_red(w, TRUE);
  else
    nd_misc_widget_set_red(w, FALSE);

  sprintf(s, _("Urgent Pointer (%u)"), htons(tcphdr->th_urp));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "th_urp");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  nd_gui_tcp_table_clear_options();

  /* Setup options GUI elements manually: */
  
  opts_len = (tcphdr->th_off*4) - 20;
  opts_done = 0;

  if (opts_len > 0)
    {
      opts_p = p->transp_data + 20;
      
      while (opts_done < opts_len)
	{
	  switch (*opts_p)
	    {
	    case TCPOPT_EOL:
	      opts_done=opts_len;
	      break;
	    case TCPOPT_NOP:
	      opts_p++;
	      opts_done++;
	      nd_gui_tcp_table_add_option("N", _("No Option"), 1);
	      break;
	    case TCPOPT_MAXSEG:
	      opts_p++;
	      o_len = *opts_p;
	      opts_p++;

	      if (o_len != TCPOLEN_MAXSEG)
		sprintf(s, _("MSS(!): %d"), (int)htons(*(u_short*)opts_p)); 
	      else
		sprintf(s, _("MSS: %d"), (int)htons(*(u_short*)opts_p)); 

	      nd_gui_tcp_table_add_option(s, s, o_len);

	      opts_p += o_len-2;
	      opts_done += o_len;
	      break;
	    case TCPOPT_WINDOW:
	      /*WSCALE is only allowed on SYN segments*/
	      opts_p++;
	      o_len = *opts_p;
	      opts_p++;

	      if (o_len != TCPOLEN_WINDOW)
		sprintf(s, _("Wscale(!): %d"), (int)htons(*(u_short*)opts_p)); 
	      else
		sprintf(s, _("Wscale: %d"), (int)htons(*(u_short*)opts_p)); 

	      nd_gui_tcp_table_add_option(s, s, o_len);

	      opts_p += o_len-2;
	      opts_done += o_len;
	      break;
	    case TCPOPT_SACK_PERMITTED:
	      /*SACK_PERMITTED is only allowed on SYN segments*/
	      opts_p++;
	      o_len = *opts_p;

	      if (o_len!=TCPOPT_SACK_PERMITTED)
		nd_gui_tcp_table_add_option("SA", _("SAck perm.(!)"), o_len);
	      else
		nd_gui_tcp_table_add_option("SA", _("SAck perm."), o_len);
	 
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case TCPOPT_SACK:
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("SA", _("SAck Option"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case TCPOPT_TIMESTAMP:
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("TS", _("Timestamp"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case TCPOPT_CC:
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("CC", _("T/TCP Connection Count"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case TCPOPT_CCNEW:
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("CN", _("T/TCP New Connection Count"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case TCPOPT_CCECHO:
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("CC", _("T/TCP Connection Count Echo"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    case 19:
	      /*MD5 SIGNATURE*/
	      /*we have to pass this or routing protocols may break!*/
	      /*XXXX - we need to not apply any other normalizations if this
		option is present*/
	      opts_p++;
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("MD5", _("MD5 Signature"), o_len);
	      opts_p+=o_len-1;
	      opts_done+=o_len;
	      break;
	    default:
	      o_len=*opts_p;
	      nd_gui_tcp_table_add_option("?", _("Unknown"), o_len);
	      opts_done+=o_len;
	      break;
	    }
	  
	  if (opts_done>opts_len)
	    {
	      printf("Warning -- bogus TCP options. ...\n");
	    }
	}
    }

  nd_trace_update_tcpdump_packet(p);

  return;
  upward = 0;
}


void 
nd_packet_set_udp(ND_Packet *p, int upward)
{
  GtkWidget      *w;
  char            s[1024];
  struct udphdr  *udphdr;
  struct servent *srvnt = NULL;

  if (!p)
    return;
  
  udphdr = nd_packet_udp(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_UDP));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_UDP);

  if ( (srvnt = getservbyport(udphdr->uh_sport, "udp")))
    sprintf(s, _("Source Port (%u, %s)"), htons(udphdr->uh_sport), srvnt->s_name);
  else
    sprintf(s, _("Source Port (%u)"), htons(udphdr->uh_sport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "uh_sport");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if ( (srvnt = getservbyport(udphdr->uh_dport, "udp")))
    sprintf(s, _("Destination Port (%u, %s)"), htons(udphdr->uh_dport), srvnt->s_name);
  else
    sprintf(s, _("Destination Port (%u)"), htons(udphdr->uh_dport));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "uh_dport");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("UDP Length (%u)"), htons(udphdr->uh_ulen));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "uh_ulen");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("Checksum (%u)"), htons(udphdr->uh_sum));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "uh_sum");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  if ((udphdr->uh_sum != nd_misc_udp_checksum(p)) &&
      p->is_complete)
    nd_misc_widget_set_red(w, TRUE);
  else
    nd_misc_widget_set_red(w, FALSE);

  nd_trace_update_tcpdump_packet(p);

  return;
  upward = 0;
}


static void
packet_set_arp_internal(ND_Packet *p)
{
  GtkWidget      *w;
  struct arphdr  *arphdr;
  char            s[1024];
  char           *sptr;
  int             i;

  if (!p)
    return;

  arphdr = nd_packet_arp(p);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  gtk_widget_show(gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ARP));
  gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ARP);

  switch (ntohs(arphdr->ar_hrd))
    {
    case ARPHRD_NETROM:
      sprintf(s, _("Hard (NET/ROM)"));
      break;
    case ARPHRD_ETHER:
      sprintf(s, _("Hard (Ethernet)"));
      break;
    case ARPHRD_EETHER:
      sprintf(s, _("Hard (Exp. Ethernet)"));
      break;
    case ARPHRD_AX25:
      sprintf(s, _("Hard (AX.25)"));
      break;
    case ARPHRD_PRONET:
      sprintf(s, _("Hard (PROnet)"));
      break;
    case ARPHRD_CHAOS:
      sprintf(s, _("Hard (Chaosnet)"));
      break;
    case ARPHRD_IEEE802:
      sprintf(s, _("Hard (IEEE 802.2)"));
      break;
    case ARPHRD_ARCNET:
      sprintf(s, _("Hard (ARCnet)"));
      break;
    case ARPHRD_APPLETLK:
      sprintf(s, _("Hard (Appletalk)"));
      break;
    case ARPHRD_DLCI:
      sprintf(s, _("Hard (DLCI)"));
      break;
    case ARPHRD_ATM:
      sprintf(s, _("Hard (ATM)"));
      break;
    case ARPHRD_METRICOM:
      sprintf(s, _("Hard (Metricom)"));
      break;
    default:
      sprintf(s, _("Hard (%i)"), ntohs(arphdr->ar_hrd));
    }

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_hard");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  packet_get_linklevel_protocol_string(s, 1024, ntohs(arphdr->ar_pro));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_prot");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("HS (%i)"), arphdr->ar_hln);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_hard_size");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  sprintf(s, _("PS (%i)"), arphdr->ar_pln);
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_prot_size");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  switch (ntohs(arphdr->ar_op))
    {
    case ARPOP_REQUEST:   
      sprintf(s, _("Op (ARP Req.)"));
      break;
    case ARPOP_REPLY:
      sprintf(s, _("Op (ARP Rep.)"));
      break;
    case ARPOP_RREQUEST:
      sprintf(s, _("Op (RARP Req.)"));
      break;
    case ARPOP_RREPLY:
      sprintf(s, _("Op (RARP Rep.)"));
      break;
    case ARPOP_InREQUEST:
      sprintf(s, _("Op (InARP Req.)"));
      break;
    case ARPOP_InREPLY:
      sprintf(s, _("Op (InARP Rep.)"));
      break;
    case ARPOP_NAK:
      sprintf(s, _("Op (ATM ARP NAK)"));
      break;
    default:
      sprintf(s, _("Op (Unknown)"));
    }

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_op");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);


  nd_gui_arp_table_adjust(arphdr->ar_hln, arphdr->ar_pln);

  sprintf(s, "%.2hhx",  ((char*)arphdr)[8]);
  sptr = s + 2;

  for (i = 1; i < arphdr->ar_hln; i++, sptr += 3)
    sprintf(sptr, ":%.2hhx",  ((char*)arphdr)[8+i]);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_sender_link");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);


  sprintf(s, "%hhu",  ((char*)arphdr)[8 + arphdr->ar_hln]);
  sptr = s + strlen(s);

  for (i = 1; i < arphdr->ar_pln; i++, sptr = s + strlen(s))
    sprintf(sptr, ".%hhu",  ((char*)arphdr)[8 + arphdr->ar_hln + i]);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_sender_net");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);


  sprintf(s, "%.2hhx",  ((char*)arphdr)[8 + arphdr->ar_hln + arphdr->ar_pln]);
  sptr = s + 2;

  for (i = 1; i < arphdr->ar_hln; i++, sptr += 3)
    sprintf(sptr, ":%.2hhx",  ((char*)arphdr)[8 + arphdr->ar_hln + arphdr->ar_pln + i]);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_target_link");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);


  sprintf(s, "%hhu",  ((char*)arphdr)[8 + 2*arphdr->ar_hln + arphdr->ar_pln]);
  sptr = s + strlen(s);

  for (i = 1; i < arphdr->ar_pln; i++, sptr = s + strlen(s))
    sprintf(sptr, ".%hhu",  ((char*)arphdr)[8 + 2*arphdr->ar_hln + arphdr->ar_pln + i]);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_target_net");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);
  
  nd_trace_update_tcpdump_packet(p);
}



void            
nd_packet_set_arp(ND_Packet *p)
{
  GtkWidget      *w;

  if (!p)
    return;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_tab");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(w), _("ARP"));

  packet_set_arp_internal(p);
}


void            
nd_packet_set_rarp(ND_Packet *p)
{
  GtkWidget      *w;

  if (!p)
    return;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "arp_tab");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(w), _("RARP"));

  packet_set_arp_internal(p);
}


void    
nd_packet_set(ND_Packet *p)
{
  GtkWidget  *w, *w2;
  int         current_page, i;

  if (!p)
    return;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
      
  current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(w));

  i = 0;
  w2 = gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), i);

  while (w2)
    {
      gtk_widget_hide(w2);
      i++;
      w2 = gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), i);
    }
      
  trace.p = p;
  
  switch (pcap_datalink(trace.pcap))
    {
    case DLT_NULL:
      p->net_data = p->link_data;
      nd_packet_set_ip(p, 1);
      break;
    default:
      nd_packet_set_link_level(p, 1);
    }

  switch (current_page)
    {
    case ND_TAB_FDDI:
      if (nd_packet_has_protocol(p, ND_PROT_ETHER))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_FDDI);
      break;
    case ND_TAB_ARP:
      if (nd_packet_has_protocol(p, ND_PROT_ARP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ARP);
      break;
    case ND_TAB_IP:
      if (nd_packet_has_protocol(p, ND_PROT_IP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_IP);
      break;
    case ND_TAB_TCP:
      if (nd_packet_has_protocol(p, ND_PROT_TCP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_TCP);
      break;
    case ND_TAB_UDP:
      if (nd_packet_has_protocol(p, ND_PROT_UDP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_UDP);
      break;
    case ND_TAB_ICMP:
      if (nd_packet_has_protocol(p, ND_PROT_ICMP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ICMP);
      break;
    case ND_TAB_ICMP_ERROR:
      if (nd_packet_has_protocol(p, ND_PROT_ICMP_ERROR_IP))
	gtk_notebook_set_page(GTK_NOTEBOOK(w), ND_TAB_ICMP_ERROR);
      break;
    default:
    }
}


ND_Packet  *
nd_packet_new(void)
{
  ND_Packet *p;

  p = (ND_Packet*) malloc(sizeof(ND_Packet));
  bzero(p, sizeof(ND_Packet));
  return (p);
}


void        
nd_packet_free(ND_Packet *p)
{
  u_char   *data;

  if (!p)
    return;

  data = nd_packet_get_data(p, NULL);

  FREE(data);      
  FREE(p);
}


ND_Packet*
nd_packet_duplicate(ND_Packet *p)
{
  ND_Packet    *copy;
  u_char       *data;

  if (!p)
    return (NULL);

  copy = nd_packet_new();
  copy->ph = p->ph;

  /* Copy the packet data and hook it into the right place */
  
  data = (u_char*) malloc(sizeof(u_char) * p->ph.caplen);
  if (p->link_prot)
    {
      memcpy(data, p->link_data, p->ph.caplen);
      copy->link_data = data;
    }
  else if (p->net_prot)
    {
      memcpy(data, p->net_data, p->ph.caplen);
      copy->net_data = data;
    }
  else if (p->transp_prot)
    {
      memcpy(data, p->transp_data, p->ph.caplen);
      copy->transp_data = data;
    }
  else
    {
      memcpy(data, p->app_data, p->ph.caplen);
      copy->app_data = data;
    }

  /* Now make sure the offsets of the upper layers are correct. */

  if (p->link_data)
    {
      if (p->net_data)
	copy->net_data = copy->link_data + (p->net_data - p->link_data);
    }
  if (p->net_data)
    {
      if (p->transp_data)
	copy->transp_data = copy->net_data + (p->transp_data - p->net_data);
    }
  if (p->transp_data)
    {
      if (p->app_data)
	copy->app_data = copy->transp_data + (p->app_data - p->transp_data);
    }

  copy->link_prot   = p->link_prot;
  copy->net_prot    = p->net_prot;
  copy->transp_prot = p->transp_prot;
  copy->app_prot    = p->app_prot;

  copy->is_hidden   = 0;

  return (copy);
}


void        
nd_packet_init(ND_Packet *p)
{
  if (!p)
    return;

  p->link_prot = p->net_prot = p->transp_prot = p->app_prot = ND_PROT_NULL;

  if (p->ph.caplen == p->ph.len)
    p->is_complete = TRUE;
  
  switch (pcap_datalink(trace.pcap))
    {
    case DLT_EN10MB:
      {
	struct ether_header *eh;

	p->link_prot = ND_PROT_ETHER;
	eh = (struct ether_header*)p->link_data;

	/* Is this an Ethernet-II frame or a Ethernet 802.3 SNAP frame?*/
	/* Check ether_type > 1500 --> Ethernet II */
	if (htons(eh->ether_type) <= ETHERMTU)
	  {
	    struct snap_header  *sh;
	    
	    sh = (struct snap_header*)(p->link_data + ETHER_HDR_LEN);

	    if (sh->ssap != 0xaa)
	      {
		/* We cannot handle this one correctly (at least for now),
		   so just interpret the ethernet header and leave the
		   rest unknown.
		*/

		return;
	      }

	    /* Check for IP payload: 0x00-00-08-00 SNAP field */
	    if (((sh->snap[0]) | (sh->snap[1]) | (sh->snap[2]) | (!sh->snap[4])) &&
		(sh->snap[3]==0x08))
	      {
		/* It's a SNAP header containing an IP datagram */		

		p->net_data = p->link_data + ETHER_HDR_LEN+SNAP_LEN;
		p->net_prot = ND_PROT_IP;

		if (!nd_packet_has_complete_header(p, ND_PROT_IP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete IP\n");
		    return;
		  }
	      }
	    else if (((sh->snap[0]) | (sh->snap[1]) | (sh->snap[2])) &&
		     (sh->snap[4]==6) && (sh->snap[3]==8))
	      {
		/* It's a SNAP header containing an ARP packet */

		p->net_data = p->link_data + ETHER_HDR_LEN+SNAP_LEN;
		p->net_prot = ND_PROT_ARP;

		if (!nd_packet_has_complete_header(p, ND_PROT_ARP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete ARP\n");
		    return;
		  }
	      }
	    else if (((sh->snap[0]) | (sh->snap[1]) | (sh->snap[2])) &&
		     (sh->snap[4]==0x35) && (sh->snap[3]==0x80))
	      {
		/* It's a SNAP header containing a RARP packet */

		p->net_data = p->link_data + ETHER_HDR_LEN+SNAP_LEN;
		p->net_prot = ND_PROT_RARP;

		if (!nd_packet_has_complete_header(p, ND_PROT_RARP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete RARP\n");
		    return;
		  }
	      }
	  }
	else 
	  {
	    if (htons(eh->ether_type) == ETHERTYPE_IP)
	      {
		/* It's Ethernet-II containing an IP datagram */
		p->net_data = p->link_data + ETHER_HDR_LEN;
		p->net_prot = ND_PROT_IP;

		if (!nd_packet_has_complete_header(p, ND_PROT_IP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete IP\n");

		    return;
		  }
	      }
	    else if (htons(eh->ether_type) == ETHERTYPE_ARP)
	      {
		p->net_data = p->link_data + ETHER_HDR_LEN;
		p->net_prot = ND_PROT_ARP;

		if (!nd_packet_has_complete_header(p, ND_PROT_ARP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete ARP\n");
		    return;
		  }
	      }
	    else if (htons(eh->ether_type) == ETHERTYPE_REVARP)
	      {
		p->net_data = p->link_data + ETHER_HDR_LEN;
		p->net_prot = ND_PROT_RARP;

		if (!nd_packet_has_complete_header(p, ND_PROT_RARP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete RARP\n");

		    return;
		  }
	      }
	  }
      }
      break;
    case DLT_SLIP:
      {
      }
      break;
    case DLT_PPP:
      {
      }
      break;
    case DLT_FDDI:
      {
	struct fddi_header  *fh;
	struct snap_header  *sh;
	
	p->link_prot = ND_PROT_FDDI;
	fh = (struct fddi_header*)p->link_data;
	sh = (struct snap_header*)(p->link_data + FDDI_HDR_LEN);
	
	/*Is this an FDDI 802.2 or an FDDI SNAP frame?*/
	if ((sh->ssap==0xaa)&&(sh->dsap==0xaa))
	  {
	    /*It's a SNAP frame*/
	    if (((sh->snap[0]) | (sh->snap[1]) | (sh->snap[2]) |
		 (sh->snap[4]==0)) &&(sh->snap[3]==8))
	      {
		/*It's a SNAP header containing an IP datagram*/
		p->net_data = p->link_data + FDDI_HDR_LEN+SNAP_LEN;
		p->net_prot = ND_PROT_IP;

		if (!nd_packet_has_complete_header(p, ND_PROT_IP))
		  {
		    p->net_data = NULL;
		    p->net_prot = ND_PROT_NULL;
		    D("Incomplete IP\n");
		    return;
		  }
	      }
	    else
	      {
		return;
	      }
	  }
	else
	  {
	    printf("XXX we don't handle FDDI 802.2 frames yet\n");
	    return;
	  }
      }
      break;
    case DLT_NULL:
      {
	p->link_prot = ND_PROT_NULL;

	/* We'll assume it's IP for now */
	p->net_data = p->link_data;
	p->net_prot = ND_PROT_IP;

	if (!nd_packet_has_complete_header(p, ND_PROT_IP))
	  {
	    p->net_data = NULL;
	    p->net_prot = ND_PROT_NULL;
	    D("Incomplete IP\n");
	    return;
	  }	
      }
      break;
    default:
    }
  
  switch (p->net_prot)
    {
    case ND_PROT_IP:
      {
	struct ip* iphdr = (struct ip*)p->net_data;
	
	/* If this is a first fragment (or normal packet), check contents: */
	if ((ntohs(iphdr->ip_off) & IP_OFFMASK) == 0 &&
	    (ntohs(iphdr->ip_off) & IP_MF) == 0)
	  {
	    switch (iphdr->ip_p)
	      {
	      case IPPROTO_ICMP:
		p->transp_data = p->net_data + 4*iphdr->ip_hl;
		p->transp_prot = ND_PROT_ICMP;

		if (!nd_packet_has_complete_header(p, ND_PROT_ICMP))
		  {
		    p->transp_data = NULL;
		    p->transp_prot = ND_PROT_NULL;
		    D("Incomplete ICMP\n");
		    return;
		  }	
		break;
	      case IPPROTO_TCP:
		p->transp_data = p->net_data + 4*iphdr->ip_hl;
		p->transp_prot = ND_PROT_TCP;
		
		if (!nd_packet_has_complete_header(p, ND_PROT_TCP))
		  {
		    p->transp_data = NULL;
		    p->transp_prot = ND_PROT_NULL;
		    D("Incomplete TCP\n");
		    return;
		  }	
		break;
	      case IPPROTO_UDP:
		p->transp_data = p->net_data + 4*iphdr->ip_hl;
		p->transp_prot = ND_PROT_UDP;

		if (!nd_packet_has_complete_header(p, ND_PROT_UDP))
		  {
		    p->transp_data = NULL;
		    p->transp_prot = ND_PROT_NULL;
		    D("Incomplete UDP\n");
		    return;
		  }	
	      default:
	      }
	  }
      }
      break;
    default:
    }

  switch (p->transp_prot)
    {
    case ND_PROT_ICMP:
    case ND_PROT_TCP:
    case ND_PROT_UDP:
      p->app_prot = ND_PROT_NULL;
      p->app_data = NULL;
      break;
    default:
    }  
}


ND_Packet  *
nd_packet_sel_next(ND_Packet *p)
{
  if (!p)
    return (NULL);

  if (trace.apply_to_all)
    return (p->next);

  return (p->sel_next);
}


u_char*
nd_packet_get_data(ND_Packet *p, ND_Layer *l)
{
  if (!p)
    return (NULL);

  if (p->link_prot)
    {
      if (l) *l = ND_LINK;
      return (p->link_data);
    }
  else if (p->net_prot)
    {
      if (l) *l = ND_NET;
      return (p->net_data);
    }
  else if (p->transp_prot)
    {
      if (l) *l = ND_TRANSP;
      return (p->transp_data);
    }
  else if (p->app_prot)
    {
      if (l) *l = ND_APP;
      return (p->app_data);
    }

  if (l) *l = ND_NULL;
  return (NULL);
}


guint       
nd_packet_hash(gconstpointer ptr)
{
  ND_Packet *p = (ND_Packet *)ptr;

  if (p->transp_prot == ND_PROT_TCP)
    {
      return(((u_int32_t)(nd_packet_tcp(p)->th_sport) ^
	      (u_int32_t)nd_packet_ip(p)->ip_src.s_addr)   ^
	     ((u_int32_t)(nd_packet_tcp(p)->th_dport) ^
	      (u_int32_t)nd_packet_ip(p)->ip_dst.s_addr));
    }
  else
    {
      printf("Hashing non-TCP packet\n");
      abort();
    }
}


gint        
nd_packet_cmp(gconstpointer ptr1, gconstpointer ptr2)
{
  ND_Packet *p1 = (ND_Packet*)ptr1;
  ND_Packet *p2 = (ND_Packet*)ptr2;

  /*
  printf("Cmp: %u %u %u %u -- %u %u %u %u\n",
	 nd_packet_ip(p1)->ip_src.s_addr,
	 nd_packet_ip(p1)->ip_dst.s_addr,
	 nd_packet_tcp(p1)->th_dport,
	 nd_packet_tcp(p1)->th_sport,
	 nd_packet_ip(p2)->ip_src.s_addr,
	 nd_packet_ip(p2)->ip_dst.s_addr,
	 nd_packet_tcp(p2)->th_dport,
	 nd_packet_tcp(p2)->th_sport);
  */
  if (((nd_packet_ip(p1)->ip_src.s_addr == nd_packet_ip(p2)->ip_src.s_addr) &&
       (nd_packet_ip(p1)->ip_dst.s_addr == nd_packet_ip(p2)->ip_dst.s_addr) &&
       (nd_packet_tcp(p1)->th_sport == nd_packet_tcp(p2)->th_sport)         &&
       (nd_packet_tcp(p1)->th_dport == nd_packet_tcp(p2)->th_dport))         ||
      ((nd_packet_ip(p1)->ip_dst.s_addr == nd_packet_ip(p2)->ip_src.s_addr) &&
       (nd_packet_ip(p1)->ip_src.s_addr == nd_packet_ip(p2)->ip_dst.s_addr) &&
       (nd_packet_tcp(p1)->th_dport == nd_packet_tcp(p2)->th_sport)         &&
       (nd_packet_tcp(p1)->th_sport == nd_packet_tcp(p2)->th_dport)))
    return TRUE;
      
  return FALSE;
}


struct ip      *
nd_packet_ip(ND_Packet *p)
{
  return (struct ip*)(p->net_data);
}


struct udphdr  *
nd_packet_udp(ND_Packet *p)
{
  return (struct udphdr*)(p->transp_data);
}


struct tcphdr  *
nd_packet_tcp(ND_Packet *p)
{
  return (struct tcphdr*)(p->transp_data);
}


struct icmp    *
nd_packet_icmp(ND_Packet *p)
{
  return (struct icmp*)(p->transp_data);
}


struct ip      *
nd_packet_icmp_error_ip(ND_Packet *p)
{
  return (struct ip*)(p->transp_data + 8);
}


struct udphdr  *
nd_packet_icmp_error_udp(ND_Packet *p)
{
  return (struct udphdr*)(p->transp_data + 8 + (nd_packet_icmp_error_ip(p)->ip_hl * 4));
}


struct arphdr  *
nd_packet_arp(ND_Packet *p)
{
  return (struct arphdr*)(p->net_data);
}


struct ether_header *
nd_packet_ethernet(ND_Packet *p)
{
  return (struct ether_header*)(p->link_data);
}


u_char         *
nd_packet_end(ND_Packet *p)
{
  if (!p)
    return (NULL);

  return (p->link_data + p->ph.caplen);
}


gboolean    
nd_packet_has_protocol(ND_Packet *p, ND_Protocol proto)
{
  if (!p)
    return FALSE;

  switch (proto)
    {
    case ND_PROT_ETHER:
      return (p->link_prot == ND_PROT_ETHER);
    case ND_PROT_FDDI:
      return (p->link_prot == ND_PROT_FDDI);
    case ND_PROT_SLIP:
      return (p->link_prot == ND_PROT_SLIP);
    case ND_PROT_PPP:
      return (p->link_prot == ND_PROT_PPP);
    case ND_PROT_ARP:
      return (p->net_prot == ND_PROT_ARP);
    case ND_PROT_RARP:
      return (p->net_prot == ND_PROT_RARP);
    case ND_PROT_IP:
      return (p->net_prot == ND_PROT_IP);
    case ND_PROT_TCP:
      return (p->transp_prot == ND_PROT_TCP);
    case ND_PROT_UDP:
      return (p->transp_prot == ND_PROT_UDP);
    case ND_PROT_ICMP:
      return (p->transp_prot == ND_PROT_ICMP);
    case ND_PROT_ICMP_ERROR_IP:
      if (p->transp_prot != ND_PROT_ICMP)
	return (FALSE);
      
      switch (nd_packet_icmp(p)->icmp_type)
	{
	case ICMP_UNREACH:
	case ICMP_REDIRECT:
	case ICMP_SOURCEQUENCH:
	case ICMP_TIMXCEED:
	case ICMP_PARAMPROB:
	  return (TRUE);
	default:
	}
      
      return (FALSE);
    default:
    }
  
  return FALSE;
}


gboolean
nd_packet_has_complete_header(ND_Packet *p, ND_Protocol proto)
{
  if (!p)
    return (FALSE);

  if (!nd_packet_has_protocol(p, proto))
    return (FALSE);

  switch (proto)
    {
    case ND_PROT_ETHER:
      return (p->ph.caplen >= ETHER_HDR_LEN);

    case ND_PROT_FDDI:
      return (p->ph.caplen >= FDDI_HDR_LEN);

    case ND_PROT_SLIP:
      /* XXX fixme */
      return (FALSE);

    case ND_PROT_PPP:
      /* XXX fixme */
      return (FALSE);

    case ND_PROT_ARP:
    case ND_PROT_RARP:
      return (p->net_data + sizeof(struct arphdr) <= nd_packet_end(p));

    case ND_PROT_IP:
      /* If we can't get the header length field, we obviously
	 don't have the full header: */

      if (p->net_data + 1 >= nd_packet_end(p))
	return (FALSE);

      /* Otherwise, lookup IP header length and check again: */
      return (p->net_data + (nd_packet_ip(p)->ip_hl * 4) <= nd_packet_end(p));
      
    case ND_PROT_UDP:
      return (p->transp_data + 8 <= nd_packet_end(p));

    case ND_PROT_TCP:
      /* If we can't get the header length field, we obviously
	 don't have the full header: */
      
      if (p->transp_data + 12 >= nd_packet_end(p))
	return (FALSE);
      
      /* Otherwise, lookup TCP header length and check again: */
      return (p->net_data + (nd_packet_tcp(p)->th_off * 4) <= nd_packet_end(p));

    case ND_PROT_ICMP:
    case ND_PROT_ICMP_ERROR_IP:

      /* Eeew, ICMP. Need to check all the different types and
	 their lengths ...
      */

      /* If we can't get the ICMP type field, we obviously
	 don't have the full header: */
      if (p->transp_data >= nd_packet_end(p))
	return (FALSE);

      switch(nd_packet_icmp(p)->icmp_type)
	{
	case ICMP_ECHOREPLY:
	case ICMP_ECHO:
	  return (p->transp_data + 8 <= nd_packet_end(p));

	case ICMP_DEST_UNREACH:
	case ICMP_SOURCE_QUENCH:
	case ICMP_REDIRECT:
	case ICMP_TIME_EXCEEDED:
	case ICMP_PARAMETERPROB:
	  {
	    struct ip *iphdr = NULL;

	    /* Okay -- ICMP error. We need to check if we have
	       the full reported error message. */
	    
	    /* If we can't get the header length field, we obviously
	       don't have the full header (the IP header's length
	       field is 9 bytes from the ICMP message begin): */
	    
	    if (p->transp_data + 9 >= nd_packet_end(p))
	      return (FALSE);
	    
	    iphdr = nd_packet_icmp_error_ip(p);

	    /* Otherwise, lookup IP header length and check again --
	       we need the full IP header plus 4 bytes of payload
	       following to be legal:
	    */
	    return ((u_char*)iphdr + (iphdr->ip_hl * 4) + 4
		    <= nd_packet_end(p));
	  }	  

	case ICMP_TIMESTAMP:
	case ICMP_TIMESTAMPREPLY:
	  return (p->transp_data + 20 <= nd_packet_end(p));

	case ICMP_INFO_REQUEST:
	case ICMP_INFO_REPLY:		
	  /* Fixme -- not sure what to do here? */
	  return (p->transp_data + 4 <= nd_packet_end(p));

	case ICMP_ADDRESS:		
	case ICMP_ADDRESSREPLY:
	  return (p->transp_data + 12 <= nd_packet_end(p));
	  
	default:
	  return (FALSE);
	}   
    default:
    }
  
  return FALSE;
}

