/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <nd.h>
#include <nd_globals.h>
#include <nd_ipfrag.h>
#include <nd_misc.h>
#include <interface.h>
#include <callbacks.h>
#include <support.h>

static GtkWidget       *dialog = NULL;
static int              size_orig, size1, size2;
static int              offset1, offset2;


static gint nd_ip_off_cmp(gconstpointer d1, gconstpointer d2);


void
nd_ipfrag_show_dialog(void)
{
  struct ip       *iphdr;
  char             s[128];
  GtkWidget       *sb1, *sb2;
  GtkLabel        *label;
  GtkObject       *adj;

  if (!nd_packet_has_protocol(trace.p, ND_PROT_IP))
    return;

  iphdr = nd_packet_ip(trace.p);

  size_orig = IP_PAYLOAD_LENGTH(iphdr);
  size1 = size_orig/2 - ((size_orig/2) % 8);
  size2 = size_orig - size1;

  if (!dialog)
    dialog = create_ip_frag_dialog();

  D_ASSERT(dialog);
  sb1 = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag1_spinbutton");
  D_ASSERT(sb1);
  sb2 = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag2_spinbutton");
  D_ASSERT(sb1);

  /* Disable callbacks for a moment ... */
  gtk_signal_handler_block_by_func(GTK_OBJECT(sb1), on_ip_frag1_spinbutton_changed, NULL);
  gtk_signal_handler_block_by_func(GTK_OBJECT(sb2), on_ip_frag2_spinbutton_changed, NULL);

  adj = gtk_adjustment_new(0, 0, size_orig, 8, 8, 1);
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(sb1), GTK_ADJUSTMENT(adj));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(sb1), (gfloat)size1);

  adj = gtk_adjustment_new(0, 0, size_orig, 8, 8, 1);
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(sb2), GTK_ADJUSTMENT(adj));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(sb2), (gfloat)size2);

  /* Re-enable callbacks ... */
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(sb1), on_ip_frag1_spinbutton_changed, NULL);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(sb2), on_ip_frag2_spinbutton_changed, NULL);

  offset1 = (ntohs(iphdr->ip_off) & IP_OFFMASK) * 8;
  offset2 = offset1 + size1;

  D_ASSERT(dialog);
  label = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag1_offset_label");
  D_ASSERT(label);
  sprintf(s, "%u", offset1);
  gtk_label_set_text(label, s);

  label = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag2_offset_label");
  D_ASSERT(label);
  sprintf(s, "%u", offset2);
  gtk_label_set_text(label, s);

  gtk_widget_show(dialog);
}


void
nd_ipfrag_hide_dialog(void)
{
  gtk_widget_hide(dialog);
}


void      
nd_ipfrag_adjust(int size_new, ND_FragNum n)
{
  char             s[128];
  GtkWidget       *sb1, *sb2;
  GtkLabel        *label;
  
  if (!dialog)
    return;

  switch (n)
    {
    case FIRST:
      size1   = size_new;
      size2   = size_orig - size1;
      break;
    case SECOND:
      size2   = size_new;
      size1   = size_orig - size2;
      break;
    default:
    }
  
  offset2 = offset1 + size1;

  D_ASSERT(dialog);
  sb1 = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag1_spinbutton");
  D_ASSERT(sb1);
  sb2 = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag2_spinbutton");
  D_ASSERT(sb2);

  /* Disable callbacks for a moment ... */
  gtk_signal_handler_block_by_func(GTK_OBJECT(sb1), on_ip_frag1_spinbutton_changed, NULL);
  gtk_signal_handler_block_by_func(GTK_OBJECT(sb2), on_ip_frag2_spinbutton_changed, NULL);

  if (n == FIRST)
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sb2), (gfloat)size2);
  else
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sb1), (gfloat)size1);

  /* Re-enable callbacks ... */
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(sb1), on_ip_frag1_spinbutton_changed, NULL);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(sb2), on_ip_frag2_spinbutton_changed, NULL);

  label = gtk_object_get_data(GTK_OBJECT(dialog), "ip_frag2_offset_label");
  D_ASSERT(label);
  sprintf(s, "%u", offset2);
  gtk_label_set_text(label, s);
}


void      
nd_ipfrag_apply(void)
{
  struct ip   *iphdr;
  ND_Packet   *p;
  ND_Layer     l;
  u_char      *payload_start, *fragment_start, *fragment_end;
  int          remainder;

  /* Create the second fragment: */

  p = nd_packet_duplicate(trace.p);
  iphdr = nd_packet_ip(p);

  payload_start = (p->net_data + (iphdr->ip_hl * 4));
  fragment_start = payload_start + size1;
  remainder = p->ph.caplen - (fragment_start - nd_packet_get_data(p, &l));

  memmove(payload_start, fragment_start, remainder);
  p->ph.caplen -= size1;
  p->ph.len -= size1;

  if (l == ND_LINK)
    {
      int net_offset;

      net_offset = p->net_data - p->link_data;
      p->link_data = realloc(p->link_data, p->ph.caplen);
      p->net_data = p->link_data + net_offset;
    }
  else /* Must be IP */
    p->net_data = realloc(p->net_data, p->ph.caplen);

  p->transp_data = p->app_data = NULL;
  iphdr = nd_packet_ip(p);

  /* If the original packet doesn't have IP_MF set,
     make sure the second fragment does not either.
     So simply write the offset into th_off.
  */
  if (ntohs(((struct ip*)trace.p->net_data)->ip_off) & IP_MF)
    {
      iphdr->ip_off = htons(((offset2/8) & IP_OFFMASK) | IP_MF);
    }
  else
    iphdr->ip_off = htons((offset2/8) & IP_OFFMASK);

  iphdr->ip_len = htons(ntohs(iphdr->ip_len) - size1);
  iphdr->ip_sum = nd_misc_ip_checksum(p);

  /* Now adjust the original packet: */

  iphdr = nd_packet_ip(trace.p);
  fragment_start = (trace.p->net_data + (iphdr->ip_hl * 4));
  fragment_end = fragment_start + size1;

  remainder = trace.p->ph.caplen - (fragment_end + size2 - nd_packet_get_data(trace.p, &l));

  memmove(fragment_end, fragment_end + size2, remainder);
  trace.p->ph.caplen -= size2;
  trace.p->ph.len -= size2;

  if (l == ND_LINK)
    {
      int net_offset;

      net_offset = trace.p->net_data - trace.p->link_data;
      trace.p->link_data = realloc(trace.p->link_data, trace.p->ph.caplen);
      trace.p->net_data = trace.p->link_data + net_offset;
    }
  else /* Must be IP */
    trace.p->net_data = realloc(trace.p->net_data, trace.p->ph.caplen);

  trace.p->transp_data = trace.p->app_data = NULL;
  p->transp_data = p->app_data = NULL;
  
  iphdr = nd_packet_ip(trace.p);

  iphdr->ip_off = htons((ntohs(iphdr->ip_off) | IP_MF) & ~IP_DF);
  iphdr->ip_len = htons(ntohs(iphdr->ip_len) - size2);
  iphdr->ip_sum = nd_misc_ip_checksum(trace.p);

  nd_packet_init(trace.p);
  nd_packet_init(p);

  /* Insert new fragment after trace.p into trace: */
  nd_trace_packet_insert_at_index(p, nd_trace_packet_get_index(trace.p) + 1);
  nd_packet_set(trace.p);
  /*nd_packet_set_gui(trace.p);*/

  nd_trace_set_dirty(TRUE);

  nd_ipfrag_hide_dialog();
}


static gint 
nd_ip_off_cmp(gconstpointer d1, gconstpointer d2)
{
  ND_Packet *p1, *p2;
  struct ip *iphdr1, *iphdr2;

  p1 = (ND_Packet*)d1;
  p2 = (ND_Packet*)d2;

  iphdr1 = nd_packet_ip(p1);
  iphdr2 = nd_packet_ip(p2);

  if ((ntohs(iphdr1->ip_off) & IP_OFFMASK) < (ntohs(iphdr2->ip_off) & IP_OFFMASK))
    return (-1);
  if ((ntohs(iphdr1->ip_off) & IP_OFFMASK) > (ntohs(iphdr2->ip_off) & IP_OFFMASK))
    return (1);

  return (0);
}


void      
nd_ipfrag_reassemble(void)
{
  struct ip        *iphdr, *iphdr_frag;
  ND_Layer          layer;
  ND_Packet        *p, *pbase = NULL;
  GtkStyle         *gs;
  GtkWidget        *w;
  int               ip_id = -1, todo = 0, trailer_len;
  int               offset, covered = -1;
  u_char           *data, *cap_end, *trailer_start, *payload_start;
  u_char           *assembled;

  GList            *fragslist = NULL;
  GList            *f;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gs = gtk_widget_get_style(GTK_WIDGET(w));

  /* Sanity-check selected packets: */

  /*
  printf("Selection dump:\n");
  for (p = trace.sel; p; p = p->sel_next)
    {
      printf("%x %i\n", p, nd_packet_ip(p)->ip_id);
    }
  printf("Selection dump done.\n");
  */

  for (p = trace.sel; p; p = p->sel_next)
    {
      if (!nd_packet_has_protocol(p, ND_PROT_IP))
	{
	  /* Selection does not even consist of only IP datagrams... */
	  nd_misc_statusbar_set(_("Reassembly failed: selection contains non-IP datagrams."));
	  return;
	}

      iphdr = nd_packet_ip(p);

      if (ip_id < 0)
	ip_id = iphdr->ip_id;
      else
	if (ip_id != iphdr->ip_id)
	  {
	    char s[LINESIZE];

	    snprintf(s, LINESIZE, _("Reassembly failed: selection contains IP "
				    "datagrams with different ID values (%u/%u)."),
		     ntohs(iphdr->ip_id), ntohs(ip_id));

	    /* IP datagrams have different IP IDs ... */
	    nd_misc_statusbar_set(s);
	    return;
	  }

      if ((ntohs(iphdr->ip_off) & IP_OFFMASK) == 0)
	pbase = p;
      else
	todo += IP_PAYLOAD_LENGTH(iphdr);

      fragslist = g_list_append(fragslist, p);
    }

  if (!pbase)
    {
      nd_misc_statusbar_set(_("Reassembly failed: No IP datagram has fragment offset zero."));
      return;
    }

  /* Things are okay so far -- sort in increasing order by offsets. */
  fragslist = g_list_sort(fragslist, nd_ip_off_cmp);

  /* Allocate data for the reassembled datagram. */

  data = nd_packet_get_data(pbase, &layer);  
  assembled = malloc(pbase->ph.caplen + todo);
  bzero(assembled, pbase->ph.caplen + todo);
  memcpy(assembled, data, pbase->ph.caplen);

  if (layer == ND_LINK)
    {
      offset = pbase->net_data - pbase->link_data;
      iphdr = (struct ip*)(assembled + offset);
    }
  else /* Must be IP */
    iphdr = (struct ip*)assembled;

  /* Move anything that could be following the IP data to the new end. */

  cap_end = (assembled + pbase->ph.caplen);
  trailer_start = (u_char*)iphdr + ntohs(iphdr->ip_len);
  trailer_len = cap_end - trailer_start;
  payload_start = (u_char*)iphdr + (iphdr->ip_hl * 4);

  if (trailer_len > 0)
    memmove(cap_end - trailer_len, trailer_start, trailer_len);

  /* Go through again and try to reassemble. */
  
  for (f = g_list_first(fragslist); f; f = g_list_next(f))
    {
      p = (ND_Packet*)f->data;
      iphdr_frag = nd_packet_ip(p);
      
      if (p == pbase)
	{
	  covered = IP_PAYLOAD_LENGTH(iphdr_frag);
	  continue;
	}

      offset = (ntohs(iphdr_frag->ip_off) & IP_OFFMASK) * 8;
      data = (u_char*)iphdr_frag + (iphdr_frag->ip_hl * 4);

      if (covered > 0)
	{
	  /* Needn't be "==", overlap is okay (but currently unchecked): */
	  if (covered < offset)
	    {
	      nd_misc_statusbar_set(_("Reassembly failed: fragments do not form a continuous block."));
	      FREE(assembled);
	      return;
	    }
	}

      memcpy(payload_start + offset, data, IP_PAYLOAD_LENGTH(iphdr_frag));
      covered = offset + IP_PAYLOAD_LENGTH(iphdr_frag);
    }

  /* Packet is reassembled -- free old data, adjust ip_len,
   * checksum, and clear fragmentation bits.
   */
  data = nd_packet_get_data(pbase, &layer);  
  FREE(data);

  if (layer == ND_LINK)
    {
      pbase->link_data = assembled;
      pbase->net_data = (u_char*)iphdr;
    }
  else
    pbase->net_data = assembled;

  pbase->ph.caplen += todo;
  pbase->ph.len += todo;

  iphdr->ip_len = htons(ntohs(iphdr->ip_len) + todo);
  iphdr->ip_off = htons(0);
  iphdr->ip_sum = nd_misc_ip_checksum(pbase);

  nd_packet_init(pbase);
  nd_packet_set(pbase);
  /*nd_packet_set_gui(pbase);*/

  nd_trace_set_dirty(TRUE);
  nd_trace_sel_delete();

  nd_misc_statusbar_set(_("Reassembly succeeded."));
}
