--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Exceptions;

pragma Warnings (Off);
with System.Memory;
with System.CRTL;
pragma Warnings (On);

package body Narval.Protected_Memory is

   protected body Buffered_Memory_Type is

      function Get_State return Full_State_Type is
      begin
         return (Buffers_State => Buffer_State,
                 Buffers_Occupancy => Memory_Occupation_Rate);
      end Get_State;

      function Dumping_Event_By_Event_Mode return Boolean is
      begin
         return Dumping_Event_By_Event;
      end Dumping_Event_By_Event_Mode;

      --------------------------
      -- Basculer_Mode_Vidage --
      --------------------------
      procedure Toggle_Dumping_Mode
        (Event_Based_Dumping : Boolean := True) is
      begin
         Dumping_Event_By_Event := Event_Based_Dumping;
         if Event_Based_Dumping then
            Full_Value := Events_Available;
         else
            Full_Value := Full;
         end if;
      end Toggle_Dumping_Mode;

      --------------------
      -- Demander_Place --
      --------------------

      entry Get_Memory_For_Filling
        (Bytes_Number : in System.Storage_Elements.Storage_Count;
         Address : out System.Address;
         Duplicate : Boolean := False)
      when Buffer_State (Buffer_In_Filling) = Memory_Available is
         use type System.Storage_Elements.Storage_Count;
         New_Memory_Occupation_Rate : System.Storage_Elements.Storage_Count;
         Requested_Size_Too_Hight : exception;
      begin
         if Bytes_Number > Bytes_Sizes then
            Ada.Exceptions.Raise_Exception
              (Requested_Size_Too_Hight'Identity,
               "Demander_Place_Remplissage, line 46 : Nombre_Octets" &
               Bytes_Number'Img & " Taille_En_Octets" & Bytes_Sizes'Img);
         end if;
         if Acquisition_Mode = Multiple_Events then
            New_Memory_Occupation_Rate := Bytes_Number +
              Memory_Occupation_Rate (Buffer_In_Filling);
            if New_Memory_Occupation_Rate > Bytes_Sizes then
               Buffer_State (Buffer_In_Filling) := Full_Value;
               if Duplicate and not Duplicated_Data_Available then
                  Duplicated_Data_Size := Memory_Occupation_Rate
                    (Buffer_In_Filling);
                  System.CRTL.memcpy (Duplicated_Address,
                                      Memory_Addresses (Buffer_In_Filling),
                                      System.CRTL.size_t
                                      (Duplicated_Data_Size));
                  Duplicated_Data_Available := True;
               end if;
               if Number_Of_Buffer_To_Send = 0 then
                  First_Buffer_To_Send := Buffer_In_Filling;
               end if;
               Number_Of_Buffer_To_Send :=
                 Number_Of_Buffer_To_Send + 1;
               if Number_Of_Buffer_To_Send =
                 Max_Buffer_Number then
                  requeue Get_Memory_For_Filling;
               end if;
               New_Memory_Occupation_Rate := Bytes_Number;
               Buffer_In_Filling := Next (Buffer_In_Filling);
            end if;
            Address := Memory_Addresses (Buffer_In_Filling) +
              Memory_Occupation_Rate (Buffer_In_Filling);
            Memory_Occupation_Rate (Buffer_In_Filling) :=
              New_Memory_Occupation_Rate;
            Buffer_State (Buffer_In_Filling) := Lock_For_Filling;
         elsif Acquisition_Mode = Event_By_Event then
            Address := Memory_Addresses (Buffer_In_Filling);
            Buffer_State (Buffer_In_Filling) := Lock_For_Filling;
         end if;
      end Get_Memory_For_Filling;

      entry Get_Memory_For_Dumping
        (Bytes_Number : in System.Storage_Elements.Storage_Count;
         Address : out System.Address)
      when Buffer_State (First_Buffer_To_Send) = Events_Available is
         use type System.Storage_Elements.Storage_Count;
         New_Memory_Dumping_Rate : System.Storage_Elements.Storage_Count;
      begin
         New_Memory_Dumping_Rate := Bytes_Number +
           Memory_Dumping_Rate (First_Buffer_To_Send);
         Buffer_State (First_Buffer_To_Send) := Lock_For_Filling;
         if New_Memory_Dumping_Rate >
           Memory_Occupation_Rate (First_Buffer_To_Send) then
            Address := System.Null_Address;
            Memory_Dumping_Rate (First_Buffer_To_Send) :=
              New_Memory_Dumping_Rate;
            return;
         end if;
         Address := Memory_Addresses (First_Buffer_To_Send) +
           Memory_Dumping_Rate (First_Buffer_To_Send);
         Memory_Dumping_Rate (First_Buffer_To_Send) :=
           New_Memory_Dumping_Rate;
      end Get_Memory_For_Dumping;

      --------------------
      -- Demander_Place --
      --------------------

      entry Get_Memory
        (Address : out System.Address;
         Available_Memory_Size : out System.Storage_Elements.Storage_Count)
      when Buffer_State (Buffer_In_Filling) = Memory_Available is
         use type System.Storage_Elements.Storage_Count;
      begin
         Buffer_State (Buffer_In_Filling) := Lock_For_Filling_Bis;
         Address := Memory_Addresses (Buffer_In_Filling);
         Available_Memory_Size := Bytes_Sizes -
           Memory_Occupation_Rate (Buffer_In_Filling);
      end Get_Memory;

      -------------------------
      -- Deverrouiller_Tampon --
      -------------------------

      procedure Unlock_Buffer
        (Locked_Buffer : in Buffer_Enumeration_Type) is
      begin
         Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send - 1;
         if Number_Of_Buffer_To_Send /= 0 then
            First_Buffer_To_Send :=
              Next (Locked_Buffer);
         end if;
         Buffer_State (Locked_Buffer) := Memory_Available;
         Memory_Occupation_Rate (Locked_Buffer) := 0;
      end Unlock_Buffer;

      ----------------
      -- Initialise --
      ----------------

      procedure Initialise
        (Buffer_Size : System.Storage_Elements.Storage_Count;
         Mode : Acquisition_Kind_Type := Multiple_Events)
      is
         procedure Memset (S : System.Address;
                           C : Integer;
                           N : System.Memory.size_t);
         pragma Import (C, Memset, "memset");
      begin
         for I in Start_Addresses'Range loop
            Memory_Addresses (I) := System.Memory.Alloc
              (System.Memory.size_t (Buffer_Size));
            Memset (S => Memory_Addresses (I),
                    C => 0,
                    N => System.Memory.size_t (Buffer_Size));
         end loop;
         Bytes_Sizes := Buffer_Size;
         Acquisition_Mode := Mode;
         Duplicated_Address := System.Memory.Alloc
           (System.Memory.size_t (Buffer_Size));
         Memset (S => Duplicated_Address,
                 C => 0,
                 N => System.Memory.size_t (Buffer_Size));
      end Initialise;

      procedure Free is
      begin
         for I in Start_Addresses'Range loop
            System.Memory.Free (Memory_Addresses (I));
         end loop;
         Bytes_Sizes := 0;
         System.Memory.Free (Duplicated_Address);
      end Free;

      --------------
      -- Notifier --
      --------------

      procedure Notify (Data : Developed_Access_Type;
                        Duplicate : Boolean := False) is
      begin
         if Number_Of_Buffer_To_Send = 0 then
            First_Buffer_To_Send := Data.Buffer_Number;
         end if;
         Memory_Occupation_Rate (Data.Buffer_Number) := Data.Size;
         Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send + 1;
         Buffer_State (Data.Buffer_Number) := Full_Value;
         if Duplicate and not Duplicated_Data_Available then
            Duplicated_Data_Size := Memory_Occupation_Rate
              (Buffer_In_Filling);
            System.CRTL.memcpy (Duplicated_Address,
                                Memory_Addresses (Buffer_In_Filling),
                                System.CRTL.size_t
                                (Duplicated_Data_Size));
            Duplicated_Data_Available := True;
         end if;
      end Notify;

      --------------------------------
      -- Recuperer_Structure_Tampon --
      --------------------------------

      function Get_Buffer_Structure
        (Manipulation_Buffer : in Buffer_Enumeration_Type)
        return Developed_Access_Type is
         Result : Developed_Access_Type;
      begin
         Result :=  (Manipulation_Buffer,
                       Memory_Occupation_Rate (Manipulation_Buffer));
         return Result;
      end Get_Buffer_Structure;

      --------------------------------
      -- Recuperer_Structure_Tampon --
      --------------------------------

      function Get_Buffer_Structure
        (Manipulation_Buffer : in Buffer_Enumeration_Type)
        return Raw_Access_Type is
         Result : Raw_Access_Type;
      begin
         Result :=  (Memory_Addresses (Manipulation_Buffer),
                       Memory_Occupation_Rate (Manipulation_Buffer));
         return Result;
      end Get_Buffer_Structure;

      --------------------
      -- Relacher_Place --
      --------------------

      procedure Release_Memory (Duplicate : Boolean := False) is
         use type System.Storage_Elements.Storage_Offset;
      begin
         if not Dumping_Event_By_Event then
            if Acquisition_Mode = Multiple_Events then
               Buffer_State (Buffer_In_Filling) := Memory_Available;
            elsif Acquisition_Mode = Event_By_Event then
               Buffer_State (Buffer_In_Filling) := Full_Value;
               if Duplicate and not Duplicated_Data_Available then
                  Duplicated_Data_Size := Memory_Occupation_Rate
                    (Buffer_In_Filling);
                  System.CRTL.memcpy (Duplicated_Address,
                                      Memory_Addresses (Buffer_In_Filling),
                                      System.CRTL.size_t
                                      (Duplicated_Data_Size));
                  Duplicated_Data_Available := True;
               end if;
               if Number_Of_Buffer_To_Send = 0 then
                  First_Buffer_To_Send := Buffer_In_Filling;
               end if;
               Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send + 1;
               Buffer_In_Filling := Next (Buffer_In_Filling);
            end if;
         else
            if Memory_Dumping_Rate (First_Buffer_To_Send) =
              Memory_Occupation_Rate (First_Buffer_To_Send) then
               Buffer_State (First_Buffer_To_Send) := Memory_Available;
               Memory_Occupation_Rate (First_Buffer_To_Send) := 0;
               Memory_Dumping_Rate (First_Buffer_To_Send) := 0;
               Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send - 1;
               if Number_Of_Buffer_To_Send /= 0 then
                  First_Buffer_To_Send :=
                    Next (First_Buffer_To_Send);
               end if;
            else
               Buffer_State (First_Buffer_To_Send) :=
                 Events_Available;
            end if;
         end if;
      end Release_Memory;

      --------------------
      -- Relacher_Place --
      --------------------

      procedure Release_Memory
        (Used_Memory_Size : in System.Storage_Elements.Storage_Count;
         Duplicate : Boolean := False) is
         use type System.Storage_Elements.Storage_Count;
      begin
         if Used_Memory_Size = 0 then
            Buffer_State (Buffer_In_Filling) := Memory_Available;
            return;
         end if;
         Buffer_State (Buffer_In_Filling) := Full_Value;
         Memory_Occupation_Rate (Buffer_In_Filling) := Used_Memory_Size;
         if Duplicate and not Duplicated_Data_Available then
            Duplicated_Data_Size := Memory_Occupation_Rate
              (Buffer_In_Filling);
            System.CRTL.memcpy (Duplicated_Address,
                                Memory_Addresses (Buffer_In_Filling),
                                System.CRTL.size_t
                                (Duplicated_Data_Size));
            Duplicated_Data_Available := True;
         end if;
         if Number_Of_Buffer_To_Send = 0 then
            First_Buffer_To_Send := Buffer_In_Filling;
         end if;
         Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send + 1;
         Buffer_In_Filling := Next (Buffer_In_Filling);
      end Release_Memory;

      -------------------
      -- Taille_Tampon --
      -------------------

      function Buffer_Size return System.Storage_Elements.Storage_Count is
      begin
         return Bytes_Sizes;
      end Buffer_Size;

      --------------
      -- Verifier --
      --------------

      procedure Check (Duplicate : Boolean := False) is
         use type System.Storage_Elements.Storage_Offset;
      begin
         if Memory_Occupation_Rate (Buffer_In_Filling) /= 0 and
           Buffer_State (Buffer_In_Filling) /= Lock_For_Filling then
            Buffer_State (Buffer_In_Filling) := Full_Value;
            if Duplicate and not Duplicated_Data_Available then
               Duplicated_Data_Size := Memory_Occupation_Rate
                 (Buffer_In_Filling);
               System.CRTL.memcpy (Duplicated_Address,
                                   Memory_Addresses (Buffer_In_Filling),
                                   System.CRTL.size_t
                                   (Duplicated_Data_Size));
               Duplicated_Data_Available := True;
            end if;
            if Number_Of_Buffer_To_Send = 0 then
               First_Buffer_To_Send := Buffer_In_Filling;
            end if;
            Number_Of_Buffer_To_Send := Number_Of_Buffer_To_Send + 1;
            Buffer_In_Filling := Next (Buffer_In_Filling);
         end if;
      end Check;

      -----------------------
      -- Verrouiller_Tampon --
      -----------------------

      entry Lock_Buffer
        (Manipulation_Buffer : out Buffer_Enumeration_Type)
      when Number_Of_Buffer_To_Send /= 0 and
        Buffer_State (First_Buffer_To_Send) = Full
      is
      begin
         Manipulation_Buffer := First_Buffer_To_Send;
      end Lock_Buffer;

      entry Available_Event
        (Remaining_Size : out System.Storage_Elements.Storage_Count) when
        Buffer_State (First_Buffer_To_Send) = Events_Available is
         use type System.Storage_Elements.Storage_Count;
      begin
         Remaining_Size := Memory_Occupation_Rate (First_Buffer_To_Send)
           - Memory_Dumping_Rate (First_Buffer_To_Send);
      end Available_Event;

      function Is_Empty return Boolean is
         Result : Boolean := True;
      begin
         for I in Buffer_State'Range loop
            Result := Result and Buffer_State (I) = Memory_Available;
         end loop;
         return Result;
      end Is_Empty;

      entry Get_Duplicated_Data
        (Size : out System.Storage_Elements.Storage_Count;
         Data : out System.Address) when Duplicated_Data_Available is
      begin
         Size := Duplicated_Data_Size;
         Data := Duplicated_Address;
      end Get_Duplicated_Data;

      procedure Release_Duplicated_Data is
      begin
         Duplicated_Data_Available := False;
      end Release_Duplicated_Data;

   end Buffered_Memory_Type;

end Narval.Protected_Memory;
