--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Unbounded;
with Ada.Unchecked_Deallocation;
with GNAT.Sockets;

with Log4ada.Loggers;

package Narval.Communication.Sockets is

   type Socket_Link_Type is
     abstract new Link_Type with record
        Port_Number : GNAT.Sockets.Port_Type;
        Address : GNAT.Sockets.Sock_Addr_Type;
        Socket : GNAT.Sockets.Socket_Type;
        Host : Ada.Strings.Unbounded.Unbounded_String;
     end record;

   function Get_Init_Info
     (Link : access Socket_Link_Type)
     return Integer;
   function Stream (Link : access Socket_Link_Type) return Stream_Access;
   procedure Free (Link : access Socket_Link_Type;
                   Stream_To_Free : in out Stream_Access);
   procedure Send
     (Link : access Socket_Link_Type;
      Data : Ada.Streams.Stream_Element_Array);
   procedure Receive
     (Link : access Socket_Link_Type;
      Data : out Ada.Streams.Stream_Element_Array);
   procedure Close (Link : access Socket_Link_Type);

   type Client_Socket_Link_Type is new Socket_Link_Type
     with null record;
   type Client_Socket_Link_Access is access all Client_Socket_Link_Type;

   procedure Connect (Link : access Client_Socket_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order);

   type Server_Socket_Link_Type;
   type Server_Socket_Link_Access is access all Server_Socket_Link_Type;
   task type Connect_Task_Type
     (Wrapper : access Server_Socket_Link_Type) is
      entry Connect (Bit_Order_To_Send : in System.Bit_Order);
   end Connect_Task_Type;

   type Server_Socket_Link_Type (First : Boolean) is
     new Socket_Link_Type with record
      Connect_Task : Connect_Task_Type
        (Server_Socket_Link_Type'Access);
      case First is
         when True =>
            Server : GNAT.Sockets.Socket_Type;
            Lock_Reference_Number : Natural := 0;
         when False =>
            Server_Access : Server_Socket_Link_Access;
      end case;
   end record;

   procedure Connect (Link : access Server_Socket_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order);
   procedure Close (Link : access Server_Socket_Link_Type);

   --  fonctions generales
   function Init_Client
     (Port_Number : Positive;
      Address : String;
      Logger : Log4ada.Loggers.Logger_Class_Access) return Link_Access;

   function Init_Server
     (Port : String;
      First : Boolean;
      Server : Server_Socket_Link_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access)
     return Server_Socket_Link_Access;

   procedure Free is new Ada.Unchecked_Deallocation
     (Client_Socket_Link_Type, Client_Socket_Link_Access);
   procedure Free is new Ada.Unchecked_Deallocation
     (Server_Socket_Link_Type, Server_Socket_Link_Access);

end Narval.Communication.Sockets;
