/*  tiffep - Ti File Format Engine & Proxy
 *  Copyright (C) 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  (Unix part)
  This unit is used to create and manage Shared Memory segments via 
  IPC System V (Inter Process Communication)
*/

#ifdef __LINUX__

#include <stdio.h>
#include <sys/ipc.h>
#include <sys/types.h>
#include <sys/shm.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "hfiles.h"
#include "tiffep.h"

static key_t ipc_key = -1;

/*
   Create a Shared Memory Segment
   - name: name of the shm object
   - size: the size of the segment
   - option: not used yet
   - id: a returned shm identifier if successfully created
   Returns 0 if successful.
*/
int shm_create(TiffepShm *s, const char* name, int size, int option)
{
  
  s->size = size;

#ifdef __WIN32__
  DISPLAY2("Process ID (pid): %d\n", _getpid());
#elif defined(__LINUX__)
  DISPLAY2("Process ID (pid): %d\n", getpid());
#endif

  /* Get an IPC key */
  if((ipc_key = ftok(SHM_NAME, '0')) == -1)
    {
      print_last_error("shm_create (ftok)");
      DISPLAY2("ipc_key = 0x%08x\n", ipc_key);
      return ERR_IPC_KEY;
    }

  /* Open (create if necessary) a shm */
  if((s->id = shmget(ipc_key, s->size, IPC_CREAT | 0666)) == -1)
    {
      print_last_error("shm_create (shmget)");
      DISPLAY2("shmid = %i\n", s->id);
      return ERR_SHM_GET;
    }

  /* Place a 'presence' file */
  open(SOCKET_NAME, O_CREAT);

  return 0;
}

/*
   Attach a Shared Memory Segment
   - id: the identifier returned by the create_shm function
   - addr: a pointer for attaching the segment
   - size: size of the shm
   - option: not used yet
*/
int shm_attach(TiffepShm *s, int option)
{
  /* Attach the segment once created */
  if((s->addr = (void *)shmat(s->id, NULL, 0)) == NULL)
    {
      print_last_error("shm_attach (shmat)");
      return ERR_SHM_ATT;
    }

  return 0;
}

/*
   Detach a Shared Memory Segment
   - id: the shm identifier
   - addr: a pointer on the attached segment
   Return 0 if successful.
*/
int shm_detach(TiffepShm *s)
{
  /* Detach segment */
  if(shmdt(s->addr) == -1)
    {
      print_last_error("shm_detach (shmdt)");
      return ERR_SHM_DTCH;
    }

  return 0;
}


/*
   Destroy a Shared Memory Segment (if no longer used)
   - id: the shm identifier
   - addr: a pointer on the attached segment
   Return 0 if successful.
*/
int shm_destroy(TiffepShm *s)
{
  if(shmctl(s->id, IPC_RMID, NULL) == -1)
    {
      print_last_error("shm_detach (shmctl)");
      return ERR_SHM_RMID;
    }

  return 0;
}


/*
  Detect if a shm already exists 
  Return 1 if it exists, 0 otherwise
*/
int shm_is_exist()
{
  //int shmid = -1;

  /* Try to open the shm in exclusive mode */
  /*
  shmid = shmget(ipc_key, 1, IPC_EXCL);
  DISPLAY2("shmid = %i, ipc_key = %x", shmid, ipc_key);
  
  return ( (shmid == -1) ? 0 : 1);
  */
  return (!access(SOCKET_NAME, F_OK));
}

#endif
