/*  tilp - link program for TI calculators
 *  Copyright (C) 1999, 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  This unit contains functions for managing TIGraphLink file format
  and some other stuffs.
*/

#include "hfiles.h"
#include "tiffep.h"

/*********************************************/
/* File content functions (write equivalent) */
/*********************************************/

int save_file_content_82_83_85_86(FILE *file, TiffepFileInfo *fi);
int save_file_content_89_92_92p(FILE *file, TiffepFileInfo *fi);

/*
  Save the content of TiffepFileInfo into a TIGraphLink file
  - filename [in]: the file to save without any extension (added)
  - fi [in]: the content stored in an allocated structure
  (must be freed when no longer used)
  - int [out]: 0 if success, an error code otherwise.
*/

TIEXPORT
int tiffep_save_file_content(gchar *filename, TiffepFileInfo *fileinfo)
{
  FILE *f;
  gint err = 0;
  TiffepFileInfo *fi;

  fi = fileinfo;
  if(FI_CALCTYPE(fi) == TYPE_NONE)
    {
      g_warning("tiffep_save_file_content: unknown file type.\n");
      return ERR_INVALID_FILE;
    }

  FI_NAME(fi) = g_strdup(g_basename(filename));
  switch(FI_FILETYPE(fi))
    {
    case FILE_SINGLE:
      FI_EXTENSION(fi) = g_strdup(tiffep_vartype_to_file_extension(FI_CALCTYPE(fi), COMMON_TYPES_TO_TIxx_TYPES(VI_TYPE(VARLIST_DATA(FI_VARLIST(fi)))-256, FI_CALCTYPE(fi))));
      break;
    case FILE_GROUP:
      FI_EXTENSION(fi) = g_strdup(tiffep_group_file_ext(FI_CALCTYPE(fi)));
      break;
    case FILE_BACKUP:
      FI_EXTENSION(fi) = g_strdup(tiffep_backup_file_ext(FI_CALCTYPE(fi)));
      break;
    case FILE_FLASH:
      break;
    default:
      return -1;
    }
  FI_FILENAME(fi) = g_strconcat(FI_NAME(fi), ".", FI_EXTENSION(fi), NULL);

  f = fopen(FI_FILENAME(fi), "wb");
  if(f == NULL)
    {
      g_error("tiffep_save_file_content: unable to open this file: <%s>\n", FI_FILENAME(fi));
      return ERR_FILE_OPEN;
    }

  switch(FI_CALCTYPE(fi))
    {
    case TYPE_TI82:
    case TYPE_TI83:
    case TYPE_TI83P:
    case TYPE_TI85:
    case TYPE_TI86:
      err = save_file_content_82_83_85_86(f, fi);
      break;
    case TYPE_TI89:
    case TYPE_TI92:
    case TYPE_TI92P:
      err = save_file_content_89_92_92p(f, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }
  fclose(f);

  return err;
}

/* Save file content for TI82..86 calcs */
int save_file_content_82_83_85_86(FILE *file, TiffepFileInfo *fi)
{
  // to do ...

  return 0;
}

int save_file_content_89_92_92p_single(FILE *file, TiffepFileInfo *fi);
int save_file_content_89_92_92p_backup(FILE *file, TiffepFileInfo *fi);
int save_file_content_89_92_92p_group(FILE *file, TiffepFileInfo *fi);
int save_file_content_89_92_92p_flash(FILE *file, TiffepFileInfo *fi);

/* 
   Save file content for TI89..92+ calcs
   - file [in]: a pointer on the file to stat 
   - fi [in]: the content to complete
   - int [out]: 0 if success, an error code otherwise.
*/
int save_file_content_89_92_92p(FILE *file, TiffepFileInfo *fi)
{
  gchar buffer[MAXCHARS];
  gchar *header = buffer;

  if(FI_FILETYPE(fi) == FILE_FLASH)
    fputs("**TIFL**", file);
  else
    {
      switch(FI_CALCTYPE(fi))
	{
	case TYPE_TI92P:
	  header = g_strdup("**TI92P*");
	  break;
	case TYPE_TI92:
	  header = g_strdup("**TI92**");
	  break;
	case TYPE_TI89:
	  header = g_strdup("**TI89**");
	  break;
	default:
	  header = g_strdup("**TI92**");
	  break;
	}
      fputs(header, file);
      g_free(header);
    }

  switch(FI_FILETYPE(fi))
    {
    case FILE_SINGLE:
      FI_NUMVARS(fi) = 1;
      return save_file_content_89_92_92p_single(file, fi);
      break;
    case FILE_GROUP:
      if(FI_NUMVARS(fi) < 2)
	return -1;
      return save_file_content_89_92_92p_group(file, fi);
      break;
    case FILE_FLASH:
      FI_NUMVARS(fi) = 1;
      return save_file_content_89_92_92p_flash(file, fi);
      break;
    case FILE_BACKUP:
      FI_NUMVARS(fi) = 1;
      return save_file_content_89_92_92p_backup(file, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }

  return 0;
}

/* 
   Save file content for a TI89..92+ single file
   - file [in]: a pointer on the file to stat 
   - fi [in]: the content to write
*/
int save_file_content_89_92_92p_single(FILE *file, TiffepFileInfo *fi)
{
  gint i;
  TiffepVarInfo *vi = (TiffepVarInfo *)(FI_VARLIST(fi)->data);
  word chk;

  fputc(0x01, file); fputc(0x00, file);

  write_string8(VI_PATH(vi), file);
  write_string_n(FI_COMMENT(fi), 40, file);
  fputc(LSB(FI_NUMVARS(fi)), file);
  fputc(MSB(FI_NUMVARS(fi)), file);
  
  VI_OFFSET(vi) = 0x52;
  fputc(LSB(LSW(VI_OFFSET(vi))), file);
  fputc(MSB(LSW(VI_OFFSET(vi))), file);
  fputc(LSB(MSW(VI_OFFSET(vi))), file);
  fputc(MSB(MSW(VI_OFFSET(vi))), file);

  write_string8(VI_CALCNAME(vi), file);
  fputc(COMMON_TYPES_TO_TIxx_TYPES(VI_TYPE(vi)-256, VI_CALCTYPE(vi)), file);
  fputc(0x00, file);
  fputc(0x00, file); fputc(0x00, file);

  fputc(LSB(LSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(MSB(LSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(LSB(MSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(MSB(MSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);

  fputc(0xA5, file); fputc(0x5A, file);
  for(i=0; i<4; i++) fputc(0x00, file);
  for(i=0; i<VI_SIZE(vi); i++)
    fputc(VI_CONTENT(vi)[i], file);
  compute_checksum(VI_CONTENT(vi), VI_SIZE(vi), &chk);
  fputc(LSB(chk), file);
  fputc(MSB(chk), file);

  return 0;
}

int save_file_content_89_92_92p_backup(FILE *file, TiffepFileInfo *fi)
{
  /************ NOT TESTED (but should work) ********************/
  gint i;
  TiffepVarInfo *vi = VARLIST_DATA(FI_VARLIST(fi));
  word chk;

  fputc(0x01, file); fputc(0x00, file);

  write_string8("\0\0\0\0\0\0\0\0\0", file);
  write_string_n(FI_COMMENT(fi), 40, file);
  fputc(LSB(FI_NUMVARS(fi)), file);
  fputc(MSB(FI_NUMVARS(fi)), file);

  fputc(LSB(LSW(VI_OFFSET(vi))), file);
  fputc(MSB(LSW(VI_OFFSET(vi))), file);
  fputc(LSB(MSW(VI_OFFSET(vi))), file);
  fputc(MSB(MSW(VI_OFFSET(vi))), file);

  write_string8(VI_CALCNAME(vi), file);
  fputc(COMMON_TYPES_TO_TI92_TYPES[VI_TYPE(vi)-256], file);
  fputc(0x00, file);
  fputc(0x00, file); fputc(0x00, file);

  fputc(LSB(LSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(MSB(LSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(LSB(MSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);
  fputc(MSB(MSW(VI_OFFSET(vi) + VI_SIZE(vi) + 6)), file);

  fputc(0xA5, file); fputc(0x5A, file);
  for(i=0; i<4; i++) fputc(0x00, file);
  for(i=0; i<VI_SIZE(vi); i++)
    fputc(VI_CONTENT(vi)[i], file);
  compute_checksum(VI_CONTENT(vi), VI_SIZE(vi), &chk);
  fputc(LSB(chk), file);
  fputc(MSB(chk), file);

  return 0;
}

int save_file_content_89_92_92p_group(FILE *file, TiffepFileInfo *fi)
{
  gint i, j, k;
  GList *vl = FI_VARLIST(fi);
  GList *dl = FI_DIRLIST(fi);
  word chk = 0;
  word main_chk = 0;
  gint num_of_entries = 0;
  gint num_of_vars = g_list_length(FI_VARLIST(fi));
  gint num_of_folders = g_list_length(FI_DIRLIST(fi));
  longword last_offset;

  fputc(0x01, file); fputc(0x00, file);
  write_string8(VI_PATH(VARLIST_DATA(vl)), file);
  write_string_n(FI_COMMENT(fi), 40, file);
  num_of_entries = num_of_vars + num_of_folders;
  fputc(LSB(num_of_entries), file);
  fputc(MSB(num_of_entries), file);

  for(k=0; k<num_of_folders; k++)
    {
      TiffepFolderInfo *di = DIRLIST_DATA(dl);

      fputc(LSB(LSW(DI_OFFSET(di))), file);
      fputc(MSB(LSW(DI_OFFSET(di))), file);
      fputc(LSB(MSW(DI_OFFSET(di))), file);
      fputc(MSB(MSW(DI_OFFSET(di))), file);

      write_string8(DI_NAME(di), file);
      fputc(COMMON_TYPES_TO_TI92_TYPES[DI_TYPE(di)-256], file);
      printf("di_type=%i\n", DI_TYPE(di));
      fputc(0x00, file);
      fputc(LSB(DI_SIZE(di)), file);
      fputc(MSB(DI_SIZE(di)), file);

      for(j=0; j<num_of_vars; j++)
	{
	  TiffepVarInfo *vi  = VARLIST_DATA(vl);

	  fputc(LSB(LSW(VI_OFFSET(vi))), file);
	  fputc(MSB(LSW(VI_OFFSET(vi))), file);
	  fputc(LSB(MSW(VI_OFFSET(vi))), file);
	  fputc(MSB(MSW(VI_OFFSET(vi))), file);

	  write_string8(VI_CALCNAME(vi), file);
	  fputc(COMMON_TYPES_TO_TI92_TYPES[VI_TYPE(vi)-256], file);
	  printf("vi_type=%i\n", VI_TYPE(vi));
	  fputc(VI_ATTRIBUTE(vi), file);
	  fputc(0x00, file);
	  fputc(0x00, file);
	  
	  vl = VARLIST_NEXT(vl);
	}

      dl = VARLIST_NEXT(dl);
    }
  vl=g_list_last(FI_VARLIST(fi));

  last_offset = VI_OFFSET(VARLIST_DATA(vl)) + VI_SIZE(VARLIST_DATA(vl)) + 6;
  fputc(LSB(LSW(last_offset)), file);
  fputc(MSB(LSW(last_offset)), file);
  fputc(LSB(MSW(last_offset)), file);
  fputc(MSB(MSW(last_offset)), file);
  fputc(0xA5, file); fputc(0x5A, file);
  for(j=0, vl=FI_VARLIST(fi); j<num_of_vars; j++)
    {
      TiffepVarInfo *vi  = (TiffepVarInfo *)(vl->data);
      for(i=0; i<4; i++) fputc(0x00, file);
      for(i=0; i<VI_SIZE(vi); i++)
	fputc(VI_CONTENT(vi)[i], file);
      compute_checksum(VI_CONTENT(vi), VI_SIZE(vi), &chk);
      main_chk += chk;
      fputc(LSB(chk), file);
      fputc(MSB(chk), file);

      vl = g_list_next(vl);
    }
  
  return 0;
}

int save_file_content_89_92_92p_flash(FILE *file, TiffepFileInfo *fi)
{
  return 0;
}

/********* Below: unused !!! **************/

/*
  Fill the TiffepFileInfo structure with the given arguments
  - filename [in]: the base name of the file (without any extension)
  - calc_type [in]: the type of calculator
  - file_type [in]: the variable type
  - comment [in]: the comment to store in the file
  - fi [in]: the structure to fill (must be allocated before)
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_create_single_file(gchar *filename, gint calc_type, gint var_type,
                              char *comment, TiffepFileInfo *fi)
{
  FI_NAME(fi) = g_strdup(g_basename(filename));
  //FI_EXTENSION(fi) = g_strdup(vartype_to_file_extension(calc_type, 
  //							      var_type));
  FI_FILENAME(fi) = g_strdup(filename);
  FI_FILETYPE(fi) = FILE_SINGLE;
  FI_CALCTYPE(fi) = calc_type;
  FI_NUMVARS(fi) = 1;
  FI_VARLIST(fi) = NULL;

  return 0;
}

/*
  Fill the TiffepFileInfo structure with the given arguments
  - filename [in]: the base name of the file (without any extension)
  - calc_type [in]: the type of calculator
  - comment [in]: the comment to store in the file
  - fi [in]: the structure to fill (must be allocated before)
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_create_group_file(gchar *filename, gint calc_type,
                             char *comment, TiffepFileInfo *fileinfo)
{
  return 0;
}

  /*
  fputc(0x01, file);
  fputc(0x00, file);
  switch(FI_FILETYPE(fi))
    {
    case FILE_FLASH:
      //to do...
      break;
    case FILE_BACKUP:
      for(i=0; i<8; i++) fputc(0x00, file);
      break;
    case FILE_SINGLE:
    case FILE_GROUP:
      s = VI_PATH(FI_VARINFO(fi));
      for(i=0; i<strlen(s); i++)
	fputc(s[i], file);
      for(j=i; j<8; j++)
	fputc(0x00, file);
      break;
    default:
      break;
    }

  fputs(FI_COMMENT(fi), file);
  fputc(LSB(FI_NUMVARS(fi)), file);
  fputc(MSB(FI_NUMVARS(fi)), file);
  */
