/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000, Romain Lievin and Thomas Corvazier
 *  (c) Copyright 2001, Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config2.h>
#endif

#include "intl.h"

#include "lib68k.h"
#include <stdlib.h>
#include "sysdeps.h"
#include <assert.h>
#include <stdio.h>
#include <ctype.h>

#include <sys/types.h>
#ifndef __WIN32__
#include <dirent.h>
#endif
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "config.h"
#include "options.h"
#include "memory.h"
#include "newcpu.h"
#include "packets.h"
#include "debug.h"
#include "hardware.h"
#include "keyboard.h"
#include "globinfo.h"
#include "callbacks.h"
#include "intlist.h"
#include "errcodes.h"
#include "cmdinterface.h"
#include "rom.h"
#include "tilibs.h"

#define DEFAULT_ROMFILE "ti92.rom"

struct GlobalInformation glob_inf;
Ti68kInfoUpdate *iupdate;

/* 
   External GUI functions
*/
callback_iv_t cb_launch_debugger = enter_debugger;
callback_vi_t cb_update_progbar = cmd_update_progbar;

callback_iv_t cb_init_specific = NULL;
callback_iv_t cb_exit_specific = NULL;
callback_iv_t cb_update_screen = NULL;
callback_iv_t cb_update_keys = NULL;
callback_iv_t cb_screen_on = NULL;
callback_iv_t cb_screen_off = NULL;
callback_vi_t cb_set_screen_ptr = NULL;
callback_ii_t cb_set_contrast = NULL;

/*
  Internal lib variables
*/
int rom_loaded = 0;
int tib_loaded = 0;
ROM_INFO current_rom_info;
TIB_INFO current_tib_info;
int bInstructionsDisplay = 0;

struct intlist *listBkptAddress = 0;
int nBkptAddress = 0;
extern int listBkptVector[16], listBkptAutoint[8], listBkptTrap[16];
extern int nBkptVector, nBkptTrap, nBkptAutoint;

#define is_num(c)   isdigit(c)
#define is_alnum(c) isalnum(c)


/******************/
/* Main functions */
/******************/

/* Set up the update functions in order to be independant of the GUI */
void ti68k_set_update(Ti68kInfoUpdate *iu,
		      void (*start)   (void),
		      void (*stop)    (void),
		      void (*refresh) (void),
		      void (*msg_box) (const char *t, char *s),
		      void (*pbar)    (void),
		      void (*label)   (void),
		      int  (*choose)  (char *cur_name, 
				       char *new_name))
{
  iu->cancel  = 0;
  iu->start   = start;
  iu->stop    = stop;
  iu->refresh = refresh;
  iu->msg_box = msg_box;
  iu->pbar    = pbar;
  iu->label   = label;
  iu->choose  = choose;

  iupdate = iu;
}

/* This is the FIRST function to call */
int ti68k_setGuiCallbacks(callback_iv_t initSpecific,
			  callback_iv_t exitSpecific,
			  callback_iv_t updateScreen,
			  callback_iv_t updateKeys,
			  callback_iv_t screenOn,
			  callback_iv_t screenOff,
			  callback_vi_t setScreenPtr,
			  callback_ii_t setContrast)
{
#ifdef ENABLE_NLS
  bindtextdomain(PACKAGE, NULL);
  textdomain(PACKAGE);
#endif

  cb_init_specific = initSpecific;
  cb_exit_specific = exitSpecific;
  cb_update_screen = updateScreen;
  cb_update_keys = updateKeys;
  cb_screen_on = screenOn;
  cb_screen_off = screenOff;
  cb_set_screen_ptr = setScreenPtr;
  cb_set_contrast = setContrast;

  memset((void *)(&current_rom_info), 0, sizeof(ROM_INFO));
  memset((void *)(&current_tib_info), 0, sizeof(TIB_INFO));

  return ERR_NONE;
}

/*
  This is the SECOND function to call unless the glob_inf structure has been
  properly initialized.
 */
int ti68k_loadDefaultConfig(void)
{
  glob_inf.background = 1;
  glob_inf.grayPlanes = 2;
  glob_inf.itick = 640;
  glob_inf.tickRate = 40000;
  glob_inf.cycleRate = 150;
  glob_inf.darkCol = 0x000000; //0x000034;
  glob_inf.lightCol = 0xffffff; //0x989898;
  glob_inf.syncOne = 0;
  glob_inf.linkParam.link_type = LINK_TIE;
  glob_inf.linkParam.io_addr = VLINK0;
  glob_inf.linkParam.timeout = 10;
  glob_inf.linkParam.delay = 5;
  //glob_inf.linkParam.baudrate(BR9600);

  return ERR_NONE;
}

/*
  This is the THIRD function to call for loading a ROM image
  at startup
*/
extern int ti68k_loadRom(char *filename);

/*
  This is the FOURTH function to call for finishing complete initialization.
  The GUI should have been initialized also before calling this function
  (init_specific & init_keyboard).
  If you have a RAM image to load, load it before or preferrably after
  this function.
  Next, call doInstructions for running the emulator periodically.
*/
int ti68k_initLib68k(void)
{
  cb_init_specific();
  cb_screen_on();
  init_hardware();

  ti68k_resetLib68k();

#ifndef __WIN32__
  calibrate_pcounter();	// crash under Win32
#endif

  /* I have noticed that TI92+ does not flicker when LCD is refresh on IRQ1 */
  if(ti68k_getCalcType() == (TI92|MODULEPLUS))
     glob_inf.syncOne = 0;

  return ERR_NONE;
}

/*
  Simply reset the emulation core without allocating/freeing memory.
*/
extern int ti68k_resetLib68k(void)
{
  rom_at_0();
  MC68000_reset();
  ram_at_0();

  return ERR_NONE;
}

/*
  Close the library:
  - free hardware ressources (linkink)
  - exit GUI
  - free memory
 */
int ti68k_closeLib68k(void)
{
  if (!rom_loaded) 
    return 1;

  exit_hardware();
  //cb_screen_off();
  cb_exit_specific();
  mem_exit();
  
  rom_loaded = 0;

  return 1;
}


/*******************/
/* State functions */
/*******************/

int loadState_old(char *filename)
{
  FILE *fp;
  int m;

  DISPLAY(_("Loading RAM image (GtkTiEmu format): %s\n"), filename);
  if(!strcmp(filename, ""))
	  return ERR_NONE;
  if( (fp = fopen(filename, "rb")) == NULL)
    return ERR_CANT_OPEN;

  fread(&m, 1, sizeof(int), fp);
  if(m != RAM_SIZE) 
    {
      fclose(fp);
      return -2;
    }
  else
    {
      fread(ti_ram, 1, RAM_SIZE, fp);
      fread(ti_io, 1, IO_SIZE, fp);
      fread(&regs, sizeof(regs), 1, fp);
      fread(&timer_init, sizeof(timer_init), 1, fp);
      fread(&specialflags, sizeof(specialflags), 1, fp);
      fclose(fp);
      MakeFromSR();
      m68k_setpc(regs.pc);
      //update_contrast();
    }
  
  return ERR_NONE;
}

int ti68k_saveState_old(char *filename)
{
  FILE *fp;
  int m = RAM_SIZE;

  if(strlen(filename)) // name exist ?
    {
      if( (fp = fopen(filename, "wb")) == NULL)
	return ERR_CANT_OPEN;
      
      m68k_setpc(m68k_getpc());
      MakeSR();
      fwrite(&m, sizeof(int), 1, fp);     // write RAM_SIZE
      fwrite(ti_ram, 1, RAM_SIZE, fp);    // write RAM mem
      fwrite(ti_io, 1, IO_SIZE, fp);      // write IO  mem
      fwrite(&regs, sizeof(regs), 1, fp); // dump registers
      fwrite(&timer_init, sizeof(timer_init), 1, fp);       // write timer
      fwrite(&specialflags, sizeof(specialflags), 1, fp); // and flags
      fclose(fp);
    }

  return ERR_NONE;
}

int loadState_vti(char *filename)
{
  FILE *fp;
  int kbmask=0;
  int comError=0;
  int cpuCompleteStop=0;
  char str[9], romName[56];
  int i;
  UBYTE* mem[256];
  int int0Count = 0;

  DISPLAY(_("Loading RAM image (VTi format): %s\n"), filename);
  if(!strcmp(filename, ""))
    return ERR_NONE;
  if( (fp = fopen(filename, "rb")) == NULL)
    return ERR_CANT_OPEN;

  fread(str,8,1,fp);
  str[8]=0;
  if( strcmp(str,"VTIv2.0 ") && strcmp(str, "GTKTIEMU") )
    return ERR_INVALID_STATE;
  fread(romName,56,1,fp);
  //  if (strcmp(romName,romImage[currentROM].name))
  //  return;
  fread(&regs,sizeof(regs),1,fp);
  MakeFromSR();
  fread(ti_ram,256,1024,fp);
  fread(&memprotect,4,1,fp); fread(&ram128,4,1,fp);
  fread(&timer_value,4,1,fp); fread(&timer_init,4,1,fp);
  fread(&int0Count,4,1,fp);
  fread(&io0Bit7,4,1,fp); fread(&io0Bit2,4,1,fp);
  fread(&kbmask,4,1,fp);
  fread(&ram_wrap,4,1,fp); fread(&(mem_and),4,1,fp);
  fread(&comError,4,1,fp);
  fread(&transflag,4,1,fp); fread(&transbyte,4,1,fp);
  fread(&transnotready,4,1,fp);
  fread(&recvflag,4,1,fp); fread(&recvbyte,4,1,fp);
  fread(&rom_ret_or,4,1,fp);
  fread(&lcd_base_addr,4,1,fp); fread(&lcd_off,4,1,fp);
  fread(&contrast,4,1,fp);
  fread(&cpuCompleteStop,4,1,fp);
  for (i=0;i<32;i++)
    mem[i]=&ti_ram[(i&(mem_and>>16))<<16];

  // GtkTiEmu specific
  fread(ti_io, 1, IO_SIZE, fp);      // write IO
  fread(&specialflags, sizeof(specialflags), 1, fp); // and flags
  fclose(fp);
  
  return ERR_NONE;
}

/*
  This function saves emulator settings in the VTi format
*/
int saveState_vti(char *filename)
{
  FILE *fp;
  int kbmask=0;
  int comError=0;
  int cpuCompleteStop=0;
  int int0Count = 0;

  if(strlen(filename)) // name exist ?
    {
      if( (fp = fopen(filename, "wb")) == NULL)
	return ERR_CANT_OPEN;

      m68k_setpc(m68k_getpc());
      fprintf(fp,"GTKTIEMU");
      fwrite(filename,56,1,fp);
      MakeSR();
      fwrite(&regs,sizeof(regs),1,fp); // struct format differs !
      fwrite(ti_ram,256,1024,fp);
      fwrite(&memprotect,4,1,fp); fwrite(&ram128,4,1,fp);
      fwrite(&timer_value,4,1,fp); fwrite(&timer_init,4,1,fp);
      fwrite(&int0Count,4,1,fp);
      fwrite(&io0Bit7,4,1,fp); fwrite(&io0Bit2,4,1,fp);
      fwrite(&kbmask,4,1,fp);
      fwrite(&ram_wrap,4,1,fp); fwrite(&(mem_and),4,1,fp);
      fwrite(&comError,4,1,fp);
      fwrite(&transflag,4,1,fp); fwrite(&transbyte,4,1,fp);
      fwrite(&transnotready,4,1,fp);
      fwrite(&recvflag,4,1,fp); fwrite(&recvbyte,4,1,fp);
      fwrite(&rom_ret_or,4,1,fp);
      fwrite(&lcd_base_addr,4,1,fp); fwrite(&lcd_off,4,1,fp);
      fwrite(&contrast,4,1,fp);
      fwrite(&cpuCompleteStop,4,1,fp);
      
      // GtkTiEmu specific
      fwrite(ti_io, 1, IO_SIZE, fp);      // write IO
      fwrite(&specialflags, sizeof(specialflags), 1, fp); // and flags
      fclose(fp);
    }

  return ERR_NONE;
}

/*
  Must be done between init_hardware and M68000_run.
  Typically called after initLib68k.
  This function (re)load the state of the calculator.
  It automagically determine the state file format.

  Return an error code if an error occured, 0 otherwise
*/
int ti68k_loadState(char *filename)
{
  FILE *fp;
  char str[MAXCHARS];

  if(!strcmp(filename, ""))
    return ERR_NONE;
  if( (fp = fopen(filename, "rb")) == NULL)
    return ERR_CANT_OPEN;
  else
    {
      fgets(str, MAXCHARS, fp);
      if(strstr(str, "VTIv2.0 ") || strstr(str, "GTKTIEMU"))
	return loadState_vti(filename);
      else
	return loadState_old(filename);
    }

  return ERR_NONE;
}

/*
  This function save the state of the calculator.
  Can be called at any time.

  Return an error code if an error occured, 0 otherwise
*/
int ti68k_saveState(char *filename)
{
  return saveState_vti(filename);
}


/*****************/
/* ROM functions */
/*****************/

void* ti68k_getRomPtr(void) 
{
  return ti_rom;
}

int ti68k_getRomSize(void) 
{
  return glob_inf.romSize;
}

const char *ti68k_getRomVersion(void)
{
  return current_rom_info.version;
}

int ti68k_isRomOk(void)
{
  return rom_loaded;
}

int ti68k_getRomInfo(ROM_INFO *ri)
{
  ri = &current_rom_info;
  return ERR_NONE;
}

int ti68k_getUpdateInfo(TIB_INFO *ri)
{
  ri = &current_tib_info;
  return ERR_NONE;
}

/*****************/
/* RAM functions */
/*****************/

void* ti68k_getRamPtr(void) 
{
  return ti_ram;
}
	
int ti68k_getRamSize(void) 
{
  return glob_inf.ramSize;
}

int ti68k_dumpRam(char *filename)
{
  FILE *f;
  int i;
  int j;
  int c;
  UBYTE *ram = ti_ram;

  f = fopen(filename, "wt");
  if(f == NULL)
    return ERR_CANT_OPEN;

  for(i=0; i<1024*glob_inf.ramSize; i+=16) // size in KB
    {
      fprintf(f, "%06X: ", i);
      for(j=0; j<16; j++)
	{
	  fprintf(f, "%02X ", ram[i+j]);
	}
      fprintf(f, "| ");
      for(j=0; j<16; j++)
        {
	  if( (ram[i+j] > 31) && (ram[i+j] < 128) )
	    c = ram[i+j];
	  else
	    c = ' ';
          fprintf(f, "%c", c);
        }
      fprintf(f, "\n");
    }

  fclose(f);

  return ERR_NONE;
}


/**********************/
/* Debugger functions */
/**********************/

void ti68k_setInstructionsDisplay(int bVal) 
{
  bInstructionsDisplay = bVal;
}

int ti68k_disasm(int addr, char *output) 
{
  CPTR nextPc;

  MC68000_disasm(addr, &nextPc, 1, output);
  output[strlen(output)-1]='\0'; // strip CR-LF
  return (nextPc-addr);
}

/*
  This function defines the passed function (callback) 
  as current debugger. If you pass NULL, the current debugger
  is restored to internal debugger.
  Return the default callback.
*/
callback_iv_t ti68k_defineDebugger(callback_iv_t new_debugger)
{
  if(new_debugger == NULL)
    {
      cb_launch_debugger = enter_debugger;
    }
  else
    {
      cb_launch_debugger = new_debugger;
    }
  
  return enter_debugger;
}

int ti68k_launchDebugger(void)
{
  return cb_launch_debugger();
}

void ti68k_getBreakpointCause(int *type, int *id, int *mode) 
{
  *type = breakType;
  *mode = breakMode;
  *id = breakId;
}

int ti68k_doSingleStep(void)
{
  return specialflags |= SPCFLAG_DBTRACE;
}

/* Replace M68000_run() */
/*
  Do 'n' instructions.
  return ERR_NONE if successful, a negative value if an error occured,
  a positive value if a breakpoint has been encountered.
*/
int ti68k_doInstructions(int n) //fait n instructions
{
  int i;
  struct intlist *l;
  //char inst[100];
  
  if (!ti68k_isRomOk()) 
    return ERR_ROM_NOT_LOADED;
  
  for(i=0; i<n; i++) 
    {
      UWORD opcode;

      opcode = nextiword();
      (*cpufunctbl[opcode])(opcode);
      do_cycles();

      if(nBkptAddress) 
	{
	  l=listBkptAddress;
	  breakId = 0;
	  while (l) 
	    {
	      if (l->val==(int)regs.pc)
		{
		  //DISPLAY("l->val=0x%06x, pc=0x%06x\n", 
		  //  l->val, (int)regs.pc);
		  breakType = BK_CAUSE_ADDRESS;
		  cb_update_screen();
		  cb_launch_debugger();
		  return 1;
		}
	      breakId++;
	      l=l->next;
	    }
	}

      /* Debug purposes */
      /*
      if (bInstructionsDisplay) 
	{
	  disasm(getPcRegister(), inst);
	  printf("disasm: %s\n", inst);
	}
      */
      /* Flag management */      
      if(specialflags) 
	{
	  if(specialflags & SPCFLAG_ADRERR) 
	    {
	      Exception(3);
	      specialflags &= ~SPCFLAG_ADRERR;
	    }
	  
	  if (specialflags & SPCFLAG_DOTRACE) 
	    {
	      Exception(9);
	    }
	  
	  while (specialflags & SPCFLAG_STOP) 
	    {
	      do_cycles();
	      if (specialflags & (SPCFLAG_INT | SPCFLAG_DOINT)) 
		{
		  int intr = intlev();
		  specialflags &= ~(SPCFLAG_INT | SPCFLAG_DOINT);
		  if (intr != -1 && intr > regs.intmask) 
		    {
		      Interrupt(intr);
		      regs.stopped = 0;
		      specialflags &= ~SPCFLAG_STOP;
		    }	    
		}
	    }		
	  
	  if (specialflags & SPCFLAG_TRACE) 
	    {
	      specialflags &= ~SPCFLAG_TRACE;
	      specialflags |= SPCFLAG_DOTRACE;
	    }	  
	  if (specialflags & SPCFLAG_DOINT) 
	    {
	      int intr = intlev();
	      specialflags &= ~(SPCFLAG_INT | SPCFLAG_DOINT);
	      if (intr != -1 && intr > regs.intmask) {
		Interrupt(intr);
		regs.stopped = 0;
	      }	    
	    }
	  if (specialflags & SPCFLAG_INT) 
	    {
	      specialflags &= ~SPCFLAG_INT;
	      specialflags |= SPCFLAG_DOINT;
	    }
	  if (specialflags & SPCFLAG_BRK) 
	    {		
	      specialflags &= ~SPCFLAG_BRK;
	      cb_update_screen();
	      cb_launch_debugger();
	      return 1;		
	    }
	  if(specialflags & SPCFLAG_DBTRACE) 
	    {
	      specialflags &= ~SPCFLAG_DBTRACE;
	      cb_update_screen();
	      cb_launch_debugger();
	    }
	}  
    }

  return ERR_NONE;
}


/***********************************/
/* Breakpoints management routines */
/***********************************/

int ti68k_setBreakpointAddress(int address) 
{
  addEnd(&listBkptAddress, address);
  nBkptAddress++;
  return (nBkptAddress-1);
}

int ti68k_setBreakpointAddressRange(int addressMin, int addressMax, int mode) 
{
  if(mode & BK_READ) 
    {		
      add2End(&listBkptAsRgR,addressMin,addressMax);
      return nBkptAsRgR++;
    }
  if(mode & BK_WRITE) 
    {
      add2End(&listBkptAsRgW,addressMin,addressMax);
      return nBkptAsRgW++;
    }
  return ERR_INVALID_MODE;
}

int ti68k_setBreakpointAccess(int address, int mode) 
{
  if(mode==BK_READ_BYTE) 
    {		
      addEnd(&listBkptAsRB, address);
      return nBkptAsRB++;
    }
  else if(mode==BK_READ_WORD) 
    {
      addEnd(&listBkptAsRW, address);
      return nBkptAsRW++;
    }
  else if(mode==BK_READ_LONG) 
    {
      addEnd(&listBkptAsRL,address);
      return nBkptAsRL++;
    }
  else if (mode==BK_WRITE_BYTE) 
    {
      addEnd(&listBkptAsWB,address);
      return nBkptAsWB++;
    }
  else if (mode==BK_WRITE_WORD) 
    {
      addEnd(&listBkptAsWW,address);
      return nBkptAsWW++;
    }
  else if (mode==BK_WRITE_LONG) 
    {
      addEnd(&listBkptAsWL,address);
      return nBkptAsWL++;
    }
  else
    return ERR_INVALID_MODE;
}

int ti68k_setBreakpointVector(int vector) 
{
  assert( (vector >= 2) && (vector < 16) );
  listBkptVector[vector] = !0;
  nBkptVector++;
  return (nBkptVector-1);
}

int ti68k_setBreakpointAutoint(int autoint)
{
  assert( (autoint >= 1) && (autoint < 8) );
  listBkptAutoint[autoint] = !0;
  nBkptAutoint++;
  return (nBkptAutoint-1);
}

int ti68k_setBreakpointTrap(int trap) 
{
  assert( (trap >= 0) && (trap < 16) );
  listBkptTrap[trap] = !0;
  nBkptTrap++;
  return (nBkptTrap-1);
}


void ti68k_delBreakpointAddress(int i) 
{
  delAt(&listBkptAddress, i);
}

void ti68k_delBreakpointAccess(int i, int mode) 
{
  if (mode==BK_READ_BYTE) 
    {		
      delAt(&listBkptAsRB,i);
      nBkptAsRB--;
    }
  if (mode==BK_READ_WORD) 
    {
      delAt(&listBkptAsRW,i);
      nBkptAsRW--;
    }
  if (mode==BK_READ_LONG) 
    {
      delAt(&listBkptAsRL,i);
      nBkptAsRL--;
    }
  if (mode==BK_WRITE_BYTE) 
    {
      delAt(&listBkptAsWB,i);
      nBkptAsWB--;
    }
  if (mode==BK_WRITE_WORD) 
    {
      delAt(&listBkptAsWW,i);
      nBkptAsWW--;
    }
  if (mode==BK_WRITE_LONG) 
    {
      delAt(&listBkptAsWL,i);
      nBkptAsWL--;
    }
}

void ti68k_delBreakpointAccessRange(int i,int mode) 
{
  if (mode & BK_READ) 
    {
      delAt(&listBkptAsRgR,i);
      nBkptAsRgR--;
    }
  else if (mode & BK_WRITE) 
    {
      delAt(&listBkptAsRgW,i);
      nBkptAsRgW--;
    }
  else
    {
      DISPLAY(_("delBreakpointAccessRange: mode is neither READ nor WRITE. This is a bug. Please report it !\n"));
      exit(-1);
    }
}

void ti68k_delBreakpointVector(int vector) 
{
  assert( (vector >= 2) && (vector < 16) );
  listBkptVector[vector] = 0;
  nBkptVector--;
  //return (nBkptVector-1);
}

void ti68k_delBreakpointAutoint(int autoint)
{
  assert( (autoint >= 1) && (autoint < 8) );
  listBkptAutoint[autoint] = 0;
  nBkptAutoint--;
  //return (nBkptAutoint-1);
}

void ti68k_delBreakpointTrap(int trap) 
{
  assert( (trap >= 0) && (trap < 16) );
  listBkptTrap[trap] = 0;
  nBkptTrap--;
  //return (nBkptTrap-1);
}

/*****************************/
/* Registers access routines */
/*****************************/

void ti68k_setDataRegister(int n,int val) 
{
  if (n>=0 && n<8) regs.d[n] = val;
}

void ti68k_setAddressRegister(int n,int val) 
{
  if (n>=0 && n<8) regs.a[n] = val;
}

void ti68k_setSpRegister(int val) 
{
  regs.usp=val;
}

void ti68k_setPcRegister(int val) 
{
  regs.pc=val;
}

void ti68k_setSrRegister(int val) 
{
  regs.sr=val;
}

void ti68k_setFlagRegister(unsigned char flag) 
{
  //TODO
}

int ti68k_getDataRegister(int n) 
{
  if (n>=0 && n<8) return regs.d[n];
  return ERR_NONE;
}

int ti68k_getAddressRegister(int n) 
{
  if (n>=0 && n<8) return regs.a[n];
  return ERR_NONE;
}
	
int ti68k_getSpRegister() 
{
  return regs.usp;
}

int ti68k_getPcRegister() 
{
  return regs.pc;
}

int ti68k_getSrRegister() 
{
  return regs.sr;
}

char *ti68k_getFlagRegister(void) 
{
  static char str_SR[128];
  sprintf(str_SR, "T=%d S=%d X=%d N=%d\nZ=%d V=%d C=%d IMASK=%d\n",
	  regs.t, regs.s, regs.x, NFLG, ZFLG, VFLG, CFLG, regs.intmask);
  return str_SR;
}

/******************/
/* Misc functions */
/******************/

void* ti68k_getLcdPtr(void) 
{
  UWORD arg = (((UWORD)ti_io[0x10])<<8)|ti_io[0x11];
  return (&ti_ram[arg<<3]);
  //  return (&ti_ram[lcd_base_addr]);
}

char ti68k_getContrast(void) 
{
  return contrast;
  //return ( ((ti_io[0x1d] & 0xf) << 1) | ((ti_io[0] >> 5) & 1) );
  //return (io_get_byte(0x1d)&0x0e);
}

int ti68k_getCalcType(void)
{
  return current_rom_info.calc_type;
}


/******************/
/* File functions */
/******************/
extern int cmdState;
extern UBYTE *backBuf;
/*
  Remark: there is a bug here... If the buffer is allocated with malloc, GtkTiEmu will segfault (backBuf address out of bounds). If the buffer is allocated on the heap as an array, there will be no problem. Moreover, this problem does not appear when we send a file via the command mode, only in GUI mode.
Then, there may be a problem of shared memory or something else...
*/
int ti68k_sendFile(char *filename)
{
  return send_ti_file(filename);
}

extern int TO_VALUE;

int ti68k_setInternalLinkTimeout(int value)
{
  TO_VALUE = value;
  return value;
}

int ti68k_getInternalLinkTimeout(int value)
{
  return TO_VALUE;
}

extern LinkCable *ilc;
extern TicalcFncts itc;

int ti68k_setLinkCable(LinkCable *lc)
{
  ilc = lc;
  switch(ti68k_getCalcType())
    {
    case TI92: ticalc_set_calc(CALC_TI92, &itc, ilc);
      break;
    case TI89: ticalc_set_calc(CALC_TI89, &itc, ilc);
      break;
    case TI92 | MODULEPLUS: ticalc_set_calc(CALC_TI92P, &itc, ilc);
      break;
    default: return ERR_INTERNAL;
      break;
    }
}

LinkCable* ti68k_getLinkCable()
{
  return ilc;
}

/*
  This function defines the passed function (callback)
  as current progress bar for link port operations. 
  If you pass NULL, the current progress bar is restored to internal 
  progress bar function.
  Return the default callback.
*/
callback_vi_t ti68k_defineProgbar(callback_vi_t new_progbar)
{
  if(new_progbar == NULL)
    {
      cb_update_progbar = cmd_update_progbar;
    }
  else
    {
      cb_update_progbar = new_progbar;
    }

  return cmd_update_progbar;
}

void ti68k_updateProgbar(int size)
{
  return cb_update_progbar(size);
}

void link_progress(int type, char *name, int size)
{
  cmd_link_progress(type, name, size);
}

