/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifndef OCICursor_H
#define OCICursor_H

#include <connect.h>
#include <vector>
#include <TeCoord2D.h>

using namespace std;

// Object SDO_GEOMETRY - Oracle Spatial
#define TYPE_OWNER				"MDSYS"
#define SDO_ORDINATE_ARRAY      TYPE_OWNER".SDO_ORDINATE_ARRAY"
#define SDO_ELEM_INFO_ARRAY     TYPE_OWNER".SDO_ELEM_INFO_ARRAY"
#define SDO_GEOMETRY			TYPE_OWNER".SDO_GEOMETRY"

#define MAX_ROWS				500	// number of lines to be fetched into memory

// type point in Oracle Spatial
struct sdo_point_type
{
   OCINumber x;
   OCINumber y;
   OCINumber z;
};

typedef struct sdo_point_type sdo_point_type;

//type geometry in oracle spatial
struct sdo_geometry
{
   OCINumber       sdo_gtype;
   OCINumber       sdo_srid;
   sdo_point_type  sdo_point;
   OCIArray        *sdo_elem_info;
   OCIArray        *sdo_ordinates;
};

typedef struct sdo_geometry SDO_GEOMETRY_TYPE;


//Indicator structures for SDO_GEOMETRY_TYPE
struct sdo_point_type_ind
{
   OCIInd _atomic;
   OCIInd x;
   OCIInd y;
   OCIInd z;
};

typedef struct sdo_point_type_ind sdo_point_type_ind;

struct SDO_GEOMETRY_ind
{
   OCIInd                    _atomic;
   OCIInd                    sdo_gtype;
   OCIInd                    sdo_srid;
   struct sdo_point_type_ind sdo_point;
   OCIInd                    sdo_elem_info;
   OCIInd                    sdo_ordinates;
};

typedef struct SDO_GEOMETRY_ind SDO_GEOMETRY_ind;

struct sb2ind
{
	sb2		sbind[MAX_ROWS];
};
typedef struct sb2ind	indarray;


//! Oracle generic Cursor class 

class OCICursor
{
protected:
	OCIConnection*		conn;			// pointer to connection, environment, etc
	OCIStmt*			stmthpToQuery;  // OCI statement handle to query
	OCIDescribe*		dschp;			//environment describe handle - contem informacoes sobre esquemas de objetos
	bool				isOpen_;		// if the cursor is open or not (if the stmthpToQuery was allocated)
	string				errorMessage;
			
	
	//-------- buffers to store data in memory - client side
	vector<void *>			buffers;	// buffer to store simple data type (varchar, number, date, etc)
	vector <indarray>		ind;		// buffer to store null indicator
				
	SDO_GEOMETRY_TYPE*		global_geom_obj[MAX_ROWS];	// buffer to store spatial object  
	SDO_GEOMETRY_ind*		global_geom_ind[MAX_ROWS];	// buffer to store spatial object indicator  

	OCILobLocator*			lobp[MAX_ROWS];				// buffer to store blob
	
	vector<OCIDefine *>		defines;	// OCIDefine to link buffers in client side and the statement handle

	string					fieldValue;

	OCIArray*				ordinates;		//store the ordinates to be used in a query (bind)

	//-------- information to handle the cursor
	short				row_Index;		// relative current row index in the memory buffer
	int					row_Cur;		// absolute current cursor row
	int					rows_Fetched;	// rows fetched from database
	int					rows_Mem;		// rows on memory buffer - client side
	bool				last_Row;		// if is last row
	

	//-------- information about columns of the cursor
	int					numColls_;  // number of columns
	vector<string>		colName_;	// name of columns
	vector<int>			colType_;	// type of columns
	vector<int>			colSize_;	// size of columns
	vector<int>			colScale_;  // sclae of columns

public:
	
	//! Cursor constructor
	OCICursor(OCIConnection* conn);

	//! Cursor destructor
	~OCICursor() { Close(); }

	//! Open the Cursor
	bool		Open();

	
	//! return the error message 
	string			getErrorMessage() { return errorMessage;}
	//! Check if the cursor is open
	bool			isOpen()	{ return isOpen_; }
	OCIConnection*	Conn()		{ return conn; }
	//! Return the current row 
	int				CurrentRow() { return row_Cur; }


	//! Close the Cursor
	void			Close();
	
	//! Define
	void		DefineByPos(int pos, void* value, int size, void* indicator, int type);

	//! Execute and Fetch
	bool		Execute();
	bool		Fetch(int rows);

	bool		Query(const string& query);
	bool		QuerySDO(const string& query);

	//! Move operations
	bool		MoveFirst();
	bool		MoveNext();
	bool		MoveTo(int pos);
	bool		MoveLast();
	
	//! Return the columns number 
	int			NumCol(void);

	//! Return the rows number 
	int			NumRows(void);
	
	//! Load all column descriptions (name, type, size and scale)
	void		loadCollDescription (); 
	
	//! Return the column type (SQLT)
	int			ColType (int colnumber);

	//! Return the column name 
	string		ColName (int colnumber);

	//! Return the column size 
	int			ColSize (int colnumber);

	//! Return the column scale (number of digits after decimal point) 
	int			ColScale (int colnumber);

	bool		AppendOrdinates(const double& val);
	bool		BindOrdinates();
	
	bool		Prepare(const string& stmt);
	int			QueryType(); 

	bool		AllocateCursor(void);
	char*		GetFieldValue(int i);
	int			GetDimArraySize();
	bool		GetDimElement(int i,int &elem);
	int			GetNumberOrdinates(void);
	bool		GetCoordinates(int i,TeCoord2D& coord);
	bool		GetCoordinates(vector<TeCoord2D>& result);
	int			GetGeometryType();
	int			GetSpatialReferenceId();
	bool		GetXYZcoord (double& x, double& y);
	void		FreeResult(void);

	bool		WriteBlob(const unsigned char* buffer, unsigned int bufsize);
	int			SizeBlob();
	bool		ReadBlob(unsigned char* buffer, unsigned int bufsize);

protected:
	// Throws exception if error found
	bool		CheckError(sword status);
};

#endif // OCICursor_H

