/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <sys/param.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>

#include "libsysactivity.h"
#include "utils.h"

static void create_keys();
static void assign(struct sa_process* dst, struct kinfo_proc2* kinfo_proc);

long pagesize;
pthread_key_t processes_key;
pthread_key_t processes_size_key;

int sa_open_process() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_processes(pid_t* number) {
	if (number == NULL)
		return EINVAL;

	int mib[6];
	size_t len;

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC2;
	mib[2] = KERN_PROC_ALL;
	mib[3] = 0;
	mib[4] = sizeof(struct kinfo_proc2);
	mib[5] = 0;

	if (sysctl(mib, 6, NULL, &len, NULL, 0) == -1)
		return ENOSYS;
// TODO sysctl says that there are more processes than actually are. I guess the second sysctl call is right.
	*number = len / sizeof(struct kinfo_proc2);
	return 0;
}

int sa_get_processes_ids(pid_t* dst, pid_t dst_size, pid_t* written) {
	if (dst == NULL || dst_size <= 0 || written == NULL)
		return EINVAL;

	int mib[6];
	size_t len;

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC2;
	mib[2] = KERN_PROC_ALL;
	mib[3] = 0;
	mib[4] = sizeof(struct kinfo_proc2);
	mib[5] = 0;

	if (sysctl(mib, 6, NULL, &len, NULL, 0) == -1)
		return ENOSYS;

	size_t processes_size = (size_t) pthread_getspecific(processes_size_key);
	struct kinfo_proc2* processes = pthread_getspecific(processes_key);
	if (processes_size < len) {
		processes = realloc(processes, len);
		if (pthread_setspecific(processes_key, processes))
			return ENOSYS;
		if (processes == NULL)
			return ENOMEM;
		processes_size = len;
		if (pthread_setspecific(processes_size_key, (void *) processes_size))
			return ENOSYS;
	}

	mib[5] = len / sizeof(struct kinfo_proc2);
	if (sysctl(mib, 6, processes, &len, NULL, 0) == -1)
		return ENOSYS;

	int i, amount;
	*written = 0;
	amount = len / sizeof(struct kinfo_proc2);
	for (i = 0; i < amount; i++) {
		if (i == dst_size)
			return ENOMEM;

		dst[i] = processes[i].p_pid;
		(*written)++;
	}
	return 0;
}

int sa_get_process(pid_t pid, struct sa_process* dst) {
	if (pid == 0 || dst == NULL)
		return EINVAL;

	int mib[6];
	struct kinfo_proc2 proc;
	size_t len = sizeof proc;

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC2;
	mib[2] = KERN_PROC_PID;
	mib[3] = pid;
	mib[4] = len;
	mib[5] = 1;
	proc.p_paddr = 0;
	if (sysctl(mib, 6, &proc, &len, NULL, 0) == -1)
		return ENOSYS; // TODO Return something more specific in case the process does not exist

	if (proc.p_paddr == 0)
		return ESRCH;

	assign(dst, &proc);
	return 0;
}

int sa_get_processes(struct sa_process* dst, pid_t dst_size, pid_t* written) {
	if (dst == NULL || dst_size <= 0 || written == NULL)
		return EINVAL;

	int mib[6];
	size_t len;

	mib[0] = CTL_KERN;
	mib[1] = KERN_PROC2;
	mib[2] = KERN_PROC_ALL;
	mib[3] = 0;
	mib[4] = sizeof(struct kinfo_proc2);
	mib[5] = 0;

	if (sysctl(mib, 6, NULL, &len, NULL, 0) == -1)
		return ENOSYS;

	size_t processes_size = (size_t) pthread_getspecific(processes_size_key);
	struct kinfo_proc2* processes = pthread_getspecific(processes_key);
	if (processes_size < len) {
		processes = realloc(processes, len);
		if (pthread_setspecific(processes_key, processes))
			return ENOSYS;
		if (processes == NULL)
			return ENOMEM;
		processes_size = len;
		if (pthread_setspecific(processes_size_key, (void *) processes_size))
			return ENOSYS;
	}

	mib[5] = len / sizeof(struct kinfo_proc2);
	if (sysctl(mib, 6, processes, &len, NULL, 0) == -1)
		return ENOSYS;

	int i, amount;
	*written = 0;
	amount = len / sizeof(struct kinfo_proc2);
	for (i = 0; i < amount; i++) {
		if (i == dst_size)
			return ENOMEM;

		assign(&dst[i], &processes[i]);
		(*written)++;
	}
	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pagesize = sysconf(_SC_PAGESIZE);
	pthread_key_create(&processes_key, free_keys);

	keys_created = 1;
}

static void assign(struct sa_process* dst, struct kinfo_proc2* proc) {
	dst->pid = proc->p_pid;
	dst->uid = proc->p_uid;
	dst->gid = proc->p_gid;
	// TODO: get the proper third parameter at compile time
	strlcpy(dst->cmdline, proc->p_comm, KI_MAXCOMLEN);
	dst->parent_pid = proc->p_ppid;
	dst->pgrp = proc->p__pgid;
	dst->sid = proc->p_sid;
	dst->tty = proc->p_tdev;
	dst->nice = proc->p_nice - 20;
	dst->start_time = proc->p_ustart_sec * 100 + proc->p_ustart_usec / 10000;
	dst->vm_size = proc->p_vm_map_size;
	dst->rss = proc->p_vm_rssize * pagesize;
}
