/* Copyright 2004, 2005, 2006 Acegi Technology Pty Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.security;

/**
 * Makes a final access control (authorization) decision.
 *
 * @author Ben Alex
 * @version $Id: AccessDecisionManager.java 2217 2007-10-27 00:45:30Z luke_t $
 */
public interface AccessDecisionManager {
    //~ Methods ========================================================================================================

    /**
     * Resolves an access control decision for the passed parameters.
     *
     * @param authentication the caller invoking the method
     * @param object the secured object being called
     * @param config the configuration attributes associated with the secured object being invoked
     *
     * @throws AccessDeniedException if access is denied as the authentication does not hold a required authority or
     *         ACL privilege
     * @throws InsufficientAuthenticationException if access is denied as the authentication does not provide a
     *         sufficient level of trust
     */
    void decide(Authentication authentication, Object object, ConfigAttributeDefinition config)
        throws AccessDeniedException, InsufficientAuthenticationException;

    /**
     * Indicates whether this <code>AccessDecisionManager</code> is able to process authorization requests
     * presented with the passed <code>ConfigAttribute</code>.<p>This allows the
     * <code>AbstractSecurityInterceptor</code> to check every configuration attribute can be consumed by the
     * configured <code>AccessDecisionManager</code> and/or <code>RunAsManager</code> and/or
     * <code>AfterInvocationManager</code>.</p>
     *
     * @param attribute a configuration attribute that has been configured against the
     *        <code>AbstractSecurityInterceptor</code>
     *
     * @return true if this <code>AccessDecisionManager</code> can support the passed configuration attribute
     */
    boolean supports(ConfigAttribute attribute);

    /**
     * Indicates whether the <code>AccessDecisionManager</code> implementation is able to provide access
     * control decisions for the indicated secured object type.
     *
     * @param clazz the class that is being queried
     *
     * @return <code>true</code> if the implementation can process the indicated class
     */
    boolean supports(Class clazz);
}
