/**
 * GUI Commands
 * Copyright 2005 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: DelegatingCommand.java,v 1.9 2007/01/11 08:28:38 pietschy Exp $
 */
package org.pietschy.command.delegate;

import org.pietschy.command.ActionCommand;
import org.pietschy.command.CommandManager;
import org.pietschy.command.ActionCommandExecutor;

import javax.swing.undo.UndoableEditSupport;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * Delegating commands allow a single command to delegate its behaviour to another object.  This is most useful when
 * the commands actaul behaviour is dependant on the current context of the UI.  A typical examples would be cut, paste
 * and print.
 * <p>
 * The behaviour of the command is provided by instances of {@link CommandDelegate} which are
 * automatically extracted from the {@link DelegateContainer} once
 * {@link #trackDelegateIn(String, java.awt.Window)} has been invoked.
 * <p>
 * DelegatingCommands are automatically bound to {@link CommandDelegate delegates} whose id match their own.
 *
 * @author andrewp
 * @version $Revision: 1.9 $
 * @see DelegateContainer
 * @see CommandDelegate
 */
public class
DelegatingCommand
extends ActionCommand
implements DelegateMediatorListener
{
   // Constants and variables
   // -------------------------------------------------------------------------
   private static final String _ID_ = "$Id: DelegatingCommand.java,v 1.9 2007/01/11 08:28:38 pietschy Exp $";

   private UndoableEditSupport undoSupport = new UndoableEditSupport(this);

   private DelegateMediator delegateManager;
   private ActionCommandExecutor delegate;
   private String delegateIdToTrack;

   private PropertyChangeListener delegateListener = new PropertyChangeListener()
   {
      public void propertyChange(PropertyChangeEvent evt)
      {
         updateState();
      }
   };

   /**
    * Creates a new DelegatingCommand with the specified id.  This command will be bound to
    * {@link CommandManager#defaultInstance()}
    *
    * @param id the id of the command.
    */
   public DelegatingCommand(String id)
   {
      super(id);
      updateState();
   }

   /**
    * Creates a new DelegatingCommand with the specified id and that is bound to the
    * specifed {@link org.pietschy.command.CommandManager}.
    *
    * @param commandManager the CommandManager to which the command is bound.
    * @param commandId      the id of the command.
    */
   public DelegatingCommand(CommandManager commandManager, String commandId)
   {
      super(commandManager, commandId);
      updateState();
   }

   /**
    * Sets the delegate id this command is to track.
    *
    * @param delegateIdToTrack the id of the delegate that this command will invoke.
    */
   protected void setDelegateIdToTrack(String delegateIdToTrack)
   {
      this.delegateIdToTrack = delegateIdToTrack;
   }

   /**
    * Gets the delegate id this command is tracking.
    *
    * @return the id of the delegate that this command will invoke.
    */
   private String getDelegateIdToTrack()
   {
      return delegateIdToTrack;
   }


   /**
    * This method configures the command to track delegates in the specified {@link java.awt.Window}.  The implementation
    * simply adds the command as a listener to the {@link DelegateManager} of the specified window.
    * I.e.
    * <pre>
    * DelegateManager manager = DelegateManager.getInstanceFor(window);
    * manager.addDelegateManagerListener(this);
    * </pre>
    *
    * @param delegateId the id of the delegate to track.
    * @param window     the window in which to track delegates.
    * @return <code>this</code> as a convenience for creating a new commannd on one line, <tt>new DelegatingCommand(...).trackDelegatesIn(win).export();</tt>.
    */
   public DelegatingCommand
   trackDelegateIn(String delegateId, Window window)
   {
      if (delegateManager != null)
      {
         delegateManager.removeDelegateTrackerListener(this);
      }
      else
      {
         delegateManager = DelegateManager.getMediatorFor(window);
         delegateManager.addDelegateTrackerListener(this);
      }

      setDelegateIdToTrack(delegateId);

      return this;
   }

   protected void handleExecute()
   {
      if (delegate != null)
      {
         delegate.execute(getHints());
      }
   }

   /**
    * Sets the delegate for this command to use.  This method is automatically called when
    * {@link #trackDelegateIn(String, java.awt.Window)} is invoked.
    *
    * and should not normally be used.
    *
    * @param newDelegate the delegate this command is to use.
    * @see #trackDelegateIn(String, java.awt.Window)
    */
   protected void
   setDelegate(ActionCommandExecutor newDelegate)
   {
      if (delegate != null)
      {
         delegate.removePropertyChangeListener(delegateListener);
      }

      delegate = newDelegate;

      if (this.delegate != null)
      {
         delegate.addPropertyChangeListener(delegateListener);
      }

      updateState();
   }

   protected ActionCommandExecutor getDelegate()
   {
      return delegate;
   }

   private void
   updateState()
   {
      setEnabled(delegate != null && delegate.isEnabled());
   }

   public void delegatesChanged(DelegateMediatorEvent event)
   {
      String delegateId = getDelegateIdToTrack();
      if (getDelegateIdToTrack() != null) ;
      {
         ActionCommandExecutor executor = event.getExcecutor(delegateId);
         setDelegate(executor);
      }
   }

}
