/***************************************************************************
                          report.cpp  -  description
                             -------------------
    begin                : Wed Dec 19 2001
    copyright            : (C) 2001 by Will DeRousse
    email                : badhack@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// for holding html functions outside of dbmusic.cpp.

#include "dbmusic.h"
/* Put here so it's not automatically included by dbinit.h */
#include "dbmusicvalidator.h"
#include <qstring.h>
#include <klocale.h>

/*
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::collectionReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	QString tlen,tnod;
	QString short1, short2;
	// zero string and then select the total number of discs and the total length based on
	// report type
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Collection Report"),i18n("collection"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"7",i18n("Collection Report"),info);

	short1="SELECT SUM(t.cdlen) FROM title t";
	short2="SELECT a.name, t.name, t.nod FROM artist a, title t";

	sqlstring.truncate(0);
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	queryBoolStatus(sqlstring,"t.collected",info);
	short1.append(sqlstring);

	cerr << "collectionReport: " << short1 << endl;
	stat=Exec(short1.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	tlen=GetValue(0,0);
	if (tlen.isEmpty())
		tlen="00:00:00";

	// add t.aid=a.aid to short2
	queryMatch(sqlstring,"a.aid","t.aid");
	// now add sorting for short2 and Exec
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append("a.name");
	if (info->sort_time)
		list->append("t.yearstart");
	else
		list->append("t.name");
	queryOrderBy(sqlstring,list,info);
	short2.append(sqlstring);

	// execute query
	cerr << "collectionReport: " << short2 << endl;
	stat=Exec(short2.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	tnod.setNum(Tuples());

	list->clear();
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("No"));
	list->append("<pre>  </pre>");
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("No"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	list->append("title");
	list->append("title");
	HTMLPopulateData(this,HTML,list,1);
	delete list;

	HTMLSumHeader(HTML,"7",i18n("Total Titles"),tnod);
	HTMLSumHeader(HTML,"7",i18n("Total Length"),tlen);

	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::dateRangeReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	QString tlen,tnod;
	QString short1, short2;
	// zero string and then select the total number of discs and the total length based on
	// report type
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Date Range Report"),i18n("daterange"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"7",i18n("Date Range Report"),info);

	short1="SELECT SUM(t.cdlen) FROM title t";
	short2="SELECT a.name, t.name, t.yearstart FROM artist a, title t";
	sqlstring.truncate(0);
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	queryBoolStatus(sqlstring,"t.collected",info);
	short1.append(sqlstring);

	cerr << "daterangeReport: " << short1 << endl;
	stat=Exec(short1.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	tlen=GetValue(0,0);
	if (tlen.isEmpty())
		tlen="00:00:00";

	// add t.aid=a.aid to short2
	queryMatch(sqlstring,"a.aid","t.aid");
	// now add sorting for short2 and Exec
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	if (info->sort_time)
		list->append("t.yearstart");
	else
		list->append("t.name");
	list->append("a.name");
	queryOrderBy(sqlstring,list,info);
	short2.append(sqlstring);

	// execute query
	cerr << "daterangeReport: " << short2 << endl;
	stat=Exec(short2.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	tnod.setNum(Tuples());

	list->clear();
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("Year"));
	list->append("<pre>  </pre>");
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("Year"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	list->append("title");
	list->append("title");
	HTMLPopulateData(this,HTML,list,1);
	delete list;

	HTMLSumHeader(HTML,"7",i18n("Total Titles"),tnod);
	HTMLSumHeader(HTML,"7",i18n("Total Length"),tlen);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::artistReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	QString tnod;
	// zero string and then select the total number of discs and the total length based on
	// report type
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Artist Report"),i18n("artist"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"7",i18n("Artist Report"),info);

	// assemble the query for the report and get the total
	sqlstring="SELECT name, first, last, follow FROM artist";
	queryRange(sqlstring,"last","first",info);
	queryBoolStatus(sqlstring,"follow",info);
	// add sorting option
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	if (info->sort_time)
		list->append("first");
	else
		list->append("name");
	queryOrderBy(sqlstring,list,info);

	// execute query
	cerr << "artisReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	tnod.setNum(Tuples());

	list->clear();
	list->append(i18n("Artist"));
	list->append(i18n("First"));
	list->append(i18n("Last"));
	list->append("<pre>  </pre>");
	list->append(i18n("Artist"));
	list->append(i18n("First"));
	list->append(i18n("Last"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	list->append("title");
	list->append("title");
	HTMLPopulateData(this,HTML,list,1);
	delete list;

	HTMLSumHeader(HTML,"7",i18n("Total Artists"),tnod);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::bteReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	QString tnod;
	// zero string and then select the total number of discs and the total length based on
	// report type
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Bands To Explore Report"),i18n("bte"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"3",i18n("Bands To Explore Report"),info);

	// assemble the query for the report
	sqlstring="SELECT name FROM explore";
	// add sorting option
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append("name");
	queryOrderBy(sqlstring,list,info);

	// execute query
	cerr << "bteReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	// get total in int form
	tnod.setNum(Tuples());

	list->clear();
	list->append(i18n("Artist"));
	list->append("<pre>  </pre>");
	list->append(i18n("Artist"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	HTMLPopulateData(this,HTML,list,1);
	delete list;

	HTMLSumHeader(HTML,"3",i18n("Total Bands To Explore"),tnod);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::statisticsReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	// this is an intense, I/O intensive function.
	// I have tried to make this as painless as possible.

	// many variables
	QString totalTrackLength, totalArtists, totalTitle, totalTrack, totalTitleCollected, totalTitleNotCollected;
	QString decadeTotal[6];
	QString decadePercent[6];
	QString ratingAverage, qualityAverage;
	float denominator;
	QString result;
	QStringList *list, *list2, *list3;
	QStringList::Iterator it;

	//---------------------------------
	// get information on artists, titles and tracks
	// total length of all tracks
	sqlstring="SELECT SUM(tr.length) FROM track tr, title t WHERE t.cdid=tr.cdid";
	queryBoolStatus(sqlstring,"t.collected",info);
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TRACK_SUM_FAIL;
	totalTrackLength=GetValue(0,0);
	if (totalTrackLength.isEmpty())
		totalTrackLength="00:00:00";

	// get total number of artists
	int s=totalArtist(totalArtists);
	if (s < 0)
		return s;

	// get total number of titles
	sqlstring="SELECT COUNT(t.cdid) FROM title t";
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	totalTitle=GetValue(0,0);

	// get total number of tracks
	sqlstring="SELECT COUNT(tr.trackid) FROM track tr, title t WHERE t.cdid=tr.cdid";
	queryBoolStatus(sqlstring,"t.collected",info);
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TRACK_SUM_FAIL;
	totalTrack=GetValue(0,0);

	// get total number of collected titles
	sqlstring="SELECT COUNT(t.cdid) FROM title t WHERE t.collected=true";
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	totalTitleCollected=GetValue(0,0);

	// calculate total number of not collected titles
	totalTitleNotCollected.setNum(totalTitle.toLong()-totalTitleCollected.toLong());

	cerr << "Calculating quality and rating averages" << endl;
	// process the rating and quality averages
	sqlstring="SELECT AVG(rating) FROM title";
	queryBoolStatus(sqlstring,"collected",info);
	queryRange(sqlstring,"yearstart","yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	ratingAverage=GetValue(0,0);

	sqlstring="SELECT AVG(quality) FROM title";
	queryBoolStatus(sqlstring,"collected",info);
	queryRange(sqlstring,"yearstart","yearstart",info);
	cerr << "statsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	qualityAverage=GetValue(0,0);

	//--------------------------------- generate totals and percentages on 6 decades
	list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1949 AND yearstart < 1960");
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1959 AND yearstart < 1970");
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1969 AND yearstart < 1980");
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1979 AND yearstart < 1990");
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1989 AND yearstart < 2000");
	list->append("SELECT COUNT(cdid) FROM title WHERE yearstart > 1999 AND yearstart < 2010");
	int i=0;
	for (it=list->begin();it!=list->end();++it)
	{
		queryBoolStatus(*it,"collected",info);
		cerr << (*it) << endl;
		stat=Exec((*it).utf8().data());
		if (execCheck() < 0)
		{
			delete list;
			return CATEGORY_SUM_FAIL;
		}
		decadeTotal[i]=GetValue(0,0);
		i++;
	}

	// process events to keep the GUI updated.
	kapp->processEvents();

	cerr << "statsReport: calculating decade percentages" << endl;
	switch(info->report_type)
	{
		case REPORT_ALL:
			sqlstring=totalTitle;
			break;
		case REPORT_YES:
			sqlstring=totalTitleCollected;
			break;
		default:
			sqlstring=totalTitleNotCollected;
	};
	for (int i=0;i<6;i++)
		decadePercent[i].setNum(100*decadeTotal[i].toFloat()/sqlstring.toFloat());

	//--------------------------------------------------------------------------------------------------
	// we must compute the demoniator based on the report type.
	cerr << "statsReport: Calculating denominator" << endl;
	switch(info->report_type)
	{
		case REPORT_ALL:
			denominator=totalTitle.toFloat();
			break;
		case REPORT_YES:
			denominator=totalTitleCollected.toFloat();
			break;
		default:
			denominator=totalTitleNotCollected.toFloat();
	};

	//-------------------------------------------------------------------- now we need to query for category totals and names
	cerr << "Calculating category totals" << endl;
	list3=new QStringList();
	if (!list3)
	{
		delete list;
		return NO_MEMORY;
	}
	list3->append("SELECT c.name, COUNT(t.cdid) FROM format c, title t WHERE t.fid=c.fid");
	list3->append("SELECT c.name, COUNT(t.cdid) FROM label c, title t WHERE t.lid=c.lid");
	list3->append("SELECT c.name, COUNT(t.cdid) FROM genre c, title t WHERE t.gid=c.gid");
	list3->append("SELECT c.name, COUNT(t.cdid) FROM type c, title t WHERE t.tid=c.tid");
	list3->append("SELECT c.name, COUNT(t.cdid) FROM method c, title t WHERE t.mid=c.mid");
	// add date range and collected stuff
	list2=new QStringList();
	if (!list2)
	{
		delete list;
		delete list3;
		return NO_MEMORY;
	}
	if (info->sort_time)
	{
		list2->append("count");
		list2->append("c.name");
	}
	else
	{
		list2->append("c.name");
		list2->append("count");
	}
	for (it=list3->begin();it!=list3->end();it++)
	{
		queryBoolStatus(*it,"t.collected",info);
		queryRange(*it,"t.yearstart","t.yearstart",info);
		(*it).append(" GROUP BY c.name");
		queryOrderBy(*it,list2,info);
		cerr << *it << endl;
	}
	it=list3->begin();

	//--------------------------------------------------------------------------------------------------
	cerr << "statsReport: Starting report" << endl;
	kapp->processEvents();
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Statistics Report"),i18n("statistics"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"2",i18n("Statistics Report"),info);

	//---------------------------------
	HTMLSimpleTableHeader(HTML,"2",i18n("TOTALS"));
	HTML.append("<TR><TD CLASS=misc>" + i18n("ALL ARTISTS") + "</TD><TD CLASS=misc>" + i18n("ALL TITLES") + "</TD></TR>\n");
	HTML.append("<TR><TD CLASS=artist>");
	HTML.append(totalArtists);
	HTML.append("</TD><TD CLASS=title>");
	HTML.append(totalTitle);
	HTML.append("</TD></TR>\n<TR><TD CLASS=misc>");
	HTML.append(i18n("TITLES COLLECTED"));
	HTML.append("</TD><TD CLASS=misc>" + i18n("TITLES NOT COLLECTED") + "</TD></TR>\n<TR><TD CLASS=title>");
	HTML.append(totalTitleCollected);
	HTML.append("</TD><TD CLASS=title>");
	HTML.append(totalTitleNotCollected);
	HTML.append("</TD></TR>\n<TR><TD CLASS=misc>");
	HTML.append(i18n("NUMBER OF TRACKS") + "</TD><TD CLASS=misc>");
	HTML.append(i18n("LENGTH OF TRACKS") + "</TD></TR>\n<TR><TD CLASS=track>");
	HTML.append(totalTrack);
	HTML.append("</TD><TD CLASS=track>");
	HTML.append(totalTrackLength);
	HTML.append("</TD></TR><TR><TD CLASS=misc>");
	HTML.append(i18n("AVERAGE RATING") + "</TD><TD CLASS=misc>");
	HTML.append(i18n("AVERAGE QUALITY") + "</TD></TR>\n");

	HTML.append("<TR><TD CLASS=title>");
	HTML.append(ratingAverage);
	HTML.append("</TD><TD CLASS=title>");
	HTML.append(qualityAverage);
	HTML.append("</TD></TR>");

	HTML.append("\n</TABLE>\n");

	//---------------------------------
	cerr << "statsReport: Starting category tables in report" << endl;
	list->clear();
	list->append(i18n("Name"));
	list->append(i18n("Total"));
	list->append(i18n("Percent"));
	list2->clear();
	list2->append("artist");
	list2->append("title");
	list2->append("title");

	HTML.append("\n<TABLE WIDTH=\"100%\"><TR><TD>\n");

	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY FORMAT"));
	HTMLColumnHeaders(HTML,list);

	stat=Exec((*it).utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		delete list2;
		delete list3;
		return CATEGORY_SUM_FAIL;
	}
	HTMLPopulateData2(this,HTML,denominator,list2,info);

	result=result.setNum(formatmap->count());
	HTMLSumHeader(HTML,"3",i18n("Total Formats"),result);
	HTML.append("</TABLE>\n\n");

	//---------------------------------
	HTML.append("\n<TD>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY LABEL"));
	HTMLColumnHeaders(HTML,list);

	it++;
	stat=Exec((*it).utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		delete list2;
		delete list3;
		return CATEGORY_SUM_FAIL;
	}
	HTMLPopulateData2(this,HTML,denominator,list2,info);

	result=result.setNum(labelmap->count());
	HTMLSumHeader(HTML,"3",i18n("Total Labels"),result);
	HTML.append("</TABLE>\n");

	//---------------------------------
	HTML.append("\n<TD>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY GENRE"));
	HTMLColumnHeaders(HTML,list);

	it++;
	stat=Exec((*it).utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		delete list2;
		delete list3;
		return CATEGORY_SUM_FAIL;
	}
	HTMLPopulateData2(this,HTML,denominator,list2,info);

	result=result.setNum(genremap->count());
	HTMLSumHeader(HTML,"3",i18n("Total Genres"),result);
	HTML.append("</TABLE>\n");
	HTML.append("</TR></TABLE>\n");

	//---------------------------------
	HTML.append("\n<TABLE WIDTH=\"100%\"><TR><TD>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY TYPE"));
	HTMLColumnHeaders(HTML,list);

	it++;
	stat=Exec((*it).utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		delete list2;
		delete list3;
		return CATEGORY_SUM_FAIL;
	}
	HTMLPopulateData2(this,HTML,denominator,list2,info);

	result=result.setNum(typemap->count());
	HTMLSumHeader(HTML,"3",i18n("Total Types"),result);
	HTML.append("</TABLE>\n");

	//---------------------------------
	HTML.append("\n<TD>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY METHOD"));
	HTMLColumnHeaders(HTML,list);

	it++;
	stat=Exec((*it).utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		delete list2;
		delete list3;
		return CATEGORY_SUM_FAIL;
	}
	HTMLPopulateData2(this,HTML,denominator,list2,info);
	delete list2;
	delete list3;

	result=result.setNum(methodmap->count());
	HTMLSumHeader(HTML,"3",i18n("Total Methods"),result);
	HTML.append("</TABLE>\n");

	//---------------------------------
	HTML.append("\n<TD>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("DISTRIBUTION BY DECADE"));
	list->clear();
	list->append(i18n("Decade"));
	list->append(i18n("Total"));
	list->append(i18n("Percent"));
	HTMLColumnHeaders(HTML,list);
	delete list;

	int j=1950;
	for (int i=0;i<6;i++)
	{
		HTML.append("\t\t<TR><TD CLASS=artist>");
		sqlstring=sqlstring.setNum(j + (i*10));
		HTML.append(sqlstring);
		HTML.append("s</TD><TD CLASS=title>");
		HTML.append(decadeTotal[i]);
		HTML.append("</TD><TD CLASS=title>");
		HTML.append(decadePercent[i]);
		HTML.append("</TD></TR>\n");
	}

	HTML.append("\n</TABLE>\n</TR></TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 * @param QString Value
 * @param report type (artist, title, track)
 */
int dbMusic::nameSearchReport(QString &HTML, const reportInfo *const info, const colorInfo *const color, const QString &sstring, const int &searchtype)
{
	if (sstring.isEmpty())
		return NO_TUPLES;
	QString tnod;

	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Name Search Report"),i18n("namesearch"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);

	switch(searchtype)
	{
		case ARTIST:
			tnod=i18n("Artists ");
			break;
		case TITLE:
			tnod=i18n("Titles ");
			break;
		default:
			tnod=i18n("Tracks ");
	};
	tnod.append(i18n(" Containing "));
	tnod.append(sstring);
	HTMLTableHeader(HTML,"4",tnod,info);

	// switch to build query for db (artist, title or track)
	// short1 is for length, short2 is for the actual query
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	switch(searchtype)
	{
		case ARTIST:
			sqlstring="SELECT t.name, t.first, t.last FROM artist t WHERE name ~* '";
			sqlstring.append(validateQString(sstring));
			sqlstring.append("'");
			queryBoolStatus(sqlstring,"t.follow",info);
			queryRange(sqlstring,"t.last","t.first",info);
			// now add sorting for short2 and Exec
			if (info->sort_time)
			{
				list->append("first");
				list->append("name");
			}
			else
			{
				list->append("name");
				list->append("first");
			}
			queryOrderBy(sqlstring,list,info);
			break;
		case TITLE:
			sqlstring="SELECT a.name, t.name, t.yearstart, t.notracks FROM artist a, title t WHERE a.aid=t.aid AND t.name ~* '";
			sqlstring.append(validateQString(sstring));
			sqlstring.append("'");
			queryBoolStatus(sqlstring,"t.collected",info);
			queryRange(sqlstring,"t.yearstart","t.yearstart",info);
			// now add sorting for short2 and Exec
			list->clear();
			if (info->sort_time)
				list->append("t.yearstart");
			else
				list->append("t.name");
			list->append("a.name");
			queryOrderBy(sqlstring,list,info);
			break;
		default:
			sqlstring="SELECT a.name, t.name, tr.name, tr.length FROM artist a, title t, track tr WHERE a.aid=t.aid and t.cdid=tr.cdid and tr.name ~* '";
			sqlstring.append(validateQString(sstring));
			sqlstring.append("'");
			queryBoolStatus(sqlstring,"t.collected",info);
			queryRange(sqlstring,"t.yearstart","t.yearstart",info);
			// now add sorting for short2 and Exec
			list->clear();
			list->append("tr.name");
			list->append("t.name");
			list->append("a.name");
			queryOrderBy(sqlstring,list,info);
	};
	//exec and get values
	cerr << "namesearch: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		return TITLE_GET_FAIL;
	}
	tnod.setNum(Tuples());

	list->clear();
	switch(searchtype)
	{
		case ARTIST:
			list->append(i18n("Artist Name"));
			list->append(i18n("First Release"));
			list->append(i18n("Last Release"));
			HTMLColumnHeaders(HTML,list);

			list->clear();
			list->append("artist");
			list->append("title");
			list->append("title");
			HTMLPopulateData(this,HTML,list,0);
			break;
		case TITLE:
			list->append(i18n("Artist Name"));
			list->append(i18n("Title Name"));
			list->append(i18n("Release"));
			list->append(i18n("No. Of Tracks"));
			HTMLColumnHeaders(HTML,list);

			list->clear();
			list->append("artist");
			list->append("title");
			list->append("title");
			list->append("title");
			HTMLPopulateData(this,HTML,list,0);
			break;
		default:
			list->append(i18n("Artist Name"));
			list->append(i18n("Title Name"));
			list->append(i18n("Track Name"));
			list->append(i18n("Track Length"));
			HTMLColumnHeaders(HTML,list);

			list->clear();
			list->append("artist");
			list->append("title");
			list->append("track");
			list->append("track");
			HTMLPopulateData(this,HTML,list,0);
	};
	delete list;
	HTMLSumHeader(HTML,"4",i18n("Total"),tnod);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 * @param int Report Type (Type, Label, Genre, etc)
 * @param QString Value
 */
int dbMusic::categorySearchReport(QString &HTML, const reportInfo *const info, const colorInfo *const color, const int &XID, const QString &VALUE)
{
	if (VALUE.isEmpty())
		return CATEGORY_GET_FAIL;
	QString tnod, tlen;
	QString short1, short2;
	short1="SELECT SUM(t.cdlen) FROM title t";
	short2="SELECT a.name, t.name, t.yearstart FROM artist a, title t";

	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Category Search Report"),i18n("categorysearch"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	switch(XID)
	{
		case KFORMAT:
			tnod=i18n("Category Search for Format: ");
			break;
		case KGENRE:
			tnod=i18n("Category Search for Genre: ");
			break;
		case KLABEL:
			tnod=i18n("Category Search for Label: ");
			break;
		case KMETHOD:
			tnod=i18n("Category Search for Method: ");
			break;
		case KTYPE:
			tnod=i18n("Category Search for Type: ");
			break;
		default:
			return NO_TUPLES;
	};
	tnod.append(VALUE);
	HTMLTableHeader(HTML,"7",tnod,info);

	sqlstring=" WHERE ";
	switch(XID)
	{
		case KFORMAT:
			sqlstring.append("t.fid=");
			sqlstring.append(formatmap->find(VALUE).data());
			break;
		case KGENRE:
			sqlstring.append("t.gid=");
			sqlstring.append(genremap->find(VALUE).data());
			break;
		case KLABEL:
			sqlstring.append("t.lid=");
			sqlstring.append(labelmap->find(VALUE).data());
			break;
		case KMETHOD:
			sqlstring.append("t.mid=");
			sqlstring.append(methodmap->find(VALUE).data());
			break;
		case KTYPE:
			sqlstring.append("t.tid=");
			sqlstring.append(typemap->find(VALUE).data());
			break;
		case KBOXSET:
		case KBTE:
		default:
			return NO_TUPLES;
	};
	queryBoolStatus(sqlstring,"t.collected",info);
	queryRange(sqlstring,"t.yearstart","t.yearstart",info);

	short1.append(sqlstring);
	cerr << "titlesearch: " << short1 << endl;
	stat=Exec(short1.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	tlen=GetValue(0,0);
	if (tlen.isEmpty())
		tlen="00:00:00";

	// add sort order to short2
	queryMatch(sqlstring,"t.aid","a.aid");
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append("a.name");
	if (info->sort_time)
		list->append("t.yearstart");
	else
		list->append("t.name");
	queryOrderBy(sqlstring,list,info);
	short2.append(sqlstring);
	cerr << "titlesearch: " << short2 << endl;
	stat=Exec(short2.utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		return TITLE_GET_FAIL;
	}
	int searchresult=Tuples();

	list->clear();
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("No"));
	list->append("<pre>  </pre>");
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("No"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	list->append("title");
	list->append("title");
	HTMLPopulateData(this,HTML,list,1);
	delete list;

	HTMLSumHeader(HTML,"7",i18n("Total Titles"),tnod.setNum(searchresult));
	HTMLSumHeader(HTML,"7",i18n("Total Length"),tlen);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 * @param QString AID
 */
int dbMusic::titlesByArtist(QString &HTML, const reportInfo *const info, const colorInfo *const color, const QString &aid)
{
	if (aid.isEmpty())
		return ARTIST_GET_FAIL;
	QString searchresult, follow;

	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Titles By Artist Report"),i18n("titlesbyartist"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"1",i18n("Titles By Artist Report"),info);

	sqlstring="SELECT name,first,last,follow,homepage,notes FROM artist WHERE aid=";
	sqlstring.append(artist->find(aid).data());

	cerr << "titlebyArtist: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return ARTIST_GET_FAIL;

	HTML.append("</TABLE>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"4",i18n("Artist Information"));
	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append(i18n("Name"));
	list->append(i18n("First"));
	list->append(i18n("Last"));
	list->append(i18n("Follow?"));
	HTMLColumnHeaders(HTML,list);

	HTML.append("<TR>");
	for (int i=0;i<3;i++)
	{
		HTML.append("<TD CLASS=artist>");
		HTML.append(GetValue(0,i));
	}
	HTML.append("<TD CLASS=artist>");
	follow=GetValue(0,3);
	if (follow=="t")
		HTML.append(i18n("Yes"));
	else
		HTML.append(i18n("No"));
	HTML.append("</TR>\n");

	HTMLSimpleTableHeader(HTML,"4",i18n("Homepage"));

	HTML.append("<TR><TD CLASS=artist COLSPAN=4>");
	HTML.append(GetValue(0,4));
	HTML.append("</TR>\n");

	HTMLSimpleTableHeader(HTML,"4",i18n("Notes"));
	HTML.append("<TR><TD CLASS=artist COLSPAN=4>");
	HTML.append(GetValue(0,5));
	HTML.append("</TR>\n</TABLE>\n");

	kapp->processEvents();

	// create query for titles
	sqlstring="SELECT t.name, t.yearstart, t.yearend, t.nod, t.notracks, t.cdlen FROM title t WHERE t.aid=";
	sqlstring.append(artist->find(aid).data());
	queryRange(sqlstring,"t.yearstart","t.yearend",info);
	queryBoolStatus(sqlstring,"t.collected",info);
	list->clear();
	if (info->sort_time)
		list->append("t.yearstart");
	else
		list->append("t.name");
	queryOrderBy(sqlstring,list,info);

	cerr << "titlesbyartist: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		return TITLE_GET_FAIL;
	}
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"6",i18n("Title Information"));
	list->clear();
	list->append(i18n("Name"));
	list->append(i18n("Year Start"));
	list->append(i18n("Year End"));
	list->append(i18n("No. Discs"));
	list->append(i18n("No. Tracks"));
	list->append(i18n("Length"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	for (int i=0;i<6;i++)
		list->append("title");
	HTMLPopulateData(this,HTML,list,0);
	delete list;

	// assemble query for counts
	sqlstring="SELECT COUNT(cdid) FROM title WHERE aid=";
	sqlstring.append(artist->find(aid).data());
	queryRange(sqlstring,"yearstart","yearend",info);
	queryBoolStatus(sqlstring,"collected",info);

	cerr << "titlesbyartist: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	searchresult=GetValue(0,0);

	HTMLSumHeader(HTML,"6",i18n("Total Titles"),searchresult);

	// assemble query for total time
	sqlstring="SELECT SUM(cdlen) FROM title WHERE aid=";
	sqlstring.append(artist->find(aid).data());
	queryRange(sqlstring,"yearstart","yearend",info);
	queryBoolStatus(sqlstring,"collected",info);

	cerr << "titlesbyartist: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if ((a=execCheck())<0)
		return a;
	searchresult=GetValue(0,0);
	if (searchresult=="")
		searchresult="00:00:00";

	HTMLSumHeader(HTML,"6",i18n("Total Length"),searchresult);
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 * @param QString CDID
 */
int dbMusic::tracksByTitle(QString &HTML, const reportInfo *const info, const colorInfo *const color, const QString &cdid)
{
	if (cdid.isEmpty())
		return TITLE_GET_FAIL;
	QString boxset, temp, collected;
	int z;

	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Tracks By Title Report"),i18n("tracksbytitle"),".20",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"6",i18n("Tracks By Title"),info);

	sqlstring="SELECT grpid FROM title WHERE cdid=";
	sqlstring.append(validateNumQString(cdid));
	cerr << "tracksbytitle: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	temp=GetValue(0,0);
	if (temp!="0")
	{
		sqlstring="SELECT b.name FROM boxset b, title t WHERE t.grpid=b.grpid AND t.cdid=";
		sqlstring.append(cdid);
		cerr << "tracksbytitle: " << sqlstring << endl;
		stat=Exec(sqlstring.utf8().data());
		if (execCheck() < 0)
			return TITLE_GET_FAIL;
		boxset=GetValue(0,0);
	}
	else
		boxset="NONE";

	sqlstring="SELECT a.name,t.name,t.yearstart,t.yearend,t.nod,t.notracks,t.cdlen,f.name,g.name,m.name,l.name,type.name,t.collected,t.rating,t.quality,t.notes FROM artist a, title t, format f, genre g, method m, label l, type type WHERE a.aid=t.aid AND f.fid=t.fid AND g.gid=t.gid AND m.mid=t.mid AND l.lid=t.lid AND type.tid=t.tid AND t.cdid=";
	sqlstring.append(validateNumQString(cdid));
	cerr << "tracksbytitle: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;

	HTML.append("</TABLE>\n");
	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"6",i18n("Artist Name"));
	HTML.append("<TR><TD COLSPAN=6 CLASS=artist>");
	HTML.append(GetValue(0,0));
	HTML.append("</TR>\n");
	HTMLSimpleTableHeader(HTML,"6",i18n("Title Information"));

	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append(i18n("Title Name"));
	list->append(i18n("Year Start"));
	list->append(i18n("Year End"));
	list->append(i18n("No. Media"));
	list->append(i18n("No. Tracks"));
	list->append(i18n("Length"));
	HTMLColumnHeaders(HTML,list);

	HTML.append("<TR>");
	for (int i=1;i<7;i++)
	{
		HTML.append("<TD CLASS=title>");
		HTML.append(GetValue(0,i));
	}
	HTML.append("</TR>\n");

	list->clear();
	list->append(i18n("Format"));
	list->append(i18n("Genre"));
	list->append(i18n("Label"));
	list->append(i18n("Method"));
	list->append(i18n("Type"));
	list->append(i18n("Collected"));
	HTMLColumnHeaders(HTML,list);

	HTML.append("<TR>");
	for (int i=7;i<12;i++)
	{
		HTML.append("<TD CLASS=title>");
		HTML.append(GetValue(0,i));
	}
	HTML.append("<TD CLASS=title>");
	collected=GetValue(0,12);
	if (collected=="t")
		HTML.append(i18n("Yes"));
	else
		HTML.append(i18n("No"));
	HTML.append("</TR>\n");

	list->clear();
	list->append(i18n("Rating"));
	list->append(i18n("Quality"));
	list->append(i18n("Belongs To Boxset"));
	HTMLColumnHeaders(HTML,list);

	HTML.append("<TR>");
	for (int i=13;i<15;i++)
	{
		HTML.append("<TD CLASS=title>");
		HTML.append(GetValue(0,i));
	}
	HTML.append("<TD COLSPAN=4 CLASS=title>");
	HTML.append(boxset);
	HTMLSimpleTableHeader(HTML,"6",i18n("Notes"));
	HTML.append("<TR><TD CLASS=title COLSPAN=6>");
	HTML.append(GetValue(0,16));
	HTML.append("</TR>\n</TABLE>\n");

	kapp->processEvents();

	sqlstring="SELECT name,length FROM track WHERE cdid=";
	sqlstring.append(cdid);
	list->clear();
	if (info->sort_time)
		list->append("length");
	else
		list->append("trackid");
	queryOrderBy(sqlstring,list,info);
	cerr << "tracksbytitle: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
	{
		delete list;
		return TRACK_GET_FAIL;
	}
	z=Tuples();

	HTMLTable(HTML);
	HTMLSimpleTableHeader(HTML,"3",i18n("Track Information"));

	list->clear();
	list->append(i18n("No."));
	list->append(i18n("Name"));
	list->append(i18n("Length"));
	HTMLColumnHeaders(HTML,list);
	delete list;

	for (int i=0;i<z;i++)
	{
		temp.setNum(i+1);
		HTML.append("\t<TR><TD WIDTH=\"5%\" CLASS=track>");
		HTML.append(temp);
		HTML.append("<TD WIDTH=\"85%\" CLASS=track>");
		HTML.append(GetValue(i,0));
		HTML.append("<TD WIDTH=\"10%\" CLASS=track>");
		HTML.append(GetValue(i,1));
		HTML.append("</TR>\n");
		if (i%75==0)
			kapp->processEvents();
	}
	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 */
int dbMusic::topCDsReport(QString &HTML, const reportInfo *const info, const colorInfo *const color)
{
	QString tavg, maxnum="15";
	float ttup;
	// zero string and then select the total number of discs and the total length based on
	// report type
	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Top Titles Report"),i18n("toptitle"),".10",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,"3",i18n("Top Title Report"),info);

	sqlstring="SELECT a.name, t.name, t.rating FROM artist a, (SELECT aid, name, rating, collected FROM title";
	queryRange(sqlstring,"yearstart","yearstart",info);
	queryBoolStatus(sqlstring,"collected",info);
	sqlstring.append(" ORDER BY rating DESC LIMIT ");
	sqlstring.append(maxnum);
	sqlstring.append(") AS t WHERE a.aid=t.aid");
	QString sortorder;
	if (info->sort_order==ASC)
		sortorder="ASC";
	else
		sortorder="DESC";
	if (info->sort_time)
		sqlstring.append(" ORDER BY t.rating " + sortorder);
	else
		sqlstring.append(" ORDER BY a.name " + sortorder + ", t.name " + sortorder);

	// execute query
	cerr << "topCDsReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_GET_FAIL;
	// need this in case there are less than 15 titles in the db
	ttup=Tuples();

	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	list->append(i18n("Artist"));
	list->append(i18n("Title"));
	list->append(i18n("Rating"));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	list->append("artist");
	list->append("title");
	list->append("title");
	HTMLPopulateData(this,HTML,list,0);
	delete list;

	sqlstring="SELECT SUM(rating) FROM (SELECT rating FROM title";
	queryRange(sqlstring,"yearstart","yearstart",info);
	queryBoolStatus(sqlstring,"collected",info);
	sqlstring.append(" ORDER BY rating DESC LIMIT ");
	sqlstring.append(maxnum);
	sqlstring.append(") as t");
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return TITLE_SUM_FAIL;
	tavg=GetValue(0,0);
	tavg.setNum(tavg.toFloat()/ttup);
	HTMLSumHeader(HTML,"3",i18n("Average"),tavg);

	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}
/**
 * Generates an HTML report.
 * @param QString HTML String
 * @param reportInfo reportInfo struct containing report variables
 * @param colorInfo colorInfo struct containing color variables
 * @param QString Custom report string
 */
int dbMusic::customReport(QString &HTML, const reportInfo *const info, const colorInfo *const color, const QString &query)
{
	if (query.isEmpty())
		return DB_UNKNOWN_ERROR;
	int tup, fld;
	QString charfld, chartup;
	sqlstring=query;

	cerr << "customReport: " << sqlstring << endl;
	stat=Exec(sqlstring.utf8().data());
	if (execCheck() < 0)
		return DB_UNKNOWN_ERROR;
	tup=Tuples();
	fld=Fields();
	chartup.setNum(tup);
	charfld.setNum(fld);

	HTMLDocType(HTML,"EN");
	HTML.append("<HTML>");
	HTMLHeader(HTML,i18n("Custom Report"),i18n("custom"),".10",info,color);
	HTMLBody(HTML);
	HTMLTable(HTML);
	HTMLTableHeader(HTML,charfld,i18n("Custom Report"),info);

	QStringList *list=new QStringList();
	if (!list)
		return NO_MEMORY;
	for (int i=0;i<fld;i++)
		list->append(FieldName(i));
	HTMLColumnHeaders(HTML,list);

	list->clear();
	for (int i=0;i<fld;i++)
		list->append("title");
	HTMLPopulateData(this,HTML,list,0);
	delete list;

	HTMLSumHeader(HTML,charfld,i18n("Total"),chartup);

	HTML.append("</TABLE>");
	HTMLSysInfo(HTML,DBName());
	HTML.append("</BODY>\n</HTML>");

	return EVERYTHING_OK;
}

